/* Copyright 93 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* 
 * tclttMisc.c
 *
 *	Miscellaneous Tooltalk functions
 *
 * $Id: tclttMisc.c,v 1.10 93/08/17 16:51:15 beust Exp Locker: beust $
 */

#include <assert.h>
#include <stdio.h>
#include "main.h"
#include "database.h"

#define MAX_TCL_CALLBACKS 32

void TclTT_property_initialize();
void TclTT_property_add(void *prop, char *value);
char * TclTT_property_get(void *prop, int n);
int TclTT_property_count(void *prop);

struct _Entry {
  void *what;    /* a message or a pattern */
  int tclProcsCount;    /* number of callbacks */
  char *tclProcs[MAX_TCL_CALLBACKS];    /* the TCL procedures to call */
};

/* Global variable */
static DataBase Callbacks;

/****************************************************************************
 * The following functions don't belong to Tooltalk's interface
 ****************************************************************************/


int
TclTT_wait_for_message(ClientData data,Tcl_Interp *interp, int argc, char **argv)
/* This function is a simple block-until-message-comes function. It is */
/* available via tcl's command TT_wait_for_message */
{
  int fd = atoi(argv[1]);
  fd_set fs;
  Tt_message msg;

  FD_ZERO(& fs);
  FD_SET(fd, & fs);
  select(20, &fs, NULL, NULL, NULL);
}

static void
mainTooltalkCallback(ClientData clientData, int mask)
{
#ifdef TCL7
  Tcl_Eval(Interp, (char *) clientData);
#else
  Tcl_Eval(Interp, (char *) clientData, 0, NULL);
#endif /* TCL 7 */
}

int
TclTT_main_tooltalk_callback(ClientData data,Tcl_Interp *interp, int argc, char **argv)
/* This function registers the specified TCL procedure as the function to be */
/* called each time a Tooltalk message arrives */
{
  char *proc = (char *) strdup(argv[1]);
#ifdef TK_EXTENDED
  Tk_CreateFileHandler(tt_fd(), NULL, TK_READABLE,
		       mainTooltalkCallback, (ClientData) proc);
#else
  Tk_CreateFileHandler(tt_fd(), TK_READABLE,
  		       mainTooltalkCallback, (ClientData) proc);
#endif
}

Tt_callback_action
TclTT_generic_pattern_callback(Tt_message m, Tt_pattern p)
/* This is our generic Tooltalk pattern callback. Each time a callback is */
/* requested, this function is registered to Tooltalk. A property has been */
/* save prealably into the session so we know what the actual Tcl function */
/* is */
{
  int callbackCount, i;
  char *tclProc, tclCommand[256];
  Tcl_Interp *interp;
  void *prop;

/*
  interp = (Tcl_Interp *)
    atol(tt_session_prop(tt_default_session(), TCLTT_PROP_INTERP, 0));
*/
  interp = Interp;
  assert(interp);
  /* If p is NULL, then it's a message callback */
  if (p) prop = (void *) p;
  else prop = (void *) m;
  callbackCount = TclTT_property_count(prop);

  for (i = 0; i < callbackCount; i++) {
    tclProc = TclTT_property_get(prop, i);
    sprintf(tclCommand, "%s %u %u", tclProc, m, p);
#ifdef TCL7
    Tcl_Eval(interp, tclCommand);
#else
    Tcl_Eval(interp, tclCommand, 0, NULL);
#endif /* TCL7 */
  }
}

static void
freeFunction(struct _Entry *entry)
{
}

static void
addProperty(struct _Entry *entry, char *value)
{
  int n = entry -> tclProcsCount;
  if (n == MAX_TCL_CALLBACKS) {
    fprintf(stderr, "*** tclttMisc : too many callbacks '%d'\n", MAX_TCL_CALLBACKS);
    assert(0);
  }
  else {
    entry -> tclProcs[n] = (char *) strdup(value);
    entry -> tclProcsCount++;
  }
}

/*
 * The TclTT_property_* functions are an emulation of properties in order
 * to store message/patterns and their related TCL callbacks. Since
 * Tooltalk's callbacks don't have a clientData, this is the only way
 * I found to solve this problem. Using ttsession to store this information
 * wouldn't work neither since this information must die when an application
 * exits, or the callbacks will be called several times
 */

void
TclTT_property_initialize()
{
  Callbacks = DB_NewDataBase(sizeof(struct _Entry), freeFunction);
}

void
TclTT_property_add(void *prop, char *value)
{
  DataBase db = Callbacks;
  struct _Entry *entry;
  int found = 0;

/*
  D(printf(printf("adding property '%u=%s'\n", prop, value)));
*/
  DB_Rewind(db);
  while (! DB_EndOfDataBase(db) && ! found) {
    entry = DB_NextEntry(db);
    if (entry -> what == prop) {
      addProperty(entry, value);
      found = 1;
    }
  }
  if (! found) {
    struct _Entry ent;
    ent.what = prop;
    ent.tclProcsCount = 0;
    addProperty(& ent, value);
    DB_AddEntry(db, & ent);
  }
}

char *
TclTT_property_get(void *prop, int n)
{
  DataBase db = Callbacks;
  struct _Entry *entry;
  DB_Rewind(db);
  while (! DB_EndOfDataBase(db)) {
    entry = DB_NextEntry(db);
    if (entry -> what == prop) {
      return entry -> tclProcs[n];
    }
  }
  return NULL;
}

int
TclTT_property_count(void *prop)
{
  DataBase db = Callbacks;
  struct _Entry *entry;
  DB_Rewind(db);
  while (! DB_EndOfDataBase(db)) {
    entry = DB_NextEntry(db);
    if (entry -> what == prop) {
      return entry -> tclProcsCount;
    }
  }
  return 0;
}



/****************************************************************************
 * The following functions belong to Tooltalk's interface
 ****************************************************************************/

int
Tcltt_open(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(0);
  interp -> result = tt_open();
  if (! interp -> result) interp -> result = "";
  return 0;
}

int
Tcltt_X_session(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(1);
  interp -> result = tt_X_session(argv[1]);
  if (! interp -> result) interp -> result = "";
  return 0;
}

int
Tcltt_close(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(0);
  DB_ClearDataBase(Callbacks);
  Tk_DeleteFileHandler(tt_fd());
  result = tt_close();
  return result;
}

int
Tcltt_status_message(ClientData data,Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(1);
  interp -> result = tt_status_message(atoi(argv[1]));
  if (! interp -> result) interp -> result = "";
  return result;
}

int
Tcltt_fd(ClientData data,Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(0);
  sprintf(Tcltt_Result, "%u", tt_fd());
  interp -> result = Tcltt_Result;
  return result;
}

int
Tcltt_prequest_create(ClientData data, Tcl_Interp *interp, int argc,char**argv)
{
  int result = 0;
  ARG(2);
  sprintf(Tcltt_Result, "%u",
	  tt_prequest_create((Tt_scope) (atoi(argv[1])), argv[2]));
  interp -> result = Tcltt_Result;
  return result;
}

int
Tcltt_pnotice_create(ClientData data, Tcl_Interp *interp, int argc,char**argv)
{
  int result = 0;
  ARG(2);
  sprintf(Tcltt_Result, "%u",
	  tt_pnotice_create((Tt_scope) (atoi(argv[1])), argv[2]));
  interp -> result = Tcltt_Result;
  return result;
}

int
Tcltt_mark(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(0);
  sprintf(Tcltt_Result, "%u", tt_mark());
  interp -> result = Tcltt_Result;
  return result;
}

int
Tcltt_release(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(1);
  tt_release(atol(argv[1]));
  return result;
}

int
Tcltt_onotice_create(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(2);
  sprintf(Tcltt_Result, "%u", tt_onotice_create(argv[1], argv[2]));
  interp -> result = Tcltt_Result;
  return result;
}

int
Tcltt_orequest_create(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(2);
  sprintf(Tcltt_Result, "%u", tt_orequest_create(argv[1], argv[2]));
  interp -> result = Tcltt_Result;
  return result;
}

int
Tcltt_objid_equal(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(2);
  sprintf(Tcltt_Result, "%u", tt_objid_equal(argv[1], argv[2]));
  interp -> result = Tcltt_Result;
  return result;
}

int
Tcltt_objid_objkey(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(1);
  sprintf(Tcltt_Result, "%u", tt_objid_objkey(argv[1]));
  interp -> result = Tcltt_Result;
  return result;
}

int
Tcltt_ptype_declare(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(1);
  result = tt_ptype_declare(argv[1]);
  return result;
}

int
Tcltt_pointer_error(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(1);
  result = tt_pointer_error((void *) atol(argv[1]));
  return result;
}

int
Tcltt_int_error(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(1);
  result = tt_int_error(atol(argv[1]));
  return result;
}

int
Tcltt_error_int(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(1);
  sprintf(Tcltt_Result, "%u", tt_error_int((Tt_status) atol(argv[1])));
  interp -> result = Tcltt_Result;
  return result;
}

int
Tcltt_error_pointer(ClientData data, Tcl_Interp *interp, int argc, char **argv)
{
  int result = 0;
  ARG(1);
  sprintf(Tcltt_Result, "%u", tt_error_pointer((Tt_status) atol(argv[1])));
  interp -> result = Tcltt_Result;
  return result;
}

