 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File:	transient.tcl

# Description:
#	Procedures for creating simple transient top-level windows.
#
#	Creating a transient window requires a number of actions to
#	inform the window manager of the window's properties.  These actions
#	have proven to be less than stable; this file provides a single place
#	to control the window manager hints for transient windows.
#
#	At present, the hints are as follows.
#
#	- The transient window is grouped with its parent
#	- The transient window is marked as being transient for its parent.
#	- The transient window is marked as being positioned by the user,
#	  and its geometry is set to center it relative to its parent.
#	- All of the above operations are protected by having the window
#	  withdrawn before applying them, and deiconified afterward.

 # $Id: transient.tcl,v 1.13 1993/11/01 18:20:46 kennykb Exp $
 # $Source: /homedisk/julius/u0/kennykb/src/tkauxlib_ship/RCS/transient.tcl,v $
 # $Log: transient.tcl,v $
 # Revision 1.13  1993/11/01  18:20:46  kennykb
 # Beta release to be announced on comp.lang.tcl
 #
 # Revision 1.12  1993/10/27  15:52:49  kennykb
 # Package for alpha release to the Net, and for MMACE 931101 release.
 #
 # Revision 1.11  1993/10/25  16:15:12  kennykb
 # Changed to make sure that the transient is on display before returning,
 # so that `focus_goToFirst' and friends will work on it.
 #
 # Revision 1.10  1993/10/21  20:43:42  kennykb
 # removed erroneous `catch' that caused `destroy' action to fail
 # sometimes.
 #
 # Revision 1.9  1993/10/20  19:10:47  kennykb
 # Alpha release #1 was thawed for bug fixes in tk 3.3.  Now frozen again at this
 # point.
 #
 # Revision 1.8  1993/10/20  19:05:11  kennykb
 # Changed copyright notice to not resemble structured comments.
 #
 # Changed so that the program waits for the visibility of transients before
 # returning to the user.
 #
 # Revision 1.7  1993/10/14  18:15:42  kennykb
 # Cleaned up alignment of log messages, to avoid problems extracting
 # structured commentary.
 #
 # Revision 1.6  1993/10/14  18:06:59  kennykb
 # Added GE legal notice to head of file in preparation for release.
 #
 # Revision 1.5  1993/10/14  14:02:02  kennykb
 # Alpha release #1 frozen at this point.
 #
 # Revision 1.4  1993/10/14  13:48:48  kennykb
 # (A) Added a FocusIn binding that had been omitted.
 # (B) Added `wm withdraw' before configuring the window, in case the window
 #     manager can't configure a top-level window.
 #
 # Revision 1.3  1993/07/21  19:44:36  kennykb
 # Finished cleaning up structured commentary.
 #
 # Revision 1.2  1993/07/16  15:58:00  kennykb
 # Renamed all commands that start with `wiget.' to either `widget_' or
 # `widget:'.  Added the code for creating composite widgets.
 #
 # Revision 1.1  1993/06/03  15:33:24  kennykb
 # Initial revision
 #

# Procedure:	transient
#
# Synopsis:
#	Create a transient top-level window, and pack a widget inside it.
#
# Usage:
#c	transient widget pathName args
#
# Parameters:
#c	widget
#		Name of a command that creates a widget, e.g., `label',
#		`fileselect'.
#c	pathName
#		Path name of a toplevel window to be created.
#c	args
#		Arguments to the widget creation command.
#
# Return value:
#	Name of the newly-created transient window.
#
# Description:
#	The `transient' procedure makes a top-level window to hold a given
#	widget.  It then creates the widget, packs it in the window, and
#	makes the window transient, centering it relative to its parent's
#	top-level window.
#
#	The widget is always named `$pathName.b'.  The class name of the
#	transient window is `Transient.'

	# Default look and feel:
option add *Transient.borderWidth 2 widgetDefault
option add *Transient.relief raised widgetDefault

bind Transient <FocusIn> { toplevel_focusIn %W %d }

proc transient {widget w args} {
	toplevel $w -class Transient
	wm withdraw $w
	uplevel 1 [list $widget $w.b] $args
	pack append $w $w.b {expand padx 4 pady 4}
	$w config -background [lindex [$w.b config -background] 4]
	transient_make $w
	widget_addBinding $w.b Destroy "
		after 1 widget_checkAndDestroy $w
	"
	return $w
}

# Procedure: transient_make
#
# Synopsis:
#	Apply the `transient window' hints to a top-level window.
#
# Usage:
#c	transient_make pathName
#
# Parameters:
#c	pathName
#		Path name of a top-level window.
#
# Return value:
#	The path name is returned.
#
# Description:
#	The transient_make procedure arranges for a window to be made
#	transient.  It is marked as `transient', owned by the top-level
#	window of its parent.  It is centered relative to the top-level
#	window of its parent.  It is withdrawn and repainted so that
#	it appears in the right place on the screen with the correct
#	window manager decorations (often, undecorated).

proc transient_make w {
	wm withdraw $w
	update idletasks
	set p [winfo parent $w]
	set t [winfo toplevel $p]
	set cx [expr [winfo rootx $t]+[winfo width $t]/2]
	set cy [expr [winfo rooty $t]+[winfo height $t]/2]
	set x [expr $cx-[winfo reqwidth $w]/2]
	set y [expr $cy-[winfo reqheight $w]/2]
	if {$x < 0} { set x 0 }
	if {$y < 0} { set y 0 }
	wm group $w $t
	wm transient $w $t
	wm positionfrom $w user
	wm geometry $w +$x+$y
	wm deiconify $w
	tkwait visibility $w
	update
}

