#---------------------------------------------------------------------------
#
#	File:	the-methods.tcl
#	Title:	Objects and Methods - Basic Support
#	Author:	Juergen Wagner (J_Wagner@iao.fhg.de)
#	Date:	93-01-19
#
#---------------------------------------------------------------------------
#
# Functions:
#
# defobject {name {super} {slots}}
#	Define a new object called "name" with super-object listed in
#	"super". The object will have an attribute value list associated
#	which is initialized to "slots". An object can be called by its
#	name just like any procedure.
#
# defsuper {name super}
#	Define additional super-objects.
#
# defmethod {name {method} {arguments} {body}}
#	Define a subcommand (method) for object "name". All objects including
#	this as a super-object may also refer to this method, unless it is
#	shadowed by another method of the same name that is found earlier
#	in the window hierarchy. Within the body of the method, the
#	variable "self" denotes the object being called. The special method
#	"DEFAULT" defines a default handler for messages. If "name" is a
#	list, the first element specifies the object name, the remaining
#	elements will be used as super-objects if the object has to be created
#	newly. If the object "name" already exists, only a method is defined.
#	The second and following elements of "name" will be ignored in that
#	case.
#
# is {object super}
#	Test whether "object" has a super-object "super".
#
# vanilla-object
#	The root of the object hierarchy. This object supports the messages
#	"help", "slot NAME", "slot NAME NEWVALUE", "clone NAME ?NEWSLOTS?",
#	"super", and "reclaim".
#
#---------------------------------------------------------------------------
#
#	Define a new object
#
if { ! $system(has_defobject) } {
  # load interpreted version
  load(lib) the-methods2
}

proc defsuper {name super} {
  global _o:${name}

  if { [lsearch [set _o:${name}()] $super] < 0 } {
    set _o:${name}() [concat [list $super] [set _o:${name}()]]
  }
  return $name
}

proc _method(slots) {name} {
  global _o:$name

  set slots {}
  catch {set slots [array names _o:$name]}
  return [lrange [lsort $slots] 1 end]
}

proc is {object super} {

  if { ${object} == $super } {
    return 1
  } elseif { [info globals _o:${object}] != "" } {
    global _o:$object
    foreach s [set _o:${object}()] {
      if { [is $s $super] } {
	return 1
      }
    }
  }
  return 0
}

#---------------------------------------------------------------------------

defobject vanilla-object {} {}

defmethod vanilla-object help {} {
  global system

  set system(_inspected) {}
  _method(inspect) $self
  set system(_inspected) {}
}

defmethod vanilla-object slots {} {

  _method(slots) $self
}

defmethod vanilla-object values {} {

  set values {}
  foreach slot [_method(slots) $self] {
    lappend values [list $slot [$self slot $slot]]
  }
  return $values
}

set system(line) \
"----------------------------------------------------------------------------"

proc _method(inspect) {object {indent "**"}} {
  global system _o:$object

  puts stdout "$indent $object"
  if { [position $object $system(_inspected)] >= 0 } {
    set len [expr 77-[string length $indent]]
    puts stdout "$indent +[string range $system(line) 0 $len]"
    return
  }
  lappend system(_inspected) $object

  set first [string length "_method($object,"]
  foreach command [info command _method($object,*)] {
    set last [expr [string length $command]-2]
    set arglist [lrange [info args $command] 1 end]
    set method [string range $command $first $last]
    puts stdout "$indent => $object $method {$arglist}"
  }
  set len [expr 76-[string length $indent]]
  puts stdout "$indent +[string range $system(line) 0 $len]"

  foreach s [set _o:${object}()] {
    _method(inspect) $s "$indent |"
  }
}

if { ! $system(has_defobject) } {

defmethod vanilla-object slot {slot args} {
  global _o:$self

  if { $args == {} } {
    set value ""
    catch {set value [set _o:${self}($slot)]}
  } {
    set value [lindex $args 0]
    set _o:${self}($slot) $value
  }
  return $value
}

}

defmethod vanilla-object clone {name {newslots {}}} {
  global _o:$name _o:$self

  foreach slot [_method(slots) $self] {
    set _o:${name}($slot) [set _o:${self}($slot)]
  }

  defobject $name $self $newslots
}

defmethod vanilla-object super {} {
  global _o:$self

  return [set _o:${self}()]
}

defmethod vanilla-object reclaim {} {
  global _o:$self

  # reclaim a single object
  unset _o:$self
  foreach proc [info command _method($self,*)] {
    rename $proc {}
  }
  rename $self {}
  catch {rename $self! {}}

  return
}

defmethod vanilla-object reclaimall {{spareme {}}} {

  # reclaim slot variables
  if { $spareme != "true" } {
    uplevel #0 [list unset _o:$self]
  }
  foreach var [info globals _o:$self.*] {
    uplevel #0 [list unset $var]
  }

  # reclaim methods
  if { $spareme != "true" } {
    foreach proc [info command _method($self,*)] {
      rename $proc {}
    }
  }
  foreach proc [info command _method($self.*)] {
    rename $proc {}
  }

  # reclaim object commands
  foreach proc [info command $self.*] {
    catch {rename $proc {}}
    catch {rename $proc! {}}
  }
  if { $spareme != "true" } {  
    rename $self {}
   catch {rename $self! {}}
  }

  return
}

defmethod vanilla-object slotvar {slot} {

  return _o:${self}($slot)
}

proc slot-variable {object {slot {}}} {

  if { $slot != {} } {
    return _o:${object}($slot)
  } {
    return _o:${object}
  }
}

defmethod vanilla-object slotappend {slot element} {
  global _o:$self

  lappend _o:${self}($slot) $element
}

defmethod vanilla-object unslot {slot} {
  global _o:$self

  unset _o:${self}($slot)
}

defmethod vanilla-object showAction {args} {
  puts stdout "Action ($self): $args"
  return
}
