/*
 * Copyright 1992 by the National Optical Astronomy Observatories(*)
 *
 * Permission to use, copy, and distribute
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * This software is provided "as is" without any express or implied warranty.
 *
 * (*) Operated by the Association of Universities for Research in
 *     Astronomy, Inc. (AURA) under cooperative agreement with the
 *     National Science Foundation.
 */
/* Program: tclipc.c
 *      This file contains the TCL interface commands for the added
 *      tclIPC functions.  All the script-level commands are 
 *      defined here.
 *
 * Created: K. Gillies 26 June 1992
 * SCCS INFO
 *      @(#)tclipc.c	1.1 9/11/92
 *
 */
#include <stdio.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>
#include <string.h>
#include "tcl.h"
#include "tclIPC.h"            /* RPC stuff */
#include "tclipcP.h"           /* Private structures */
#include "tclipc.h"            /* Public interface */
#include "ipcHandles.h"        /* Handle table interface */

int ipcDescriptorTableSize;    /* Used in dispatch.c */

/* Used for tcl commands */
#define STREQU(str1, str2) \
        ((str1[0] == str2[0]) && (strcmp (str1, str2) == 0))
#define STRNEQU(str1, str2, cnt) \
        ((str1[0] == str2[0]) && (strncmp (str1, str2, cnt) == 0))

/* The name of this host--used globally in the library */
static char thisHostName[MAXHOSTNAMELEN];
char *ipcThisHostName = thisHostName;

/* A time structure for timeouts, if used */
static struct itimerval atime;

/* In file prototypes */
static int cmdSender _ANSI_ARGS_((ClientData *clientData, Tcl_Interp *interp, 
 	int argc, char *argv[]));
static int cmdReceiver _ANSI_ARGS_((ClientData *clientData, 
				    Tcl_Interp *interp,
				    int argc, char *argv[]));
static int cmdSend _ANSI_ARGS_((ClientData *clientData, Tcl_Interp *interp, 
		int argc, char *argv[]));
static int cmdDump _ANSI_ARGS_((ClientData *clientData, Tcl_Interp *interp, 
		int argc, char *argv[]));
static int cmdPmDump _ANSI_ARGS_((ClientData *clientData, Tcl_Interp *interp, 
		int argc, char *argv[]));

/*
 *--------------------------------------------------------------
 *
 * ipcInterpInit
 *
 *      Takes a tcl interperter and adds the new tcl/ipc
 *      commands.
 *
 * Results:
 *      void
 *--------------------------------------------------------------
 */
void ipcInterpInit(interp)
Tcl_Interp *interp;
{
  /* Just make sure */
  assert (interp != NULL);

  Tcl_CreateCommand(interp, "sender", (Tcl_CmdProc *)cmdSender,
                    (ClientData) NULL,
	            (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "receiver", (Tcl_CmdProc *)cmdReceiver,
                    (ClientData) NULL,
	            (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "isend", (Tcl_CmdProc *)cmdSend,
                    (ClientData) NULL,
	            (Tcl_CmdDeleteProc *) NULL);
  /* Other commands */
  Tcl_CreateCommand(interp, "srdump", (Tcl_CmdProc *)cmdDump, 
                    (ClientData) NULL,
	            (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "pmdump", (Tcl_CmdProc *)cmdPmDump, 
                    (ClientData) NULL,
	            (Tcl_CmdDeleteProc *) NULL);
}

/*
 *--------------------------------------------------------------
 *
 * ipcSendInit
 *
 *      Initialize the message receiving code.  Set the time
 *      between message checks and alternate eval and listen 
 *      functions.
 *
 * Results:
 *      void
 *--------------------------------------------------------------
 */
void ipcSendInit(timeoutsecs, timeoutusecs, 
	    receivesecs, receiveusecs, 
	    evalfunc, listener, connecter, destroyer)
long timeoutsecs, timeoutusecs;
long receivesecs, receiveusecs;
evalfunction evalfunc;
listenfunction listener;
connectfunction connecter;
connectfunction destroyer;
{
  DispatchTable tclDTable;

  ipcHandleTableInit(5);         /* Number of commands */
  /* Set up ipc default response */
  tclDTable.cmdRecvPtr = evalfunc;
  /* Set up ipc default listener */
  tclDTable.listenFunctionPtr = listener;
  /* Set up tcp connection monitor */
  tclDTable.connectFunctionPtr = connecter;
  /* Set up tcp connection destroyer */
  tclDTable.destroyConnectFunctionPtr = destroyer;
  /* Enter them  */
  BuildDispatchTable(&tclDTable);
  
  /* Set the select timeout values */
  setReceiveTimeout(receivesecs, receiveusecs);

  /* To not use the timer BOTH times must be 0 */
  if (timeoutsecs != 0 || timeoutusecs != 0) {
    atime.it_interval.tv_sec = timeoutsecs;
    atime.it_interval.tv_usec = timeoutusecs;
    atime.it_value.tv_sec = timeoutsecs;
    atime.it_value.tv_usec = timeoutusecs;
    signal(SIGALRM, receiverAlarmTimeout);
    setitimer(ITIMER_REAL, &atime, (struct itimerval *)NULL);
  }
  /* Get size of descriptor table for later use by select() */
  ipcDescriptorTableSize = getdtablesize(); 

  /* Get the name of the local host for possible PortMgr location */
  if (gethostname(&thisHostName[0], MAXHOSTNAMELEN) != 0) {
    fprintf(stderr, "Could not get the local host name.  Exiting...\n");
  }
}

/*
 *--------------------------------------------------------------
 *
 * ipcNewSender
 *
 *      Programmer callable function to create a new sender.
 *
 * Results:
 *      Returns a new sender or NULL if it fails.
 *--------------------------------------------------------------
 */
Sender ipcNewSender(interp, program, hostname, portmanhost)
Tcl_Interp *interp; 
char *program;
char *hostname;
char *portmanhost;
{
  Port myPort;                         /* Used for new connection */
  Sender newSender = NULL;
  Port mgrPort;                        /* Used for connecton to a port mgr */
  Sender portMgrSender;
  int result;
  PortMgrResult pmresult;
  
  /* Check for valid args */
  if (hostname == NULL || program == NULL || portmanhost == NULL) {
    Tcl_AppendResult(interp, 
	       "Could not create a new sender because of NULL arguments.", 
		     (char *) NULL);
    return (Sender)NULL;
  }

  /* First contact the portmanager */
  mgrPort.hostName = portmanhost;
  mgrPort.portNumber = PortMgrPortNumber;
 
  /* Create a new portmanager sender */
  portMgrSender = NewSender(interp, &mgrPort);
  if (portMgrSender == (Sender)NULL) {
    Tcl_AppendResult(interp, "Could not connect to PortManager on host: ",
		     portmanhost, ".", (char *) NULL);
    return (Sender)NULL;
  }

  /* It was successful, now get info for the target of the send */
  myPort.hostName = hostname;
  myPort.appName = program;
  myPort.portNumber = 0;
  result = pmGetPortFromName(portMgrSender, &myPort, &pmresult);
  if (result != TCL_OK) {
    Tcl_AppendResult(interp, 
		     "The PortManager could not find your program: ", 
		     program, ".", (char *) NULL);
    /* Close the port manager connection if portfromname failed */
    DestroySender(portMgrSender);
    pmDestroyResult(pmresult);
    return (Sender)NULL;
  }
  /* Close the port manager connection, no longer need it. */
  DestroySender(portMgrSender);

  /* Pick the first program as a default.  A more clever selection
   * method could be used here. Load balancing?
   */
  result = pmGetPortFromResult(interp, 0, 
			pmresult, &myPort);
  if (result != TCL_OK) {
    Tcl_AppendResult(interp, 
		     "The PortManager could not find your program: ", 
		     program, ".", (char *) NULL);
    pmDestroyResult(pmresult);
    return (Sender)NULL;
  }

  /* Get rid of allocated apps space */
  pmDestroyResult(pmresult);

  /* Now open a sender for the new app/port/host */
  newSender = NewSender(interp, &myPort); 
  if (newSender == (Sender)NULL) {
    Tcl_AppendResult(interp, "Could not connect to ", program, 
		     " on host: ", portmanhost, ".", (char *) NULL);
    return newSender;
  }

  /* We are all set so now set up the table entry */
  /* For senders, the hostname is the location of destination */
  Tcl_SetResult(interp,
		ipcHandleAddEntry(newSender, NULL,
				  myPort.hostName, program), TCL_STATIC);

  /* Free that port hostname that was strdup'd */
  free(myPort.hostName);
  /* We made it! */
  return newSender;
}

/*
 *--------------------------------------------------------------
 *
 * ipcDestroySenderFromHandle
 *
 *      Programmer callable function to destroy a new sender.
 *
 * Results:
 *      Returns TCL_OK or TCL_ERROR
 *--------------------------------------------------------------
 */
int ipcDestroySenderFromHandle(interp, handleName)
Tcl_Interp *interp;
char *handleName;
{
  sendTableEntry entry;

  if ((entry = ipcHandleGet(interp, handleName)) == NULL) {
    Tcl_AppendResult(interp, ".  Could not free sender.", (char *) NULL);
    return TCL_ERROR;
  }

  /* If it's not a sender, return an error */
  if (entry->s_sender == NULL) {
    Tcl_AppendResult(interp, 
	       handleName, " is not a sender.", (char *)NULL);
    return TCL_ERROR;
  }
  /* Only need to destroy sender storage, close connection and 
   * remove the handle from the table.
   */
  return ipcDestroySender(interp, entry->s_sender);
}

/*
 *--------------------------------------------------------------
 *
 * ipcDestroySender
 *
 *      Programmer callable function to destroy a new sender.
 *
 * Results:
 *      Returns TCL_OK or TCL_ERROR
 *--------------------------------------------------------------
 */
int ipcDestroySender(interp, sender)
Tcl_Interp *interp;
Sender sender;
{
  char tmphandle[IPCHLENGTH];

  /* Check for valid args */
  if (sender == NULL) {
    Tcl_AppendResult(interp, 
	       "Could not destroy a sender because of NULL arguments.", 
		     (char *) NULL);
    return TCL_ERROR;
  }
  /* Store the handle name for later destruction */
  strcpy(tmphandle, sender->handleName);

  /* Only need to destroy sender storage, close connection and 
   * remove the handle from the table.
   */
  DestroySender(sender);
  ipcHandleDestroy(interp, sender->handleName);

  return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * cmdSender
 *
 *      TCL command for creating new senders.  See docs for
 *      arguments.
 *
 * Results:
 *      TCL_OK or TCL_ERROR and result is problem or new sender
 *      handle.
 *--------------------------------------------------------------
 */
static int cmdSender(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  char *program = NULL;
  char *hostname = NULL;
  char *portmanhost = NULL;
  char *subcommand;
  
  if (argc < 3 || argc > 5) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
      argv[0], " new label | destroy handle\"", 
      (char *) NULL);
    return TCL_ERROR;
  }

  /* What is needed? */
  subcommand = argv[1];
  if (STREQU(subcommand, "new")) {
    if (argc == 3) {
      /* Use localhost, argv[2] is the sender label */
      program = argv[2];
      hostname = ANYHOST;
      portmanhost = ipcThisHostName;
    } else
    if (argc == 4) {
      /* They have given a host for the program too as argv[3] */
      program = argv[2];
      hostname = argv[3];
      portmanhost = ipcThisHostName;
    } else
    if (argc == 5) {
      /* They have given a PortManager machine too as argv[4] */
      program = argv[2];
      hostname = argv[3];
      portmanhost = argv[4];
    }
    return ((ipcNewSender(interp, program, hostname, portmanhost) == NULL) ?
            TCL_ERROR : TCL_OK);
  }
  if (STREQU(subcommand, "destroy")) {
    if (argc == 3) {
      /* Use localhost, argv[2] is the sender label */
      program = argv[2];
      return ipcDestroySenderFromHandle(interp, program);
    } else {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], "destroy sendername.", 
		       (char *) NULL);
      return TCL_ERROR;
    }
  }
  Tcl_AppendResult(interp, "wrong command args: should be \"",
      argv[0], " new label | destroy handle\"", 
      (char *) NULL);

  return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ipcNewReceiver
 *
 *      Programmer interface for creating a new Receiver.
 *
 * Results:
 *      Returns a new receiver or NULL if it fails.
 *--------------------------------------------------------------
 */
Receiver ipcNewReceiver(interp, label, portManHost)
Tcl_Interp *interp; 
char *label;
char *portManHost;
{
  Receiver newReceiver = NULL;
  Port mgrPort;                        /* Used for connecton to a port mgr */
  Sender portMgrSender;

  /* Return if arguments are bad */
  if (label == NULL || portManHost == NULL) {
    Tcl_AppendResult(interp, 
	       "Could not create a new receiver because of NULL arguments.", 
		     (char *) NULL);
    return (Receiver)NULL;
  }

  /* First contact the portmanager */
  mgrPort.hostName = portManHost;
  mgrPort.portNumber = PortMgrPortNumber;
  portMgrSender = NewSender(interp, &mgrPort);
  if (portMgrSender == (Sender)NULL) {
    Tcl_AppendResult(interp, "Could not connect to PortManager on host: ",
		     portManHost, ".", (char *)NULL);
    return (Receiver)NULL;
  }

  /* Now create the new Receiver */
  newReceiver = NewReceiver(portMgrSender, label, AnyPort); 
  if (newReceiver == (Receiver)NULL) {
    Tcl_AppendResult(interp, "Could not register ", label, 
		     " with PortManager on host: ", portManHost, 
		     ".", (char *)NULL);

    /* Close the port manager connection */
    DestroySender(portMgrSender);
    return newReceiver;
  }

  /* Close the port manager connection */
  DestroySender(portMgrSender);

  /* We are all set so now set up the table entry.  For Receivers,
   * the hostname is the location of the portManager where this app is
   * registered. 
   */
  Tcl_SetResult(interp,
		ipcHandleAddEntry(NULL, newReceiver,
				  portManHost,
				  label), TCL_STATIC);
  return newReceiver;
}

/*
 *--------------------------------------------------------------
 *
 * ipcDestroyReceiverFromHandle
 *
 *      Destroy the receiver represented by handleName and remove
 *      it from the PortManager on the host given.
 *
 * Results:
 *      TCL_ERROR or TCL_OK
 *--------------------------------------------------------------
 */
int ipcDestroyReceiverFromHandle(interp, handleName)
Tcl_Interp *interp; 
char *handleName;
{
  sendTableEntry entry;

  /* Return if NULL arguments */
  if (handleName == NULL) {
    Tcl_AppendResult(interp, 
	     "Could not destroy a receiver because of NULL arguments.", 
		     (char *)NULL);
    return TCL_ERROR;
  }

  /* Get the entry for handleName */
  if ((entry = ipcHandleGet(interp, handleName)) == NULL) {
    Tcl_AppendResult(interp, ".  Could not free receiver.", (char *)NULL);
    return TCL_ERROR;
  }
  /* If it's not a receiver, return an error */
  if (entry->s_receiver == NULL) {
    Tcl_AppendResult(interp, 
	       handleName, " is not a receiver.", (char *)NULL);
    return TCL_ERROR;
  }
  /* Do the actual desctruction */
  return ipcDestroyReceiver(interp, entry->s_receiver);
}

/*
 *--------------------------------------------------------------
 *
 * ipcDestroyReceiver
 *
 *      Destroy the receiver represented by handleName and remove
 *      it from the PortManager on the host given.
 *
 * Results:
 *      TCL_ERROR or TCL_OK
 *--------------------------------------------------------------
 */
int ipcDestroyReceiver(interp, receiver)
Tcl_Interp *interp; 
Receiver receiver;
{
  Port mgrPort;                    /* Used for connecton to a port mgr */
  Sender portMgrSender;
  char tmphandle[IPCHLENGTH];
  sendTableEntry entry;

  /* Need the entry to get the PortManager information */
  entry = ipcHandleGet(interp, receiver->handleName);
  if (entry == (sendTableEntry)NULL) {
    Tcl_AppendResult(interp, 
	       "The receiver handle is invalid.", (char *)NULL);
    return TCL_ERROR;
  }

  /* First contact the portmanager */
  mgrPort.hostName = entry->s_hostname;
  mgrPort.portNumber = PortMgrPortNumber;
  portMgrSender = NewSender(interp, &mgrPort);
  if (portMgrSender == (Sender)NULL) {
    Tcl_AppendResult(interp, "Could not connect to PortManager on host: ",
		     entry->s_hostname, ".", (char *)NULL);
    return TCL_ERROR;
  }
  /* Save the handle name */
  strcpy(tmphandle, receiver->handleName);
  
  /* Only need to destroy sender storage and close connection */
  DestroyReceiver(portMgrSender, receiver);
  
  /* Rid ourselves of the Port Manager sender and the table handle */
  DestroySender(portMgrSender);
  ipcHandleDestroy(interp, receiver->handleName);

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * cmdReceiver
 *
 *      Tcl script level interface for creating new Receivers.
 *      See the manual for arg possibilities.
 *
 * Results:
 *      Returns a new receiver or NULL if it fails.
 *--------------------------------------------------------------
 */
static int cmdReceiver(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  char *label = NULL;
  char *portmanhost = NULL;
  char *subcommand;
  
  /* Check number of args */
  if (argc < 3 || argc > 4) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
      argv[0], " new label | destroy handle\"", 
      (char *)NULL);
    return TCL_ERROR;
  }
  subcommand = argv[1];
  /* Use argv[2] is the receiver label */
  label = argv[2];

  /* Create a new receiver */
  if (STREQU(subcommand, "new")) {
    /* If there is 4 args, they have given a portmanager as argv[3], */
    /* otherwise, the local host is used. */
    portmanhost = (argc == 3) ? ipcThisHostName : argv[3];
    return ((ipcNewReceiver(interp, label, portmanhost) == NULL) ?
             TCL_ERROR : TCL_OK);
  }
  if (STREQU(subcommand, "destroy")) {
    if (argc == 3) {
      /* Use localhost, argv[2] is the receiver handle */
      label = argv[2];
      return ipcDestroyReceiverFromHandle(interp, label);
    } else {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], "destroy handlename", 
		       (char *)NULL);
    }
  }
  return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * cmdSend
 *
 *      TCL script level interface to the send command.  Takes
 *      an ipc handle name and the command.
 *
 * Results:
 *      Returns a new receiver or NULL if it fails.
 *--------------------------------------------------------------
 */
static int cmdSend(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  char *handleName = NULL;
  char *command = NULL;
  sendTableEntry entry;
  int result;
  
  /* Must have at least -send ipcxx command- */
  if (argc < 3) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
      argv[0], " sender arg ?arg ...?\"", 
      (char *)NULL);
    return TCL_ERROR;
  }
  /* Destination */
  handleName = argv[1];
  /* If the command is multiple arguments, make it one string */
  command = (argc == 3) ? argv[2] : Tcl_Concat(argc-2, argv+2);

  /* Make sure the handle is valid and a sender */
  if ((entry = ipcHandleGet(interp, handleName)) == NULL) {
    Tcl_AppendResult(interp, " or is not a valid ipc handle.", 
                     (char *)NULL);
    result = TCL_ERROR;
    goto exitpoint;
  }
  /* Is the handle a sender? */
  if (entry->s_sender == NULL) {
    Tcl_AppendResult(interp, 
	       handleName, " is not a sender.", (char *)NULL);
    result = TCL_ERROR;
    goto exitpoint;
  }

  /* Do the command */
  result = ipcRemoteSend(entry->s_sender, command, NULL);

  /* Free the command memory if needed. */
exitpoint:
  if (command != argv[2]) {
    /* It was ckalloc'ed */
    ckfree(command);
  }
  return result;
}

/*
 *--------------------------------------------------------------
 *
 * cmdDump
 *
 *      TCL command to dump the handle table.
 *
 * Results:
 *      TCL_OK or TCL_ERROR
 *--------------------------------------------------------------
 */
static int cmdDump(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  if (argc != 1) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
      argv[0], (char *)NULL);
    return TCL_ERROR;
  }
  return ipcHandleDump(interp);
}

/*
 *--------------------------------------------------------------
 *
 * sCreatePmDumpContext
 *
 *      Dump a list of available apps on a group of PortManagers.
 *
 * Results:
 *      TCL_OK or TCL_ERROR
 *--------------------------------------------------------------
 */
static int sCreatePmDumpContext(interp, portmanhost)
Tcl_Interp *interp;
char *portmanhost;
{
  Port mgrPort;                        /* Used for connecton to a port mgr */
  Sender portMgrSender;
  int result;
  PortMgrResult pmresult;

  /* First contact the portmanager on portmanhost */
  mgrPort.hostName = portmanhost;
  mgrPort.portNumber = PortMgrPortNumber;

  portMgrSender = NewSender(interp, &mgrPort);
  if (portMgrSender == (Sender)NULL) {
    Tcl_AppendResult(interp, "Could not connect to PortManager on host: ",
		     portmanhost, ".", (char *)NULL);
    return TCL_ERROR;
  }

  /* Now get the list of available apps */
  result = pmGetOpenApps(portMgrSender, &pmresult);
  if (result != TCL_OK) {
    Tcl_AppendResult(interp, 
		     "The PortManager could not list applications on: ", 
		     portmanhost, ".", (char *)NULL);
  }
  /* Close the port manager connection */
  DestroySender(portMgrSender);
  pmDestroyResult(pmresult);
 
  return result;
}
  

/*
 *--------------------------------------------------------------
 *
 * cmdPmDump
 *
 *      List the available apps on a PortManager
 *
 * Results:
 *      Returns a new receiver or NULL if it fails.
 *--------------------------------------------------------------
 */
static int cmdPmDump(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp; 
int argc;
char *argv[];
{
  /* Check for number of args.  For now, only support one
   * PortManager at a time.
   */
  if (argc != 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
      argv[0], " portmanhost\"", (char *)NULL);
    return TCL_ERROR;
  }
  return sCreatePmDumpContext (interp, argv[1]);
}
