/*
 * Copyright 1990 Regents of the University of California
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appears in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

/* @(#) tcl_gdbm.c 1.2@(#) University of California Berkeley 10/13/90 */

#include <stdio.h>
#include <malloc.h>
#include <time.h>

#include "tcl.h"
#include "gdbm.h"

#define HASHSIZE            31
#define TCA_EXACT           1
#define TCA_ATLEAST         2
#define TCA_ATMOST          3

#define content_setup(datum, string)	(datum).dptr = string; \
				(datum).dsize = strlen(string) + 1;
#define key_setup(datum, string)	(datum).dptr = string; \
				(datum).dsize = strlen(string) + 1;

#define bit(n)		(1 << (n))

#define GD_READ	0
#define GD_WRITE	1
#define GD_CLOSE	-1

#define GD_MODE(m)		(((m) == GDBM_READER) ? GD_READ : GD_WRITE)
#define GD_CHKMODES(m1, m2)	((GD_MODE(m1) != GD_WRITE) && \
				 (GD_MODE(m2) != GD_WRITE))

typedef struct _argcv {
          int       ar_argc;
          char      **ar_argv;
} argcv;

typedef struct _gdbm_list {
	char			*gl_name;
	int			gl_mode;
	GDBM_FILE		gl_dbf;
	struct _gdbm_list	*gl_next;
	struct _gdbm_list	*gl_prev;
} gdbm_list;

static gdbm_list	*gdbml[HASHSIZE];

static char	*gd_errstr;

static char	invalid[] =	"%s: invalid gdbm file \"%s\"";
static char	args1[] =	"%s: \"%s\": error: %s";
static char	args2[] =	"%s: \"%s\" \"%s\": error: %s";
static char	args3[] =	"%s: \"%s\" \"%s\" \"%s\": error: %s";

char                *strsave (char *str);
static int          gd_open(char *name, int mode);
static GDBM_FILE    get_gdbm(char *str, int mode);
static gdbm_list    *gl_alloc();
static unsigned long gd_hash(char *str);
static char         *gdbm_perror();
static void         gd_fatal(char *str);
static int          tcl_chk_mode(char *str);
argcv               *ar_alloc();
void                ar_add(argcv *ar, char *str);
void                ar_free(argcv *ar);

/* template 
extern Tcl_CmdProc xxxx;
*/
extern Tcl_CmdProc tcl_gdbm_close;
extern Tcl_CmdProc tcl_gdbm_delete;
extern Tcl_CmdProc tcl_gdbm_fetch;
extern Tcl_CmdProc tcl_gdbm_insert;
extern Tcl_CmdProc tcl_gdbm_list;
extern Tcl_CmdProc tcl_gdbm_open;
extern Tcl_CmdProc tcl_gdbm_reorganize;
extern Tcl_CmdProc tcl_gdbm_replace;
extern Tcl_CmdProc tcl_gdbm_store;
extern Tcl_CmdProc tcl_localtime;
extern Tcl_CmdProc tcl_gettimeofday;
extern Tcl_CmdProc tcl_ctime;

/*
 * get the time of day.
 *
 * no args.
 *
 * returns number of seconds since midnight
 * January 1, 1970.
 */
int tcl_gettimeofday(client_data, interp, argc, argv)
ClientData          client_data;
Tcl_Interp          *interp;
int                 argc;
char                **argv;
{
   struct timeval      tv;
   struct timezone     tz;
   char                buf[512];

   if (tcl_chkarg(interp, argv[0], TCA_EXACT, 1, argc))
      return(TCL_ERROR);

   if (gettimeofday(&tv, &tz) == -1) {
      sprintf(buf, "%s: gettimeofday failed", argv[0]);
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   sprintf(buf, "%lu", tv.tv_sec);

   Tcl_AppendResult (interp,buf,(char *)0);

   return(TCL_OK);
}

/*
 * convert a time as returned by gettimeofday
 * into 25 character ascii string.  all fields
 * have constant width.
 *
 * expects 1 arg which is the time.
 *
 * returns string 25 characters long.
 */

int tcl_ctime(client_data, interp, argc, argv)
ClientData          client_data;
Tcl_Interp          *interp;
int                 argc;
char                **argv;
{
   extern char         *ctime();
   extern char         *index();
   long                clock;
   char                *str, *cp;

   if (tcl_chkarg(interp, argv[0], TCA_EXACT, 2, argc)) {
      Tcl_AppendResult (interp,"\nctime <gettimeofday>\n",
         "returns: 25 chars strings of 'Sun Sep 16 01:03:52 1973'\n",
         (char *)0);
      return(TCL_ERROR);
   }

   if (sscanf(argv[1], "%ld", &clock) != 1) {
      Tcl_AppendResult (interp,"ctime: bad argument ",argv[1],"\n",(char *)0);
      return(TCL_ERROR);
   }

   str = ctime(&clock);
   if ((cp = index(str, '\n')) != NULL)
      *cp = NULL;

   Tcl_AppendResult (interp,str,(char *)0);

   return(TCL_OK);
}

/*
 * convert a time as returned by gettimeofday into a 9 element
 * list of numbers: seconds minutes hour month-day month year
 * week-day year-day is-daylight-savings.  These quantities give
 * the time on a 24-hour clock, the day of the month (1-31), the
 * month of the year (0-11), the day of the week (Sunday = 0),
 * the year minus 1900, the day of the year (0-365, also known as
 * the Julian date), and either a 1 or a 0 if daylight saving time
 * is in effect.
 *
 * expects 1 arg which is the time.
 *
 * returns a list as described above.
 */

int tcl_localtime(client_data, interp, argc, argv)
ClientData                    client_data;
Tcl_Interp                    *interp;
int                           argc;
char                          **argv;
{
   extern struct tm    *localtime();
   struct tm           *tm;
   argcv                         *ar;
   long                          clock;
   char                          buf[128];

   if (tcl_chkarg(interp, argv[0], TCA_EXACT, 2, argc)) {
      Tcl_AppendResult (interp,"\nlocaltime <gettimeofday>\n",
                        "returns: seconds minutes hour(0-23)\n",
                        "         month-day(1-31) month(0-11) year(x-1990)\n",
                        "         week-day(Sunday=0) daylight-saving(0,1)\n",
                        (char *)0);
      return(TCL_ERROR);
   }

   if (sscanf(argv[1], "%ld", &clock) != 1) {
      sprintf(buf, "%s: malformed argument", argv[0]);
      Tcl_Return(interp, strsave(buf), TCL_DYNAMIC);
      return(TCL_ERROR);
   }

   ar = ar_alloc();

   tm = localtime(&clock);

   sprintf(buf, "%d", tm->tm_sec);
   ar_add(ar, buf);
   sprintf(buf, "%d", tm->tm_min);
   ar_add(ar, buf);
   sprintf(buf, "%d", tm->tm_hour);
   ar_add(ar, buf);
   sprintf(buf, "%d", tm->tm_mday);
   ar_add(ar, buf);
   sprintf(buf, "%d", tm->tm_mon);
   ar_add(ar, buf);
   sprintf(buf, "%d", tm->tm_year);
   ar_add(ar, buf);
   sprintf(buf, "%d", tm->tm_wday);
   ar_add(ar, buf);
   sprintf(buf, "%d", tm->tm_yday);
   ar_add(ar, buf);
   sprintf(buf, "%d", tm->tm_isdst);
   ar_add(ar, buf);

   Tcl_SetResult (interp,Tcl_Merge(ar->ar_argc,ar->ar_argv),TCL_DYNAMIC);
   ar_free(ar);

   return(TCL_OK);
}

/*
 * gdbm_open <name> [<mode>]
 *
 * mode is either "r", "w", "wc", or "n".
 */
int tcl_gdbm_open(client_data, interp, argc, argv)
ClientData	client_data;
Tcl_Interp	*interp;
int		argc;
char		**argv;
{
   int		mode;
   char		buf[512];

   if (tcl_chkarg(interp, argv[0], TCA_ATMOST, 3, argc)) {
      Tcl_AppendResult (interp,"\ngdbm_open <name> [<mode>]\n",(char *)0);
      return(TCL_ERROR);
   }

   mode = GDBM_READER;

   if (argc==3) {
      if ((mode=tcl_chk_mode(argv[2]))==-1) {
         sprintf(buf, "%s: bad mode \"%s\"", argv[0], argv[2]);
         Tcl_AppendResult (interp,buf,(char *)0);
	return(TCL_ERROR);
      }
   }

   gd_errstr = "???";

   if (gd_open(argv[1],mode)==-1) {
      if (argc == 3)
         sprintf(buf, args2, argv[0], argv[1], argv[2], gdbm_perror());
      else
         sprintf(buf, args1, argv[0], argv[1], gdbm_perror());
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }
   return(TCL_OK);
}

/*
 * gdbm_close <name>
 */
int tcl_gdbm_close(client_data, interp, argc, argv)
ClientData	client_data;
Tcl_Interp	*interp;
int		argc;
char		**argv;
{
   GDBM_FILE	gfr;
   char		buf[1024];

   if (tcl_chkarg(interp, argv[0], TCA_EXACT, 2, argc)) {
      Tcl_AppendResult (interp,"\ngdbm_close <name>\n",(char *)0);
      return(TCL_ERROR);
   }

   if ((gfr = get_gdbm(argv[1], GD_CLOSE)) == NULL) {
      sprintf(buf, invalid, argv[0], argv[1]);
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   return(TCL_OK);
}

/*
 * gdbm_insert <name> <key> <content>
 */
int tcl_gdbm_insert(client_data, interp, argc, argv)
ClientData	client_data;
Tcl_Interp	*interp;
int		argc;
char		**argv;
{
   GDBM_FILE	gfw;
   datum		key, content;
   char		buf[512];

   if (tcl_chkarg(interp, argv[0], TCA_EXACT, 4, argc)) {
      Tcl_AppendResult (interp,"\ngdbm_insert <name> <key> <data>\n",(char *)0);
      return(TCL_ERROR);
   }

   if ((gfw = get_gdbm(argv[1], GD_WRITE)) == NULL) {
      sprintf(buf, invalid, argv[0], argv[1]);
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   key_setup(key, argv[2]);
   content_setup(content, argv[3]);

   if (gdbm_store(gfw, key, content, GDBM_INSERT) != 0) {
      sprintf(buf,args3,argv[0],argv[1],argv[2],argv[3],gdbm_perror());
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   return(TCL_OK);
}

/*
 * gdbm_replace <name> <key> <content>
 */
int tcl_gdbm_replace(client_data, interp, argc, argv)
ClientData	client_data;
Tcl_Interp	*interp;
int		argc;
char		**argv;
{
   GDBM_FILE	gfw;
   datum		key, content;
   char		buf[512];

   if (tcl_chkarg(interp, argv[0], TCA_EXACT, 4, argc)) {
      Tcl_AppendResult (interp,"\ngdbm_replace <name> <key> <data>\n",(char *)0);
      return(TCL_ERROR);
   }

   if ((gfw = get_gdbm(argv[1], GD_WRITE)) == NULL) {
      sprintf(buf, invalid, argv[0], argv[1]);
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   key_setup(key, argv[2]);
   content_setup(content, argv[3]);

   if (gdbm_store(gfw, key, content, GDBM_REPLACE) != 0) {
      sprintf(buf,args3,argv[0],argv[1],argv[2],argv[3],gdbm_perror());
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   return(TCL_OK);
}

/*
 * gdbm_store <name> <key> <content>
 */
int tcl_gdbm_store(client_data, interp, argc, argv)
ClientData	client_data;
Tcl_Interp	*interp;
int		argc;
char		**argv;
{
   GDBM_FILE	gfw;
   datum		key, content;
   char		buf[512];
   int		ret;

   if (tcl_chkarg(interp, argv[0], TCA_EXACT, 4, argc)) {
      Tcl_AppendResult (interp,"\ngdbm_store <name> <key> <data>\n",(char *)0);
      return(TCL_ERROR);
   }

   if ((gfw = get_gdbm(argv[1], GD_WRITE)) == NULL) {
      sprintf(buf, invalid, argv[0], argv[1]);
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   key_setup(key, argv[2]);
   content_setup(content, argv[3]);

   if ((ret=gdbm_store(gfw, key, content, GDBM_INSERT)) == 0)
      return(TCL_OK);

   if (gdbm_store(gfw, key, content, GDBM_REPLACE) != 0) {
      sprintf(buf, args3, argv[0], argv[1], argv[2], argv[3], gdbm_perror());
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   return(TCL_OK);
}

/*
 * gdbm_fetch <name> <key>
 */
int tcl_gdbm_fetch(client_data, interp, argc, argv)
ClientData	client_data;
Tcl_Interp	*interp;
int		argc;
char		**argv;
{
   GDBM_FILE	gfr;
   datum		key, content;
   char		buf[512];

   if (tcl_chkarg(interp, argv[0], TCA_EXACT, 3, argc)) {
      Tcl_AppendResult (interp,"\ngdbm_fetch <name> <key>\n",(char *)0);
      return(TCL_ERROR);
   }

   if ((gfr=get_gdbm(argv[1], GD_READ)) == NULL) {
      sprintf(buf, invalid, argv[0], argv[1]);
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   key_setup(key, argv[2]);

   content = gdbm_fetch(gfr, key);

   if (content.dptr != NULL)
      Tcl_AppendResult (interp,content.dptr,(char *)0);

   return(TCL_OK);
}

/*
 * gdbm_delete <name> <key>
 */
int tcl_gdbm_delete(client_data, interp, argc, argv)
ClientData	client_data;
Tcl_Interp	*interp;
int		argc;
char		**argv;
{
   GDBM_FILE	gfw;
   datum		key;
   char		buf[512];

   if (tcl_chkarg(interp, argv[0], TCA_EXACT, 3, argc)) {
      Tcl_AppendResult (interp,"\ngdbm_delete <name> <key>\n",(char *)0);
      return(TCL_ERROR);
   }

   if ((gfw=get_gdbm(argv[1], GD_WRITE)) == NULL) {
      sprintf(buf, invalid, argv[0], argv[1]);
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   key_setup(key, argv[2]);

   if (gdbm_delete(gfw, key) == -1) {
      sprintf(buf, args2, argv[0], argv[1], argv[2], gdbm_perror());
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   return(TCL_OK);
}

/*
 * gdbm_list <name>
 */
int tcl_gdbm_list(client_data, interp, argc, argv)
ClientData	client_data;
Tcl_Interp	*interp;
int		argc;
char		**argv;
{
   argcv		*key_list;
   GDBM_FILE	gfr;
   datum		key;
   char		*str;
   char		buf[1024];

   if (tcl_chkarg(interp, argv[0], TCA_EXACT, 2, argc)) {
      Tcl_AppendResult (interp,"\ngdbm_list <name>\n",(char *)0);
      return(TCL_ERROR);
   }

   if ((gfr=get_gdbm(argv[1], GD_READ)) == NULL) {
      sprintf(buf, invalid, argv[0], argv[1]);
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   key = gdbm_firstkey(gfr);

   if (key.dptr == NULL)
      return(TCL_OK);

   key_list = (argcv *)ar_alloc();

   ar_add(key_list, key.dptr);

   strcpy(buf, key.dptr);
   free(key.dptr);
   key.dptr = buf;

   for (;;) {
      key = gdbm_nextkey(gfr, key);
      if (key.dptr == NULL)
         break;
      ar_add(key_list, key.dptr);
      strcpy(buf, key.dptr);
      free(key.dptr);
      key.dptr = buf;
   }

   str = Tcl_Merge(key_list->ar_argc, key_list->ar_argv);

   ar_free(key_list);

   Tcl_AppendResult (interp,str,(char *)0);

   return(TCL_OK);
}

/*
 * gdbm_reorganize <name>
 */
int tcl_gdbm_reorganize(client_data, interp, argc, argv)
ClientData	client_data;
Tcl_Interp	*interp;
int		argc;
char		**argv;
{
   GDBM_FILE	gfw;
   char		buf[512];

   if (tcl_chkarg(interp, argv[0], TCA_EXACT, 2, argc)) {
      Tcl_AppendResult (interp,"\ngdbm_reorganize <name>\n",(char *)0);
      return(TCL_ERROR);
   }

   if ((gfw=get_gdbm(argv[1], GD_WRITE)) == NULL) {
      sprintf(buf, invalid, argv[0], argv[1]);
      Tcl_AppendResult (interp,buf,(char *)0);
      return(TCL_ERROR);
   }

   gdbm_reorganize(gfw);

   return(TCL_OK);
}

static int gd_open(name, mode)
char	*name;
int	mode;
{
   extern void	gd_fatal();
   gdbm_list	*gl, *hgl;
   GDBM_FILE	dbf;

   /*
    * see if already open.
    * if opening for reading then find a gl that's already
    * opened for reading and return its dbf.  if opening for
    * writing then check all gl's to see if one with the same
    * name is already opened and bomb if so.
    */
   hgl = gdbml[gd_hash(name)];
   for (gl = hgl->gl_next; gl != hgl; gl = gl->gl_next) {
      if ((gl->gl_name != NULL) && strcmp(gl->gl_name, name) == 0) {
         if (mode == GDBM_READER && gl->gl_mode == GDBM_READER)
            return(0);
         return(-1);
      }
   }

   /* not on the list */

   gl = gl_alloc();

   if ((dbf=gdbm_open(name, 0, mode, 00664, gd_fatal)) == NULL) {
      free((char *) gl);
      return(-1);
   }

   gl->gl_dbf = dbf;
   gl->gl_name = strsave(name);
   gl->gl_mode = mode;

   /* link in */
   gl->gl_prev = hgl;
   gl->gl_next = hgl->gl_next;
   hgl->gl_next->gl_prev = gl;
   hgl->gl_next = gl;

   return(0);
}

static GDBM_FILE gd_close(gl)
gdbm_list	*gl;
{
   GDBM_FILE	dbf;

   dbf = gl->gl_dbf;

   gdbm_close(gl->gl_dbf);
   gl->gl_dbf = NULL;

   if (gl->gl_name != NULL)
      free(gl->gl_name);

   gl->gl_name = NULL;
   gl->gl_mode = GD_CLOSE;

   gl->gl_prev->gl_next = gl->gl_next;
   gl->gl_next->gl_prev = gl->gl_prev;

   return(dbf);
}

static GDBM_FILE get_gdbm(str, mode)
char		*str;
int		mode;
{
   gdbm_list	*gl, *hgl;

   if (str == NULL)
      return(NULL);

   hgl = gdbml[gd_hash(str)];
   for (gl = hgl->gl_next; gl != hgl; gl = gl->gl_next) {
      if (gl->gl_name == NULL)
         continue;
      if (strcmp(gl->gl_name, str) == 0)
         break;
   }

   /* not found */
   if (gl == hgl)
      return(NULL);

   if (mode == GD_CLOSE)
      return(gd_close(gl));

   if (mode == GD_MODE(gl->gl_mode))
      return(gl->gl_dbf);

   /* wrong mode */
   return(NULL);
}

void init_gdbml() {
   int		i;
   gdbm_list	*gl;

   for (i = 0; i < HASHSIZE; i++) {
      gl = gl_alloc();
      gl->gl_name = NULL;
      gl->gl_mode = GD_CLOSE;
      gl->gl_dbf = NULL;
      gl->gl_next = gl;
      gl->gl_prev = gl;
      gdbml[i] = gl;
   }
}

static gdbm_list *gl_alloc() {
   gdbm_list	*gl;

   if ((gl = (gdbm_list *) malloc(sizeof(gdbm_list))) == NULL) {
      perror("malloc");
      shutdown();
   }

   gl->gl_name = NULL;
   gl->gl_dbf = NULL;
   gl->gl_mode = GD_CLOSE;
   gl->gl_next = NULL;
   gl->gl_prev = NULL;

   return(gl);
}

static unsigned long gd_hash(str)
char		*str;
{
   unsigned long	ch;
   unsigned long	guard;

   ch = 0;

   while (*str != NULL) {
      ch = (ch << 1) + *str++;
      if ((guard=(ch & 0xf0000000)) != 0) {
         ch ^= (guard >> 24);
         ch ^= guard;
      }
   }

   ch %= 211;

   return(ch % HASHSIZE);
}

void close_gdbm() {
   gdbm_list	*gl, *hgl;
   int		i;

   for (i = 0; i < HASHSIZE; i++) {
      hgl = gdbml[i];
      for (gl = hgl->gl_next; gl != hgl; gl = gl->gl_next) {
         if (gl->gl_mode == GD_CLOSE)
            continue;

         if (gl->gl_dbf == NULL)
            continue;

         gdbm_close(gl->gl_dbf);
      }
   }
}

static void gd_fatal(str)
char	*str;
{
   perror("gdbm");
   fprintf(stderr, "fatal gdbm error: %s\n", str);
   gd_errstr = str;
}

static char *gdbm_perror() {
   static char *gdbm_errors[] = {
		"no error",
		"malloc error",
		"block size error",
		"file open error",
		"file write error",
		"file seek error",
		"file read error",
		"bad magic number",
		"empty database",
		"can't be reader",
		"can't be writer",
		"reader can't delete",
		"reader can't store",
		"reader can't reorganize",
		"unknown update",
		"key not found",
		"reorganize failed",
		"cannot replace",
                    "illegal data",
                    "opt already set",
                    "illegal opt"
	};
   extern gdbm_error	gdbm_errno;

   if (gdbm_errno < 0)
      return("bogus error");

   if (gdbm_errno > (sizeof(gdbm_errors) / sizeof(gdbm_errors[0])))
      return("unknown error");

   return(gdbm_errors[gdbm_errno]);
}

static int tcl_chk_mode(str)
char	*str;
{
   if (strcmp(str, "r") == 0)
      return(GDBM_READER);

   if (strcmp(str, "w") == 0)
      return(GDBM_WRITER);

   if (strcmp(str, "wc") == 0)
      return(GDBM_WRCREAT);

   if (strcmp(str, "n") == 0)
      return(GDBM_NEWDB);

   return(-1);
}

int tcl_chkarg(interp, name, type, cnt, argc)
Tcl_Interp      *interp;
char            *name;
int             type;
int             cnt;
int             argc;
{
   char            buf[512];

   if (argc == cnt)
      return(0);

   if ((type != TCA_ATLEAST) && (argc > cnt)) {
      sprintf(buf, "%s: too many args", name);
      Tcl_AppendResult (interp,buf,(char *)0);
      return(1);
   }

   if ((type != TCA_ATMOST) && (argc < cnt)) {
      sprintf(buf, "%s: not enough args", name);
      Tcl_AppendResult (interp,buf,(char *)0);
      return(1);
   }

   return(0);
}

/*
 * argcv variables are used to
 * collect the multi-line output
 * from the mud server.
 */
argcv *ar_alloc() {
   argcv               *ar;

   if ((ar = (argcv *) malloc(sizeof(argcv))) == NULL) {
      perror("malloc");
      exit(1);
   }

   if ((ar->ar_argv = (char **) malloc(sizeof(char *))) == NULL) {
      perror("malloc");
      exit(1);
   }

   ar->ar_argc = 0;

   return(ar);
}

/*
 * add another line to an argcv
 */
void ar_add(ar, str)
argcv           *ar;
char            *str;
{
   ar->ar_argc++;

   if ((ar->ar_argv=(char **)realloc((char *)ar->ar_argv,
                    (unsigned)ar->ar_argc*sizeof(char *)))==NULL) {
      perror("realloc");
      exit(1);
   }

   ar->ar_argv[ar->ar_argc - 1] = strsave(str);
}

/*
 * free an argcv and all its elements.
 */
void ar_free(ar)
argcv           *ar;
{
   int             i;

   for (i = 0; i < ar->ar_argc; i++)
      free(ar->ar_argv[i]);

   free((char *) ar->ar_argv);
   free((char *) ar);
}

char *strsave(str)
char                *str;
{
   char                *cp;

   if ((cp = malloc((unsigned) strlen(str) + 1)) == NULL) {
      fprintf(stderr, "strsave: ");
      perror("malloc");
      exit(1);
   }

   strcpy(cp, str);

   return(cp);
}

int gdbm_Init (interp)
Tcl_Interp *interp;
{
   char buffer[16];

   /* template
   Tcl_CreateCommand (interp, "", xxxx,  yyyy,
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);
   */

   Tcl_CreateCommand (interp, "gettimeofday", tcl_gettimeofday,
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);
   Tcl_CreateCommand (interp, "localtime", tcl_localtime,
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);
   Tcl_CreateCommand (interp, "ctime", tcl_ctime,
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);

   Tcl_CreateCommand (interp, "gdbm_close", tcl_gdbm_close, 
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);
   Tcl_CreateCommand (interp, "gdbm_delete", tcl_gdbm_delete, 
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);
   Tcl_CreateCommand (interp, "gdbm_fetch", tcl_gdbm_fetch, 
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);
   Tcl_CreateCommand (interp, "gdbm_insert", tcl_gdbm_insert, 
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);
   Tcl_CreateCommand (interp, "gdbm_list", tcl_gdbm_list, 
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);
   Tcl_CreateCommand (interp, "gdbm_open", tcl_gdbm_open, 
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);
   Tcl_CreateCommand (interp, "gdbm_reorganize", tcl_gdbm_reorganize, 
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);
   Tcl_CreateCommand (interp, "gdbm_replace", tcl_gdbm_replace, 
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);
   Tcl_CreateCommand (interp, "gdbm_store", tcl_gdbm_store, 
                      (ClientData)0, (Tcl_CmdDeleteProc *)0);

   init_gdbml();

   return TCL_OK;
}
