/*
 * tmConverters.c -
 *	This module implements all the Xt Converters to and
 *	from Strings needed by Tm
 *
 * Copyright 1993 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

/* Copyright (C) 1992 by Gustaf Neumann, Stefan Nusser

 *      Wirtschaftsuniversitaet Wien, 
 *      Abteilung fuer Wirtschaftsinformatik
 *      Augasse 2-6, 
 *      A-1090 Vienna, Austria
 *      neumann@wu-wien.ac.at, nusser@wu-wien.ac.at

 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appears in all copies and that both that
 * copyright notice and this permission notice appear in all supporting
 * documentation.  This software is provided "as is" without expressed or
 * implied warranty.
 */

#include "tmFuncs.h"
#ifndef MOTIF11
#include <Xm/RepType.h>
#include <Xm/DragDrop.h>
#endif
#include <X11/IntrinsicP.h>
#include <X11/CoreP.h>		/* for CoreRec */
#include <string.h>
#include <ctype.h>

/* escape char for XmString formatting */
#define TM_FORMAT_CHAR '@'

#define done(address, type)                               \
    {                                                   \
        if(toVal->addr)                                 \
        {                                               \
            if(toVal->size < sizeof(type))              \
            {                                           \
                toVal->size = sizeof(type);             \
                return( FALSE) ;                        \
            }                                           \
            else                                        \
            {   *((type *) (toVal->addr)) = *address;   \
            }                                           \
        }                                               \
        else                                            \
        {                                               \
            toVal->addr = (XPointer) address;           \
        }                                               \
        toVal->size = sizeof(type);                     \
    }


/*
 * so that we can keep tabs on the size of an XmStringTable
 * see function Tm_MaybeSetXmStringTableSize
 */
int Tm_XmStringTableSize;

/*
 *--------------------------------------------------------------
 *
 * InstallReverseRepTypes --
 *
 *	Install the converters from internal representation
 *	types to their string values.
 *
 * Results:
 *
 * Side effects:
 *	Modifies the Motif RepType tables
 *
 *--------------------------------------------------------------
 */

static void
InstallReverseRepTypes()
{
#ifndef MOTIF11
    XmRepTypeId id;
    XmRepTypeList start, list;
    int n;

    start = list = XmRepTypeGetRegistered();

    while (list->rep_type_name != NULL) {
        if ((id = XmRepTypeGetId(list->rep_type_name)) != XmREP_TYPE_INVALID) {
	    XmRepTypeAddReverse(id);
        } else {
	    fprintf(stderr, "Failed to install %s converter\n", 
			list->rep_type_name);
        }
	list++;
    }
    XtFree((char *)start);
#endif
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToWidget --
 *
 *	Converter from String to Widget type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToWidget(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    Tcl_Interp *interp = (Tcl_Interp *) args->addr;
    Tm_Widget *info;
    static Widget w;

    info = Tm_WidgetInfoFromPath(interp, (char *) fromVal->addr);
    if (info == NULL) {
	return False;
    }
    w = info->widget;

    done(&w, Widget);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtWidgetToString --
 *
 *	Converter from Widget to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtWidgetToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    Tm_Widget *info;
    Widget w = *(Widget *) fromVal->addr;
    static char *buf;
    extern char *Tm_Resource; 

    /* double hack going on here. because of Motif
       bug, some Widget resources are of type Window.
       When we *really* get a window to convert,
       just fail :-((
     */
    if (strcmp(Tm_Resource, "windowGroup") == 0)
	return False;

    /* it is valid to ask for a NULL widget e.g. topWidget in a Form
       when the thing is attached to the Form (default is NULL)
     */
    if (w == NULL) {
        buf = "";
        done(&buf, String);
	return True;
    }
    info = NULL;
    XtVaGetValues(w, XmNuserData, &info, NULL);
    if (info == NULL) {
	return False;
    }

    buf = info->pathName;
    done(&buf, String);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtWidgetListToString --
 *
 *	Converter from WidgetList to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtWidgetListToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    Tm_Widget *info;
    Widget w = *(Widget *) args[0].addr;
    WidgetList children = *(WidgetList *) fromVal->addr;
    Cardinal num_children, final_num_children;
    String *widgetNames;
    int n, len;
    char *end, *p;
    static char *buf = NULL;
    int buf_size;

    buf_size = 32;
    buf = XtRealloc(buf, buf_size);

    if ( !XtIsComposite(w)) {
	buf[0] = '\0';
        done(&buf, char *);
        return True;
    }

    XtVaGetValues(w, 
	XmNnumChildren, &num_children,
	XmNchildren, &children, NULL);
    if (num_children == 0) {
	buf[0] = '\0';
        done(&buf, char *);
        return True;
    }

    widgetNames = (String *) XtMalloc(num_children * sizeof(String));
    len = 0;
    final_num_children = 0;
    for (n = 0; n < num_children; n++) {
	info = NULL;
        XtVaGetValues(children[n], XmNuserData, &info, NULL);
        if (info == NULL) {
	    continue;
	}
	widgetNames[final_num_children] = info->pathName;
	len += strlen(info->pathName) + 2; /* for ", " */
	final_num_children++;
    }

    /* grow if needed */
    if (buf_size < len + 1) {
	do {
	    buf_size *= 2;
	}   while (buf_size < len + 1);
	buf = XtRealloc(buf, buf_size);
    }

    if (len == 0) {
        *buf = '\0';
        done(&buf, char *);
        return True;
    }
    end = buf;
    for (n = 0; n < final_num_children; n++) {
        p = widgetNames[n];
        while (*end++ = *p++)
        ;
        *(end - 1) = ',';
        *end++ = ' ';
    }
    *(end - 2) = '\0';

    done(&buf, char *);
    XtFree((char *) widgetNames);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToTransferStatus --
 *
 *	Converter from String to TransferStatus type
 *	(see transferProc widget0
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToTransferStatus(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    char *buf;
    int n;
    static unsigned char status;

    buf = (char *) (fromVal->addr);
    for (n = 0; n < strlen(buf); n++) {
	buf[n] = tolower(buf[n]);
    }
#   ifdef DEBUG
    fprintf(stderr, "converted string to %s\n", buf);
#   endif
    if (strcmp(buf, "transfer_success") == 0) {
	status = 1;
    } else {
	status = 0;
    }

    done(&status, unsigned char);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToDropTransfers--
 *
 *	Converter from String to DropTransfers type
 *	(see transferProc widget0
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToDropTransfers(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
#ifndef MOTIF11
    char *buf;
    char **argv, **argvv;
    int argc, argcc;
    int n;
    static unsigned char status;
    static XmDropTransferEntryRec transferEntries[32];
    static XmDropTransferEntryRec *p;
    Tm_Widget *wPtr;
    Tcl_Interp *interp = (Tcl_Interp *) args->addr;

    buf = (char *) (fromVal->addr);
#   ifdef DEBUG
    fprintf(stderr, "converted string to %s\n", buf);
#   endif
    if (Tcl_SplitList(interp, buf, &argc, &argv) == TCL_ERROR) {
	fprintf(stderr, "failed to convert list\n");
    }
    for (n = 0; n < argc; n++) {
	if (Tcl_SplitList(interp, argv[n], &argcc, &argvv) == TCL_ERROR) {
	    fprintf(stderr, "failed to convert internal list\n");
	}
	if (argcc != 2) {
	    fprintf(stderr, "list should have two elements\n");
	}
	transferEntries[n].target = XInternAtom(display, argvv[0], False);
/*
	wPtr = Tm_WidgetInfoFromPath(interp, argvv[1]);
	transferEntries[n].client_data = wPtr->widget;
*/
	transferEntries[n].client_data = argvv[1];
    }

    p = transferEntries;
    done(&p, XtPointer);

#endif
    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtXmStringToString --
 *
 *	Converter from XmString to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	destroys the XmString
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtXmStringToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    XmStringContext context;
    static char *buf = NULL;
    char *p, *text;
    XmStringCharSet charset;
    XmStringDirection direction;
    Boolean separator;
    int size;
    int current_len, segment_len;
    XmString xmstr;

    /* whatever buf has grown to, or started from, 
       make it a reasonable size
     */
    size = 128;
    buf = XtRealloc(buf, size);
    p = buf;

    xmstr = *(XmString *) fromVal->addr;
    if (!XmStringInitContext(&context, xmstr)) {
	return False;
    }
    while (XmStringGetNextSegment(context, &text, &charset,
		&direction, &separator)) {
	XtFree(charset);
	segment_len = strlen(text);
	if (p + segment_len >= buf + size) {
	    /* string too large, double space */
	    current_len = p - buf;
	    size *= 2;
	    buf = XtRealloc(buf, size);
	    p = buf + current_len;
	}
	strcpy(p, text);
	p += segment_len;
	if (separator == True) {
	    *p++ = '\n';
	    *p = 0;
	}
	XtFree(text);
    }
    XmStringFreeContext(context);

    /* destroy the XmString we started from */
    XmStringFree(xmstr);
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_XmStringToString --
 *
 * Results:
 *
 *	returns the Ascii string rep of the XmString
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */
String
Tm_XmStringToString(xmstr)
    XmString xmstr;
{
    XmStringContext context;
    static char *buf = NULL;
    char *p, *text;
    XmStringCharSet charset;
    XmStringDirection direction;
    Boolean separator;
    int size;
    int current_len, segment_len;

    /* whatever buf has grown to, or started from, 
       make it a reasonable size
     */
    size = 128;
    buf = XtRealloc(buf, size);
    p = buf;

    if (!XmStringInitContext(&context, xmstr)) {
	return False;
    }
    while (XmStringGetNextSegment(context, &text, &charset,
		&direction, &separator)) {
	XtFree(charset);
	segment_len = strlen(text);
	if (p + segment_len >= buf + size) {
	    /* string too large, double space */
	    current_len = p - buf;
	    size *= 2;
	    buf = XtRealloc(buf, size);
	    p = buf + current_len;
	}
	strcpy(p, text);
	p += segment_len;
	if (separator == True) {
	    *p++ = '\n';
	    *p = 0;
	}
	XtFree(text);
    }
    XmStringFreeContext(context);

    return buf;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtXmStringTableToString --
 *
 *	Converter from XmStringTable to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtXmStringTableToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    String *items;
    static String buf = NULL;
    int n, len, item_count;
    char *p, *end;
    char *dummy;
    XmString *xmstrs = *(XmString **) fromVal->addr;
    Arg arg;
    Widget w = *((Widget *) args[0].addr);
    char *size_resource;
    /* this should come in via args, but Xt doesn't give it */
    extern char *Tm_Resource;

    if (Tm_Resource== NULL)
	return False;
    if (strcmp (Tm_Resource, XmNitems) == 0) {
        size_resource = XmNitemCount;
    } else
    if (strcmp (Tm_Resource, XmNselectedItems) == 0) {
        size_resource = XmNselectedItemCount;
    } else
    return False;

    XtSetArg(arg, size_resource, &item_count);
    XtGetValues(w, &arg, 1);
    Tm_Resource = NULL; /* make sure under our control */

    len= 0;
    items = NULL;

    if (item_count != 0) {
        items = (String *) XtMalloc(item_count * sizeof(String));
        for (n = 0; n < item_count; n++) {
	    items[n] = Tm_XmStringToString(xmstrs[n]);
	    items[n] = XtNewString(items[n]);
            len += strlen(items[n]) + 2; /* for , */
        }
    }
#ifdef TM_MOTIF_XMSTRINGTABLE
    buf = XtRealloc(buf, len + 1); /* we overshoot by 1 */
    if (len == 0) {
        *buf = '\0';
        done(&buf, char *);
        return True;
    }
    end = buf;
    for (n = 0; n < item_count; n++) {
        p = items[n];
        while (*end++ = *p++)
    	;
        *(end - 1) = ',';
        *end++ = ' ';
        XtFree(items[n]);
    }
    *(end - 2) = '\0';
#else
    if (buf != NULL)
	free(buf);
    buf = Tcl_Merge(item_count, items);
    for (n = 0; n < item_count; n++) {
        XtFree(items[n]);
    }
#endif

    done(&buf, char *);

    if (items != NULL) {
	XtFree((char *) items);
    }

    return True;
}


Boolean
Tm_CvtStringToXmStringTable(display, args, num_args, 
		fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    Tcl_Interp *interp = (Tcl_Interp *) args->addr;
    Tm_Widget *info;
    char **argv;
    int argc;
    static XmStringTable tbl = NULL;
    int n;

    if (Tcl_SplitList(interp, (char *) fromVal->addr,
		&argc, &argv) == TCL_ERROR) {
	return False;
    }

    if (argc == 0) {
	tbl = NULL;
	done(&tbl, XmStringTable);
	free((char *) argv);
	return True;
    }

    tbl = (XmStringTable) XtMalloc(argc * sizeof(XmString));
    Tm_AddToFreeResourceList((char *) tbl, (Tm_FreeProc) XtFree);

    for (n = 0; n < argc; n++) {
	tbl[n] = XmStringCreateLocalized(argv[n]);
        /* allow this to be garbage collected when possible */
        Tm_AddToFreeResourceList((char *) tbl[n],
                (Tm_FreeProc) XmStringFree);
    }
    done(&tbl, XmStringTable);
    free((char *) argv);

    return True;
}

static XmString
Tm_XmStringConcatAndFree(front, back)
    XmString front;
    XmString back;
{
    XmString result;

    result = XmStringConcat(front, back);
    XmStringFree(front);
    XmStringFree(back);

    return result;
}

static XmString
Tm_XmStringAppendString(oldCompStr, text, charSet, dir, sep)
XmString           oldCompStr;
char              *text;
XmStringCharSet    charSet;
XmStringDirection  dir;
Boolean            sep;
    {
    XmString  addSeg, newString;
    
    if (*text == '\0' && sep)
	return(oldCompStr);

/*
    fprintf(stderr, "Generated segment with <%s>\n", text);
*/

    if (oldCompStr == NULL)
	return(XmStringSegmentCreate(text, charSet, dir, sep));
    else
	{
	addSeg = XmStringSegmentCreate(text, charSet, dir, sep);
	newString = XmStringConcat(oldCompStr, addSeg);
	XmStringFree(addSeg);
	XmStringFree(oldCompStr);
	return(newString);
	}
    }

static XmString    
Tm_StringToXmString(interp, inString)
    char  *inString;
    Tcl_Interp *interp;
{
    char               *segBuf;
    XmStringCharSet    charSet = XmFONTLIST_DEFAULT_TAG; 
    XmStringDirection  dir = XmSTRING_DIRECTION_L_TO_R;
    XmString           result = NULL;
    XmString	       xm_dir;
    char              *segPtr;
    char              *ptr;
    char              *top;
    int		      len;
    char	      separator;
    int 	      argc;
    char	      **argv;
    int		      n;
    Boolean	      segment_started = True;

    len = strlen(inString); 
    segPtr = segBuf =  XtMalloc(len + 1);

    /* In Motif 2.0 the segment stuff will vanish, leaving
       the component stuff only
     */
    *segPtr = '\0';

    Tcl_SplitList(interp, inString, &argc, &argv);
    for (n = 0; n < argc; n++) {
	ptr = argv[n];
	switch (*ptr) {
	case '\n':
	    /* NewLine: Finish current segment */

	    *segPtr = '\0';
            result = Tm_XmStringAppendString(result, segBuf, charSet, dir, True);
	    segPtr = segBuf;
	    segment_started = True;
	    break;
	    
	case TM_FORMAT_CHAR:
            /* Enter command mode */
	    ptr++;

	    if (*ptr == '\0') {
		fprintf(stderr, 
		    "XmString Conversion: Cannot handle trailing 'TM_FORMAT_CHAR'\n");
		return NULL;
	    }

	    if (*ptr == TM_FORMAT_CHAR) {
		/* Two TM_FORMAT_CHAR => leave command mode, 
		   insert one TM_FORMAT_CHAR */
		if (*segPtr != '\0')
		    *segPtr++ = ' ';
		*segPtr++ = TM_FORMAT_CHAR;
	    } else

	    switch (*ptr) {
		case 'n':
	    	    *segPtr = '\0';
            	    result = Tm_XmStringAppendString(result, segBuf, 
					   charSet, dir, True);
	    	    segPtr = segBuf;
	    	    segment_started = True;
		    break;
		case 'F':
		case 'f':
		    /* save string collected so far */
		    *segPtr = '\0';
            	    result = Tm_XmStringAppendString(result, segBuf, 
					   charSet, dir, False);

		    /* now make new font */
		    segPtr = segBuf;
		    ptr++;
		    if (*ptr == '{') {
			/* @f{font-name} */
			ptr++;
			while (*ptr != '}') {
			    *segPtr++ = *ptr++;
			}
			ptr++;
		    } else
		    if (*ptr == '(') {
			/* @fXY - 2 char font name XY */
			ptr++;
			*segPtr++ = *ptr++;
			*segPtr++ = *ptr;
		    } else {
			/* @fX - 1 char font name X */
			*segPtr++ = *ptr;
		    }
		    *segPtr = '\0';
		    /* replace these 3 lines in 2.0 */
		    charSet = XtNewString(segBuf);
		    result = Tm_XmStringAppendString(result, "", 
                                           charSet, dir, False);

		    segPtr = segBuf;
	    	    segment_started = True;
		    *segPtr = '\0';
		    break;
		case 'L':
		case 'l':
	    	    *segPtr = '\0';
		    /* replace these 3 lines in 2.0 */
            	    result = Tm_XmStringAppendString(result, segBuf, 
					   charSet, dir, False);
		    dir = XmSTRING_DIRECTION_L_TO_R;
		    xm_dir = XmStringDirectionCreate(dir);
		    result = Tm_XmStringConcatAndFree(result, xm_dir);

		    segPtr = segBuf;
	    	    segment_started = True;
		    break;
		case 'R':
		case 'r':
	    	    *segPtr = '\0';
		    /* replace these 3 lines in 2.0 */
            	    result = Tm_XmStringAppendString(result, segBuf, 
					   charSet, dir, False);
		    dir = XmSTRING_DIRECTION_R_TO_L;
		    xm_dir = XmStringDirectionCreate(dir);
		    result = Tm_XmStringConcatAndFree(result, xm_dir);

		    segPtr = segBuf;
	    	    segment_started = True;
		    break;
		default:
		    break;
	    }
	    break;
	    	    
	default:
	    /* insert a space if not first */
	    if ( ! segment_started)
		*segPtr++ = ' ';
	    /* and copy string across */
	    while (*ptr)
	        *segPtr++ = *ptr++;
	    segment_started = False;
	    break;

	} /* switch */	    
    } /* for */

    *segPtr = '\0';
    result = Tm_XmStringAppendString(result, segBuf, charSet, dir, False);

    free((char *) argv);
    XtFree(segBuf);
    return result;
}



static Boolean
Tm_CvtStringToXmString(dpy, args, num_args, fromVal, toVal,
                                converter_data)
    Display   *dpy;
    XrmValue  *args;
    Cardinal  *num_args;
    XrmValue  *fromVal;
    XrmValue  *toVal;
    XtPointer *converter_data;
{
    XmString result;
    Tcl_Interp *interp = (Tcl_Interp *) args->addr;


    if ((result = Tm_StringToXmString(interp, (char *)fromVal->addr)) == NULL) {
	XtDisplayStringConversionWarning(dpy, (char *)fromVal->addr, 
					 "XmString");
	return(False);
    }
    
    /* allow this to be garbage collected when possible */
    Tm_AddToFreeResourceList((char *) result,
		(Tm_FreeProc) XmStringFree);

    done(&result, XmString);
    return(True);
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToString --
 *
 *	Converter from String to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;

    buf = *(char **) (fromVal->addr);
#   ifdef DEBUG
    fprintf(stderr, "converted string to %s\n", buf);
#   endif

    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtBooleanToString --
 *
 *	Converter from Boolean to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtBooleanToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Boolean boolVal;
 
    boolVal = *(Boolean *) (fromVal->addr);

    if (boolVal) {
	buf = "true";
    } else {
	buf = "false";
    } 
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtIntToString --
 *
 *	Converter from Int to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtIntToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf = NULL;
    int intVal;
 
    intVal = *(int *) (fromVal->addr);

    if (buf == NULL) {
        buf = XtMalloc(16);
    }
    sprintf(buf, "%d", intVal);
 
    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtUnsignedIntToString --
 *
 *	Converter from UnsignedInt to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtUnsignedIntToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf = NULL;
    unsigned int intVal;
 
    intVal = *(int *) (fromVal->addr);

    if (buf == NULL)
        buf = XtMalloc(16);
    sprintf(buf, "%u", intVal);
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtDimensionToString --
 *
 *	Converter from Dimension to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtDimensionToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf = NULL;
    Dimension dimVal;
 
    dimVal = *(Dimension *) (fromVal->addr);

    if (buf == NULL)
        buf = XtMalloc(16);
    sprintf(buf, "%hu", dimVal);
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtShortToString --
 *
 *	Converter from Short to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtShortToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf = NULL;
    Dimension shortVal;
 
    shortVal = *(short *) (fromVal->addr);

    if (buf == NULL)
        buf = XtMalloc(16);
    sprintf(buf, "%hd", shortVal);
 
    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtLongToString --
 *
 *	Converter from Long to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtLongToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf = NULL;
    Dimension shortVal;
 
    shortVal = *(short *) (fromVal->addr);

    if (buf == NULL)
        buf = XtMalloc(16);
    sprintf(buf, "%ld", shortVal);
 
    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtPixelToString --
 *
 *	Converter from Pixel to String type
 *	gives result in #RGB form
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtPixelToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf = NULL;
    Pixel pixelVal;
    Screen *screen;
    Colormap colormap;
    XColor color;
 
    color.pixel = *(unsigned long *) fromVal->addr;
    screen = *((Screen **) args[0].addr);
    colormap = *((Colormap *) args[1].addr);

    XQueryColor(DisplayOfScreen(screen), colormap, &color);

    if (buf == NULL)
        buf = XtMalloc(32);
    sprintf(buf, "#%04x%04x%04x", color.red, color.green, color.blue);

    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToPixmap --
 *
 *	Converter from String to Pixmap type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToPixmap(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    Widget w = *((Widget *) args[0].addr);

    /* use the Motif registered converter for this! (why reinvent?) */
    if (XtConvertAndStore(w, XmRString, fromVal, 
		XmRPrimForegroundPixmap, toVal)) {
        return True;
    }
    return False;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToFunction --
 *
 *	Converter from String to XmRProc type
 *	(for dropProc in D&D)
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToFunction(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
#ifndef MOTIF11
    char *command;
    extern Tm_Widget *Tm_HackDragWidgetPtr;
    Tm_Widget *wPtr = Tm_HackDragWidgetPtr;
    Widget w = *((Widget *) args[1].addr);
    WidgetClass Class = *((WidgetClass *) args[2].addr);
    static XtConvertSelectionProc p = Tm_ConvertProcHandler;

    /* set our proc in widgets user data */
    command = (char *) (fromVal->addr);
/*
    XtVaGetValues(w, XmNuserData, &wPtr, NULL);
*/
    wPtr->convertProc = XtNewString(command);
    XtVaSetValues(w, XmNuserData, wPtr, NULL);
    
    done(&p, XtConvertSelectionProc);

    return True;
#endif
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToProc --
 *
 *	Converter from String to XmRFunc type
 *	(for convertProc resource in D&D)
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToProc(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
#ifndef MOTIF11
    char *command;
    Tm_Widget *wPtr;
    Widget w = *((Widget *) args[1].addr);
    WidgetClass Class = *((WidgetClass *) args[2].addr);
    static XtCallbackProc p = Tm_DropProcHandler;

   /* note these two hacks because XmRProc
      is not a unique enough type */
    extern Boolean Tm_SettingDropResources;
    extern Boolean Tm_SettingDropProc;
 
    if (Tm_SettingDropResources) {
        /* set our proc in widgets user data */
        command = (char *) (fromVal->addr);
        XtVaGetValues(w, XmNuserData, &wPtr, NULL);
	if (Tm_SettingDropProc) {
            wPtr->dropProc = XtNewString(command);
	} else {
	    return False;
	}
        XtVaSetValues(w, XmNuserData, wPtr, NULL);
    
        done(&p, XtCallbackProc);

        return True;
    }
#endif /* not MOTIF11 */
    return False;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToCallbackProc --
 *
 *	Converter from String to XmRFunc type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToCallbackProc(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
#ifndef MOTIF11
    char *command;
    extern Tm_Widget *Tm_HackWidgetPtr;
    Tm_Widget *wPtr = Tm_HackWidgetPtr;
    Widget w = *((Widget *) args[1].addr);
    WidgetClass Class = *((WidgetClass *) args[2].addr);
    static XtSelectionCallbackProc p = Tm_DropTransferHandler;

/*
    if (Tm_SettingDropTransferResources) {
*/
        /* set our proc in widgets user data */
        command = (char *) (fromVal->addr);
        wPtr->transferProc = XtNewString(command);
    
        done(&p, XtCallbackProc);

        return True;
/*
    }
    return False;
*/
#endif
}


/*
 *--------------------------------------------------------------
 *
 * Tm_RegisterConverters --
 *
 *	Register all extra converters with Xt.
 *
 * Results:
 *
 * Side effects:
 *
 *	Modifies the Xt converter lists.
 *--------------------------------------------------------------
 */

void
Tm_RegisterConverters(interp, appContext)
    Tcl_Interp *interp;
    XtAppContext appContext;
{
    XtConvertArgRec convert_arg;
    XtConvertArgRec convert_args[128];
 
    /*
     * Register the type converters we need
     * This implementation installs everything, 
     * even if it is never used. A smarter method
     * would be to install on demand. Maybe someday...
     */

    InstallReverseRepTypes();

    /* 
     * String to Widget
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XtRWidget,
		Tm_CvtStringToWidget,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /*
     * Motif 1.2.1 MainW.c has bugs in it - resources like menuBar are declared
     * as type Window not Widget. So we have to have another converter just
     * to handle this bug
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XtRWindow,
		Tm_CvtStringToWidget,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * String to MenuWidget
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XmRMenuWidget,
		Tm_CvtStringToWidget,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * Widget to String
     */

    XtSetTypeConverter(
		XtRWidget, XmRString,
		Tm_CvtWidgetToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * Widget to String - this is registered as Window to String because
     * of the buggy resource decls mentioned before
     */

    XtSetTypeConverter(
		XtRWindow, XmRString,
		Tm_CvtWidgetToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * WidgetList to String
     */
    convert_args[0].address_mode = XtWidgetBaseOffset;
    convert_args[0].address_id = (XtPointer) XtOffsetOf(CoreRec, core.self);
    convert_args[0].size = sizeof(CoreWidget);

    XtSetTypeConverter(
		XtRWidgetList, XmRString,
		Tm_CvtWidgetListToString,
		convert_args,
		1,
		XtCacheNone,
		NULL);

#ifndef MOTIF11
    /* 
     * String to DropTransfers
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XmRDropTransfers,
		Tm_CvtStringToDropTransfers,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * String to TransferStatus (as in dropTransfer widget
     */

    XtSetTypeConverter(
		XtRString, XmRTransferStatus,
		Tm_CvtStringToTransferStatus,
		NULL,
		0,
		XtCacheNone,
		NULL);
#endif

    /* 
     * String to Pixmap
     */
    convert_args[0].address_mode = XtWidgetBaseOffset;
    convert_args[0].address_id = (XtPointer) XtOffsetOf(CoreRec, core.self);
    convert_args[0].size = sizeof(CoreWidget);

    XtSetTypeConverter(
		XmRString, XtRPixmap,
		Tm_CvtStringToPixmap,
		convert_args,
		1,
		XtCacheNone,
		NULL);
    /* 
     * XmString to String
     */
    XtSetTypeConverter(
		XmRXmString, XtRString,
		Tm_CvtXmStringToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * XmStringTable to String
     */
    convert_arg.address_mode = XtWidgetBaseOffset;
    convert_arg.address_id = (XtPointer) XtOffsetOf(CoreRec, core.self);
    convert_arg.size = sizeof(CoreWidget);

    XtSetTypeConverter(
		XmRXmStringTable, XtRString,
		Tm_CvtXmStringTableToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * String to XmString
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XmRString, XmRXmString,
		Tm_CvtStringToXmString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /*
     * String to XmStringTable
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XmRString, XmRXmStringTable,
		Tm_CvtStringToXmStringTable,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * String to String
     * I don't know why there isn't already one of these,
     * but getting value out of Text seems to need it.
     *
     * it doesn't seem to work anyway - maybe something
     * special case away X to X convertions
     */
/*
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XtRString,
		Tm_CvtStringToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);
*/

    /* 
     * Int to String
     */
    XtSetTypeConverter(
		XmRInt, XtRString,
		Tm_CvtIntToString,
		NULL,
		0,
		XtCacheNone,
		NULL);


    /* 
     * Cardinal to String
     */
    XtSetTypeConverter(
		XmRCardinal, XtRString,
		Tm_CvtUnsignedIntToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * HorizontalInt to String
     */
    XtSetTypeConverter(
		XmRHorizontalInt, XtRString,
		Tm_CvtIntToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * VerticalInt to String
     */
    XtSetTypeConverter(
		XmRVerticalInt, XtRString,
		Tm_CvtIntToString,
		NULL,
		0,
		XtCacheNone,
		NULL);


    /* 
     * Boolean to String
     */
    XtSetTypeConverter(
		XmRBoolean, XtRString,
		Tm_CvtBooleanToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * Dimension to String
     */
    XtSetTypeConverter(
		XmRDimension, XtRString,
		Tm_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * BooleanDimension to String
     */
    XtSetTypeConverter(
		XmRBooleanDimension, XtRString,
		Tm_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * HorizontalDimension to String
     * Buggy? should use UnitType resource
     */
    XtSetTypeConverter(
		XmRHorizontalDimension, XtRString,
		Tm_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * VerticalDimension to String
     * Buggy? should use UnitType resource
     */

    XtSetTypeConverter(
		XmRVerticalDimension, XtRString,
		Tm_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * short to String
     */
    XtSetTypeConverter(
		XmRShort, XtRString,
		Tm_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * Position to String
     */
    XtSetTypeConverter(
		XmRPosition, XtRString,
		Tm_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * HorizontalPosition to String
     */
    XtSetTypeConverter(
		XmRHorizontalPosition, XtRString,
		Tm_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * VerticalPosition to String
     */
    XtSetTypeConverter(
		XmRVerticalPosition, XtRString,
		Tm_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * TextPosition to String
     */
    XtSetTypeConverter(
		XmRTextPosition, XtRString,
		Tm_CvtLongToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * String to XmProc
     */
    convert_args[0].address_mode = XtImmediate;
    convert_args[0].address_id = (XtPointer) interp;
    convert_args[0].size = sizeof(XtPointer);

    convert_args[1].address_mode = XtWidgetBaseOffset;
    convert_args[1].address_id = (XtPointer) XtOffsetOf(CoreRec, core.self);
    convert_args[1].size = sizeof(CoreWidget);

    convert_args[2].address_mode = XtWidgetBaseOffset;
    convert_args[2].address_id = (XtPointer) XtOffsetOf(CoreRec, core.widget_class);
    convert_args[2].size = sizeof(CoreWidget);

    XtSetTypeConverter(
		XmRString, XmRProc,
		Tm_CvtStringToProc,
		convert_args,
		3,
		XtCacheNone,
		NULL);

    /* 
     * String to XmCallbackProc
     */
    convert_args[0].address_mode = XtImmediate;
    convert_args[0].address_id = (XtPointer) interp;
    convert_args[0].size = sizeof(XtPointer);

    convert_args[1].address_mode = XtWidgetBaseOffset;
    convert_args[1].address_id = (XtPointer) XtOffsetOf(CoreRec, core.self);
    convert_args[1].size = sizeof(CoreWidget);

    convert_args[2].address_mode = XtWidgetBaseOffset;
    convert_args[2].address_id = (XtPointer) XtOffsetOf(CoreRec, core.widget_class);
    convert_args[2].size = sizeof(CoreWidget);

    XtSetTypeConverter(
		XmRString, XmRCallbackProc,
		Tm_CvtStringToCallbackProc,
		convert_args,
		3,
		XtCacheNone,
		NULL);

    /* 
     * String to XmFunc
     */
    convert_args[0].address_mode = XtImmediate;
    convert_args[0].address_id = (XtPointer) interp;
    convert_args[0].size = sizeof(XtPointer);

    convert_args[1].address_mode = XtWidgetBaseOffset;
    convert_args[1].address_id = (XtPointer) XtOffsetOf(CoreRec, core.self);
    convert_args[1].size = sizeof(CoreWidget);

    convert_args[2].address_mode = XtWidgetBaseOffset;
    convert_args[2].address_id = (XtPointer) XtOffsetOf(CoreRec, core.widget_class);
    convert_args[2].size = sizeof(CoreWidget);

    XtSetTypeConverter(
		XmRString, XmRFunction,
		Tm_CvtStringToFunction,
		convert_args,
		3,
		XtCacheNone,
		NULL);

    /* 
     * Pixel to String
     */
    convert_args[0].address_mode = XtWidgetBaseOffset;
    convert_args[0].address_id = (XtPointer) XtOffsetOf(WidgetRec, core.screen);
    convert_args[0].size = sizeof(Screen *);

    convert_args[1].address_mode = XtWidgetBaseOffset;
    convert_args[1].address_id = (XtPointer) XtOffsetOf(CoreRec, core.colormap);
    convert_args[1].size = sizeof(Colormap);

    XtSetTypeConverter(
		XmRPixel, XmRString,
		Tm_CvtPixelToString,
		convert_args,
		2,
		XtCacheNone,
		NULL);
}
