/*
 * tmConverters.c -
 *	This module implements all the Xt Converters to and
 *	from Strings needed by Tm
 *
 * Copyright 1993 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include "tmFuncs.h"
#ifndef MOTIF11
#include <Xm/RepType.h>
#include <Xm/DragDrop.h>
#endif
#include <X11/IntrinsicP.h>
#include <X11/CoreP.h>		/* for CoreRec */
#include <string.h>

#define done(address,type) \
	{	toVal->addr = (XtPointer) address; \
		toVal->size = sizeof(type); \
	}

/*
 * so that we can keep tabs on the size of an XmStringTable
 * see function Tm_MaybeSetXmStringTableSize
 */
int Tm_XmStringTableSize;

/*
 *--------------------------------------------------------------
 *
 * InstallReverseRepTypes --
 *
 *	Install the converters from internal representation
 *	types to their string values.
 *
 * Results:
 *
 * Side effects:
 *	Modifies the Motif RepType tables
 *
 *--------------------------------------------------------------
 */

static void
InstallReverseRepTypes()
{
#ifndef MOTIF11
    XmRepTypeId id;
    XmRepTypeList start, list;
    int n;

    start = list = XmRepTypeGetRegistered();

    while (list->rep_type_name != NULL) {
        if ((id = XmRepTypeGetId(list->rep_type_name)) != XmREP_TYPE_INVALID) {
	    XmRepTypeAddReverse(id);
        } else {
	    fprintf(stderr, "Failed to install %s converter\n", 
			list->rep_type_name);
        }
	list++;
    }
    XtFree((char *)start);
#endif
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToWidget --
 *
 *	Converter from String to Widget type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToWidget(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    Tcl_Interp *interp = (Tcl_Interp *) args->addr;
    Tm_Widget *info;
    static Widget w;

    info = Tm_WidgetInfoFromPath(interp, (char *) fromVal->addr);
    if (info == NULL) {
	return False;
    }
    w = info->widget;

    done(w, Widget);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtWidgetToString --
 *
 *	Converter from Widget to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtWidgetToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    Tm_Widget *info;
    Widget w = *(Widget *) fromVal->addr;
    static char *buf;

    XtVaGetValues(w, XmNuserData, &info, NULL);
    if (info == NULL) {
	return False;
    }

    buf = info->pathName;
    done(&buf, String);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToTransferStatus --
 *
 *	Converter from String to TransferStatus type
 *	(see transferProc widget0
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToTransferStatus(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    char *buf;
    int n;
    static unsigned char status;

    buf = (char *) (fromVal->addr);
    for (n = 0; n < strlen(buf); n++) {
	buf[n] = tolower(buf[n]);
    }
#   ifdef DEBUG
    fprintf(stderr, "converted string to %s\n", buf);
#   endif
    if (strcmp(buf, "transfer_success") == 0) {
	status = 1;
    } else {
	status = 0;
    }

    done(&status, unsigned char);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToDropTransfers--
 *
 *	Converter from String to DropTransfers type
 *	(see transferProc widget0
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToDropTransfers(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
#ifndef MOTIF11
    char *buf;
    char **argv, **argvv;
    int argc, argcc;
    int n;
    static unsigned char status;
    static XmDropTransferEntryRec transferEntries[32];
    static XmDropTransferEntryRec *p;
    Tm_Widget *wPtr;
    Tcl_Interp *interp = (Tcl_Interp *) args->addr;

    buf = (char *) (fromVal->addr);
#   ifdef DEBUG
    fprintf(stderr, "converted string to %s\n", buf);
#   endif
    if (Tcl_SplitList(interp, buf, &argc, &argv) == TCL_ERROR) {
	fprintf(stderr, "failed to convert list\n");
    }
    for (n = 0; n < argc; n++) {
	if (Tcl_SplitList(interp, argv[n], &argcc, &argvv) == TCL_ERROR) {
	    fprintf(stderr, "failed to convert internal list\n");
	}
	if (argcc != 2) {
	    fprintf(stderr, "list should have two elements\n");
	}
	transferEntries[n].target = XInternAtom(display, argvv[0], False);
/*
	wPtr = Tm_WidgetInfoFromPath(interp, argvv[1]);
	transferEntries[n].client_data = wPtr->widget;
*/
	transferEntries[n].client_data = argvv[1];
    }

    p = transferEntries;
    done(&p, XtPointer);

#endif
    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtXmStringToString --
 *
 *	Converter from XmString to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtXmStringToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    XmStringContext context;
    static char *buf;
    char *p, *text;
    XmStringCharSet charset;
    XmStringDirection direction;
    Boolean separator;
    int size = 128;
    int current_len, segment_len;

    XmString xmstr = *(XmString *) fromVal->addr;
    p = buf = XtMalloc(size);
    if (!XmStringInitContext(&context, xmstr)) {
	XtFree(p);
	return False;
    }
    while (XmStringGetNextSegment(context, &text, &charset,
		&direction, &separator)) {
	segment_len = strlen(text);
	if (p + segment_len >= buf + size) {
	    /* string too large, double space */
	    current_len = p - buf;
	    size *= 2;
	    buf = XtRealloc(buf, size);
	    p = buf + current_len;
	}
	strcpy(p, text);
	p += segment_len;
	if (separator == True) {
	    *p++ = '\n';
	    *p = 0;
	}
	XtFree(text);
    }
    XmStringFreeContext(context);
 
    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtXmStringTableToString --
 *
 *	Converter from XmStringTable to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtXmStringTableToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    String *items;
    static String buf;
    int n, len, item_count;
    char *p, *end;
    char *dummy;
    XmString *xmstrs = *(XmString **) fromVal->addr;

    extern Boolean _XmStringIsString _ANSI_ARGS_((XmString));

    len= 0;
    item_count = Tm_XmStringTableSize;
    items = NULL;

    if (item_count != 0) {
        items = (String *) XtMalloc(item_count * sizeof(String));
        for (n = 0; n < item_count; n++) {
            XmStringGetLtoR(xmstrs[n],
    		XmFONTLIST_DEFAULT_TAG, items + n);
            len += strlen(items[n]) + 2; /* for , */
        }
    }
    buf = XtMalloc(len + 1); /* we overshoot by 1 */
    if (len == 0) {
        *buf = '\0';
        done(&buf, char *);
        return True;
    }
    end = buf;
    for (n = 0; n < item_count; n++) {
        p = items[n];
        while (*end++ = *p++)
    	;
        *(end - 1) = ',';
        *end++ = ' ';
        XtFree(items[n]);
    }
    *(end - 2) = '\0';

    done(&buf, char *);

    if (items != NULL) {
	XtFree((char *) items);
    }

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToString --
 *
 *	Converter from String to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;

    buf = *(char **) (fromVal->addr);
#   ifdef DEBUG
    fprintf(stderr, "converted string to %s\n", buf);
#   endif

    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtBooleanToString --
 *
 *	Converter from Boolean to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtBooleanToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Boolean boolVal;
 
    boolVal = *(Boolean *) (fromVal->addr);

    if (boolVal) {
	buf = "true";
    } else {
	buf = "false";
    } 
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtIntToString --
 *
 *	Converter from Int to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtIntToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    int intVal;
 
    intVal = *(int *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%d", intVal);
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtDimensionToString --
 *
 *	Converter from Dimension to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtDimensionToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Dimension dimVal;
 
    dimVal = *(Dimension *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%hu", dimVal);
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtShortToString --
 *
 *	Converter from Short to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtShortToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Dimension shortVal;
 
    shortVal = *(short *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%hd", shortVal);
 
    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtLongToString --
 *
 *	Converter from Long to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtLongToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Dimension shortVal;
 
    shortVal = *(short *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%ld", shortVal);
 
    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtPixelToString --
 *
 *	Converter from Pixel to String type
 *	gives result in #RGB form
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtPixelToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Pixel pixelVal;
    Screen *screen;
    Colormap colormap;
    XColor color;
 
    color.pixel = *(unsigned long *) fromVal->addr;
    screen = *((Screen **) args[0].addr);
    colormap = *((Colormap *) args[1].addr);

    XQueryColor(DisplayOfScreen(screen), colormap, &color);

    /* memory leak here */
    buf = XtMalloc(32);
    sprintf(buf, "#%04x%04x%04x", color.red, color.green, color.blue);

    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToFunction --
 *
 *	Converter from String to XmRProc type
 *	(for dropProc in D&D)
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToFunction(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
#ifndef MOTIF11
    char *command;
    extern Tm_Widget *Tm_HackDragWidgetPtr;
    Tm_Widget *wPtr = Tm_HackDragWidgetPtr;
    Widget w = *((Widget *) args[1].addr);
    WidgetClass class = *((WidgetClass *) args[2].addr);
    static XtConvertSelectionProc p = Tm_ConvertProcHandler;

    /* set our proc in widgets user data */
    command = (char *) (fromVal->addr);
/*
    XtVaGetValues(w, XmNuserData, &wPtr, NULL);
*/
    wPtr->convertProc = XtNewString(command);
    XtVaSetValues(w, XmNuserData, wPtr, NULL);
    
    done(&p, XtConvertSelectionProc);

    return True;
#endif
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToProc --
 *
 *	Converter from String to XmRFunc type
 *	(for convertProc resource in D&D)
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToProc(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
#ifndef MOTIF11
    char *command;
    Tm_Widget *wPtr;
    Widget w = *((Widget *) args[1].addr);
    WidgetClass class = *((WidgetClass *) args[2].addr);
    static XtCallbackProc p = Tm_DropProcHandler;

   /* note these two hacks because XmRProc
      is not a unique enough type */
    extern Boolean Tm_SettingDropResources;
    extern Boolean Tm_SettingDropProc;
 
    if (Tm_SettingDropResources) {
        /* set our proc in widgets user data */
        command = (char *) (fromVal->addr);
        XtVaGetValues(w, XmNuserData, &wPtr, NULL);
	if (Tm_SettingDropProc) {
            wPtr->dropProc = XtNewString(command);
	} else {
	    return False;
	}
        XtVaSetValues(w, XmNuserData, wPtr, NULL);
    
        done(&p, XtCallbackProc);

        return True;
    }
#endif /* not MOTIF11 */
    return False;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToCallbackProc --
 *
 *	Converter from String to XmRFunc type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToCallbackProc(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
#ifndef MOTIF11
    char *command;
    extern Tm_Widget *Tm_HackWidgetPtr;
    Tm_Widget *wPtr = Tm_HackWidgetPtr;
    Widget w = *((Widget *) args[1].addr);
    WidgetClass class = *((WidgetClass *) args[2].addr);
    static XtSelectionCallbackProc p = Tm_DropTransferHandler;

/*
    if (Tm_SettingDropTransferResources) {
*/
        /* set our proc in widgets user data */
        command = (char *) (fromVal->addr);
        wPtr->transferProc = XtNewString(command);
    
        done(&p, XtCallbackProc);

        return True;
/*
    }
    return False;
*/
#endif
}

/*
 *--------------------------------------------------------------
 *
 * Tm_RegisterConverters --
 *
 *	Register all extra converters with Xt.
 *
 * Results:
 *
 * Side effects:
 *
 *	Modifies the Xt converter lists.
 *--------------------------------------------------------------
 */

void
Tm_RegisterConverters(interp, appContext)
    Tcl_Interp *interp;
    XtAppContext appContext;
{
    XtConvertArgRec convert_arg;
    XtConvertArgRec convert_args[128];
 
    /*
     * Register the type converters we need
     * This implementation installs everything, 
     * even if it is never used. A smarter method
     * would be to install on demand. Maybe someday...
     */

    InstallReverseRepTypes();

    /* 
     * String to Widget
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XtRWidget,
		Tm_CvtStringToWidget,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /*
     * Motif 1.2.1 MainW.c has bugs in it - resources like menuBar are declared
     * as type Window not Widget. So we have to have another converter just
     * to handle this bug
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XtRWindow,
		Tm_CvtStringToWidget,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * String to MenuWidget
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XmRMenuWidget,
		Tm_CvtStringToWidget,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * Widget to String
     */

    XtSetTypeConverter(
		XtRWidget, XmRString,
		Tm_CvtWidgetToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * Widget to String - a bug in Form upto and including Motif 1.2.0
       used a Window instead of a Widget in topWidget, etc, resources
     */

    XtSetTypeConverter(
		XtRWindow, XmRString,
		Tm_CvtWidgetToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

#ifndef MOTIF11
    /* 
     * String to DropTransfers
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XmRDropTransfers,
		Tm_CvtStringToDropTransfers,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * String to TransferStatus (as in dropTransfer widget
     */

    XtSetTypeConverter(
		XtRString, XmRTransferStatus,
		Tm_CvtStringToTransferStatus,
		NULL,
		0,
		XtCacheNone,
		NULL);
#endif

    /* 
     * XmString to String
     */
    XtSetTypeConverter(
		XmRXmString, XtRString,
		Tm_CvtXmStringToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * XmStringTable to String
     */
    XtSetTypeConverter(
		XmRXmStringTable, XtRString,
		Tm_CvtXmStringTableToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * String to String
     * I don't know why there isn't already one of these,
     * but getting value out of Text seems to need it.
     *
     * it doesn't seem to work anyway - maybe something
     * special case away X to X convertions
     */
/*
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XtRString,
		Tm_CvtStringToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);
*/

    /* 
     * Int to String
     */
    XtSetTypeConverter(
		XmRInt, XtRString,
		Tm_CvtIntToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * HorizontalInt to String
     */
    XtSetTypeConverter(
		XmRHorizontalInt, XtRString,
		Tm_CvtIntToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * VerticalInt to String
     */
    XtSetTypeConverter(
		XmRVerticalInt, XtRString,
		Tm_CvtIntToString,
		NULL,
		0,
		XtCacheNone,
		NULL);


    /* 
     * Boolean to String
     */
    XtSetTypeConverter(
		XmRBoolean, XtRString,
		Tm_CvtBooleanToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * Dimension to String
     */
    XtSetTypeConverter(
		XmRDimension, XtRString,
		Tm_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * BooleanDimension to String
     */
    XtSetTypeConverter(
		XmRBooleanDimension, XtRString,
		Tm_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * HorizontalDimension to String
     * Buggy? should use UnitType resource
     */
    XtSetTypeConverter(
		XmRHorizontalDimension, XtRString,
		Tm_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * VerticalDimension to String
     * Buggy? should use UnitType resource
     */

    XtSetTypeConverter(
		XmRVerticalDimension, XtRString,
		Tm_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * short to String
     */
    XtSetTypeConverter(
		XmRShort, XtRString,
		Tm_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * Position to String
     */
    XtSetTypeConverter(
		XmRPosition, XtRString,
		Tm_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * HorizontalPosition to String
     */
    XtSetTypeConverter(
		XmRHorizontalPosition, XtRString,
		Tm_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * VerticalPosition to String
     */
    XtSetTypeConverter(
		XmRVerticalPosition, XtRString,
		Tm_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * TextPosition to String
     */
    XtSetTypeConverter(
		XmRTextPosition, XtRString,
		Tm_CvtLongToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * String to XmProc
     */
    convert_args[0].address_mode = XtImmediate;
    convert_args[0].address_id = (XtPointer) interp;
    convert_args[0].size = sizeof(XtPointer);

    convert_args[1].address_mode = XtWidgetBaseOffset;
    convert_args[1].address_id = (XtPointer) XtOffsetOf(CoreRec, core.self);
    convert_args[1].size = sizeof(CoreWidget);

    convert_args[2].address_mode = XtWidgetBaseOffset;
    convert_args[2].address_id = (XtPointer) XtOffsetOf(CoreRec, core.widget_class);
    convert_args[2].size = sizeof(CoreWidget);

    XtSetTypeConverter(
		XmRString, XmRProc,
		Tm_CvtStringToProc,
		convert_args,
		3,
		XtCacheNone,
		NULL);

    /* 
     * String to XmCallbackProc
     */
    convert_args[0].address_mode = XtImmediate;
    convert_args[0].address_id = (XtPointer) interp;
    convert_args[0].size = sizeof(XtPointer);

    convert_args[1].address_mode = XtWidgetBaseOffset;
    convert_args[1].address_id = (XtPointer) XtOffsetOf(CoreRec, core.self);
    convert_args[1].size = sizeof(CoreWidget);

    convert_args[2].address_mode = XtWidgetBaseOffset;
    convert_args[2].address_id = (XtPointer) XtOffsetOf(CoreRec, core.widget_class);
    convert_args[2].size = sizeof(CoreWidget);

    XtSetTypeConverter(
		XmRString, XmRCallbackProc,
		Tm_CvtStringToCallbackProc,
		convert_args,
		3,
		XtCacheNone,
		NULL);

    /* 
     * String to XmFunc
     */
    convert_args[0].address_mode = XtImmediate;
    convert_args[0].address_id = (XtPointer) interp;
    convert_args[0].size = sizeof(XtPointer);

    convert_args[1].address_mode = XtWidgetBaseOffset;
    convert_args[1].address_id = (XtPointer) XtOffsetOf(CoreRec, core.self);
    convert_args[1].size = sizeof(CoreWidget);

    convert_args[2].address_mode = XtWidgetBaseOffset;
    convert_args[2].address_id = (XtPointer) XtOffsetOf(CoreRec, core.widget_class);
    convert_args[2].size = sizeof(CoreWidget);

    XtSetTypeConverter(
		XmRString, XmRFunction,
		Tm_CvtStringToFunction,
		convert_args,
		3,
		XtCacheNone,
		NULL);

    /* 
     * Pixel to String
     */
    convert_args[0].address_mode = XtWidgetBaseOffset;
    convert_args[0].address_id = (XtPointer) XtOffsetOf(WidgetRec, core.screen);
    convert_args[0].size = sizeof(Screen *);

    convert_args[1].address_mode = XtWidgetBaseOffset;
    convert_args[1].address_id = (XtPointer) XtOffsetOf(CoreRec, core.colormap);
    convert_args[1].size = sizeof(Colormap);

    XtSetTypeConverter(
		XmRPixel, XmRString,
		Tm_CvtPixelToString,
		convert_args,
		2,
		XtCacheNone,
		NULL);
}
