#
# Open Registration Client
# ------------------------
#
# This module presents one possible Registrar Client (see rc.tcl for 
# information on Registrar Clients in general).  Within an open Registrar
# Client, any user can create a conference, and users can join any 
# conference.  Users can only delete themselves from the conference,
# and the last user leaving a conference closes down the entire conference
# (default behavior for Registrar Clients; see the userLeft proc in rc.tcl).
#
# There are two parts to this module, the underlying protocol part, and a
# Tk interface to interact with the protocol.
#

initGenericRC

PollConferences

#
# This routine allows a user to join a conference and is called in 
#    response to double clicking an existing conference in the interface
#
# This routine tells the Registrar we want to join the conference, and then
# instructs our conference (which hasn't yet been created; the joinTo proc
# calls toConf which will queue the messages until it is created) to join to
# join to the existing users of the conference.  The global list "confs_joined"
# keeps track of conferences we've already joined, so as not to join them
# twice.
#
# Note: we no longer join existing conferences; this is moved to foundNewUser
#

set confs_joined ""

proc doJoin whichconf {     global conflist userlist confs_joined
    set conf [lindex $conflist $whichconf]
    if {[lsearch $confs_joined [keylget conf confnum]] == -1} {
	lappend confs_joined [keylget conf confnum]
	callJoinConference [keylget conf confnum]
	after 100 PollUsers [keylget conf confnum]
    }
}


#
# This is called by updatelist when a new conference is found.  Add the
# conference to the global conflist.  If we created the conference, as
# specified by the originator key, tell the Registrar we'd like to join
# to it; under this policy we automatically join to conferences we create
#

proc foundNewConf conf {     global conflist host myport userlist
    PollUsers [keylget conf confnum]
    lappend conflist $conf
    set userlist([keylget conf confnum]) {}
    if {[keylget conf originator] == [concat $host$myport]} {
	callJoinConference [keylget conf confnum]
	PollUsers [keylget conf confnum]
    }
}


#
# This is called by updatelist when a new user is found.  Add the user to 
# the appropriate user list.  If the new user is in fact ourself, create
# the conference application.
#
# Note: when we get our own entry, we also join to other users in the conference.
#   Because joins are essentially serialized through the registrar, after we
#   receive our own "newuser" announcement, there will not be any announcements
#   for "previously joined" users.
#

proc foundNewUser user {    global host myport userlist
    set confnum [keylget user confnum]
    lappend userlist($confnum) $user
    if {([keylget user host] == $host) && ([keylget user port] == $myport)} {
	createConference [findconfinfo $confnum] [keylget user usernum]
	foreach i $userlist($confnum) {
	    if {[keylget i usernum] < [keylget user usernum]} {
		joinTo $i $user
	    }
	}
    }
}


#
# This is called by updatelist when a deleted user is found.  Delete the
# user from the list and tell the conference to remove the user which may
# even be ourself (because under the protocol we obey all delete requests).
# If the user is ourself, delete the conference from the confs_joined list
# so we can join it again if desired.
#

proc foundDeletedUser {conf user} {  global userlist confs_joined
    set idx 0; while {$idx < [llength $userlist($conf)]} {
	set i [lindex $userlist($conf) $idx]
	if {[keylget i usernum] == $user} {
	    catch {toConf $conf "removeUser [list [lindex $userlist($conf) $idx]]"}
	    set userlist($conf) [lreplace $userlist($conf) $idx $idx]
	} else { incr idx }
    }
    set c ""; catch {set c [findconf $conf]}
    if {$c != ""} {
	set unum [keylget c usernum]
	if {$unum == $user} {
	    set idx [lsearch $confs_joined $conf]
	    if {$idx != -1} { 
		set confs_joined [lreplace $confs_joined $idx $idx]
	    }
	}
    }
    updateuserlist blah
}


#
# This is called by updatelist when a deleted conference is found.  Delete
# it from the list and tell the conference to delete itself (if we are
# running the conference).
#

proc foundDeletedConf conf { global conflist userlist
    set idx 0; foreach i $conflist {
	if {[keylget i confnum] == $conf} {
	    set conflist [lreplace $conflist $idx $idx]
	    set userlist($conf) ""
	    toConf $conf "deleteconf"
	} else { incr idx }
    }
}


#
# The remainder of this module specifies the Tk interface for the Open
# Registrar Client.  
#
# This first part specifies the main window.  It contains two scrollable
# lists, one containing a list of all the conferences (their names actually)
# and the second a list of users for a particular conference (selected in
# the first list).  Double clicking on a conference allows you to join
# the conference.  A 'new' button pops up a dialog box allowing you to
# create new conferences.
#

pack append [frame .buttons -borderwidth 10] \
    [button    .buttons.new -text New -command "newConfDlg"] {left expand}\

pack append [frame .conflist] \
    [label     .conflist.label -text Conferences] top \
    [scrollbar .conflist.scroll -relief sunken \
	       -command ".conflist.list yview"] {right filly} \
    [listbox   .conflist.list -yscroll ".conflist.scroll set" \
	       -relief sunken -exportselection false -setgrid 1] {left expand fill}

pack append [frame .userlist] \
    [label     .userlist.label -text Users] top \
    [scrollbar .userlist.scroll -relief sunken  \
               -command ".userlist.list yview"] {right filly} \
    [listbox   .userlist.list -yscroll ".userlist.scroll set" \
               -relief sunken -exportselection false -setgrid 1] {left expand fill}

gkDefaultRCMenu .menu

pack append . \
    .menu {top fillx} \
    .buttons {bottom expand fill} \
    .conflist left \
    .userlist right

bind .conflist.list <1> {%W select from [%W nearest %y]; updateuserlist blat}
bind .conflist.list <Double-1> {doJoin [.conflist.list curselection]}
tk_listboxSingleSelect .conflist.list


#
# The interface monitors both the userlist and the conflist, and when they
# change calls routines to update the interface.
#

trace variable conflist w updateConfList 
trace variable userlist w updateuserlist


#
# When the conference list changes, rebuild the entire conferences listbox.
# Not overly efficient, but hey, it works.
#

proc updateConfList args {
    global conflist .conflist.list
    .conflist.list delete 0 end
    foreach i $conflist {
	set blah $i
	.conflist.list insert end [keylget blah confname]
    }
    return
}


#
# When the user list changes, update the user listbox.  This listbox displays
# the users of the conference selected in the conferences listbox.
#

proc updateuserlist args {     global conflist userlist
    set sel [.conflist.list curselection]
    .userlist.list delete 0 end
    if {$sel != ""} {
	set conf [lindex $conflist $sel]
	set confnum [keylget conf confnum]
	foreach i $userlist($confnum) {
	    set blah $i
	    .userlist.list insert end [keylget blah userid]
	}
    }
    return
}


#
# Pop up a dialog box allowing the user to create a new conference.  The
# box prompts for the name of the dialog box as well as the type of conference
# to be created.  The list of conference types is obtained from the .Xdefaults
# file (see the documentation for proc createConference in rc.tcl for the
# format of the particular resources used).
#
# Changed to support the gk_program tcl variable rather than the
# Xdefaults form.
#

proc newConfDlg {} {    global confType confName gk_program
    toplevel .dlg
    wm title .dlg "New Conference"
    set confName "Conference name"

    pack append [frame .dlg.buttons] \
	[button .dlg.buttons.cancel -text Cancel \
	        -command "destroy .dlg"] right \
	[button .dlg.buttons.ok -text Ok -command doNewConf] right

    frame .dlg.types -borderwidth 20
    
    set j 1; foreach i [array names gk_program] {
	if {$j==1} { set confType [set i] }
	pack append .dlg.types \
	    [radiobutton .dlg.types.r$j -text [set i] -variable confType -value [set i]] \
	    {top expand pady 5 fillx frame w}
	incr j
    }

#    set i 1; while {$i <= [option get . conferenceTypes conferenceTypes]} {
#	pack append .dlg.types \
#	    [radiobutton .dlg.types.r$i \
#	                 -text [option get . conf${i}-desc blah] \
#		         -variable confType -value $i] \
#	            {top expand pady 5 fillx frame w}
#	incr i
#    }

    pack append .dlg \
	[label .dlg.lbl -text "Conference name"] top \
	[entry .dlg.name -relief sunken -textvariable confName] \
	           {top pady 10 fillx} \
	.dlg.buttons {bottom fillx pady 20} \
	.dlg.types top
}


#
# This command is called in response to pressing the "Ok" button in the
# new conferences dialog box.  Create the desired conference.
#

proc doNewConf {} {	global confName confType host myport
    keylset conf confname $confName confType $confType originator $host$myport

    if {[string compare $confName "Conference name"] == 0} {
	keylset conf confname [keylget conf confType]}
    callNewConference $conf
    PollConferences
    destroy .dlg
}

