# Title: Voting Booth
# Author: David Marwood
# Date: May 13, 1993
# Description: Creates a conference that allows the conference members
# to propose questions to be voted on.  A window is displayed to
# each conference member asking them to vote "yes" or "no" on the
# question.  Votes are tallied and displayed as the votes come in.

#source "conf.tcl"

initConf $argv



# Broadcasts the question (calls NewQuestion) in .newQuestion to 
# all applications in this conference (incl this one).
# It generate a unique ID based on the usernum and NextQuestNum.  
# The text in .newQuestion is cleared.
proc SendQuestion {} {
    global NextQuestNum
    global ouropts
    global others

    set QuestionID q[keylget ouropts usernum]n$NextQuestNum
    incr NextQuestNum
    NewQuestion $QuestionID [.newQuestion get 0.0 end] [expr [llength $others]+1]
    Broadcast "NewQuestion $QuestionID \"[.newQuestion get 0.0 end]\" [expr [llength $others]+1]"
    .newQuestion delete 0.0 end
}


# Called when a new question is created, either by this or another 
# conference application.  Creates a voter's window for the question
# and allows the user to select yes or no.  QuestionID must be a name
# starting with a lower case alpha and unique to all questions.
# QuestionText is the text of the question itself, e.g. "Should we 
# go out for dinner?"  NumVoters is the number of votes expected.  
# It initializes $QuestionID.undecidedVotes.
# Three global variables are created and used to keep track of the
# votes: $QuestionID.yesVotes, $QuestionID.noVotes, and 
# $QuestionID.undecidedVotes.
proc NewQuestion {QuestionID QuestionText NumVoters} {
    global Questions
    global NextQuestNum
    global $QuestionID.yesVotes
    global $QuestionID.noVotes
    global $QuestionID.undecidedVotes
    set $QuestionID.yesVotes 0
    set $QuestionID.noVotes 0
    set $QuestionID.undecidedVotes $NumVoters

    # Set Title: "Voter's Question"
    toplevel .$QuestionID
    wm title .$QuestionID "Voter's Question"

    # Create question title and pack at top.
    label .$QuestionID.qtitle -text "Question:" -font "*-times-bold-r-normal--*-180-*"
    pack append .$QuestionID .$QuestionID.qtitle top

    # Create window for question and pack at top.
    message .$QuestionID.question -text $QuestionText -width 10c
    pack append .$QuestionID .$QuestionID.question top
    
    # Create results title and pack at top.
    label .$QuestionID.rtitle -text "Results:" -font "*-times-bold-r-normal--*-180-*"
    pack append .$QuestionID .$QuestionID.rtitle top

    # Create labels and such to display the results in and pack at top. 
    frame .$QuestionID.values
    label .$QuestionID.values.yesVotesTitle -text "Yes:"
    label .$QuestionID.values.yesVotes -textvariable $QuestionID.yesVotes
    label .$QuestionID.values.noVotesTitle -text "No:"
    label .$QuestionID.values.noVotes -textvariable $QuestionID.noVotes
    label .$QuestionID.values.undecidedVotesTitle -text "Undecided:"
    label .$QuestionID.values.undecidedVotes -textvariable $QuestionID.undecidedVotes
    pack append .$QuestionID.values .$QuestionID.values.yesVotesTitle left .$QuestionID.values.yesVotes left .$QuestionID.values.noVotesTitle left .$QuestionID.values.noVotes left .$QuestionID.values.undecidedVotesTitle left .$QuestionID.values.undecidedVotes left
    pack append .$QuestionID .$QuestionID.values top
    
    # Create title for voting buttons and pack at top.
    label .$QuestionID.vtitle -text "Your Vote:" -font "*-times-bold-r-normal--*-180-*"
    pack append .$QuestionID .$QuestionID.vtitle top

    # Create "Yes" and "no" buttons and pack at top.
    frame .$QuestionID.buttons
    button .$QuestionID.buttons.yes -text Yes -command "HandleResponse $QuestionID Yes"
    button .$QuestionID.buttons.no -text No -command "HandleResponse $QuestionID No"
    pack append .$QuestionID.buttons .$QuestionID.buttons.yes left .$QuestionID.buttons.no right
    pack append .$QuestionID .$QuestionID.buttons top

    # Add Question ID and number of voters to Questions list.
    keylset Questions $QuestionID $NumVoters
}


# Broadcasts a response (calls NewResponse) to question $QuestionID 
# to all applications (incl this one).  Changes the "yes" and "no" 
# buttons to a label, either "Yes" or "No", according to Response.
proc HandleResponse {QuestionID Response} {
    NewResponse $QuestionID $Response
    Broadcast "NewResponse $QuestionID $Response"
    pack unpack .$QuestionID.buttons
    pack append .$QuestionID [label .$QuestionID.thisResult -text $Response] top
}


# Called when a new response to an existing question is recieved, 
# either locally or externally.
# QuestionID is the ID of the question being responded to.
# Response is either "Yes" or "No"
proc NewResponse {QuestionID Response} {
    global Questions
    global $QuestionID.yesVotes
    global $QuestionID.noVotes
    global $QuestionID.undecidedVotes
    incr $QuestionID.undecidedVotes -1

    if {[string compare $Response Yes] == 0} then {
	incr $QuestionID.yesVotes
    } else {
	incr $QuestionID.noVotes
    }
}


# Broadcast simply send the message Message to all the other 
# applications in this conference ($others).
proc Broadcast {Message} {
    global others
    foreach i $others {
	RDO [keylget i filedesc] eval $Message
    }
}


# Initialization code.
# Creates the main "Question Creation" window and allows new questions 
# to be created. 
wm title . "Question Creation"
label .newQuestionTitle -text "Question"  -font "*-times-bold-r-normal--*-180-*" 
text .newQuestion  -width 40 -height 15 -relief sunken -wrap word
button .sendQuestion -text "Create Question" -command {SendQuestion}
pack append . .newQuestionTitle top .newQuestionTitle fillx .newQuestion top .sendQuestion top .sendQuestion fillx

# Questions is a keyed list (keylget, keylset, etc) of question ID's
# and the number of voters for that question.
set Questions {}

# Each time a question is sent out, NextQuestNum is set to a new unique
# number (incremented, in this case).
set NextQuestNum 0
