/*
 *-----------------------------------------------------------------------------
 *  Copyright (c) 1993-1995 European Synchrotron Radiation Facility
 *
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the "Terms and Conditions of Distribution", given in the documentation
 *  to this software, are applicable and the above copyright message appears
 *  in each copy.
 *
 *-----------------------------------------------------------------------------
 *
 *
 *  CTAXT:      Combine Tcl/Tk with arbitrary X Toolkits 
 *              (into a single application)
 *
 *  Module:	ctaxtWin.c
 *
 *  Purpose:    Functions for window creation
 *
 *  22.10.1993
 * -25.10.1993	ds/hp	first version
 *  26.10.1993  hp      CTAXT_CreateTopLevelWindow ():
 *                      Calling Tk_CreateMainWindow for initialization of
 *                      the Tk-Stuff but causing it to produce an error before
 *                      any windows are created.
 *  27.10.1993  hp      CTAXT_CreateMainWindow ():
 *                      Saving the XContext before exiting the function
 *                      Calling TkWmMapWindow for initialization of the Tk-
 *                      Stuff for the WindowManager
 *  28.10.1993  hp      Got rid of call to TkWmMapWindow since it slowed down
 *                      the initialization-phase a lot and didn't seem to have
 *                      any useful effects
 *  03.11.1993  hp      since Tk considers its main-window to be the
 *                      application-main-window we have to compute the absolute
 *                      x- and y-coordinates of the window
 *  04.11.1993  hp      Positioning and Resizing is now done by the Generic-
 *                      Event-Handler, so there's no more need for explicit
 *                      Call-Back-Functions for Resizing and therefore no more
 *                      need for using the Motif-DrawingArea as the placeholder
 *                      for the Tk-Main-Window.
 *                      Functions for Explicit-Event-Handling
 *  08.11.1993  hp      Hiding the Tcl-related stuff
 *                      Preparing the Interface for handling multiple Tk-Main-
 *                      Windows and multiple Tcl-Interpreters
 *  10.11.1993  hp      Put in some debug-outputs (conditional compilation)
 *                      Made the Interface independent from the Xt-Library by
 *                      removing the XtDispatch()-Call. If application
 *                      designers want their part of the application
 *                      to response to user interaction (or even to appear on
 *                      the screen) they have to install an explicit event
 *                      handler and dispatch the events their by calling their
 *                      favorite event dispatcher
 *  15.11.1993  hp      Mechanism for adding and freeing extensions to a Tcl 
 *                      interpreter
 *  29.11.1993  hp      changes to make compatible to Tk3.6
 *  06.12.1993  hp      changes to make the TK_LIBRARY define auto-configurable
 *  09.12.1993  hp      changed CTAXT_GetScreen(), so that it doesn't allocate
 *                      a new display structure, if there already exists one
 *                      for the given display.
 *  10.12.1993  hp      changed the generic event handlers to keep track of the
 *                      focus
 *                      Moved all the stuff related to the generic event 
 *                      handlers into a new module
 *  21.12.1993  hp      changed the returned error messages to support the new
 *                      result protocol in ctaxt.c
 *                      fixed a bug in the call to Tk_CreateMainWindow(), the
 *                      function was called with one more argument than 
 *                      necessary (which didn't cause a problem)
 *
 *
 *
 *  inherited from
 *
 *
 * tkWindow.c --
 *
 *	This file provides basic window-manipulation procedures,
 *	which are equivalent to procedures in Xlib (and even
 *	invoke them) but also maintain the local Tk_Window
 *	structure.
 *
 * Copyright (c) 1989-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */




#include <tkConfig.h>
#include <tkInt.h>
#include <patchlevel.h>
#include "ctaxtInt.h"




/*
 *  For autoconfiguration of the TK_LIBRARY define
 */

#ifdef CTAXTX
#define TK_LIBRARY TKX_LIB
#endif
#ifdef CTAXT
#define TK_LIBRARY TK_LIB
#endif




/*
 * Count of number of main windows currently open in this process.
 */

extern int tk_NumMainWindows;


/*
 * First in list of all main windows managed by this process.
 */

extern TkMainInfo *tkMainWindowList;


/*
 * List of all displays currently in use.
 */

extern TkDisplay *tkDisplayList;


/*
 * Context information used to map from X window id's to
 * TkWindow structures (during event handling, for example):
 */

extern XContext tkWindowContext;


/*
 * The variables below hold several uid's that are used in many places
 * in the toolkit.
 */

extern Tk_Uid tkDisabledUid;
extern Tk_Uid tkActiveUid;
extern Tk_Uid tkNormalUid;


/*
 * Default values for "changes" and "atts" fields of TkWindows.  Note
 * that Tk always requests all events for all windows, except StructureNotify
 * events on internal windows:  these events are generated internally.
 */

static XWindowChanges defChanges = {
    0, 0, 1, 1, 0, 0, Above
};
#define ALL_EVENTS_MASK \
    KeyPressMask|KeyReleaseMask|ButtonPressMask|ButtonReleaseMask| \
    EnterWindowMask|LeaveWindowMask|PointerMotionMask|ExposureMask| \
    VisibilityChangeMask|SubstructureNotifyMask| \
    FocusChangeMask|PropertyChangeMask|ColormapChangeMask
static XSetWindowAttributes defAtts= {
    None,			/* background_pixmap */
    0,				/* background_pixel */
    CopyFromParent,		/* border_pixmap */
    0,				/* border_pixel */
    ForgetGravity,		/* bit_gravity */
    NorthWestGravity,		/* win_gravity */
    NotUseful,			/* backing_store */
    ~0,				/* backing_planes */
    0,				/* backing_pixel */
    False,			/* save_under */
    ALL_EVENTS_MASK,		/* event_mask */
    0,				/* do_not_propagate_mask */
    False,			/* override_redirect */
    CopyFromParent,		/* colormap */
    None			/* cursor */
};


/*
 * The following structure defines all of the commands supported by
 * Tk, and the C procedures that execute them.
 */

typedef struct {
    char *name;			/* Name of command. */
    int (*cmdProc) _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp,
	    int argc, char **argv));
				/* Command procedure. */
} TkCmd;

extern TkCmd commands[];




/*
 *  Protoypes for Forward-References
 */

static Tk_Window CTAXT_CreateTopLevelWindow _ANSI_ARGS_((Tcl_Interp *interp,
							 Display *display,
							 Window window));
static TkDisplay *CTAXT_GetScreen _ANSI_ARGS_((Tcl_Interp *interp,
					       Display *display,
					       int *screenPtr));
static TkWindow *CTAXT_NewWindow _ANSI_ARGS_ ((TkDisplay *dispPtr,
					       int screenNum,
					       Window window));
int CTAXT_MoveResizeMainWindow _ANSI_ARGS_ ((Tk_Window *tkwin));




/*
 *----------------------------------------------------------------------
 *
 * CTAXT_CreateTopLevelWindow --
 *
 *      Create a TK-internal-Window and map it to an existing X-Window.
 *      Tk considers this window to be a toplevel-window.
 *
 * Results:
 *	The return value is a token for the new window, or NULL if
 *	an error prevented the new window from being created.  If
 *	NULL is returned, an error message will be left in
 *	interp->result.
 *
 * Side effects:
 *	A new window structure is allocated locally.  An X
 *	window is NOT initially created, and will not be created
 *	since it should already exist and be mapped.
 *
 *----------------------------------------------------------------------
 */

static Tk_Window
CTAXT_CreateTopLevelWindow(interp, display, window)
    Tcl_Interp *interp;		/* Interpreter to use for error reporting. */
    Display *display;           /* X-Display-Information of already existing
				 * window */
    Window window;              /* X-Window-ID of already existing window */
{
    Tk_Window tkwin;
    register TkWindow *winPtr;
    register TkDisplay *dispPtr;
    int screenId;


    if (tk_NumMainWindows == 0) {

      /*
       *  Call Tk_CreateMainWindow (TK) once so that TK gets initialized
       *  Make sure, that this function will exit with an error-message
       *  before (!!!) creating any windows. This can in the current version
       *  Tk3.3b3 be done by calling the function with a NULL-ScreenName-
       *  Pointer and an Environment not containing a DISPLAY-Entry.
       */

       static char buf[1024];
       char        *p;
       Tk_Window   result;


       p = getenv ("DISPLAY");
       if (p != NULL) {
	 putenv ("DISPLAY=");
       }
#if CTAXT_DEBUG_LEVEL > 1
       fprintf (stderr,"\nBefore Tk_CreateMainWindow");
#endif
       result = Tk_CreateMainWindow (interp,NULL,"","Toplevel");
#if CTAXT_DEBUG_LEVEL > 1
       fprintf (stderr,"\nAfter Tk_CreateMainWindow");
#endif
       if (p != NULL) {
	 sprintf (buf,"DISPLAY=%s",p);
	 putenv (buf);
       }
       Tcl_ResetResult (interp);
       if (result != NULL) {
	 Tcl_AppendResult (interp->result,
			   "FCould not cause Tk_CreateMainWindow() to exit "
			   "with an error in CTAXT_CreateTopLevelWindow()!",
			   NULL);
	 return ((Tk_Window) NULL);
       }
/*
 *  26.10.1993  hp   The following is now done by Tk_CreateMainWindow ()
 *
 *	tkWindowContext = XUniqueContext();
 *	tkActiveUid = Tk_GetUid("active");
 *	tkDisabledUid = Tk_GetUid("disabled");
 *	tkNormalUid = Tk_GetUid("normal");
 */
    }


    /*  create the Tk-Structures for the display-information  */

#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nBefore CTAXT_GetScreen");
#endif
    dispPtr = CTAXT_GetScreen(interp, display, &screenId);
#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nAfter CTAXT_GetScreen");
#endif
    if (dispPtr == NULL) {
      return (Tk_Window) NULL;
    }


    /*  create the new window  */

#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nBefore CTAXT_NewWindow");
#endif
    winPtr = CTAXT_NewWindow(dispPtr, screenId, window);
#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nAfter CTAXT_NewWindow");
#endif


    /*
     * Internal windows don't normally ask for StructureNotify events,
     * since we can generate them internally.  However, for top-level
     * windows we need to as for the events because the window could
     * be manipulated externally.
     */

    winPtr->atts.event_mask |= StructureNotifyMask;


    /*
     * (Need to set the TK_TOP_LEVEL flag immediately here;  otherwise
     * Tk_DestroyWindow will core dump if it is called before the flag
     * has been set.)
     */

    winPtr->flags |= TK_TOP_LEVEL;
#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nBefore TkWmNewWindow");
#endif
    TkWmNewWindow(winPtr);
#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nAfter TkWmNewWindow");
#endif


    return (Tk_Window) winPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * CTAXT_GetScreen --
 *
 *      Create a Tk-Display-Structure for an already opened Display
 *
 * Results:
 *	The return value is a pointer to information about the display,
 *	or NULL if the display couldn't be opened.  In this case, an
 *	error message is left in interp->result.  The location at
 *	*screenPtr is overwritten with the screen number parsed from
 *	screenName.
 *
 * Side effects:
 *	A new TkDisplay data structure is also setup, if necessary.
 *
 *----------------------------------------------------------------------
 */

static TkDisplay *
CTAXT_GetScreen(interp, display, screenPtr)
    Tcl_Interp *interp;		/* Place to leave error message. */
    Display *display;           /* X-Display-Information about the already
				 * opened Display-Connection */
    int *screenPtr;		/* Where to store screen number. */
{
    register TkDisplay *dispPtr;
    char *p,*screenName;
    int length, screenId, i;


    /*
     * Separate the screen number from the rest of the display
     * name.  ScreenName is assumed to have the syntax
     * <display>.<screen> with the dot and the screen being
     * optional.
     */

    screenName = getenv("DISPLAY");
    if (screenName == NULL) {
      interp->result =
	"Fno display name and no $DISPLAY environment variable";
      return (TkDisplay *) NULL;
    }
    length = strlen(screenName);
    screenId = 0;
    p = screenName+length-1;
    while (isdigit(UCHAR(*p)) && (p != screenName)) {
	p--;
    }
    if ((*p == '.') && (p[1] != '\0')) {
	length = p - screenName;
	screenId = strtoul(p+1, (char **) NULL, 10);
    }

    /*
     * See if we already have a connection to this display.  If not,
     * then open a new connection.
     */

    for (dispPtr = tkDisplayList; ;dispPtr = dispPtr->nextPtr) {
      if (dispPtr == NULL) {

	/*  create the Tk-Display-Structure  */

	dispPtr = (TkDisplay *) ckalloc(sizeof(TkDisplay));
	dispPtr->display = display;
	dispPtr->nextPtr = tkDisplayList;
	dispPtr->name = (char *) ckalloc((unsigned) (length+1));
	dispPtr->lastEventTime = CurrentTime;
	strncpy(dispPtr->name, screenName, length);
	dispPtr->focusTopLevelPtr = NULL;
	dispPtr->focussedOnEnter = 0;
	dispPtr->name[length] = '\0';
	dispPtr->bindInfoStale = 1;
	dispPtr->numModKeyCodes = 0;
	dispPtr->modKeyCodes = NULL;
	dispPtr->errorPtr = NULL;
	dispPtr->deleteCount = 0;
	dispPtr->defaultHandler = NULL;
	dispPtr->commWindow = NULL;
	dispPtr->serverSecure = 0;
	dispPtr->selectionOwner = NULL;
	dispPtr->selectionSerial = 0;
	dispPtr->multipleAtom = None;
	dispPtr->atomInit = 0;
	dispPtr->cursorFont = None;
	dispPtr->grabWinPtr = NULL;
	dispPtr->eventualGrabWinPtr = NULL;
	dispPtr->buttonWinPtr = NULL;
	dispPtr->serverWinPtr = NULL;
	dispPtr->firstGrabEventPtr = NULL;
	dispPtr->lastGrabEventPtr = NULL;
	dispPtr->grabFlags = 0;
	dispPtr->colorModels = (Tk_ColorModel *) ckalloc((unsigned)
		(ScreenCount(display)*sizeof(Tk_ColorModel)));
	for (i = ScreenCount(display)-1; i >= 0; i--) {
	    if (DisplayPlanes(display, i) <= 4) {
		dispPtr->colorModels[i] = TK_MONO;
	    } else {
		dispPtr->colorModels[i] = TK_COLOR;
	    }
	}
	tkDisplayList = dispPtr;

#if CTAXT_DEBUG_LEVEL > 1
	fprintf (stderr,"\nBefore Tk_CreateFileHandler");
#endif
	Tk_CreateFileHandler(ConnectionNumber(display),
		TK_READABLE|TK_IS_DISPLAY, (void (*)()) NULL,
		(ClientData) display);
#if CTAXT_DEBUG_LEVEL > 1
	fprintf (stderr,"\nAfter Tk_CreateFileHandler");
#endif

      }  /*  Matches if (dispPtr == NULL)  */

      if ((strncmp (dispPtr->name,screenName,length) == 0) &&
          (dispPtr->name[length] == '\0')) {
        break;
      }

    }  /*  Matches for (dipPtr = ....; ; ....)  */

    if (screenId >= ScreenCount(dispPtr->display)) {
	sprintf(interp->result, "Ebad screen number \"%d\"", screenId);
	return (TkDisplay *) NULL;
    }
    *screenPtr = screenId;
    return dispPtr;
}


/*
 *--------------------------------------------------------------
 *
 * CTAXT_NewWindow --
 *
 *	This procedure creates and initializes a TkWindow structure for
 *      an application-main-window mapping an already existing X-Window to.
 *
 * Results:
 *	The return value is a pointer to the new window.
 *
 * Side effects:
 *	A new window structure is allocated and all its fields are
 *	initialized.
 *
 *--------------------------------------------------------------
 */

static TkWindow *
CTAXT_NewWindow(dispPtr, screenNum, window)
    TkDisplay *dispPtr;		/* X-Display-Info about the already opened
				 * Display-Connection */
    int screenNum;		/* Index of screen for new window. */
    Window window;              /* X-Window-ID for already existing window */
{
    register TkWindow *winPtr;
    XWindowAttributes xattrbs;


    /*  create the TkWindow-Structure  */

    winPtr = (TkWindow *) ckalloc(sizeof(TkWindow));
    winPtr->display = dispPtr->display;
    winPtr->dispPtr = dispPtr;
    winPtr->screenNum = screenNum;

    
    /*  get Info from X about already existing window  */


#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nBefore XGetWindowAttributes");
#endif
    XGetWindowAttributes (winPtr->display,window,&xattrbs);
#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nAfter XGetWindowAttributes");
#endif


    /*  initialize the Tk-Structure to match the current context for window */

    winPtr->visual = xattrbs.visual;
    winPtr->depth = xattrbs.depth;
    winPtr->window = window;
    winPtr->childList = NULL;
    winPtr->lastChildPtr = NULL;
    winPtr->parentPtr = NULL;
    winPtr->nextPtr = NULL;
    winPtr->mainPtr = NULL;
    winPtr->pathName = NULL;
    winPtr->nameUid = NULL;
    winPtr->classUid = NULL;
    winPtr->changes.x = xattrbs.x;
    winPtr->changes.y = xattrbs.y;
    winPtr->changes.width = xattrbs.width;
    winPtr->changes.height = xattrbs.height;
    winPtr->changes.border_width = xattrbs.border_width;
    winPtr->changes.sibling = 0;
    winPtr->changes.stack_mode = Above;
    winPtr->dirtyChanges = 0;
    winPtr->atts = defAtts;
#ifdef new_stuff
    winPtr->atts.bit_gravity = xattrbs.bit_gravity;
    winPtr->atts.win_gravity = xattrbs.win_gravity;
    winPtr->atts.backing_store = xattrbs.backing_store;
    winPtr->atts.backing_planes = xattrbs.backing_planes;
    winPtr->atts.backing_pixel = xattrbs.backing_pixel;
    winPtr->atts.do_not_propagate_mask = xattrbs.do_not_propagate_mask;
    winPtr->atts.override_redirect = xattrbs.override_redirect;
    winPtr->atts.colormap = xattrbs.colormap;
#else
    winPtr->atts.colormap = DefaultColormap(dispPtr->display, screenNum);
#endif
    winPtr->dirtyAtts = 0;
    winPtr->flags |= TK_MAPPED;
    winPtr->handlerList = NULL;
    winPtr->focusProc = NULL;
    winPtr->focusData = NULL;
    winPtr->optionLevel = -1;
    winPtr->selHandlerList = NULL;
    winPtr->selClearProc = NULL;
    winPtr->selClearData = NULL;
    winPtr->geomProc = NULL;
    winPtr->geomData = NULL;
    winPtr->reqWidth = xattrbs.width;
    winPtr->reqHeight = xattrbs.height;
    winPtr->internalBorderWidth = xattrbs.border_width;
    winPtr->wmInfoPtr = NULL;

    return winPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * CTAXT_CreateMainWindow --
 *
 *	Make a new main window.  A main window is a special kind of
 *	top-level window used as the outermost window in an
 *	application. Map the window to an existing X-Window
 *
 * Results:
 *	The return value is a token for the new window, or NULL if
 *	an error prevented the new window from being created.  If
 *	NULL is returned, an error message will be left in
 *	interp->result.
 *
 * Side effects:
 *	A new window structure is allocated locally;  "interp" is
 *	associated with the window and registered for "send" commands
 *	under "baseName".  BaseName may be extended with an instance
 *	number in the form "#2" if necessary to make it globally
 *	unique.  Tk-related commands are bound into interp.  An X
 *	window is NOT initially created, but will be created the
 *	first time the window is mapped.
 *
 *----------------------------------------------------------------------
 */

Tk_Window
CTAXT_CreateMainWindow(interp, display, window, baseName, className)
    Tcl_Interp *interp;		/* Interpreter to use for error reporting. */
    Display *display;           /* X-Display-Info about already opened
				 * Display-Connection */
    Window window;              /* X-Window-ID for already existing window */
    char *baseName;		/* Base name for application;  usually of the
				 * form "prog instance". */
    char *className;            /* Class to use for application (same as class
                                 * for main window). */
{
    Tk_Window tkwin;
    int result, dummy;
    Tcl_HashEntry *hPtr;
    register TkMainInfo *mainPtr;
    register TkWindow *winPtr;
    register TkCmd *cmdPtr;
    char *libDir;
    char buffer[30];
    XEvent event;


    /*
     * Create the basic TkWindow structure.
     */

#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nBefore CTAXT_CreateTopLevelWindow");
#endif
    tkwin = CTAXT_CreateTopLevelWindow(interp, display, window);
    if (tkwin == NULL) {
	return NULL;
    }
#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nAfter CTAXT_CreateTopLevelWindow");
#endif


    /*
     * Create the TkMainInfo structure for this application, and set
     * up name-related information for the new window.
     */

    winPtr = (TkWindow *) tkwin;
    mainPtr = (TkMainInfo *) ckalloc(sizeof(TkMainInfo));
    mainPtr->winPtr = winPtr;
    mainPtr->interp = interp;
    Tcl_InitHashTable(&mainPtr->nameTable, TCL_STRING_KEYS);
    mainPtr->bindingTable = Tk_CreateBindingTable(interp);
    mainPtr->focusPtr = winPtr;
    mainPtr->focusDefaultPtr = NULL;
    mainPtr->optionRootPtr = NULL;
    mainPtr->nextPtr = tkMainWindowList;
    tkMainWindowList = mainPtr;
    winPtr->mainPtr = mainPtr;
    hPtr = Tcl_CreateHashEntry(&mainPtr->nameTable, ".", &dummy);
    Tcl_SetHashValue(hPtr, winPtr);
    winPtr->pathName = Tcl_GetHashKey(&mainPtr->nameTable, hPtr);


    /*
     * Register the interpreter for "send" purposes.  If baseName isn't
     * already unique, find a unique suffix to add to it to make it
     * unique.  Change the window's name to contain the suffix.
     */

#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nBefore Tk_RegisterInterp");
#endif
    result = Tk_RegisterInterp(interp, baseName, tkwin);
#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nAfter Tk_RegisterInterp");
#endif
    if (result == TCL_OK) {
	winPtr->nameUid = Tk_GetUid(baseName);
    } else {
	char newName[110];
	int i;

	for (i = 2; ; i++) {
	    sprintf(newName, "%.100s #%d", baseName, i);
	    Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);
	    result = Tk_RegisterInterp(interp, newName, tkwin);
	    if (result == TCL_OK) {
		break;
	    }
	    if (i >= 100) {
		Tcl_SetResult(interp,
			"Ecouldn't generate unique name to register "
			"application",
			TCL_STATIC);
		Tk_DestroyWindow(tkwin);
	    }
	}
	winPtr->nameUid = Tk_GetUid(newName);
    }


    /*
     * Bind in Tk's commands.
     */

    for (cmdPtr = commands; cmdPtr->name != NULL; cmdPtr++) {
	Tcl_CreateCommand(interp, cmdPtr->name, cmdPtr->cmdProc,
		(ClientData) tkwin, (void (*)()) NULL);
    }


    /*
     * Set variables for the intepreter.
     */

    libDir = getenv("TK_LIBRARY");
    if (libDir == NULL) {
	libDir = TK_LIBRARY;
    }
    Tcl_SetVar(interp, "tk_library", libDir, TCL_GLOBAL_ONLY);
    sprintf(buffer, "%d", TK_PATCH_LEVEL);
    Tcl_SetVar(interp, "tk_patchLevel", buffer, TCL_GLOBAL_ONLY);
    Tcl_SetVar(interp, "tk_version", TK_VERSION, TCL_GLOBAL_ONLY);
    Tcl_SetVar(interp, "tkVersion", TK_VERSION, TCL_GLOBAL_ONLY);


    /*  Set the Class for the window  */

#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nBefore Tk_SetClass");
#endif
    Tk_SetClass(tkwin,className);
#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nAfter Tk_SetClass");
#endif


    /*  Create a Frame-Widget for the Window  */

#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nBefore TkInitFrame");
#endif
    if (TkInitFrame(interp, tkwin, 1, 0, NULL) != TCL_OK) {
        return NULL;
    }
#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nAfter TkInitFrame");
#endif

    tk_NumMainWindows++;


    /*  
     *  Call XSaveContext because that's done by Tk in the normal window
     *  creation procedure (by Tk_MakeWindowExist)
     */

    XSaveContext (winPtr->display,
		  winPtr->window,
		  tkWindowContext,
		  (caddr_t) winPtr);
    winPtr->dirtyAtts = 0;
    winPtr->dirtyChanges = 0;


#ifdef original_stuff
    /*
     *  Call also TkWmMapWindow because that's what Tk does in the normal
     *  window-creation-procedure. Also create the event as Tk_MapWindow
     *  does.
     *  Doesn't seem to be necessary, couldn't recognize any change in be-
     *  havior apart from slowing down the start-up-process of the application.
     *  Also the initial size of the Tk-widgets inside the existing X-Window
     *  isn't set up correctly when TkWmMapWindow is called, so better leave 
     *  it out.
     */

    TkWmMapWindow (winPtr);

    event.type = MapNotify;
    event.xmap.serial = LastKnownRequestProcessed(winPtr->display);
    event.xmap.send_event = False;
    event.xmap.display = winPtr->display;
    event.xmap.event = winPtr->window;
    event.xmap.window = winPtr->window;
    event.xmap.override_redirect = winPtr->atts.override_redirect;
    Tk_HandleEvent(&event);
#endif


    /*  to get the position of the window right  */

    CTAXT_MoveResizeMainWindow (tkwin);


    return tkwin;
}
