/* Bos_Tcl.c - BOS/Tcl Interface
 *
 * Copyright (C) 1992,1993 Engineering Design Research Center
 *
 * Author: Sean Levy (snl+@cmu.edu)
 *         n-dim Group
 *         Engineering Design Research Center
 *         Carnegie Mellon University
 *         5000 Forbes Ave / PGH, PA / 51221
 *
 *         Fax: (412) 268-5229
 *         Voice: (412) 268-5226
 */

/*
 * Bos_Tcl.c,v 1.7 1992/08/03 20:50:04 snl Exp
 *
 * Bos_Tcl.c,v
 * Revision 1.7  1992/08/03  20:50:04  snl
 * Debugged new storage implementation
 *
 * Revision 1.6  1992/07/30  20:02:46  snl
 * bug fixes to new storage code
 *
 * Revision 1.5  1992/07/30  17:53:48  snl
 * Re-implemented GDBM-based storage scheme
 *
 * Revision 1.4  1992/07/22  17:15:11  snl
 * More tweaks for storage and evanescence
 *
 * Revision 1.3  1992/07/14  03:01:55  snl
 * Added evanescence everywhere, plus small changes to Storage
 *
 */

#ifndef lint
static char rcsID[] = "Bos_Tcl.c,v 1.7 1992/08/03 20:50:04 snl Exp";
#endif

#include <stdio.h>
#include "bosInt.h"
#include "bosSearch.h"

/*
 * Forwards
 */

Bos_Slot_Type _Bos_ParseSlotType _ARGS_((char *str));
static Bos_Slot_Pri parse_slot_pri _ARGS_((char *str));

/* Bos_InitInterp - create a world bound to an interpreter
 */

Bos_World *Bos_InitInterp(interp, local_only_p)
     Tcl_Interp *interp;
     int local_only_p;
{
  Bos_World *the_world;
  Tcl_HashTable *world;

  the_world = (Bos_World *)ckalloc(sizeof(Bos_World));
  the_world->objects = world = (Tcl_HashTable *)ckalloc(sizeof(Tcl_HashTable));
  Tcl_InitHashTable(world, TCL_STRING_KEYS);
  the_world->methods = interp; /* XXX for now... */

  Bos_InitUtilities(interp);

  Tcl_CreateCommand(interp, "%bosCreateObject", Bos_CreateObjectCmd,
		    (ClientData)the_world, (void (*)())NULL);
  Tcl_CreateCommand(interp, "%bosSend", Bos_SendCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosCopy", Bos_CopyCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosDestroy", Bos_DestroyCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosAddSlot", Bos_AddSlotCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosAddOrSetSlot", Bos_AddSlotCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosRemoveSlot", Bos_RemoveSlotCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosSetSlot", Bos_SetSlotCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "who", Bos_WhoCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosSave", Bos_SaveCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosLoad", Bos_LoadCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosRemove", Bos_RemoveCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosInStore", Bos_InStoreCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosCloseObjectFiles", Bos_ShutdownStoreCmd,
		    (ClientData)the_world, (void (*)())NULL);
  Tcl_CreateCommand(interp, "%bosCompactStorage", Bos_CompactCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosDumpStorageStatus", Bos_DumpStorageStatusCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosListStoredObjects", Bos_ListStoredObjectsCmd,
		    (ClientData)the_world, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "%bosGetSlots", Bos_GetSlotsCmd,
                    (ClientData)the_world, (void (*)())NULL);
  Tcl_CreateCommand(interp, "%bosLocalObjects", Bos_LocalObjectsCmd,
                    (ClientData)the_world, (void (*)())NULL);
  Tcl_CreateCommand(interp, "%bosSearch", Bos_SearchCmd,
		    (ClientData)the_world, (void (*)())NULL);
  Tcl_SetVar(interp, "bosVersion", BOS_VERSION, TCL_GLOBAL_ONLY);
#ifdef DISTRIBUTED_BOS
  if (!local_only_p)
    Bos_RegisterWorld(interp, world);
#endif
  return the_world;
}

#define USAGE(_e,_u) \
Tcl_AppendResult(interp,argv[0],": ",_e," -- usage: \"",argv[0]," ",_u,"\"",0);\
return TCL_ERROR

int
Bos_DumpStorageStatusCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_DumpStorageStatus(stdout);
  return TCL_OK;
}

int
Bos_SaveCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int ret, xx;

  if (argc != 4) {
    USAGE("wrong # args","obj_name dir lock");
  }
  if (sscanf(argv[3], "%d", &xx) != 1) {
    USAGE("lock param must be 0 or 1","obj_name dir lock");
  } else if (xx != 0 && xx != 1) {
    USAGE("lock param must be 0 or 1","obj_name dir lock");
  }
  ret = Bos_StoreObject(world, argv[1], argv[2], xx);
  if (ret != BOS_OK)
    Tcl_AppendResult(interp, argv[0], ": error storing object ", argv[1], 0);
  return ret? TCL_ERROR: TCL_OK;
}

int Bos_LoadCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int ret, xx;

  if (argc != 4) {
    USAGE("wrong # args","obj_name dir lock");
  }
  if (sscanf(argv[3], "%d", &xx) != 1) {
    USAGE("lock param must be 0 or 1","obj_name dir lock");
  } else if (xx != 0 && xx != 1) {
    USAGE("lock param must be 0 or 1","obj_name dir lock");
  }
  ret = Bos_FetchObject(world, argv[1], argv[2], xx);
  if (ret != BOS_OK)
    Tcl_AppendResult(interp, argv[0], ": error loading object ", argv[1], 0);
  return ret? TCL_ERROR: TCL_OK;
}

int
Bos_RemoveCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int ret, xx;

  if (argc != 4) {
    USAGE("wrong # args","obj_name dir lock");
  }
  if (sscanf(argv[3], "%d", &xx) != 1) {
    USAGE("lock param must be 0 or 1","obj_name dir lock");
  } else if (xx != 0 && xx != 1) {
    USAGE("lock param must be 0 or 1","obj_name dir lock");
  }
  ret = Bos_RemoveObject(argv[1], argv[2], xx);
  if (ret != BOS_OK)
    Tcl_AppendResult(interp, argv[0], ": error removing object ", argv[1], 0);
  return ret? TCL_ERROR: TCL_OK;
}

int
Bos_InStoreCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int ret, xx;

  if (argc != 4) {
    USAGE("wrong # args","obj_name dir lock");
  }
  if (sscanf(argv[3], "%d", &xx) != 1) {
    USAGE("lock param must be 0 or 1","obj_name dir lock");
  } else if (xx != 0 && xx != 1) {
    USAGE("lock param must be 0 or 1","obj_name dir lock");
  }
  sprintf(interp->result, "%d", Bos_IsObjectStored(argv[1], argv[2], xx));
  return TCL_OK;
}

int Bos_ShutdownStoreCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  sprintf(interp->result, "%d",
	  Bos_ShutdownStorage(0));
  return TCL_OK;
}

int Bos_CompactCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  int err;

  if (argc != 2) {
    USAGE("wrong # args","dir");
  }
  err = Bos_CompactStorage(argv[1]);
  if (err)
    sprintf(interp->result, "GDBM error %d compacting %s", err, argv[1]);
  return err? TCL_ERROR: TCL_OK;
}

int Bos_WhoCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int s;

  if (argc < 3) {
    sprintf(interp->result,
   "wrong # args: \"%.50s [defines|sends|calls|points|isa] string\"",
	    argv[0]);
    return TCL_ERROR;
  }
  if (!strcmp(argv[1], "defines") || !strcmp(argv[1], "sends") ||
      !strcmp(argv[1], "points") || !strcmp(argv[1], "calls") ||
      !strcmp(argv[1], "isa") || !strcmp(argv[1], "is-a")) {
    Tcl_HashEntry *oentry;
    Tcl_HashSearch osearch;
    int what;
    char *tName = (char *)0;

#define WHO_DEFINES 0		/* objects that define slot argv[2] */
#define WHO_SENDS 1		/* " with methods that mention argv[2] */
#define WHO_CALLS 2		/* same as SENDS but only returns obj names */
#define WHO_POINTS 3		/* objs that have pointers to argv[2] */
#define WHO_IS_A 4		/* objs that point at argv[2]Traits */

    if (!strcmp(argv[1], "defines"))
      what = WHO_DEFINES;
    else if (!strcmp(argv[1], "sends"))
      what = WHO_SENDS;
    else if (!strcmp(argv[1], "calls"))
      what = WHO_CALLS;
    else if (!strcmp(argv[1], "isa") || !strcmp(argv[1], "is-a")) {
      tName = (char *)ckalloc(strlen(argv[2]) + 7); /* "Traits": 6 chars + \0 */
      sprintf(tName, "%sTraits", argv[2]);
      what = WHO_IS_A;
    } else
      what = WHO_POINTS;
    for (oentry = Tcl_FirstHashEntry(Bos_Objects(world), &osearch);
	 oentry != (Tcl_HashEntry *)0;
	 oentry = Tcl_NextHashEntry(&osearch)) {
      Bos_Object *o;
      Tcl_HashEntry *sentry;
      Tcl_HashSearch ssearch;

      o = (Bos_Object *)Tcl_GetHashValue(oentry);
      for (sentry = Tcl_FirstHashEntry(o->slots, &ssearch);
	   sentry != (Tcl_HashEntry *)0;
	   sentry = Tcl_NextHashEntry(&ssearch)) {
	Bos_Slot *s;

        s = (Bos_Slot *)Tcl_GetHashValue(sentry);
	if (what == WHO_DEFINES) {
	  if (!strcmp(s->name, argv[2])) {
	    Tcl_AppendResult(interp, o->name, " ", 0);
	    break;
	  }
	} else if ((what == WHO_SENDS || what == WHO_CALLS) &&
		   Bos_PlainSlotType(s->type) == Bos_SLOT_METHOD) {
	  Bos_Method *m = (Bos_Method *)s->value;
	  char *sindex();

	  if (sindex(m->body, argv[2]) != (char *)0) {
	    if (what == WHO_SENDS)
	      Tcl_AppendResult(interp, "{", o->name, " ", s->name, "} ", 0);
	    else
	      /* XXX This should filter out dups. */
	      Tcl_AppendResult(interp, o->name, " ", 0);
	    break;
	  }
	} else if (what == WHO_POINTS && Bos_PlainSlotType(s->type) == Bos_SLOT_OBJECT) {
	  if (!strcmp((char *)s->value, argv[2])) {
	    Tcl_AppendResult(interp, o->name, " ", 0);
	    break;
	  }
	} else if (what == WHO_IS_A && Bos_PlainSlotType(s->type) == Bos_SLOT_OBJECT) {
	  if (!strcmp((char *)s->value, tName)) {
	    Tcl_AppendResult(interp, o->name, " ", 0);
	    break;
	  }
	}
      }
    }
    if (tName != (char *)0)
      ckfree(tName);
    s = TCL_OK;
  } else {
    strcpy(interp->result, "usage: who [defines|sends|points] string");
    s = TCL_ERROR;
  }
  return s;
}

int Bos_WhoMentionsCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
}


/* Bos_ObjectCmd -- Procedure bound to all objects.
 */

int Bos_ObjectCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int s;

  if (argc < 2) {
    sprintf(interp->result,
	    "wrong # args: should be \"%.50s message [args...]\"", argv[0]);
    return TCL_ERROR;
  }
  s = Bos_Sendv(world, interp, argv[0], argc-1, &argv[1]);
  switch (s) {
  case BOS_OK:
    s = TCL_OK;
    break;
  case BOS_NOT_FOUND:
    s = TCL_ERROR;
    break;
  case BOS_TCL_ERROR:
  case BOS_ERROR:
    s = TCL_ERROR;
    break;
  default:
    Bos_FatalError("Bos_ObjectCmd: impossible # 1");
    break;
  }
  if (s != TCL_OK && interp->result[0] == '\0') {
    sprintf(interp->result, "unspecified error during message send");
  }
  return s;
}

int Bos_CreateObjectCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int s;

  if (argc < 2) {
    sprintf(interp->result, "wrong # of args: \"%.50s name\"", argv[0]);
    return TCL_ERROR;
  }
  if (Bos_CreateNewObject(world, argv[1]) != (Bos_Object *)0) {
    int len = strlen(argv[1]);

    Tcl_CreateCommand(interp, argv[1], Bos_ObjectCmd,
		      clientData, (void (*)())NULL);
    if (len < Bos_USE_DYNAMIC_RETURN)
      sprintf(interp->result, argv[1]);
    else {
      interp->result = (char *)ckalloc(len + 1);
      strcpy(interp->result, argv[1]);
      interp->freeProc = TCL_DYNAMIC;
    }
    s = TCL_OK;
  } else {
    sprintf(interp->result, "object \"%.50s\" already exists", argv[1]);
    s = TCL_ERROR;
  }
  return s;
}

int Bos_SendCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int s;

  if (argc < 3) {
    sprintf(interp->result,
	    "wrong # args: should be \"%.50s object message [args...]\"",
	    argv[0]);
    return TCL_ERROR;
  }
  s = Bos_Sendv(world, interp, argv[1], argc-2, &argv[2]);
  switch (s) {
  case BOS_OK:
    s = TCL_OK;
    break;
  case BOS_NOT_FOUND:
    s = TCL_ERROR;
    break;
  case BOS_TCL_ERROR:
    break;
  default:
    Bos_FatalError("Bos_SendCmd: impossible # 1");
    break;
  }
  return s;
}

int Bos_CopyCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int s;

  if (argc < 3) {
    sprintf(interp->result,
	    "wrong # args: should be \"%.50s object newName\"", argv[0]);
    return TCL_ERROR;
  }
  if (Bos_Copy(world, argv[1], argv[2]) != (Bos_Object *)0) {
    int len = strlen(argv[2]);

    Tcl_CreateCommand(interp, argv[2], Bos_ObjectCmd,
		      clientData, (void (*)())NULL);
    if (len < Bos_USE_DYNAMIC_RETURN)
      strcpy(interp->result, argv[2]);
    else {
      interp->result = (char *)ckalloc(len + 1);
      strcpy(interp->result, argv[2]);
      interp->freeProc = TCL_DYNAMIC;
    }
    s = TCL_OK;
  } else {
    if (Bos_Find(world, argv[2]) != (Bos_Object *)0)
      sprintf(interp->result, "object already exists: \"%.50s\"", argv[2]);
    else
      sprintf(interp->result, "no such object: \"%.50s\"", argv[1]);
    s = TCL_ERROR;
  }
  return s;
}

int Bos_DestroyCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int s;

  if (argc < 2) {
    sprintf(interp->result,
	    "wrong # args: should be \"%.50s object\"", argv[0]);
    return TCL_ERROR;
  }
  if (Bos_Destroy(world, argv[1]) != (Bos_Object *)0)  {
    Tcl_DeleteCommand(interp, argv[1]);
    s = TCL_OK;
  } else {
    sprintf(interp->result, "no such object: \"%.50s\"", argv[1]);
    s = TCL_ERROR;
  }
  return s;
}

int Bos_AddSlotCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int s = TCL_OK, add_or_set;
  Bos_Object *obj;

  if (argc != 6) {
    sprintf(interp->result,
	    "wrong # argc: should be \"%.50s object sName sType sPri sVal\"",
	    argv[0]);
    return TCL_ERROR;
  }
  if (!strcmp(argv[0], "%bosAddOrSetSlot"))
    add_or_set = 1;
  else
    add_or_set = 0;
  obj = Bos_Find(world, argv[1]);
  if (obj == (Bos_Object *)0) {
    sprintf(interp->result, "no such object: \"%.50s\"", argv[1]);
    s = TCL_ERROR;
  } else {
    Bos_Slot_Type stype;

    stype = _Bos_ParseSlotType(argv[3]);
    if (stype == Bos_SLOT_ILLEGAL) {
      sprintf(interp->result, "bad slot type: \"%.50s\"", argv[3]);
      s = TCL_ERROR;
    }
    if (stype != Bos_SLOT_ILLEGAL) {
      Bos_Slot_Pri spri;

      spri = parse_slot_pri(interp, argv[4], stype);
      if (spri == Bos_PRI_ILLEGAL) {
	s = TCL_ERROR;
        if (Bos_PlainSlotType(stype) == Bos_SLOT_FOREIGN)
	  sprintf(interp->result, "bad foreign subtype \"%.50s\"", argv[4]);
	else
	  sprintf(interp->result, "bad priority \"%.50s\"", argv[4]);
      } else {
        _VoidPtr value;
	int retstat;

	if (Bos_PlainSlotType(stype) == Bos_SLOT_FOREIGN)
          value = Bos_ParseCSlotString(argv[5], spri);
        else if (Bos_PlainSlotType(stype) == Bos_SLOT_CMETHOD)
	  value = Bos_GetCMethodPointer(argv[5]);
	else
	  value = (_VoidPtr)argv[5];
	if (add_or_set) {
	  retstat = Bos_AddSlot(obj, argv[2], stype, spri, value);
	  if (retstat == BOS_ALREADY) {
	    int op_mask;

	    op_mask = Bos_SET_VALUE;
	    if (stype & Bos_SLOT_EVANESCENT_MASK)
	      op_mask |= Bos_SET_EVANESCENT_ON;
	    retstat = Bos_SetSlot(obj, op_mask, argv[2], 0, 0, value);
	  }
	} else
	  retstat = Bos_AddSlot(obj, argv[2], stype, spri, value);
	switch (retstat) {
	case BOS_OK:
          {
	    int len = strlen(argv[2]);

	    if (len < Bos_USE_DYNAMIC_RETURN)
	      strcpy(interp->result, argv[2]);
	    else {
	      interp->result = (char *)ckalloc(len + 1);
	      strcpy(interp->result, argv[2]);
	      interp->freeProc = TCL_DYNAMIC;
	    }
	  }
	  s = TCL_OK;
	  break;
	case BOS_ALREADY:
	  strcpy(interp->result,
		 "already a slot with that priority in that object");
	  s = TCL_ERROR;
	  break;
	case BOS_ERROR:
	  strcpy(interp->result, "BOS error setting/adding slot.");
	  s = TCL_ERROR;
	  break;
        default:
	  { char msg[200];

	    sprintf(msg, "Bos_AddSlotCmd: impossible # 1 (%d)", retstat);
	    Bos_FatalError(msg);
	    break;
	  }
	}
        if (Bos_PlainSlotType(stype) == Bos_SLOT_FOREIGN)
	  Bos_FreeCSlotValue(value, spri);
      }
    }
  }
  return s;
}

int Bos_RemoveSlotCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int s;
  Bos_Object *obj;

  if (argc < 3) {
    sprintf(interp->result,
	    "wrong # argc: should be \"%.50s object slotName\"",
	    argv[0]);
    return TCL_ERROR;
  }
  s = TCL_OK;
  obj = Bos_Find(world, argv[1]);
  if (obj == (Bos_Object *)0) {
    sprintf(interp->result, "no such object: \"%.50s\"", argv[1]);
    s = TCL_ERROR;
  } else {
    s = Bos_RemoveSlot(obj, argv[2]);
    if (s != BOS_OK) {
      sprintf(interp->result,
              "no such slot \"%.50s\" in object \"%.50s\"",argv[2], argv[1]);
      s = TCL_ERROR;
    } else
      s = TCL_OK;
  }
  return s;
}

int Bos_SetSlotCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int s, ncmds;
  Bos_Object *obj;
  char *obj_name, **cmds;

  if (argc < 5) {
   sprintf(interp->result,
           "wrong # args: should be \"%.50s object <slot option arg>...\"",
	   argv[0]);
   return TCL_ERROR;
  }
  s = TCL_OK;
  obj_name = argv[1];
  obj = Bos_Find(world, obj_name);
  if (obj == (Bos_Object *)0) {
    strcpy(interp->result, "no such object: \"%.50s\"", obj_name);
    return TCL_ERROR;
  } 
  ncmds = argc - 2;
  cmds = &argv[2];
  while (ncmds >= 3) {
    char *slot_name, *option, *arg;
    Bos_Slot *_BosFindSlot(), *sl;

    slot_name = *cmds++;
    option = *cmds++;
    arg = *cmds++;
    ncmds -= 3;
    sl = _BosFindSlot(obj, slot_name);
    if (sl == (Bos_Slot *)0) {
      s = TCL_ERROR;
      sprintf(interp->result, "no such slot as \"%.50s\" in \"%.50s\"",
              slot_name, obj_name);
      break;
    }
    if (!strcmp(option, "type")) {
      Bos_Slot_Type t;
      int op_mask;

      t = _Bos_ParseSlotType(arg);
      if (t == Bos_SLOT_ILLEGAL) {
        s = TCL_ERROR;
	sprintf(interp->result,
	        "bad slot type \"%.50s\" given for slot \"%.50s\"",
		arg, slot_name);
	break;
      }
      op_mask = Bos_SET_TYPE;
      if (t & Bos_SLOT_EVANESCENT_MASK)
	op_mask |= Bos_SET_EVANESCENT_ON;
      else
	op_mask |= Bos_SET_EVANESCENT_OFF;
      s = Bos_SetSlot(obj, op_mask, slot_name, t, 0, 0);
    } else if (!strcmp(option, "pri") || !strcmp(option, "subtype")) {
      Bos_Slot_Pri slot_pri = 0;

      slot_pri = parse_slot_pri(interp, arg, sl->type);
      if (slot_pri == Bos_PRI_ILLEGAL) {
        s = TCL_ERROR;
	sprintf(interp->result,
	        "priority did not scan: \"%.50s\"", arg);
        break;
      }
      s = Bos_SetSlot(obj, Bos_SET_PRI, slot_name, 0, slot_pri, 0);
    } else if (!strcmp(option, "value")) {
      _VoidPtr val = (_VoidPtr)0;

      if (Bos_PlainSlotType(sl->type) == Bos_SLOT_FOREIGN) {
        val = Bos_ParseCSlotString(arg, sl->pri);
        if (val == (_VoidPtr)0) {
          char *tname;

	  tname = Bos_GetCSlotTypeName(sl->pri);
	  if (tname == (char *)0)
	    tname = "unknown subtype";
	  s = TCL_ERROR;
	  sprintf(interp->result,
	          "could not parse \"%.50s\" (subtype %d, %s) slot \"%.50s\"",
		  arg, sl->pri, tname, slot_name);
	  break;
	}
      } else if (Bos_PlainSlotType(sl->type) == Bos_SLOT_CMETHOD) {
        val = Bos_GetCMethodPointer(arg);
	if (val == (_VoidPtr)0)
	  fprintf(stderr, "Bos/Tcl WARNING: \"%s\" not defined as C Method.\n",
	          arg);
      } else
        val = (_VoidPtr)arg;
      s = Bos_SetSlot(obj, Bos_SET_VALUE, slot_name, 0, 0, val);
    } else if (!strcmp(option, "name"))
      s = Bos_SetSlot(obj, Bos_SET_NAME, slot_name, 0, 0, (_VoidPtr)arg);
    else {
      sprintf(interp->result, "unknown option: \"%.50s\"", option);
      s = TCL_ERROR;
      break;
    }
    switch (s) {
    case BOS_OK:
      s = TCL_OK;
      break;
    case BOS_NOT_FOUND:
      sprintf(interp->result, "no such slot \"%.50s\" in \"%.50s\"",
             slot_name, obj_name);
      s = TCL_ERROR;
      break;
    default:
      sprintf(interp->result, "unknown result %d from Bos_SetSlot", s);
      s = TCL_ERROR;
      break;
    }
    if (s != TCL_OK)
      break;
  }
  return s;
}

int Bos_GetSlotsCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  Bos_Object *obj;
  int s;

  if (argc < 2) {
    sprintf(interp->result,
            "wrong # args: should be \"%.50s object [slot slot...]\"",
	    argv[0]);
    return TCL_ERROR;
  }
  obj = Bos_Find(world, argv[1]);
  if (obj == (Bos_Object *)0) {
    sprintf(interp->result,
            "no such object: \"%.50s\"", argv[1]);
    s = TCL_ERROR;
  } else {
    void getAllSlots();
    char *appendSlot();
    int error = 0;

    Tcl_AppendResult(interp, "{ ", 0);
    s = TCL_OK;
    if (argc == 2)
      getAllSlots(interp, obj);
    else {
      int i;

      for (i = 2; i < argc; i++)
	if (appendSlot(interp, obj, argv[i]) == (char *)0) {
	  error = 1;
	  break;
	}
      if (error) {
        *interp->result = '\0';
        Tcl_AppendResult(interp, "no such slot as: \"", argv[i], "\"", 0);
        s = TCL_ERROR;
      }
    }
    if (!error)
      Tcl_AppendResult(interp, " }", 0);
  }
  return s;
}

int Bos_LocalObjectsCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  Tcl_HashSearch search;
  Tcl_HashEntry *entry;

  for (entry = Tcl_FirstHashEntry(Bos_Objects(world), &search);
       entry != (Tcl_HashEntry *)0;
       entry = Tcl_NextHashEntry(&search)) {
    Bos_Object *obj;

    obj = (Bos_Object *)Tcl_GetHashValue(entry);
    Tcl_AppendResult(interp, obj->name, " ", (char *)0);
  }
  return TCL_OK;
}

int Bos_SearchCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  int arg, retval;
  char *old_argv0;
  Tcl_HashTable *results = (Tcl_HashTable *)0;

  if (argc % 4) {
    sprintf(interp->result,
	    "wrong # args: should be \"%.50s slotName boolOp value [AND|OR slotName boolOp value ...]\"",
	    argv[0]);
    return TCL_ERROR;
  }
  old_argv0 = argv[0];
  argv[0] = "or";
  arg = 0;
  retval = TCL_OK;
  while (arg < argc) {
    char *slot_name, *val;
    Bos_Search_Boolean bool, parse_boolean();
    int and;

    if (!strcasecmp(argv[arg], "and"))
      and = 1;
    else if (!strcasecmp(argv[arg], "or"))
      and = 0;
    else {
      sprintf(interp->result,
	      "saw \"%.50s\" where AND or OR was expected", argv[arg]);
      retval = TCL_ERROR;
      break;
    }
    slot_name = argv[arg+1];
    bool = parse_boolean(argv[arg+2]);
    if (bool == Bos_SEARCH_ILLEGAL) {
      sprintf(interp->result, "bad boolean \"%.50s\"", argv[arg+2]);
      retval = TCL_ERROR;
      break;
    }
    val = argv[arg+3];
    arg += 4;
    results = Bos_Search1(and? results: Bos_Objects(world),
			  results, slot_name, bool, val);
  }
  if (results != (Tcl_HashTable *)0 && retval == TCL_OK) {
    Tcl_HashSearch search;
    Tcl_HashEntry *entry;
    
    for (entry = Tcl_FirstHashEntry(results, &search);
	 entry != (Tcl_HashEntry *)0;
	 entry = Tcl_NextHashEntry(&search)) {
      Bos_Object *obj;
      
      obj = (Bos_Object *)Tcl_GetHashValue(entry);
      Tcl_AppendResult(interp, obj->name, " ", (char *)0);
    }
  }
  argv[0] = old_argv0;
  Bos_DestroySearchResults(results);
  return retval;
}

/*
 * Local routines
 */

static Bos_Search_Boolean parse_boolean(str)
     char *str;
{
  static struct { int bool; char *name; } bools[] = {
    { Bos_SEARCH_STRING_EQ, "string==" },
    { Bos_SEARCH_STRING_NE, "string!=" },
    { Bos_SEARCH_STRING_GT, "string>" },
    { Bos_SEARCH_STRING_LT, "string<" },
    { Bos_SEARCH_STRING_GE, "string>=" },
    { Bos_SEARCH_STRING_LE, "string<=" },
    { Bos_SEARCH_INT_EQ, "int==" },
    { Bos_SEARCH_INT_NE, "int!=" },
    { Bos_SEARCH_INT_GT, "int>" },
    { Bos_SEARCH_INT_LT, "int<" },
    { Bos_SEARCH_INT_GE, "int>=" },
    { Bos_SEARCH_INT_LE, "int<=" },
    { Bos_SEARCH_TYPE_EQ, "type==" },
    { Bos_SEARCH_TYPE_NE, "type!=" },
    { Bos_SEARCH_STRING_EQ, "==" },
    { Bos_SEARCH_STRING_NE, "!=" },
    { Bos_SEARCH_INT_GT, ">" },
    { Bos_SEARCH_INT_LT, "<" },
    { Bos_SEARCH_INT_GE, ">=" },
    { Bos_SEARCH_INT_LE, "<=" },
    { Bos_SEARCH_ILLEGAL, (char *)0 },
  };
  int i;

  for (i = 0; bools[i].bool != Bos_SEARCH_ILLEGAL; i++)
    if (!strcasecmp(str, bools[i].name))
      return bools[i].bool;
  return Bos_SEARCH_ILLEGAL;
}

static Bos_Slot_Pri parse_slot_pri(interp, str, type)
     Tcl_Interp *interp;
     char *str;
     Bos_Slot_Type type;
{
  int spri;

  type = Bos_PlainSlotType(type);
  if (type == Bos_SLOT_FOREIGN) {
    spri = Bos_GetCSlotType(str);
    if (spri == Bos_INVALID_SLOT_SUBTYPE)
      spri = Bos_PRI_ILLEGAL;
  } else if (!strcmp(str, "highest"))
    spri = Bos_PRI_HIGHEST;
  else if (!strcmp(str,"lowest"))
    spri = Bos_PRI_LOWEST;
  else if (sscanf(str, "%d", &spri) != 1) {
    sprintf(interp->result, "priority did not scan: \"%.50s\"", str);
    spri = Bos_PRI_ILLEGAL;
  }
  return (Bos_Slot_Pri)spri;
}

static void getAllSlots(interp, obj)
     Tcl_Interp *interp;
     Bos_Object *obj;
{
  Tcl_HashSearch search;
  Tcl_HashEntry *entry;

  for (entry = Tcl_FirstHashEntry(obj->slots, &search);
       entry != (Tcl_HashEntry *)0;
       entry = Tcl_NextHashEntry(&search)) {
    Bos_Slot *slot;
    char *slotTypeString(), *slotPriString(), *slotValString();

    slot = (Bos_Slot *)Tcl_GetHashValue(entry);
    Tcl_AppendResult(interp, "{", slot->name, " ",
                     slotTypeString(slot->type), " ",
		     slotPriString(slot->pri, slot->type), " ",
		     slotValString(slot), "} ", 0);
  }
}

static char *appendSlot(interp, obj, slot_name)
     Tcl_Interp *interp;
     Bos_Object *obj;
     char *slot_name;
{
  Tcl_HashEntry *e;
  char *ret;

  e = Tcl_FindHashEntry(obj->slots, slot_name);
  if (e == (Tcl_HashEntry *)0)
    ret = (char *)0;
  else {
    Bos_Slot *slot;
    char *slotTypeString(), *slotPriString(), *slotValString();

    slot = (Bos_Slot *)Tcl_GetHashValue(e);
    Tcl_AppendResult(interp, "{", slot->name, " ",
                     slotTypeString(slot->type), " ",
		     slotPriString(slot->pri, slot->type), " ",
		     slotValString(slot), "} ", 0);
    ret = slot->name;
  }
  return ret;
}

static char *slotTypeString(type)
     Bos_Slot_Type type;
{
  static char ret[30];

  ret[0] = '\0';
  if (type & Bos_SLOT_EVANESCENT_MASK) {
    strcat(ret, ".");
    type = Bos_PlainSlotType(type);
  }
  switch (type) {
  case Bos_SLOT_NORMAL:
    strcat(ret, "normal");
    break;
  case Bos_SLOT_METHOD:
    strcat(ret, "method");
    break;
  case Bos_SLOT_CMETHOD:
    strcat(ret, "cmethod");
    break;
  case Bos_SLOT_OBJECT:
    strcat(ret, "object");
    break;
  case Bos_SLOT_REFERENCE:
    strcat(ret, "reference");
    break;
  case Bos_SLOT_FOREIGN:
    strcat(ret, "foreign");
    break;
  default:
    sprintf(&ret[strlen(ret)], "%d", type);
    break;
  }
  return ret;
}

static char *slotPriString(pri, type)
     Bos_Slot_Pri pri;
     Bos_Slot_Type type;
{
  static char ret[30];

  type = Bos_PlainSlotType(type);
  if (type == Bos_SLOT_FOREIGN) {
    char *st;

    st = Bos_GetCSlotTypeName(pri);
    if (st == (char *)0)
      sprintf(ret, "{UNKNOWN FOREIGN #%d}", pri);
    else
      sprintf(ret, "%s", st);
  } else if (pri == Bos_PRI_HIGHEST)
    strcpy(ret, "highest");
  else if (pri == Bos_PRI_LOWEST)
    strcpy(ret, "lowest");
  else
    sprintf(ret, "%d", pri);
  return ret;
}

static char *slotValString(slot)
     Bos_Slot *slot;
{
  static char _buf[30];
  char *ret;
  Bos_Slot_Type plain_type;

  plain_type = Bos_PlainSlotType(slot->type);
  if (plain_type == Bos_SLOT_CMETHOD || plain_type == Bos_SLOT_FOREIGN) {
    sprintf(_buf, "%x", slot->value);
    ret = _buf;
  } else if (slot->value == (_VoidPtr)0) {
    strcpy(_buf, "null");
    ret = _buf;
  } else if (plain_type != Bos_SLOT_METHOD)
    ret = (char *)slot->value;
  else {
    Bos_Method *method = (Bos_Method *)slot->value;

    ret = method->body;
  }
  return ret;
}
