/*
     libscheme	
     Copyright (C) 1994 Brent Benson

     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 1, or (at your option)
     any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* This set of routines was written by 

       Shriram Krishnamurthi <shriram@cs.rice.edu>

   based on the code in scheme_read.c.  Please convey bugs and improvements
   to the current author as well as to the original one.
*/

/* The interface routine for the string-based read routine is

       Scheme_Object  * string_scheme_read_caller ( char  * stringArg ) .

   Do not call string_scheme_read() directly.
*/

#include "scheme.h"
#include <stdlib.h>
#include <ctype.h>

#define MAX_STRING_SIZE 1024
#define MAX_NUMBER_SIZE 64
#define MAX_SYMBOL_SIZE 255

int  stringArgPosn ;

/* local function prototypes */

static Scheme_Object *string_scheme_read ( char  * stringArg ) ;

static Scheme_Object *string_read_char ( char  * stringArg );
static Scheme_Object *string_read_list ( char  * stringArg );
static Scheme_Object *string_read_string ( char  * stringArg );
static Scheme_Object *string_read_quote ( char  * stringArg );
static Scheme_Object *string_read_vector ( char  * stringArg );
static Scheme_Object *string_read_number ( char  * stringArg );
static Scheme_Object *string_read_hex_number ( char  * stringArg );
static Scheme_Object *string_read_binary_number ( char  * stringArg );
static Scheme_Object *string_read_octal_number ( char  * stringArg );
static Scheme_Object *string_read_symbol ( char  * stringArg );
static Scheme_Object *string_read_character ( char  * stringArg );
static Scheme_Object *string_read_quasiquote ( char  * stringArg );
static Scheme_Object *string_read_unquote ( char  * stringArg );
static Scheme_Object *string_read_unquote_splicing ( char  * stringArg );
static void string_skip_whitespace_comments ( char  * stringArg );
static int string_peek_char ( char  * stringArg );
static int string_double_peek_char ( char  * stringArg );
static int string_match_chars ( char  * stringArg , char *str);

Scheme_Object *
string_scheme_read_caller ( char  * stringArg )
{
  stringArgPosn = 0 ;
  return ( string_scheme_read ( stringArg ) ) ;
}

Scheme_Object *
string_scheme_read ( char  * stringArg )
{
  int ch;

 start_over:
  ch = stringArg [ stringArgPosn ++ ] ;
  while (isspace(ch))
    {
      ch = stringArg [ stringArgPosn ++ ] ;
    }
  switch ( ch )
    {
    case EOF: return (scheme_eof);
    case ')': scheme_signal_error ("read: unexpected ')'");
    case '(': return (string_read_list ( stringArg ));
    case '"': return (string_read_string ( stringArg ));
    case '\'': return (string_read_quote ( stringArg ));
    case '`': return (string_read_quasiquote ( stringArg ));
    case ',':
      if (string_peek_char ( stringArg ) == '@')
	{
	  ch = stringArg [ stringArgPosn ++ ] ;
	  return (string_read_unquote_splicing ( stringArg ));
	}
      else
	{
	  return (string_read_unquote ( stringArg ));
	}
    case ';':
      while ((ch = stringArg [ stringArgPosn ++ ]) != '\n')
	{
	  if (ch == EOF)
	    {
	      return (scheme_eof);
	    }
	}
      goto start_over;
    case '+':
    case '-':
      if (isdigit (string_peek_char ( stringArg )))
	{
	  stringArg [ -- stringArgPosn ] = ch ; /* ungetc */
	  return (string_read_number ( stringArg ));
	}
      else
	{
	  stringArg [ -- stringArgPosn ] = ch ; /* ungetc */
	  return (string_read_symbol ( stringArg ));
	}
    case '#':
      ch = stringArg [ stringArgPosn ++ ];
      switch ( ch )
	{
	case '(': return (string_read_vector ( stringArg ));
	case '\\': return (string_read_character ( stringArg ));
	case 't': return (scheme_true);
	case 'f': return (scheme_false);
	case 'x': return (string_read_hex_number ( stringArg ));
	case 'b': return (string_read_binary_number ( stringArg ));
	case 'o': return (string_read_octal_number ( stringArg ));
	case '|':
	  do
	    {
	      ch = stringArg [ stringArgPosn ++ ];
	      if (ch == EOF)
		{
		  scheme_signal_error ("read: end of file in #| comment");
		}
	      if ((ch == '|') && (string_peek_char( stringArg ) == '#'))
		{
		  ch = stringArg [ stringArgPosn ++ ];
		  goto start_over;
		}
	    }
	  while ( 1 );
	  break;
	default:
	  scheme_signal_error ("read: unexpected `#'");
	}
    default:
      if (isdigit (ch))
	{
	  stringArg [ -- stringArgPosn ] = ch ;  /* ungetc */
	  return (string_read_number ( stringArg ));
	}
      else
	{
	  stringArg [ -- stringArgPosn ] = ch ;  /* ungetc */
	  return (string_read_symbol ( stringArg ));
	}
    }
}

static Scheme_Object *
string_read_char ( char  * stringArg )
{
  int ch;

  ch = stringArg [ stringArgPosn ++ ];
  if (ch == EOF)
    {
      return (scheme_eof);
    }
  else
    {
      return (scheme_make_char (ch));
    }
}

/* "(" has already been read */
static Scheme_Object *
string_read_list ( char  * stringArg )
{
  Scheme_Object *obj, *car, *cdr;
  int ch;

  string_skip_whitespace_comments ( stringArg );
  if (string_peek_char( stringArg ) == ')')
    {
      ch = stringArg [ stringArgPosn ++ ];
      return (scheme_null);
    }
  car = string_scheme_read ( stringArg );
  string_skip_whitespace_comments ( stringArg );
  if (string_peek_char( stringArg ) == ')')
    {
      ch = stringArg [ stringArgPosn ++ ];
      cdr = scheme_null;
    }
  else if ((string_peek_char( stringArg ) == '.') &&
	   isspace (string_double_peek_char( stringArg )))
    {
      ch = stringArg [ stringArgPosn ++ ];
      cdr = string_scheme_read ( stringArg );
      string_skip_whitespace_comments ( stringArg );
      if (string_peek_char( stringArg ) != ')')
	{
	  scheme_signal_error ("read: malformed list");
	}
      ch = stringArg [ stringArgPosn ++ ];
    }
  else
    {
      cdr = string_read_list ( stringArg );
    }
  return (scheme_make_pair (car, cdr));
}

/* '"' has already been read */
static Scheme_Object *
string_read_string ( char  * stringArg )
{
  char ch, buf[MAX_STRING_SIZE];
  int i;

  i = 0;
  while ((ch = stringArg [ stringArgPosn ++ ]) != '"')
    {
      if (ch == '\\')
	{
	  ch = stringArg [ stringArgPosn ++ ];
	}
      if (i > MAX_STRING_SIZE)
	{
	  scheme_signal_error ("read: string too long for reader");
	}
      buf[i++] = ch;
    }
  buf[i] = '\0';
  return (scheme_make_string (buf));
}

/* "'" has been read */
static Scheme_Object *
string_read_quote ( char  * stringArg )
{
  Scheme_Object *obj;

  obj = string_scheme_read ( stringArg );
  return (scheme_make_pair (scheme_quote_symbol, 
			    scheme_make_pair (obj, scheme_null)));
}

/* "#(" has been read */
static Scheme_Object *
string_read_vector ( char  * stringArg )
{
  Scheme_Object *obj, *vec;
  int len, i;
  
  obj = string_read_list ( stringArg );
  len = scheme_list_length (obj);
  vec = scheme_make_vector (len, NULL);
  for ( i=0 ; i<len ; ++i )
    {
      SCHEME_VEC_ELS(vec)[i] = SCHEME_CAR(obj);
      obj = SCHEME_CDR(obj);
    }
  return (vec);
}

/* nothing has been read */
static Scheme_Object  *
string_read_number ( char  * stringArg )
{
  char buf[MAX_NUMBER_SIZE];
  int i, is_float, is_negative, ch;

  i = 0;
  is_negative = is_float = 0;
  ch = stringArg [ stringArgPosn ++ ];
  if (ch == '+')
    {
      ch = stringArg [ stringArgPosn ++ ];
    }
  else if (ch == '-')
    {
      is_negative = 1;
      ch = stringArg [ stringArgPosn ++ ];
    }
  do
    {
      if (i > MAX_NUMBER_SIZE)
	{
	  scheme_signal_error ("read: number too long for reader");
	}
      if ((ch == '.') || (ch == 'e') || (ch == 'E'))
	{
	  is_float = 1;
	}
      buf[i++] = ch;
    }
  while (isdigit (ch = stringArg [ stringArgPosn ++ ]) || (ch == '.') ||
	 (ch == 'e') || (ch == 'E'));
  stringArg [ -- stringArgPosn ] = ch ;  /* ungetc */
  buf[i] = '\0';
  if ( is_float )
    {
      double d;
      d = atof (buf);
      if (is_negative)
	{
	  d = -d;
	}
      return (scheme_make_double (d));
    }
  else
    {
      int i;

      i = atoi (buf);
      if (is_negative)
	{
	  i = -i;
	} 
      return (scheme_make_integer (i));
    }
}

static Scheme_Object *
string_read_hex_number ( char  * stringArg )
{
  int ch, i;

  i = 0;
  while ( 1 )
    {
      ch = stringArg [ stringArgPosn ++ ];
      if (ch >= '0' && ch <= '9')
	{
	  i *= 16;
	  i += ch - '0';
	}
      else if ((ch >= 'a' && ch <= 'f') || (ch >= 'A' && ch <= 'F'))
	{
	  i *= 16;
	  if (ch >= 'a' && ch <= 'f')
	    i += ch - 'a' + 10;
	  else
	    i += ch - 'A' + 10;
	}
      else
	{
	  stringArg [ -- stringArgPosn ] = ch ;  /* ungetc */
	  return (scheme_make_integer (i));
	}
    }
}

static Scheme_Object *
string_read_binary_number ( char  * stringArg )
{
  int ch, i;

  i = 0;
  while ( 1 )
    {
      ch = stringArg [ stringArgPosn ++ ];
      if (ch == '0' || ch == '1')
	{
	  i *= 2;
	  i += ch - '0';
	}
      else
	{
	  stringArg [ -- stringArgPosn ] = ch ;  /* ungetc */
	  return (scheme_make_integer (i));
	}
    }
}

static Scheme_Object *
string_read_octal_number ( char  * stringArg )
{
  int ch, i;

  i = 0;
  while ( 1 )
    {
      ch = stringArg [ stringArgPosn ++ ];
      if (ch >= '0' && ch <= '7')
	{
	  i *= 8;
	  i += ch - '0';
	}
      else
	{
	  stringArg [ -- stringArgPosn ] = ch ;  /* ungetc */
	  return (scheme_make_integer (i));
	}
    }
}

/* nothing has been read */
static Scheme_Object *
string_read_symbol ( char  * stringArg )
{
  char buf[MAX_SYMBOL_SIZE];
  int i, ch;

  i = 0;
  while ((!isspace (ch = stringArg [ stringArgPosn ++ ]))
	 && (ch != '(')
	 && (ch != ')')
	 && (ch != '"')
	 && (ch != ';')
	 && (ch != EOF))
    {
      buf[i++] = ch;
    }
  if (ch != EOF)
    {
      stringArg [ -- stringArgPosn ] = ch ;  /* ungetc */
    }
  buf[i] = '\0';
  return (scheme_intern_symbol ((char *)&buf));
}

/* "#\" has been read */
static Scheme_Object *
string_read_character ( char  * stringArg )
{
  int ch;

  ch = stringArg [ stringArgPosn ++ ];
  switch (ch)
    {
    case 'n': /* maybe `newline' */
    case 'N':
      if ((string_peek_char( stringArg ) == 'e') || (string_peek_char( stringArg ) == 'E'))
	{
	  if (! string_match_chars ( stringArg , "ewline"))
	    {
	      scheme_signal_error ("read: bad character constant");
	    }
	  else
	    {
	      return (scheme_make_char ('\n'));
	    }
	}
      else
	{
	    return (scheme_make_char (ch));
	}
    case 's': /* maybe `space' */
    case 'S':
      if ((string_peek_char( stringArg ) == 'p') || (string_peek_char( stringArg ) == 'P'))
	{
	  if (! string_match_chars ( stringArg , "pace"))
	    {
	      scheme_signal_error ("read: bad character constant");
	    }
	  else
	    {
	      return (scheme_make_char (' '));
	    }
	}
      else
	{
	  return (scheme_make_char (ch));
	}
    case 'r': /* maybe `rubout' */
    case 'R':
      if ((string_peek_char( stringArg ) == 'u') || (string_peek_char( stringArg ) == 'U'))
	{
	  if (! string_match_chars ( stringArg , "ubout"))
	    {
	      scheme_signal_error ("read: bad character constant");
	    }
	  else
	    {
	      return (scheme_make_char (0x7f));
	    }
	}
      else if ((string_peek_char( stringArg ) == 'e') || (string_peek_char( stringArg ) == 'E'))
	{
	  if (! string_match_chars ( stringArg , "eturn"))
	    {
	      scheme_signal_error ("read: bad character constant");
	    }
	  else
	    {
	      return (scheme_make_char ('\r'));
	    }
	}
      else
	{
	  return (scheme_make_char (ch));
	}
    case 'p': /* maybe `page' */
    case 'P':
      if ((string_peek_char( stringArg ) == 'a') || (string_peek_char( stringArg ) == 'A'))
	{
	  if (! string_match_chars ( stringArg , "age"))
	    {
	      scheme_signal_error ("read: bad character constant");
	    }
	  else
	    {
	      return (scheme_make_char ('\f'));
	    }
	}
      else
	{
	  return (scheme_make_char (ch));
	}
    case 't': /* maybe `tab' */
    case 'T':
      if ((string_peek_char( stringArg ) == 'e') || (string_peek_char( stringArg ) == 'E'))
	{
	  if (! string_match_chars ( stringArg , "ab"))
	    {
	      scheme_signal_error ("read: bad character constant");
	    }
	  else
	    {
	      return (scheme_make_char ('\t'));
	    }
	}
      else
	{
	  return (scheme_make_char (ch));
	}
    case 'b': /* maybe `backspace' */
    case 'B':
      if ((string_peek_char( stringArg ) == 'e') || (string_peek_char( stringArg ) == 'e'))
	{
	  if (! string_match_chars ( stringArg , "ackspace"))
	    {
	      scheme_signal_error ("read: bad character constant", NULL);
	    }
	  else
	    {
	      return (scheme_make_char ('\b'));
	    }
	}
      else
	{
	  return (scheme_make_char (ch));
	}
    case 'l': /* maybe `linefeed' */
    case 'L':
      if ((string_peek_char( stringArg ) == 'e') || (string_peek_char( stringArg ) == 'E'))
	{
	  if (! string_match_chars ( stringArg , "inefeed"))
	    {
	      scheme_signal_error ("read: bad character constant");
	    }
	  else
	    {
	      return (scheme_make_char ('\n'));
	    }
	}
      else
	{
	  return (scheme_make_char (ch));
	}
    default:
      return (scheme_make_char (ch));
    }
}

/* "`" has been read */
static Scheme_Object *
string_read_quasiquote ( char  * stringArg )
{
  Scheme_Object *quoted_obj, *ret;
  
  quoted_obj = string_scheme_read ( stringArg );
  ret = scheme_make_pair (scheme_quasiquote_symbol, 
			  scheme_make_pair (quoted_obj, scheme_null));
  return (ret);
}

/* "," has been read */
static Scheme_Object *
string_read_unquote ( char  * stringArg )
{
  Scheme_Object *obj, *ret;

  obj = string_scheme_read ( stringArg );
  ret = scheme_make_pair (scheme_unquote_symbol, 
			  scheme_make_pair (obj, scheme_null));
  return (ret);
}

/* ",@" has been read */
static Scheme_Object *
string_read_unquote_splicing ( char  * stringArg )
{
  Scheme_Object *obj, *ret;

  obj = string_scheme_read ( stringArg );
  ret = scheme_make_pair (scheme_unquote_splicing_symbol, 
			  scheme_make_pair (obj, scheme_null));
  return (ret);
}

/* utilities */

static void
string_skip_whitespace_comments ( char  * stringArg )
{
  int ch;

 start_over:
  while (isspace(ch = stringArg [ stringArgPosn ++ ]));
  if ( ch == ';' )
    {
      while ((ch = stringArg [ stringArgPosn ++ ]) != '\n');
      goto start_over;
    }
  stringArg [ -- stringArgPosn ] = ch ;  /* ungetc */
}

static int
string_peek_char ( char  * stringArg )
{
  int ch;

  ch = stringArg [ stringArgPosn ] ;
  return (ch);
}

static int 
string_double_peek_char ( char  * stringArg )
{
  int ch1, ch2;

  ch2 = stringArg [ stringArgPosn + 1 ] ;
  return (ch2);
}

static int
string_match_chars ( char  * stringArg , char *str)
{
  int i;
  int ch;

  i = 0;
  while (str[i])
    {
      ch = stringArg [ stringArgPosn ++ ];
      if (tolower(ch) != tolower(str[i]))
	{
	  return (0);
	}
      i++;
    }
  return (1);
}
