#
#  Tk/Tcl widget tour, Version 2.0
#
#	Andrew Payne
#	payne@crl.dec.com
#

option add Tk.geometry 	"+25+405" startupFile
option add Tk.demo-geometry "300x300+25+25" startupFile

option add *Entry*BorderWidth	2
option add *Entry*Background	white
option add *Entry*Relief	sunken
option add *Entry*Font 		-*-courier-bold-r-*-*-14-*-*-*-*-*-*-*
option add *Entry*Width 	40

if {$argc < 1} {
	puts stderr "wtour:  lesson directory expected"
	exit 1
}
set lessondir [lindex $argv 0]

#
#  Check Tk version, need 3.3 or later (because we use the new packer syntax)
#
if {$tk_version < 3.3} {
	puts stderr "Wtour 2.0 requires Tk version 3.3 or later!"
	exit 1
}

#
#  Find and load support scripts
#
set appdir [file dirname [info script]]
foreach f {emacs.tcl} {
	if {![file exists $appdir/$f]} {
		puts stdout "wtour:  can't find support file $appdir/$f"
		exit
	}
	source $appdir/$f
}

#
#  Start up the demo proces
#
set wdemo "demo-[pid]"
set wish_bin wish
catch {set wish_bin $env(WISH)}
set wdemopid [exec $wish_bin -name $wdemo -file /dev/null &]

#
#  Make a text widget with an attached scrollbar
#
proc mkText {w} {
	frame $w
	scrollbar $w.scroll -relief flat -command "$w.text yview"
	pack $w.scroll -side right -fill y 
	text $w.text -bd 1 -relief raised -yscroll "$w.scroll set" -wrap none
	pack $w.text -expand yes -fill both
	return $w.text
}

#
#  Set up main window
#
wm title . "Tk Widget Tour"
wm geometry . [option get . geometry Wtour]
wm minsize . 100 100

set mframe [frame .menu -relief raised -borderwidth 1]
pack $mframe -fill x

menubutton $mframe.help -text "Help" -menu $mframe.help.menu
pack $mframe.help -side right
set m [menu $mframe.help.menu]
$m add command -label "Help!" -command {mkHelp}

menubutton $mframe.file -text "File" -menu $mframe.file.menu
pack $mframe.file -side left
set m [menu $mframe.file.menu]
$m add command -label "New" -command {do-new}
$m add command -label "Open..." -command {do-open}
$m add command -label "Save..." -command {do-saveas}
$m add separator
$m add checkbutton -label "Command Window" -variable cmd_window -command {
	if {$cmd_window} {
		pack .rmt -expand yes -fill both
		rmt-prompt
	} else {
		.rmt.t delete 1.0 end
		pack forget .rmt
	}
}
set cmd_window 0

$m add cascade -label "Screen Font" -menu $mframe.file.menu.fonts
$m add separator
$m add command -label "Exit" -command {do-exit}

set m [menu $mframe.file.menu.fonts]
$m add command -label "Small" -command {
	set-font -*-courier-medium-r-*-*-12-*-*-*-*-*-*-* \
		 -*-courier-bold-r-*-*-12-*-*-*-*-*-*-*
}
$m add command -label "Medium" -command {
	set-font -*-courier-medium-r-*-*-14-*-*-*-*-*-*-* \
		 -*-courier-bold-r-*-*-14-*-*-*-*-*-*-*
}
$m add command -label "Large" -command {
	set-font -*-courier-medium-r-*-*-18-*-*-*-*-*-*-* \
		 -*-courier-bold-r-*-*-18-*-*-*-*-*-*-*
}

set t [mkText .text]
pack .text -expand yes -fill both

bind $t <Any-Key-Menu> "apply-changes"
bind $t <Any-Key-Prior> "adjust-lesson -1"
bind $t <Any-Key-Next> "adjust-lesson 1"
focus $t

set f [frame .buttons -relief raised -borderw 1]
pack $f -side bottom -fill x
button $f.apply -text " Apply " -command {apply-changes}
button $f.next -text " Next  " -command {adjust-lesson 1}
button $f.prev -text " Prev " -command {adjust-lesson -1}
pack $f.apply $f.next $f.prev -side left -padx 7 -pady 7

#
#  Set the font of both text windows
#
proc set-font {reg bold} {
	global t
	$t configure -font $reg
	.rmt.t configure -font $reg
	.rmt.t tag configure bold -font $bold
}

#
#  Make a new dialog toplevel window
#
proc mkDialogWindow {w} {
	catch "destroy $w"
	toplevel $w -class Dialog -bd 0
	wm title $w "Dialog box"
	wm iconname $w "Dialog"
	wm geometry $w +425+300
	grab $w
	focus $w
	return $w
}

#
#  Center a window on the screen
#
proc centerwindow {w} {
	wm withdraw $w
	update idletasks
	set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
		- [winfo vrootx [winfo parent $w]]]
	set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
		- [winfo vrooty [winfo parent $w]]]
	wm geom $w +$x+$y
	wm deiconify $w
}

#
#  Show a help window
#
proc mkHelp {} {
	set w [mkDialogWindow .help]
        wm title $w "Window Tour Help"
	set t [mkText $w.t]
	pack $w.t
	set f [frame $w.buttons -relief raised -borderw 1]
	pack $f -side bottom -fill x
	button $f.close -text " Close " -command "destroy $w"
	pack $f.close -side right -padx 7 -pady 7
        $t insert current {
Wtour is an interactive tour of Tk widgets.

The main window displays a short Tcl/Tk program, and the demo window
displays the results of running the program.

You can make changes to the program and apply those changes by clicking
on the "Apply" button or pressing the "Do" button.

You can navigate through the tour with the "Prev" and "Next" buttons.  Or,
you can go directly to a specified lesson with the drop down menus.

There is also a command window that can be used to send individual commands
to the demo process.  You can toggle the command window on and off with an
option under the "File" menu.

Andrew Payne
payne@crl.dec.com
	}
	$t configure -state disabled
	centerwindow $w
}

#
#  Make a one-line query dialog box
#
proc mkEntryQuery {w prompt var} {
	global action

	set w [mkDialogWindow $w]
	set t [frame $w.top -relief raised -border 1]
	set b [frame $w.bot -relief raised -border 1]
	pack $t $b -fill both

	label $t.lab -text $prompt
	set e [entry $t.ent -textvar $var]
	bind $e <Any-Return> "set action ok"
	pack $t.lab $e -side left -padx 3m -pady 3m

	button $b.ok -text "Ok" -command "set action ok"
	frame $b.default -relief sunken -bd 1
	raise $b.ok $b.default
	pack $b.default -in $w.bot -side left -expand 1 -padx 3m -pady 2m
	pack $b.ok -in $b.default -padx 2m -pady 2m -ipadx 2m -ipady 1m
	button $b.cancel -text "Cancel" -command "set action cancel"
	pack $b.cancel -side left -padx 3m -pady 3m -ipadx 2m -ipady 1m \
		-expand yes
	centerwindow $w
	focus $e
	tkwait variable action
	destroy $w
	return $action
}

#
#  Write the edit buffer to the specified file
#
proc write-file {fname} {
	global t errorCode

	if [catch {set f [open $fname w]}] {
		set msg "Can't write the file `$fname'"
		if {[lindex $errorCode 0] == "UNIX"} {
			append msg " -- [lindex $errorCode 2]"
		}
		if {[tk_dialog .error "Error" $msg error 0 Retry Cancel] == 0} {
			write-file $fname
		}
	} {
		puts $f [$t get 1.0 end]
		close $f
	}
}

#
#  Write the edit buffer out to a file, first checking for file existence
#
proc do-save-file {fname} {
	if [file exists $fname] {
		if {[tk_dialog .error "Error" \
			"File '$fname' exists.  Overwrite?" error 0 \
			Ok Cancel] == 1} {
				return
		}
	}
	write-file $fname
}

#
#  Clear everything out
#
proc do-new {} {
	global t

	$t delete 1.0 end
	apply-changes
}

#
# Save contents to a named file
#
proc do-saveas {} {
	global filename

	if {[mkEntryQuery .dialog "Enter save file name:" filename] == "ok"} {
		do-save-file $filename
	}
}

#
#  Load the edit buffer from a file
#
proc do-open-file {fname} {
	global t filename errorCode

	if [catch {set f [open $fname]}] {
		set msg "Error: can't open the file `$fname'"
		if {[lindex $errorCode 0] == "UNIX"} {
			append msg " -- [lindex $errorCode 2]"
		}
		if {[tk_dialog .error "Error" $msg error 0 Retry Cancel] == 0} {
			do-open-file $fname
		}
	} {
		set filename $fname
		$t delete 1.0 end
		$t insert current [read $f]
		close $f
	}
	apply-changes
}

#
# Prompt user for filename and load a file
#
proc do-open {} {
	global filename

	if {[mkEntryQuery .dialog "Enter file name to load:" filename] == "ok"} {
		do-open-file $filename
	}
}

#
#  Apply the changes to the demo process
#
proc apply-changes {} {
	global t wdemo

	send $wdemo {
		foreach w [winfo children .] {
			destroy $w
		}
		option clear
	}
	send $wdemo [$t get 1.0 end]
}

#
#  Define a new lesson
#
proc lesson {mname name file} {
	global Lessons nlessons mframe Menus menu_bar

	set Lessons($nlessons) $file
	if {[catch {set mb $Menus($mname)}]} {
		set mb $mframe.$nlessons
		menubutton $mb -text $mname -menu $mb.menu
		pack $mb -side left
		set Menus($mname) [menu $mb.menu]
		lappend menu_bar $mb
	}
	if {$name != ""} {
		$Menus($mname) add command -label $name \
			-command "set-lesson $nlessons"
		incr nlessons
	} {
		$mb add separator
	}
}

#
#  Set the current lesson
#
proc set-lesson {n} {
	global Lessons curlesson lessondir

	set curlesson $n
	do-open-file $lessondir/$Lessons($n)
}

#
#  Adjust the current lesson by some increment
#
proc adjust-lesson {i} {
	global curlesson nlessons

	incr curlesson $i
	if {$curlesson >= $nlessons} {
		tk_dialog .info "Warning" "That was the last lesson!" warning \
			0 Dismiss
		set curlesson [expr $nlessons-1]
	}
	if {$curlesson < 0} {
		tk_dialog .info "Warning" "That was the first lesson!" warning \
			0 Dismiss
		set curlesson 0
	}
	set-lesson $curlesson
}


#
# Clean up and exit
#
proc do-exit {} {
	global wdemo
	send $wdemo {after 1 {exit}}
	exit
}

#
#  Set up the remote control window.  Need routines to handle backspacing,
#  the prompt, and command invocation (from John's book).
#
proc rmt-backspace {} {
	if {[.rmt.t index promptEnd] != [.rmt.t index {insert - 1 char}]} {
		.rmt.t delete {insert - 1 char} insert
		.rmt.t yview -pickplace insert
	}
}

proc rmt-prompt {} {
	.rmt.t insert insert "demo: "
	.rmt.t mark set promptEnd {insert - 1 char}
	.rmt.t tag add bold {insert linestart} promptEnd
}

proc rmt-invoke {} {
	global wdemo

	set cmd [.rmt.t get {promptEnd + 1 char} insert]
	if [info complete $cmd] {
		catch [list send $wdemo $cmd] msg
		if {$msg != ""} {
			.rmt.t insert insert $msg\n
		}
		rmt-prompt
	}
	.rmt.t yview -pickplace insert
}

frame .rmt
scrollbar .rmt.s -relief flat -command ".rmt.t yview"
pack .rmt.s -side right -fill y
text .rmt.t -relief raised -bd 1 -yscrollcommand ".rmt.s set" \
	-height 12

pack .rmt.t -side left -fill both -expand yes

bind .rmt.t <1> {
	set tk_priv(selectMode) char
	.rmt.t mark set anchor @%x,%y
	if {[lindex [%W config -state] 4] == "normal"} {focus %W}
}

bind .rmt.t <Double-1> {
	set tk_priv(selectMode) word
	tk_textSelectTo .rmt.t @%x,%y
}

bind .rmt.t <Triple-1> {
	set tk_priv(selectMode) line
	tk_textSelectTo .rmt.t @%x,%y
}

bind .rmt.t <Return> {.rmt.t insert insert \n; rmt-invoke}
bind .rmt.t <BackSpace> {rmt-backspace}
bind .rmt.t <Control-h> {rmt-backspace}
bind .rmt.t <Delete> {rmt-backspace}
bind .rmt.t <Button-2> {
	.rmt.t insert insert [selection get]
	.rmt.t yview -pickplace insert
	if [string match *.0 [.rmt.t index insert]] {
		rmt-invoke
	}
}

set-font -*-courier-medium-r-*-*-12-*-*-*-*-*-*-* \
		 -*-courier-bold-r-*-*-12-*-*-*-*-*-*-*

#
# Configure the demo process
#
while {[lsearch [winfo interps] $wdemo] == -1} {
	after 50
}

#
#  Check to see if "send" works
#
update idletasks
if [catch {send $wdemo "winfo interps"}] {
	puts stderr {*** Fatal error:  'send' doesn't work!

Tk versions 3.3 and later disable the "send" command (needed by the widget 
tour) if the X server is running simple host based access control.  Solutions 
are:

     1) Use xauth-style authorization on your server (see the X
        documentation or your local X guru on how to do this)

     2) Build a custom version of Tk 3.3 without this feature (set the
        cpp macro "TK_NO_SECURITY" when building)

Sorry!
}
	exec kill $wdemopid	
	exit
}

send $wdemo {
	wm title . "Demo Window"
	wm iconname . "Demo Window"
	wm minsize . 100 100
}
send $wdemo "wm geometry . [option get . demo-geometry Wtour]"

#
#  Load in the lesson index
#
set nlessons 0
send $wdemo "cd $lessondir"
source $lessondir/index
set-lesson 0
eval "tk_menuBar $mframe $mframe.file $menu_bar $mframe.help"
