#############################################################################
#
# folders.tcl: This file contains all procedures related to folders
#
# CreateFol {Fol}
#       Create the Folder with some checks.
#
# FindLastFile {Fol}
#       Find the last file in a folder, and return it.
#
# GetMsgNum {Fol maxnum}
#       Get the message number from the selected message in the Folder 
#       list and return it. maxnum is the maximum number of messages,
#       defaults to 1.
#
# OpenFol {Fol}
#       Create a window in which the headers from messages in $Fol can
#       be displayed, use ScanFol to read the messages.
#
# PackFol {Fol}
#       Renumber the files in the folder so that they will go from
#       1-nummessages, also delete files starting with a , which are
#       backup files from deleted messages.
#       TODO: implement it
#
# ScanFol {Fol}
#       Fill the Folder window with the message headers, using the MH scan 
#       command.
#
# RemoveFol {Fol}
#       Remove the folder, ask for confirmation
#
# RenameFol {Fol}
#       Rename $Fol.
#
# SortFol {Fol}
#       Sort a folder according to the sort options in the tkmh_profile
#       and the settings in the configuration box
#       TODO: write this all, including configuration box entries, and so
#       temporarily use sortm from mh to sort by date
#
# CopyMsg {FromFol}{Msg} {ToFol}
#       Copy $Msg from $FromFol to $ToFol, append as last
#
# MoveMsg {FromFol} {Msg} {ToFol}
#       Move $Msg from $FromFol to $ToFol, append as last
#
#############################################################################

proc CreateFol {Fol} {
    global MH cmd

    # Create the folder, needs a check on a "/", split it up between 
    # the slashes and create in turn, if they don't exist already.
    # also check on spaces and return an error message if found any
    if {[string first { } $Fol] != -1} {
        MsgWindow "No spaces allowed in foldernames"
        return
    }
    set cwd [pwd]
    cd [RF $MH(Path)]
    foreach dir [split $Fol "/"] {
        if [file isfile $dir] {
            MsgWindow "$dir already exists as a file"
            cd $cwd
            return
        }
        if [file isdirectory $dir] {
            cd $dir
            continue
        }
        eval exec $cmd(mkdir) $dir
        cd $dir
    }
    cd $cwd
}

proc FindLastFile {Fol} {
    global MH

    set cwd [pwd]
    cd [RF $MH(Path)/$Fol]
    set List [lsort -integer \
        [glob -nocomplain {[0-9]} {[0-9][0-9]} {[0-9][0-9][0-9]}]]
    set LastMsg [lindex $List [expr [llength $List] - 1]]
    if {$LastMsg == ""} {
        set LastMsg {0}
    }
    cd $cwd
    return $LastMsg
}

proc GetMsgNum {Fol {maxnum 1}} {
    set Result {}
    set Msgs [GetListboxSel .fol$Fol.t $maxnum]
    if {$Msgs == ""} {
        return {}
    }
    foreach Msg $Msgs {
        regexp {[0-9]+} $Msg Msg
        lappend Result $Msg 
    }
    return $Result
}


proc OpenFol {Fol} {
    global FolList Aliases
    global FONT PREF

    if [winfo exists .fol$Fol] {
        ScanFol $Fol
        return
    }
    toplevel .fol$Fol
    wm title .fol$Fol $Fol

    scrollbar .fol$Fol.s -relief raised -command ".fol$Fol.t yview"
    listbox .fol$Fol.t -yscroll ".fol$Fol.s set" -geometry 80x15 -setgrid true \
      -font $FONT(FixB) -relief raised -borderwidth 2

    frame .fol$Fol.m -borderwidth 2 -relief raised
    checkbutton .fol$Fol.m.backup -variable PREF(BackupMsg) -relief flat \
      -anchor w -text "Backup on delete"
    menubutton .fol$Fol.m.folder -text "Folder" -menu .fol$Fol.m.folder.m
    menu .fol$Fol.m.folder.m
      .fol$Fol.m.folder.m add command -label "Rescan" -command "ScanFol $Fol"
      .fol$Fol.m.folder.m add command -label "Pack" -command "PackFol $Fol; ScanFol $Fol"
#      .fol$Fol.m.folder.m add command -label "Pick Sequence" -command "PickFol $Fol"
      .fol$Fol.m.folder.m add command -label "Sort by Date" -command "SortFol $Fol; ScanFol $Fol"

    frame .fol$Fol.b
    button .fol$Fol.b.done -text "Done" -command "destroy .fol$Fol"
    button .fol$Fol.b.help -text "Help" -command "Help Folder"
    button .fol$Fol.b.show -text "Show" \
      -command "set Msg \[GetMsgNum $Fol\]; if {\$Msg != {}} {ShowMsg $Fol \$Msg}"
    button .fol$Fol.b.print -text "Print" \
      -command "set Msgs \[GetMsgNum $Fol 100\]
                if {\$Msgs != {}} {PrintMsg $Fol \$Msgs}"
    button .fol$Fol.b.delete -text "Delete" \
      -command "set Msgs \[GetMsgNum $Fol 100\]
                if {\$Msgs != {}} {DeleteMsg $Fol \$Msgs; ScanFol $Fol}"
    button .fol$Fol.b.reply -text "Reply" \
      -command "set Msg \[GetMsgNum $Fol\]; if {\$Msg != {}} {ReplyMsg $Fol \$Msg}"
    menubutton .fol$Fol.b.forw -text "Forward To" -relief raised -menu .fol$Fol.b.forw.m
    menubutton .fol$Fol.b.move -text "Move To" -relief raised -menu .fol$Fol.b.move.m
    menubutton .fol$Fol.b.copy -text "Copy To" -relief raised -menu .fol$Fol.b.copy.m

    menu .fol$Fol.b.move.m
    menu .fol$Fol.b.copy.m
    foreach F $FolList {
      .fol$Fol.b.move.m add command -label "$F" -command \
        "if {\[set Msg \[GetMsgNum $Fol 100\]\] != {}} {
           MoveMsg $Fol \$Msg $F; ScanFol $Fol
           if \[winfo exists .fol$F\] {ScanFol $F}}"
       .fol$Fol.b.copy.m add command -label "$F" -command \
         "if {\[set Msg \[GetMsgNum $Fol 100\]\] != {}} {
           CopyMsg $Fol \$Msg $F; if \[winfo exists .fol$F\] {ScanFol $F}}"
    }
            
    menu .fol$Fol.b.forw.m
      .fol$Fol.b.forw.m add command -label *NEW* \
         -command "set Msg \[GetMsgNum $Fol\]
                   if {\$Msg != {}} {ForwardMsg $Fol \$Msg}"
    foreach A $Aliases {
       .fol$Fol.b.forw.m add command -label "$A" -command  \
          "set Msg \[GetMsgNum $Fol\]
           if {\$Msg != {}} {
             if \$PREF(ExpandAlias) {
               ForwardMsg $Fol \$Msg \"\[SearchAlias $A 1\]\"
             } else {
               ForwardMsg $Fol \$Msg $A
             }
           }"
    }

    pack append .fol$Fol.m \
      .fol$Fol.m.folder {left} \
      .fol$Fol.m.backup {right}

    pack append .fol$Fol.b \
      .fol$Fol.b.show {top expand fill} \
      .fol$Fol.b.print {top expand fill} \
      .fol$Fol.b.reply {top expand fill} \
      .fol$Fol.b.forw {top expand fill} \
      .fol$Fol.b.copy {top expand fill} \
      .fol$Fol.b.move {top expand fill} \
      .fol$Fol.b.delete {top expand fill} \
      .fol$Fol.b.help {top expand fill} \
      .fol$Fol.b.done {top expand fill} 

    pack append .fol$Fol \
      .fol$Fol.m {top fillx} \
      .fol$Fol.t {left expand fill} \
      .fol$Fol.s {left filly} \
      .fol$Fol.b {left filly} 

    bind .fol$Fol.t <Double-1> ".fol$Fol.b.show invoke"
    bind .fol$Fol.t <3> \
      ".fol$Fol.t select clear
      .fol$Fol.t select from \[.fol$Fol.t nearest %y\]
      .fol$Fol.b.reply invoke"

    ScanFol $Fol
}

proc PackFol {Fol} {
    global MH
    catch {exec $MH(bindir)/folder -pack +$Fol}
}

proc ScanFol {Fol} {
    global MH

    # This is the format for the scan command (see scan(1mh) and
    # mh_format(1mh) for details). You can change it to whatever you 
    # want, but you should leave the message number the first 
    # number in the file. The procedure that reads out the message
    # number from the listbox entry relies on that....
    set Format "%3(msg) %02(mon{date})/%02(mday{date}) \
      %<(mymbox{from})To:%14(friendly{to})%|%17(friendly{from})%> \
      %{subject}%<{body}<<%{body}%>"

    set curYview [lindex [.fol$Fol.s get] 2]
    .fol$Fol.t delete 0 end
    # Read in the Messages in a long string.
    # The entry for every message is on a separate line. Split them up.
    if [catch {set Msgs [exec $MH(bindir)/scan -format $Format +$Fol]}] {
        return 
    }
    set Msgs [split $Msgs "\n"]
    set lastMsg [expr [llength $Msgs] - 1]
    for {set i 0} {$i <= $lastMsg} {incr i} {
        .fol$Fol.t insert end [lindex $Msgs $i]
    }
    .fol$Fol.t yview $curYview
}

proc RemoveFol {Fol} {
    global MH cmd

    if {$Fol == $MH(Inbox) || $Fol == $MH(Draft-Folder)} {
        MsgWindow "You can't delete your drafts or inbox folders!\n"
        return 0
    }
    if [Confirm "Remove $Fol?"] {
        eval exec $cmd(rmdir) [RF $MH(Path)/$Fol]
        return 1        
    }
    return 0 
}

proc RenameFol {Fol} {
    global MH cmd

    # Maybe use the same check as in CreateFol to create needed dirs...
    set NewName [Change Folder $Fol "Rename to" {} 15]
    if {$NewName == ""} {
        return 0
    }
    eval exec $cmd(mv) [RF $MH(Path)/$Fol] [RF $MH(Path)/$NewName]
    return 1
}

proc SortFol {Fol} {
    global MH

    catch {exec $MH(bindir)/sortm +$Fol -noverbose}
}

proc CopyMsg {FromFol Msgs ToFol} {
    global MH cmd

    foreach Msg $Msgs {
    set NewMsg [expr [FindLastFile $ToFol] + 1]
    eval exec $cmd(cp) [RF $MH(Path)/$FromFol/$Msg] \
                       [RF $MH(Path)/$ToFol/$NewMsg]
    }
}

proc MoveMsg {FromFol Msgs ToFol} {
    global MH cmd
    
    foreach Msg $Msgs {
        set NewMsg [expr [FindLastFile $ToFol] + 1]
        eval exec $cmd(mv) [RF $MH(Path)/$FromFol/$Msg] \
                           [RF $MH(Path)/$ToFol/$NewMsg]
    }
}

