/*
 * tmConverters.c -
 *	This module implements all the Xt Converters to and
 *	from Strings needed by Tm
 *
 * Copyright 1993 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include "tmFuncs.h"
#include <Xm/RepType.h>

#define done(address,type) \
	{	toVal->addr = (XtPointer) address; \
		toVal->size = sizeof(type); \
	}

/*
 * so that we can keep tabs on the size of an XmStringTable
 * see function Tm_MaybeSetXmStringTableSize
 */
int Tm_XmStringTableSize;

/*
 *--------------------------------------------------------------
 *
 * InstallReverseRepTypes --
 *
 *	Install the converters from internal representation
 *	types to their string values.
 *
 * Results:
 *
 * Side effects:
 *	Modifies the Motif RepType tables
 *
 *--------------------------------------------------------------
 */

static void
InstallReverseRepTypes()
{
    XmRepTypeId id;
    XmRepTypeList start, list;
    int n;

    start = list = XmRepTypeGetRegistered();

    while (list->rep_type_name != NULL) {
        if ((id = XmRepTypeGetId(list->rep_type_name)) != XmREP_TYPE_INVALID) {
	    XmRepTypeAddReverse(id);
        } else {
	    fprintf(stderr, "Failed to install %s converter\n", 
			list->rep_type_name);
        }
	list++;
    }
    XtFree((char *)start);
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToWidget --
 *
 *	Converter from String to Widget type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToWidget(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    Tcl_Interp *interp = (Tcl_Interp *) args->addr;
    Tm_Widget *info;
    Widget w;

    info = Tm_WidgetInfoFromPath(interp, (char *) fromVal->addr);
    if (info == NULL) {
	return False;
    }
    w = info->widget;

/*
    toVal->addr = (XtPointer) w;
    toVal->size = sizeof(Widget);
*/
    done(w, Widget);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtXmStringToString --
 *
 *	Converter from XmString to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtXmStringToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    XmStringContext context;
    static char *buf;
    char *p, *text;
    XmStringCharSet charset;
    XmStringDirection direction;
    Boolean separator;
    int size = 128;
    int current_len, segment_len;
    XmString xmstr = *(XmString *) fromVal->addr;

    p = buf = XtMalloc(size);
    if (!XmStringInitContext(&context, xmstr)) {
	return False;
    }
    while (XmStringGetNextSegment(context, &text, &charset,
		&direction, &separator)) {
	segment_len = strlen(text);
	if (p + segment_len >= buf + size) {
	    current_len = p - buf;
	    buf = XtRealloc(buf, 2*size);
	    p = buf + current_len;
	}
	strcpy(p, text);
	p += segment_len;
	if (separator == True) {
	    *p++ = '\n';
	    *p = 0;
	}
	XtFree(text);
    }
    XmStringFreeContext(context);
 
    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * Tm_CvtXmStringTableToString --
 *
 *	Converter from XmStringTable to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtXmStringTableToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    String *items, buf;
    int n, len, item_count;
    char *p, *end;
    char *dummy;
    XmString *xmstrs = *(XmString **) fromVal->addr;
    Tcl_Interp *interp = (Tcl_Interp *) args->addr;

    extern Boolean _XmStringIsString _ANSI_ARGS_((XmString));

    /* pass through once, trying to find out size of list */
/*
    for (n = 0; TRUE; n++) {
        if (_XmStringIsXmString(xmstrs[n]) ) {
           XtFree(dummy);
        } else { 
	    break;
	}
    }
*/
    len= 0;
/*
    item_count = n;
    item_count = atoi(Tcl_GetVar(interp, "XmStringTableSize", 0));
*/
    item_count = Tm_XmStringTableSize;
    items = (char **) XtMalloc(sizeof(char *) * item_count);

    for (n = 0; n < item_count; n++) {
        XmStringGetLtoR(xmstrs[n],
    		XmFONTLIST_DEFAULT_TAG, items + n);
        len += strlen(items[n]) + 2; /* for , */
    }
    buf = XtMalloc(len + 1); /* we overshoot by 1 */
    if (len == 0) {
        *buf = '\0';
        done(&buf, char *);
        return True;
    }
    end = buf;
    for (n = 0; n < item_count; n++) {
        p = items[n];
        while (*end++ = *p++)
    	;
        *(end - 1) = ',';
        *end++ = ' ';
        XtFree(items[n]);
    }
    *(end - 2) = '\0';

    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtStringToString --
 *
 *	Converter from String to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtStringToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;

    buf = *(char **) (fromVal->addr);
#   ifdef DEBUG
    fprintf(stderr, "converted string to %s\n", buf);
#   endif

    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtBooleanToString --
 *
 *	Converter from Boolean to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtBooleanToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Boolean boolVal;
 
/*
    bcopy(fromVal->addr, &boolVal, sizeof(Boolean));
*/
    boolVal = *(Boolean *) (fromVal->addr);

    if (boolVal) {
	buf = "true";
    } else {
	buf = "false";
    } 
 
/*
    toVal->addr = &buf;
    toVal->size = sizeof(char *);
*/
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtIntToString --
 *
 *	Converter from Int to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtIntToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    int intVal;
 
    intVal = *(int *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%d", intVal);
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtDimensionToString --
 *
 *	Converter from Dimension to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtDimensionToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Dimension dimVal;
 
    dimVal = *(Dimension *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%hu", dimVal);
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CvtShortToString --
 *
 *	Converter from Short to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
Tm_CvtShortToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Dimension shortVal;
 
    shortVal = *(short *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%hd", shortVal);
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_RegisterConverters --
 *
 *	Register all extra converters with Xt.
 *
 * Results:
 *
 * Side effects:
 *
 *	Modifies the Xt converter lists.
 *--------------------------------------------------------------
 */

void
Tm_RegisterConverters(interp, appContext)
    Tcl_Interp *interp;
    XtAppContext appContext;
{
    XtConvertArgRec convert_arg;
 
    /*
     * Register the type converters we need
     * This implementation installs everything, 
     * even if it is never used. A smarter method
     * would be to install on demand. Maybe someday...
     */

    InstallReverseRepTypes();

    /* 
     * String to Widget
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XtRWidget,
		Tm_CvtStringToWidget,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /*
     * Motif 1.2.1 MainW.c has bugs in it - resources like menuBar are declared
     * as type Window not Widget. So we have to have another converter just
     * to handle this bug
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XtRWindow,
		Tm_CvtStringToWidget,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * String to MenuWidget
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XmRMenuWidget,
		Tm_CvtStringToWidget,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * XmString to String
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XmRXmString, XtRString,
		Tm_CvtXmStringToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * XmStringTable to String
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XmRXmStringTable, XtRString,
		Tm_CvtXmStringTableToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * String to String
     * I don't know why there isn't already one of these,
     * but getting value out of Text seems to need it.
     *
     * it doesn't seem to work anyway - maybe something
     * special case away X to X convertions
     */
/*
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XtRString,
		Tm_CvtStringToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);
*/

    /* 
     * Int to String
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XmRInt, XtRString,
		Tm_CvtIntToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);


    /* 
     * Boolean to String
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XmRBoolean, XtRString,
		Tm_CvtBooleanToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * Dimension to String
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XmRDimension, XtRString,
		Tm_CvtDimensionToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * HorizontalDimension to String
     * Buggy? should use UnitType resource
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XmRHorizontalDimension, XtRString,
		Tm_CvtDimensionToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * VerticalDimension to String
     * Buggy? should use UnitType resource
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XmRVerticalDimension, XtRString,
		Tm_CvtDimensionToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);

    /* 
     * short to String
     */
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XmRShort, XtRString,
		Tm_CvtShortToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);
}
