#
# Copyright (c) 1994 Open Software Foundation, Inc.
# 
# Permission is hereby granted to use, copy, modify and freely distribute
# the software in this file and its documentation for any purpose without
# fee, provided that the above copyright notice appears in all copies, and
# that both the copyright notice and this permission notice appear in
# supporting documentation.  Further, provided that the name of Open
# Software Foundation, Inc. ("OSF") not be used in advertising or
# publicity pertaining to distribution of the software without prior
# written permission from OSF.  OSF makes no representations about the
# suitability of this software for any purpose.  It is provided "AS IS"
# without express or implied warranty.
#

# countTable - generate count tables.
#
#
# Generate n-dimensional count tables.
#
# Usage:
#   ot_bugs -p<proj> -x '[countTable "Table 1: CRs by engineer" {code} "" ]'
# Arguments to countTable procedure are
#	Title for table - this is mandatory and must be unique w/in x string
#	Brace-enclosed, blank-separated list of keywords repr. table dimensions
#	String representing filter for counts (e.g. "[stat open]" counts only
#	    open bugs - this can be the blank string as above
# 	
# ./ot_bugs -p not -x '[countTable "bugs by engineer" {code prior} "[stat closed]"; countTable "by status" {stat} ""]'
# 
#
# You can generate multiple tables - separate the individual countTable cmds
# w/ semi-colons and surrounded them w/ square brackets, as in Boolean search
# argument to ot_bugs -x flag.
#
# ot_bugs -p<proj -x '[countTable "T1" {code} ""; countTable "T2" {stat} ""]'
#
# open bugs by engineer --> -x '[countTable "by engr" {code} "[stat open]"]'
# all bugs by priority  --> -x '[countTable "priority" {prior} ""]'
#
# Keep a global table of all possible values of all fields (i.e.
#	$status(open) = 1
#	$status(closed) = 1
# etc.
# Keep a global table 'tabElem' of counts and subcounts so for command
#	countTable "mytable" {code stat} ""
# there are tables like
#
# mytable_	-> count of all bugs so far <filtered through condition, here
#			null
# mytable_pnh	-> count of all bugs assigned to pnh
# mytable_pnh_open -> count of all open bugs assigned to pnh
# etc.
#

proc countTable { tableTitle fldlist cond } {

    # OT's tcl support has 'begin', 'end' primitives (think of awk)
    if { [begin] } then { set t("fields") $fldlist ; quiet "1"  ; return "0" }
    if { [end  ] } then { 
	puts stdout [format "%s\n" $tableTitle]
	printTable $fldlist $tableTitle ; return "0"    
    }

    # nb because ot_bugs handles -x string as tcl expression this procedure
    # MUST return "1" or "0", otherwise a syntax error results

    # two interesting TCL features-
    #	"upvar #0 x y" binds variable x at absolute level 0 (global) to local
    # 		variable y
    #   "info exists x" returns "1" if x has been assigned to - dereferencing
    #           unassigned variables is an error in TCL, that is why we check
    #           so often

    upvar #0 tabElem t
    set tabArrayName $tableTitle

    # if condition is false, continue
    if { $cond == "" } {
	set cond "1"
    }

    if $cond then { 
    } else {
	return "0"
    }

    if { ![info exists t($tabArrayName)] } then {
    	set t($tabArrayName) "1"
    } else {
	set t($tabArrayName) [expr "$t($tabArrayName) + 1"]
    }

    foreach element $fldlist {
	upvar #0 $element ${element}T
	
	set fieldVal [$element]
	if { $fieldVal == "" } then {
	    set fieldVal null
	}
	set ${element}T($fieldVal) 1
	if { $fieldVal == "" } then {
	    set tabArrayName ${tabArrayName}_null
	} else {
	    set tabArrayName ${tabArrayName}_${fieldVal}
	}

        if { ![info exists t($tabArrayName)] } then {
	    set t($tabArrayName) "1"
        } else {
	    set t($tabArrayName) [expr "$t($tabArrayName) + 1"]
        }
    }

    return "0"
}

proc printTable { fldlist prefix } {

    # recursive routine to print titles, subtables

    if { [expr "[llength $fldlist] <= 2"] } then {
	printOneTable $fldlist $prefix
    } else {
	set elem [lindex $fldlist 0]
	upvar #0 $elem e

	foreach value [lsort [array names e]] {
	    puts stdout [format "%s:" $value]
	    printTable [lrange $fldlist 1 end] ${prefix}_${value}
	}
    }
    puts stdout [format "\n\n"]
}

proc printOneTable { fldlist prefix } {

    upvar #0 tabElem t

   # special case the situation where there is only one row e.g.
   # 	ot_bugs -x '[countTables "Status" {stat} ""]'

    if { [expr "[llength $fldlist] == 1"] } then {

	set fieldVal [lindex $fldlist 0]
	upvar #0 $fieldVal ${fieldVal}T
	set y [lsort [array names ${fieldVal}T]]
	puts stdout [format "\t%s" $fieldVal] nonewline

	foreach k $y {	# Write header
	    puts stdout [format "\t%s" $k] nonewline
	}
	puts stdout [format "\ttotal\n\t"] nonewline
	foreach k $y {	# Write values
	    set arrayName ${prefix}_${k}
	    if { [info exists t($arrayName)] } then {
		puts stdout [format "\t%d" $t($arrayName) ] nonewline
	    } else {
		puts stdout [format "\t0"] nonewline
	    }
	}
	set arrayName ${prefix}
	if { [info exists t($arrayName)] } then {
	    puts stdout [ format "\t%d" $t($arrayName) ]
	} else {
	    puts stdout [ format "\t0"]
	}

    } else {

    # print a table - first determining rows and columns
	set row [lindex $fldlist 0]
	upvar #0 $row ${row}T
	set rowNames [lsort [array names ${row}T]]
	set column [lindex $fldlist 1]
	upvar #0 $column ${column}T
	set columnNames [lsort [array names ${column}T]]	

    # print column headings
	set columnName [$column label]
	puts stdout [format "%s" $columnName] nonewline
	foreach k $columnNames {
	    if { [expr "[string length $k] > 7"] } then {
		puts stdout [format "\t%s" [string range $k 0 6]] nonewline
	    } else {
		puts stdout [format "\t%s" $k] nonewline
	    }
	}
	puts stdout "\ttotal\n"

    # print rows
	set rowName [$row label]
	puts stdout [format "%s\n" $rowName]

	foreach k $rowNames {
	    if { [expr "[string length $k] > 15"] } then {
		puts stdout [format "%s" [string range $k 0 14]] nonewline
	    } else {
		puts stdout [format "%s" $k] nonewline
	    }
	    foreach m $columnNames {
		set arrayName ${prefix}_${k}_${m}
		if { [info exists t($arrayName)] } then {
		    set currentCount $t($arrayName)
		} else {
		    set currentCount "0"
		}
    # create intermediate totals per column
		puts stdout [format "\t%d" $currentCount] nonewline
		if { [info exists colTotal($m)] } then {
		    set colTotal($m) [expr "$colTotal($m) + $currentCount"]
		} else {
		    set colTotal($m) $currentCount
		}
	    }
	    set arrayName ${prefix}_${k}
	    if { [info exists t($arrayName)] } then {
		puts stdout [ format "\t%d" $t($arrayName) ]
	    } else {
		puts stdout [ format "\t0"]
	    }
	}
	puts stdout [format "\ntotal"] nonewline
	foreach m $columnNames {
	    puts stdout [format "\t%d" $colTotal($m)] nonewline
	}
	set arrayName ${prefix}
	puts stdout [format "\t%d\n" $t($arrayName)]
    }

}
