/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

/*
 * OT 3.0.2
 */

/*
 *	otValidate.c
 *
 */


#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <tcl.h>
#include <tclInt.h>
#include "ot.h"
#include "otInt.h"


#ifndef LINT
static char RCSid_otValidate[] =
    "$RCSfile: otValidate.c,v $ $Revision: 1.1.6.3 $ $Date: 1994/02/25 19:11:01 $";

#endif


/*
 *		 List of functions defined in the otValidate.c
 *
 *	otValidateLhs()
 *	otValidateData()
 *	otValidateDepend()
 *	
 *
 */

#define VAL(t, cp, prj, op)    (*(otTypes[getTypeIndex((t)->type)].tVal))((cp),(prj),(t),(op))
#define MAX(a,b) (((a)>(b))?(a):(b))

/*
 * Temporary stubs.
 */
OTErr otValidateEnum();
OTErr otValidateElement();
OTErr otValidateTrue();
OTErr otValidateDate();
OTErr otValidateIdnum();
OTErr otValidateNum();

OTErr ValidateLhsMandatory();
OTErr ValidateLhsUser();
OTErr ValidateUserByMeta();
OTErr ValidateSeries();

void strTable();

struct {
    char *  tname;      /* text for template keyword */
    otType  tcode;      /* internal type code */
    OTErr   (*tVal)();  /* validating routine */
} otTypes[] = {
    { "enum",           TYPE_ENUM,      otValidateEnum          },
    { "list",           TYPE_LIST,      otValidateElement	},
    { "text",           TYPE_TEXT,      otValidateTrue		},
    { "mailname",       TYPE_MAILNAME,  otValidateTrue		},
    { "mm/dd/yy",       TYPE_DATE,      otValidateDate		},
    { "date",           TYPE_DATE,      otValidateDate		},
    { "idnum",          TYPE_IDNUM,     otValidateIdnum		},
    { "num",            TYPE_NUM,       otValidateNum		},
    { "",	        TYPE_NULL,      otValidateTrue		}
};

/* otGetType() - return a type in return for its name. */
otType
otGetType(n)
char *n;
{
    register int i;
    otType type = TYPE_ENUM;		/* returns TYPE_ENUM by default */

    if ( !n || ! *n ) {
	type = TYPE_NULL;
    } else  {
	for (i=0; otTypes[i].tname[0]; i++) {
	    if ( ! strcmp(otTypes[i].tname, n) ) {
		type = otTypes[i].tcode;
		break;
	    }
	}
    }

    DBUG_MED((stderr, "getType: %s -> %d\n", n, type));

    return type;
}

/* getTypeIndex() - return an index into otTypes[] for element with type t. */

int getTypeIndex(t)
otType t;
{
    register int i;

    for (i=0; otTypes[i].tname; i++) {

	if ( otTypes[i].tcode == t ) {

	    DBUG_MED((stderr, "getTypeIndex: %d -> %d\n", t, i));
	    return i;
	}
    }
    return -1;
}

/* otGetTypeName() - Takes type code (e.g., TYPE_ENUM) and returns typename. */

char *
otGetTypeName(type)
otType type;
{
    register int i;
    char *cp;

    cp = 0;
    if ( (i = getTypeIndex(type)) != -1 ) {
	cp = otTypes[i].tname;
    }
    DBUG_MED((stderr, "otGetTypeName: %d -> %s\n", type, cp));

    return cp;
}





OTErr
otValidateData()
{
    OTErr valErr, finalErr;
    OTTemplate *tStruct;
    OTHeaderField *hfld;
    OTControlBlock * control;
    OTProject *pStruct;
    char *startp, *nextp, *cp, *tmp, *tmcp;
    int i;

    control = otCB;
    pStruct = control->cb_pcb->pcb_project;
    tStruct = control->cb_ecb->ecb_tStruct;
    valErr = finalErr = OT_SUCCESS;
    /*
     * Calls internal function ValidateData() for each header field.
     * Find template in cb->cb_ecb->ecb_tStruct.
     */
    for ( hfld = tStruct->tr; hfld->field && *hfld->field; hfld++ ) {

        DBUG_MIN((stderr,"otValidateData %s val %s\n", hfld->field, hfld->value));
	DBUG_MIN((stderr,"\t(%d/%d)\n", hfld->optionality, hfld->type));

	if ( !hfld->value || !*hfld->value ) {
	    if (hfld->optionality == OPT_MAND) {
		otPutPostedMessage( OT_MANDATORY_FIELD , hfld->field );
		valErr = OT_INVALID;
	    } 
	    else
		valErr = OT_SUCCESS;
	} else if ( hfld->degree == DEGREE_MANY ) {
	    if ( (tmp = startp = nextp = strdup(hfld->value)) == NULL ) {
		otPutPostedMessage( OT_MALLOC_LOCATION, "otValidateData()" );
		return OT_MALLOC_LOCATION;
	    }
	    if ( tmcp = strchr(nextp, '#') )
	        *tmcp = 0;
	    otTrimWs(nextp);
	    while (nextp && *nextp && !OT_WARN(valErr) && !OT_SYSTEM(valErr)) {
		startp = nextp;
		if ( nextp = strchr(startp, ',') ) {
		    *nextp++ = 0;
		    while (isspace(*nextp))
			nextp++;
		}
		valErr = VAL( hfld, startp, pStruct, &(control->cb_operation) );
		if ( valErr )
		    finalErr = OT_INVALID;
	    }
	    free(tmp);
	} else if ( strchr(hfld->value, ',') && (hfld->type != TYPE_TEXT) 
	    && (hfld->type != TYPE_NULL) ) {
	    otPutPostedMessage( OT_NOT_MULTIPLE, hfld->field );
	    valErr = OT_INVALID;
	} else {
	    if ( (tmp = strdup(hfld->value)) == NULL ) {
		otPutPostedMessage( OT_MALLOC_LOCATION, "otValidateData()" );
		return OT_MALLOC_LOCATION;
	    }
	    if ( tmcp = strchr(tmp, '#') )
	        *tmcp = 0;
	    otTrimWs(tmp);

	    valErr = VAL( hfld, tmp, pStruct, &(control->cb_operation) );
	    free(tmp);
	}
	
	if ( OT_WARN(valErr) || OT_SYSTEM(valErr) )
	    return valErr;
	if ( valErr )
	    finalErr = OT_INVALID;
    }
	

#ifdef notdef
    /*
     * Now validate dates in notes.
     */
    for (i=0; tStruct->notep[i].who[0]; i++) {
	OTDateComparison valueD;

	if (otParseDate( tStruct->notep[i].date, &valueD )) {
	    otPutPostedMessage( OT_INVALID_FIELD_DATE, "Note field", tStruct->notep[i].date );
	    finalErr = OT_INVALID;
	}
    }
#endif

    return finalErr;

}


/*
 * otValidateDate - Validate the date.
 *
 * Parameter list for these: 
 * 	first, char *, cannot be just template->value because it is a value
 * 	   parsed from (possibly) multiple value string by strtok()
 *	second, struct ot_project *, may be necessary for baselines, proj_plan
 *	   field 
 *	third, struct ot_t_record *, may be necessary for list member (when
 *         validating against a list in template)
 */
OTErr
otValidateDate(datestr, proj, hfld, oper )
char          *datestr;
OTProject     *proj;
OTHeaderField *hfld;
OTOperation   *oper;
{
    OTDateComparison dummy;
    OTErr dateErr;

    DBUG_MED((stderr, "otValidateDate %s\n", datestr));

    if (dateErr = otParseDate(datestr, &dummy))
        otPutPostedMessage( OT_INVALID_FIELD_DATE, hfld->field, datestr ? datestr : "" );

    return dateErr;

}




/* otValidateEnum - Validate user-specified type. */

OTErr otValidateEnum(cp, proj, hfld, oper)
char *cp;
OTProject *proj;
OTHeaderField *hfld;
OTOperation * oper;
{
    OTErr enumErr = OT_SUCCESS;
    char *elist;

    DBUG_MED((stderr, "otValidateEnum %s\n", cp));
    if ( cp && *cp ) {
        if (elist = otGetEnumtypeList(hfld->list, proj)) {
	    if ( enumErr = otCompare( cp, TYPE_ENUM, elist ) ) {
		if ( enumErr == OT_NOMATCH ) {
		    otPutPostedMessage( OT_INVALID_FIELD_ENUMTYPE, hfld->field,
			cp ? cp : "", hfld->list );
		    enumErr = OT_INVALID_FIELD_ENUM;
		}
	    }
	}
	else if ( enumErr = otCompare( cp, TYPE_ENUM, hfld->list ) ) {
	    otPutPostedMessage( OT_INVALID_FIELD_ENUM, hfld->field, 
		cp ? cp : "", hfld->list );
	    enumErr = OT_INVALID_FIELD_ENUM;
	}
    }

    DBUG_MED((stderr, "otValidateEnum ret\n"));
    return enumErr;
}


/*
 * otValidateNum - Validate datum of type num against a regular expression.
 */
OTErr otValidateNum(cp, proj, hfld, oper)
char *cp;
OTProject *proj;
OTHeaderField *hfld;
OTOperation * oper;
{
    OTErr numErr = OT_SUCCESS;
    char *startp;

    startp = cp;
    if ( cp )
	for ( ;  *cp; cp++)
	    if (!isdigit(*cp)) {
                DBUG_MED((stderr, "otValidateNum %c is not digit\n", *cp));
		break;
   	    }

    if ( cp && *cp ) {
        numErr = OT_INVALID_FIELD_NUM;
	otPutPostedMessage( OT_INVALID_FIELD_NUM, hfld->field, startp ? startp : "" );
    }
    DBUG_MED((stderr, "otValidateNum ck %s %s\n", cp, numErr == OT_SUCCESS ? "succeeded" : "failed"));

    return numErr;

}

/*
 * otValidateIdnum
 * Validate datum of type idnum against six 'n' and numbers
 */
OTErr
otValidateIdnum(cp, proj, hfld, oper)
char *cp;
OTProject *proj;
OTHeaderField *hfld;
OTOperation * oper;
{
    char *msg;
    OTErr idnumErr = OT_SUCCESS;

    if ( cp ) {
	if ( *oper == ENTER ) {
	    if (strcmp(cp, "nnnnnn")) {
		otPutPostedMessage( OT_INVALID_FIELD_IDNUM, hfld->field, cp ? cp : "", "nnnnnn");
		idnumErr = OT_INVALID_FIELD_IDNUM;
	    }
	}
	else {
            for ( ;  *cp; cp++)
                if (!isdigit(*cp)) {
                    DBUG_MED((stderr, "otValidateIdnum %c non digit\n", *cp));
                    break;
                }
	    if ( cp && *cp ) {
		otPutPostedMessage( OT_INVALID_FIELD_IDNUM, hfld->field, cp ? cp : "", "an integer");
		idnumErr = OT_INVALID_FIELD_NUM;
	    }
	}

    }

    DBUG_MED((stderr, "otValidateIdnum %s %s\n", cp, idnumErr == OT_SUCCESS ? "succeeded" : "failed"));

    return idnumErr;
}

/*
 * otValidateElement
 * Validate datum of a list type.  This is now obsolete as enumerated types
 * are always converted to type TYPE_ENUM.
 */
OTErr
otValidateElement(cp, proj, hfld, oper)
char *cp;
OTProject *proj;
OTHeaderField *hfld;
OTOperation * oper;
{
    OTErr elemErr = OT_SUCCESS;

    DBUG_MED((stderr, "otValidateElement %s\n", cp));
    if ( cp && *cp ) {
	if ( elemErr = otCompare( cp, TYPE_LIST, hfld->list ) ) {
	    otPutPostedMessage( OT_INVALID_FIELD_ELEM, hfld->field, 
		cp ? cp : "", hfld->list );
	    elemErr = OT_INVALID_FIELD_ELEM;
	}
    }

    DBUG_MED((stderr, "otValidateElement ret\n"));
    return elemErr;
}

/*
 * otValidateTrue
 * Validate any datum as true.
 * This is used for elements with no validation information in the
 * template (e.g., Ticket Number :).
 */
OTErr
otValidateTrue(cp, proj, hfld, oper)
char *cp;
OTProject *proj;
OTHeaderField *hfld;
OTOperation *oper;
{

    DBUG_MED((stderr, "otValidateTrue\n"));
    return OT_SUCCESS;

}



OTErr
otValidateDepend()
{
    int tclRes, i, j;
    char *newStatp, *oldStatp;
    char date[NAMELEN];
    OTTemplate *tStruct, *origStruct;
    OTOperation op;
    OTProject *pStruct;
    OTErr evalStat, depStat;
    Tcl_Interp *interp;
    bool k;

    depStat = OT_SUCCESS;
    tStruct = otCB->cb_ecb->ecb_tStruct;
    origStruct = otCB->cb_ecb->ecb_origStruct;
    interp = otCB->cb_pcb->pcb_interp;
    op = otCB->cb_operation;

    evalStat = otTclEvalProc("validateDepend");
    if ( OT_WARN( evalStat ) || OT_SYSTEM( evalStat ) ) {
        /*
	 * OT_MALLOC_LOCATION or OT_TCL_CALLER are meaningful warnings.
	 */
	depStat = evalStat;
    }
    else if (evalStat == OT_SUCCESS && !(interp->result[0])) {
	/*
	 * This means the empty string was returned: valid.
	 */
        depStat = OT_SUCCESS;
    }
    else if (evalStat == OT_SUCCESS && interp->result[0]) {
        otPutPostedMessage( OT_INVALID, interp->result );
        /*
	 * This means a diagnostic string was returned from the TCL
	 * procedure: return OT_INVALID.
	 */
	depStat = OT_INVALID;
    }

    return depStat;
}


OTErr
otPostProcess()
{
    int tclRes;
    OTErr evalStat, retStat;
    Tcl_Interp *interp;

    retStat = OT_SUCCESS;
    interp = otCB->cb_pcb->pcb_interp;

    evalStat = otTclEvalProc("postProcess");

    if (evalStat == OT_TCL_NOPROC) {
	/*
	 * No "postsProcess" procedure defined by the administrator
	 */
	 return retStat;
    }

    if ( OT_WARN( evalStat ) || OT_SYSTEM( evalStat ) ) {
	/*
	 * OT_MALLOC_LOCATION or OT_TCL_CALLER are meaningful warnings.
	 */
	 retStat = evalStat;
    }
    else if (evalStat == OT_SUCCESS && !(interp->result[0])) {
	/*
	 * This means the empty string was returned: valid.
	 */
	 retStat = OT_SUCCESS;
    }
    else if (evalStat == OT_SUCCESS && interp->result[0]) {
	otPutPostedMessage( OT_INFORMATION, interp->result );
	/*
	 * This means an informational diagnostic string was returned from 
	 * the TCL procedure: return OT_INFORMATION.
	 */
	 retStat = OT_INFORMATION;
    }

    return retStat;	
}

/*
 *                           V a l i d a t e L h s
 *
 * Validate the left-hand side as follows:
 *
 * 1. Any mandatory field in the metatemplate missing in the user template
 *    causes an error, unless the project file has
 *    "strict lhs compliance" set to off.
 * 2. Each field in the user template with a name which is identical to a
 *    field name in the metatemplate must have paren/bracket/degree info
 *    identical to that of the corresponding metatemplate field.  If it
 *    doesn't it causes an error, unless the project file has "strict
 *    lhs compliance" set to off.
 *    NB the user template may have fields not present in the metatemplate.
 *    That is never an error.
 * 3. If a field in the user template and a field in the metatemplate have
 *    the same name but different types and the project file has "strict
 *    lhs compliance" turned off, then make sure that if the
 *    user template lhs is a list, then 1) the original lhs for the
 *    corresponding field (i.e., the lhs of the template which the user
 *    is updating) is a list and 2) each of the user template
 *    lhs' elements is a member of the either the original lhs list or
 *    the metatemplate lhs elements.  With strict compliance, the elements
 *    of the user lhs list must be the same as the metatemplate lhs list,
 *    although they need not be in the same order.
 */
OTErr
otValidateLhs()
{
    OTErr lhsMandStatus, lhsUserStatus, lhsEndStatus;
    
    lhsMandStatus = ValidateLhsMandatory();
    lhsUserStatus = ValidateLhsUser();

    /*
     * ValidateLhsMandatory() returns either OT_LHS_INVALID or OT_SUCCESS.
     * ValidateLhsUser() returns OT_LHS_INVALID, OT_MALLOC_LOCATION or 
     *    OT_SUCCESS.
     */

    if (lhsUserStatus == OT_MALLOC_LOCATION)
	lhsEndStatus = OT_MALLOC_LOCATION;
    else if (lhsMandStatus == OT_LHS_INVALID || lhsUserStatus == OT_LHS_INVALID )
	lhsEndStatus = OT_LHS_INVALID;
    else 
        lhsEndStatus = OT_SUCCESS;
    return lhsEndStatus;

}


/*
 * ValidateLhsMandatory - Validate left-hand side for mandatory fields.
 */

OTErr
ValidateLhsMandatory()
{
    register int j, k;
    bool matchToUserField, templateValid;
    OTProject *pStruct;
    OTMetaTemplate *metaFields;
    OTTemplate *origtmpl;          /* 0 -> enter, !0 -> update */
    OTTemplate *tmpl;

    origtmpl   = otCB->cb_ecb->ecb_origStruct;
    tmpl       = otCB->cb_ecb->ecb_tStruct;
    pStruct    = otCB->cb_pcb->pcb_project;
    metaFields = pStruct->otmp;

    if (origtmpl && !strcmp(pStruct->compliance, "loose"))
	return OT_SUCCESS;
    
    templateValid = TRUE;

    /*
     * Are all mandatory fields really in the template?
     */

    for ( j=0; metaFields[j].field[0]; j++) {

	DBUG_MED((stderr, "metafields[%d].field = %s\n", j, metaFields[j].field));

	if (!strcmp(metaFields[j].field, "!"))
	    continue;

	if ( metaFields[j].optionality == OPT_MAND ) {

	    matchToUserField = FALSE;

	    DBUG_MED((stderr, "tmpl field %s\n", 0, tmpl->tr[0].field));

	    for ( k=0 ; tmpl->tr[k].field && tmpl->tr[k].field[0] && !matchToUserField; k++) {
		DBUG_MED((stderr, "tmpl->tr[%d].field = %s\n", k, tmpl->tr[k].field));
		if ( !strcmp(tmpl->tr[k].field, metaFields[j].field) ) {
		    DBUG_MED((stderr, "Mandatory field %s, value %s\n", tmpl->tr[k].field, tmpl->tr[k].value));
		    if (tmpl->tr[k].optionality != OPT_MAND) {
			otPutPostedMessage(OT_LHS_MANDATORY,tmpl->tr[k].field);
			templateValid = FALSE;
		    }
		    matchToUserField = TRUE;
		}
	    }

	    if (!matchToUserField) {
		otPutPostedMessage(OT_LHS_MANDATORY_USER, metaFields[j].field);
		templateValid = FALSE;
	    }
	}

    } /* for all fields in metatemplate */

    for (j = 0; tmpl->tr[j].field && tmpl->tr[j].field[0]; j++) {
	for (k=j+1;  tmpl->tr[k].field && tmpl->tr[k].field[0];  k++) {

	    if ( (tmpl->tr[k].field[0] == tmpl->tr[j].field[0]) && 
		!strcmp(tmpl->tr[k].field,  tmpl->tr[j].field) ) {
		otPutPostedMessage( OT_DUPLICATE_FIELD, tmpl->tr[j].field );
		templateValid = FALSE;
	    }
	}
    }


    DBUG_MED((stderr, "ValidateLhsMandatory returns %d\n", templateValid));
    if (!templateValid)
	return OT_LHS_INVALID;
    else
        return OT_SUCCESS;

}

/* ValidateLhsUser - Compare user's template fields to the metatemplate */

OTErr ValidateLhsUser()
{
    register int h, j, k;
    int idnum_count = 0;
    bool templateValid = TRUE;
    bool matchToMetaField;
    OTProject *pStruct;
    OTMetaTemplate *metaFields;
    OTTemplate *origtmpl;
    OTTemplate *tmpl;
    OTErr elementErr;

    origtmpl = otCB->cb_ecb->ecb_origStruct;
    tmpl     = otCB->cb_ecb->ecb_tStruct;
    pStruct  = otCB->cb_pcb->pcb_project;
    metaFields = pStruct->otmp;
    /*
     * Now check each existing field in the user template against the
     * metatemplate.  The user can add fields not in the metatemplate, but
     * any field with the same name as a field in the metatempl. must have
     * the same validation criteria.
     */
    for ( k=0; tmpl->tr[k].field && tmpl->tr[k].field[0] ; k++ ) {

	if ( tmpl->tr[k].type == TYPE_IDNUM )
	    if ( ++idnum_count == 2 ) {
	        otPutPostedMessage( OT_LHS_ONE_IDNUM );
		templateValid = FALSE;
	    }

	h = origtmpl ? find_template_field(origtmpl, tmpl->tr[k].field) : -1;
	matchToMetaField = FALSE;

	for (j=0; metaFields[j].field[0] && !matchToMetaField;  j++)
	    if (!strcmp( tmpl->tr[k].field, metaFields[j].field)) {
		bool elementsValid = FALSE;

		DBUG_MED((stderr, "User field %s matched %s\n", tmpl->tr[k].field, metaFields[j].field));

		matchToMetaField = TRUE;

		elementErr = ValidateUserByMeta(pStruct, &tmpl->tr[k], 
		    origtmpl ? &origtmpl->tr[h] : 0, &metaFields[j]); 

		if ( elementErr == OT_MALLOC_LOCATION ) 
		    return OT_MALLOC_LOCATION;
		else if ( elementErr == OT_SUCCESS )
		    elementsValid = TRUE;
		else
		    elementsValid = FALSE;
		templateValid = templateValid && elementsValid;
	    }

	if ( !matchToMetaField ) {
	    /*
	     * Right now, added fields are OK so do nothing.
	     */
	}
    }

    DBUG_MED((stderr, "ValidateLhsUser returns %d\n", templateValid));
    if (!templateValid)
        return OT_LHS_INVALID;
    else
        return OT_SUCCESS;

}

/* ValidateUserByMeta */

OTErr ValidateUserByMeta(pStruct,user,orig,meta)
OTProject *pStruct;
OTHeaderField *user;
OTHeaderField *orig;
OTMetaTemplate *meta;
{
    bool fieldValid = TRUE;
    bool seriesValid = TRUE;
    bool loose;
    OTErr serErr;

    loose = (orig && !strcmp(pStruct->compliance, "loose"));

    DBUG_MED((stderr,"Entering ValidateUserByMeta compli %s\tuser %s meta %s\n", pStruct->compliance, user->field, meta->field ));

    if (user->type != meta->type && !loose) { 
        otPutPostedMessage( OT_LHS_TYPE, user->field, user->list, meta->list );
	fieldValid = FALSE;
    }

    if (user->optionality == OPT_MAND && meta->optionality == OPT_OPTL && 
	!loose) {
        otPutPostedMessage( OT_LHS_OPT, user->field);
	fieldValid = FALSE;
    }

    if (user->degree != meta->degree && !loose) {
	DBUG_MED((stderr, "Degree: user %d original %d metatemplate %d\n", user->degree, (orig ? orig->degree : DEGREE_NONE), meta->degree));

	otPutPostedMessage( OT_LHS_DEGREE, user->field,
	    meta->degree == DEGREE_ONE ? "one value (no + before colon)" :
	    "multiple valued (+ before colon)" );
	fieldValid = FALSE;
    }

    if (user->type == TYPE_ENUM) {
	if (!strcmp(user->list, meta->list)) {
	    /* Perfect match */
	}
	else {

	    DBUG_MAX((stderr, "\tUser list: %s\tmetatemplate list: %s\n", user->list, meta->list));
	    DBUG_MAX((stderr, "\t\tOrig list: %s\n", (orig ? orig->list: "(null)")));

	    /*
	     * Match series of elements where the metatemplate
	     * and user template have the same elements, but
	     * in a different order, or the elements are really
	     * different.
	     */
	    serErr = ValidateSeries(pStruct, user->field, user->list, meta->list, orig ? orig->list : "");

	    if (serErr == OT_MALLOC_LOCATION)
		return serErr;
	    else if (serErr == OT_SUCCESS)
	        seriesValid = TRUE;
	    else 
	        seriesValid = FALSE;
	}
    }
    if ( fieldValid && seriesValid )
        return OT_SUCCESS;
    else
        return OT_LHS_INVALID;

}


/* ValidateSeries - Insure series have same elements */
OTErr ValidateSeries(pStruct, fieldName, userV, metaV, origV)
OTProject *pStruct;
char *fieldName;
char *userV;
char *metaV;
char *origV;
{
    char **userTable, **metaTable, **origTable;
    char *userVal, *metaVal, *origVal;
    char *cp;
    int u, m, x;
    int uc, mc, xc;    
    int tabLen;
    bool metaFound, origFound, userFound;
    int seriesValid = TRUE;

    for(uc = 0, cp=userV; *cp; *cp++==','?uc++:0 );
    for(mc = 0, cp=metaV; *cp; *cp++==','?mc++:0 );
    for(xc = 0, cp=origV; *cp; *cp++==','?xc++:0 );
    tabLen = MAX( MAX(uc,mc), xc) + 2;

    userTable = metaTable = origTable = (char **) 0;
    userTable = (char **)malloc(tabLen * sizeof(char *));
    metaTable = (char **)malloc(tabLen * sizeof(char *));
    origTable = (char **)malloc(tabLen * sizeof(char *));

    if ( !userTable || !metaTable || !origTable ) {
        if ( userTable ) free ( userTable );
	if ( metaTable ) free ( metaTable );
	if ( origTable ) free ( origTable );
	otPutPostedMessage( OT_MALLOC_LOCATION, "ValidateSeries()" );
    }

    DBUG_MED((stderr, "ValidateSeries compliance %s field %s user values /%s/ metatemplate values /%s/ original values /%s/\n", pStruct->compliance, fieldName, userV, metaV, origV));

    /*
     * Make temporary list element strings.
     */
    userVal = metaVal = origVal = 0;

    userVal = strdup(userV);
    metaVal = strdup(metaV);
    origVal = strdup(origV);
    
    if ( !userVal || !metaVal || !origVal ) {
	if ( userVal ) free (userVal) ;
	if ( metaVal ) free (metaVal) ;
	if ( origVal ) free (origVal) ;
	otPutPostedMessage( OT_MALLOC_LOCATION, "ValidateSeries()" );
	return ;
    }

    u = m = x = 0;
    strTable(userTable, userVal);
    strTable(metaTable, metaVal);
    strTable(origTable, origVal);

    DBUG_MED((stderr, "ValidateSeries %s %s ...\n", userTable[0], metaTable[0]));
    if (*origV != '\000' && pStruct && !strcmp(pStruct->compliance, "loose")) {

	/*
	 * This is an update operation with loose compliance.
	 *
	 * Check each of the user's values against the ones in the
	 * metatemplate.
	 * Even w/ loose compliance it is an error to have a value in
	 * the series which is neither in the original template nor
	 * in the metatemplate.
	 */
	for (u = 0; userTable[u]; u++) {

	    metaFound = origFound = FALSE;

	    for(m = 0; metaTable[m] && !metaFound; m++)
		if (!strcmp(userTable[u], metaTable[m]) )
		    metaFound = TRUE;
	    for (x = 0; origTable[x] && !origFound; x++)
	        if (!strcmp(userTable[u], origTable[x]) )
		    origFound = TRUE;

	    if ( !metaFound && !origFound ) {
		otPutPostedMessage( OT_LHS_BOTHVAL_MISSING, fieldName, userTable[u]);
		seriesValid = FALSE;
	    }
	}
    }
    else {
	/*
	 * Enter operation, or update operation with strict compliance.
	 * This template comes directly from the current
	 * template so there should be no change in the list elements.
	 */
	for (m = 0; metaTable[m]; m++) {
	    userFound = FALSE;

	    for (u = 0; userTable[u] && !userFound; u++)
		if ( !strcmp(metaTable[m], userTable[u]) )
		    userFound = TRUE;

	    if (!userFound) {
	        otPutPostedMessage( OT_LHS_VAL_MISSING, fieldName, metaTable[m] );
		seriesValid = FALSE;
	    }
	}

	for (u = 0; userTable[u]; u++) {
	    metaFound = FALSE;
	    for (m = 0; metaTable[m] && !metaFound; m++)
		if ( !strcmp(userTable[u], metaTable[m]) )
		    metaFound = TRUE;

	    if (!metaFound) {
	        otPutPostedMessage( OT_LHS_NOTPRESENT, fieldName, userTable[u]);
		seriesValid = FALSE;
	    }
	}
    }

    free(userVal);
    free(metaVal);
    free(origVal);
    if (seriesValid == TRUE)
        return OT_SUCCESS;
    else
        return OT_LHS_INVALID;

}

/* find_template_field - Return index of template fld corresponding to name. */

int
find_template_field(tmpl, name)
OTTemplate *tmpl;
char *name;
{
    register int i, ret_val;

    if ( !tmpl || !name )
	ret_val = -1;
    else {
	for ( i=0; tmpl->tr[i].field && tmpl->tr[i].field[0] ; i++ ) {

	    if (!strcmp(tmpl->tr[i].field,name)) {
		ret_val = i;
		break;
	    }
	}
    }

    DBUG_MED((stderr, "find_template_field returns %d\n", ret_val));

    return ret_val;
}

/* strTable - create array of words from string of ws-delimited words */
void strTable(table, string)
char *table[];
char *string;
{
    char *np;
    int i = 0;
    
    if (!string)
	return;

    for (np = strtok(string, ", "); np; np = strtok(0, ", "))
	table[i++] = np;

    table[i] = 0;

}
