# listbox.tcl --
#
# This file contains Tcl procedures used to manage Tk listboxes.
#
# $Header: /user6/ouster/wish/library/RCS/listbox.tcl,v 1.3 93/07/01 13:42:05 ouster Exp $ SPRITE (Berkeley)
#
# Copyright (c) 1992-1993 The Regents of the University of California.
# All rights reserved.
#
# Permission is hereby granted, without written agreement and without
# license or royalty fees, to use, copy, modify, and distribute this
# software and its documentation for any purpose, provided that the
# above copyright notice and the following two paragraphs appear in
# all copies of this software.
#
# IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
# DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
# OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
# CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
# AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
#

# The procedure below may be invoked to change the behavior of
# listboxes so that only a single item may be selected at once.
# The arguments give one or more windows whose behavior should
# be changed;  if one of the arguments is "Listbox" then the default
# behavior is changed for all listboxes.

proc tk_listboxSingleSelect args {
    foreach w $args {
	bind $w <B1-Motion> {%W select from [%W nearest %y]} 
	bind $w <Shift-1> {%W select from [%W nearest %y]}
	bind $w <Shift-B1-Motion> {%W select from [%W nearest %y]}
    }
}

############ Listbox scrolling functions ################
#
# These functions (LBscroll_sb & LBscroll_drag) vastly improve
# the action of listboxes when they are scrolled around.  Out of
# the box TK lets you drag the listbox down to a point where there's
# only one item at the top of the screen, whereas it is more normal
# and better UI design to drag only to where the last item in the
# listbox is at the bottom (rather than the top).
#
# These functions implement this policy.  They do this by calculating
# where the window needs to end up.
#
#
#
# LBscroll_sb list scrollbar which total window first last
#
#	This is to scroll the list by means of the scrollbar.
#	This is meant to be used as so:
#
#		listbox .lb	-relief sunken \
#			-yscrollcommand "LBscroll_sb .lb .vs y" \
#			-xscrollcommand "LBscroll_sb .lb .hs x"
#
#	list:	The listbox widget
#
#	scrollbar: The relavent scrollbar widget
#
#	which:	Either `x' or `y' and is used to
#		generate the `yview' or `xview'
#		subcommand.
#
#	total, window, first, last: Provided
#		by the listbox widget.
#
# LBscroll_kb listbox which
#
#	Scrolls by the keyboard.  This is used when keyboard
#	focus has traversed to the listbox.  Out of the box
#	TK does not support this, but should as it is a normal
#	part of Motif and TK is moving very strongly to the
#	Motif L&F.
#
#	It is meant to be used as so:
#
#		bind .lb <Key-Up>   "LBscroll_kb %W Up"
#		bind .lb <Key-Down> "LBscroll_kb %W Down"
#		bind .lb <Key-F27>  "LBscroll_kb %W Home"
#		bind .lb <Key-F29>  "LBscroll_kb %W PgUp"
#		bind .lb <Key-F35>  "LBscroll_kb %W PgDn"
#		bind .lb <Key-R13>  "LBscroll_kb %W End"
#
#	The F27/F29/F35/R13 are generated by my Sun type 4
#	keyboard while running MIT X11R5 (pl19?).  I've seen
#	other keysyms generated from other keyboards.  The
#	ShowKey function below is useful in determining what
#	the keysyms are on your keyboard (as TK sees them).
#
# LBbindScroll listbox
#
#	Sets up bindings as described for LBscroll_kb.
#

proc LBscroll_sb {list sb which total window first last} {
	if {[expr $first+$window] > $total} {
		set first [expr $total-$window]
		set last  [expr $first+$window]
	}
	$list ${which}view $first
	$sb set $total $window $first $last
}

proc LBscroll_kb {lb which} {

	set cur  [$lb nearest 0]
	set last [$lb nearest [winfo height $lb]]
	set sz   [$lb size]
	set disp [expr "($last - $cur) + 1"]

	switch -- $which {
	Up	{
		set cur [expr "$cur <= 0 ? $cur : $cur - 1"]
		$lb yview $cur
		}
	Down	{
		incr cur
		set newend [expr "$cur + $disp"]
		if {$newend >= $sz} { set cur [expr "$sz - $disp"] }
		$lb yview $cur
		}
	PgUp	{
		incr cur "-$disp"
		if {$cur < 0} {set cur 0}
		$lb yview $cur
		}
	PgDn	{
		incr cur $disp
		set newend [expr "$cur + $disp"]
		if {$newend > $sz} { set cur [expr "$sz - $disp"] }
		$lb yview $cur
		}
	Home	{
		$lb yview 0
		}
	End	{
		set cur  [expr "$sz - $disp"]
		$lb yview $cur
		}
	default	{
		error "Unknown scroll request '$lb $which'." \
			"" \
			[list PWMERROR "" -toplevel $lb.error]
		}
	}
}

proc LBbindScroll {} {
	bind Listbox <Key-Up>   "LBscroll_kb %W Up"
	bind Listbox <Key-Down> "LBscroll_kb %W Down"
	bind Listbox <Key-F27>  "LBscroll_kb %W Home"
	bind Listbox <Key-F29>  "LBscroll_kb %W PgUp"
	bind Listbox <Key-F35>  "LBscroll_kb %W PgDn"
	bind Listbox <Key-R13>  "LBscroll_kb %W End"
}

proc ShowKey {} {
	wm title    .showKey "Show Keypresses"
	wm geometry .showKey 500x200

	label .showKey.l -relief flat
	button .showKey.b -text OK -command { destroy .showKey. }
	pack .showKey.l -in .showKey -fill both -expand 1 -side top
	pack .showKey.b -in .showKey -fill x    -expand 0 -side top -padx 5 -pady 5

	focus .showKey.l

	bind .showKey.l <Any-KeyPress> { 
		.showKey.l configure -text "KeyCode: %k; KeySym: %K;"
	}
}



