#!/afs/ece/usr/tcl/bin/wisha -f
# Top-level beth calling routine

# To run this program, you should make sure the top line reflects the location
# of wish on your system, and the following variable represent the location
# of the beth source directory.

if {![info exists beth_dir]} {
	set beth_dir /afs/ece/usr/svoboda/src/beth
}

#
# Beth takes any number of arguments. Args consist of options and configurations
# Each option brings up a separate Beth window. Configs modify all subsequent
# Beth window.
#
# Options
#	file		File to edit.
#	|cmd		Command to execute, load output
#	|cmd&		Command to execute, load output gradually
#	- 		stdin
#	=		stdin (gradually)
#	X		X-selection
#	0		nothing
# Configs
#	-w configs	Configs to be passed to text widget. (such as -height)
#	-c 		controls input (via a global grab)
#	-i 		iconified (overrides -c)
#	-b 		browse (instead of edit)
#	-t title	(to add to current window title)
#	-l script	Load script (which lives in either pwd or beth_dir)
#

# Initial .beth file locations
set config_file ".beth"
set config_dir1 $env(HOME)
set config_dir2 [pwd]

# Tempfile
set default_tmpfile "/tmp/beth.output"

# Traverse through the arguments and fork off some other Beth procs.
# This one will take the first option; no processes are created if only
# one option exists.

if {![info exists beth_exec]} {
	set beth_exec "beth"
}

if {(![info exists option])} {set option ""}
if {(![info exists configs])} {set configs ""}
set other_configs ""
set i 0
while {$i < $argc} {
	set item [lindex $argv $i]
	if {($item == "-t") || ($item == "-l") || ($item == "-w")} {
		incr i
		lappend other_configs $item [lindex $argv $i]
	} elseif {[string match -? $item]} {
		lappend other_configs $item
	} elseif {$option == ""} {
		set option $item
		set configs $other_configs
	} elseif {[string match \|* $item]} {
		eval exec $beth_exec $other_configs \"$item\" &
	} else {eval exec $beth_exec $other_configs $item &}
	incr i
}

if {($option == "") && ![info exists embedded]} {
	puts \
{Beth options: [-c] [-i] [-b] [-t title] [-l script] [-w tcl-text-configs]
               [0 X - = file |cmd |cmd&]}
	exit 1
}

# Tell user what is being instantiated.
# puts "[pid]: beth $configs $option"

proc return_option_assocs {option list} {
	set result ""
	set append_flag 0
	foreach item $list {
		if $append_flag {set result [lappend result $item]}
		set append_flag 0
		if {$item == $option} {set append_flag 1}
	}
	return $result
}

# Instantiate configs.
if {([lsearch $configs -i] >= 0)} {wm iconify .
} elseif {([lsearch $configs -c] >= 0)} {after 1000 {update ; take_control $frame}}
set title_comment [return_option_assocs "-t" $configs]
set load_modules [return_option_assocs "-l" $configs]
set text_configs [return_option_assocs "-w" $configs]
set cwd [pwd]

# Get text and entry widgets
foreach module {strings.tcl bindings.tcl widgets.tcl entrybind.tcl} {
	source $beth_dir/$module
}


# Alter look if we are editing vs browsing
if {([lsearch $configs -b] >= 0)} {
	set edit_flag 0} else {set edit_flag 1}

# Name of this interp
set me [winfo name .]

catch {
if {($edit_flag) && [file exists $option] && ![file writable $option]} {
	set edit_flag 0
	flash_label $frame -text "File is unwritable"
}}

# Load other modules
foreach module {dialog.tcl browsebind.tcl windowbind.tcl commandbind.tcl
		searchbind.tcl interpbind.tcl keybind.tcl} {
	source $beth_dir/$module
}
if {$edit_flag} {foreach module {filebind.tcl editbind.tcl replacebind.tcl} {
	source $beth_dir/$module
}}
set beth_lib_dir $beth_dir/lib
source $beth_lib_dir/libraries.tcl


foreach module $load_modules {
	if {[file exists $cwd/$module]} {source $cwd/$module
	} else {load_library_module $module
}}


# sources file in either $config_dir1 or $config_dir2
proc source_config_file {file} {
	global config_dir1 config_dir2
	if {[file exists $config_dir1/$file]} {
		uplevel #0 source $config_dir1/$file}
	if {($config_dir1 != $config_dir2) &&
	    [file exists $config_dir2/$file]} {
		uplevel #0 source $config_dir2/$file}
}
source_config_file $config_file

proc set_window_title {wm_name} {
	global edit_flag title_comment
	if {[info globals dont_change_title] != ""} {return}

	if {$title_comment != ""} {
		set title_comment [string trim [lindex $title_comment  0] { }]
		set title_comment "$title_comment "}
	if {$edit_flag} {
		wm title . "$title_comment\Edit: $wm_name"
		wm iconname . "$title_comment\E $wm_name"

	} else {wm title . "$title_comment\Browse: $wm_name"
		wm iconname . "$title_comment\B $wm_name"
}}


# Fill text widget with initial material, based on option
if {($option == "-")} {
	set path $cwd
	set name ""
	append wm_name $option
	set_window_title $wm_name
	after 0 beth_busy $text read_file_handler $text end stdin 0 \; $text mark set insert 1.0
} elseif {($option == "=")} {
	set path $cwd
	append wm_name $option
	set_window_title $wm_name
	after 0 beth_busy $text read_file_handler $text end stdin 1 \; $text mark set insert 1.0
} elseif {($option == "X")} {
	set path $cwd
	append wm_name $option
	set_window_title $wm_name
	if {[catch {$text insert insert "[selection get]"}]} {
		flash_label $frame -text "Cannot get X selection"
	}
	$text mark set insert 1.0
} elseif {($option == "0")} {
		set path $cwd
		set name ""
} elseif {($option != "")} {
	if {([string match \|* $option])} {
		set name $option
		set path $cwd
	} else {set name [file tail $option]
		if {([string index $option 0] == "/")} {
			set path [file dirname $option]
		} else {set path $cwd/[file dirname $option]}
	}
	append wm_name $name
	set_window_title $wm_name
	if {[file tail $path] == "."} {set path [file dirname $path]}
	after 0 load_file $text end $frame \"$path\" \"$name\" \; $text mark set insert 1.0
} else {set dont_change_title 1
	set path $cwd
}
cd $path
set cwd [pwd]

catch {set fsBox(name) $name}
catch {set fsBox(path) $path}

# Default generic shell command
set sh_cmd ""
# Make command
set make_cmd "$beth_exec -t {Make} -l makebind.tcl {| make |& cat &} &"
# Print command
set print_cmd "| lpr -m -p -T [lindex [wm title .] 1]"
# Cmd to browse the X region
set browse_cmd "| $beth_exec -t {X-region} - &"
# Cmd to view help
set help_cmd "$beth_exec -b $beth_dir/HELP &"
# Default command to visit a new file
set visit_cmd "$beth_exec -i"
