#
# Strings module
#

# Unless stated otherwise, the following string functions should return the
# following format: {result message}. If result is empty, the command failed.

# The scope in which to perform commands to yield completion results.
# (One could use local or a send command instead, I suppose).
set uplevel_prefix "uplevel #0"

# Replaces s with its history event
proc string_history_event {s} {
	if {$s == ""} {return [list "" ""]}
	if {[catch "history event $s" result]} {
		return [list "" "History search failed because:\n$result"]
	} else {return [list [string trimright $result \n] ""]
}}

# Replaces s with its Tcl result (in global mode) (s = tcl-command])
proc string_tcl_result {s} {
	if {![string match *\] $s]} {return [list "" ""]}
# If the user entered an erroneous Tcl command (i.e. it was *meant* to be
# Tcl (the braces indicate such), let the Tk error handle the problem.
	global uplevel_prefix
	if {[catch "$uplevel_prefix [string trimright $s \]]" result]} {
		return [list "" "Tcl Command failed because:\n$result"]
	} else {return [list $result ""]
}}

# Replaces s with its Tcl value (s == "varname" or "arrayname(index)")
proc string_global_value {s} {
	if {$s == ""} {return [list "" ""]}
	global uplevel_prefix
	if {![catch "$uplevel_prefix [list set $s]" result]} {return [list $result ""]}
	if {![catch "$uplevel_prefix [list array size $s]"]} {return [list "\$$s\(" ""]
	} else {return [list "" ""]
}}

# Replaces s with its Tcl glob equivalent.
proc string_glob_files {s} {
	if {$s == ""} {return [list "" ""]}
	global uplevel_prefix
	if {[catch "$uplevel_prefix glob -nocomplain -- [list $s]" result]} {
		return [list "" ""]}
	if {($result == $s) || ($result == "")} {return [list "" ""]
	} else {return [list " $result" ""]
}}


# Breaks a string by the last occurrence of delimiter.
# Returns a list of {head delimiter tail}
# or {head "" ""} if no delimiter exists.
proc divide_string {s {delimiter " "}} {
	set list [split $s $delimiter]
	set l [llength $list]
	if {$l == 1} {return [list $list "" ""]
	} else {return [list [join [lrange $list 0 [expr $l - 2]] $delimiter] \
			$delimiter [lindex $list [expr $l - 1]]]}
}

# Replaces the substring [delimiter-end], with the substring through filter.
proc replace_substring {filter delimiter s} {
	set list [divide_string $s $delimiter]
	if {([lindex $list 1] == "")} {return [list "" ""]}
	set replacement [$filter [lindex $list 2]]
	if {([lindex $replacement 0] == "")} {return $replacement
	} else {return [lreplace $replacement 0 0 [join [lreplace $list 1 2 [lindex $replacement 0]] ""]]
}}


# String completion

# Removes everything before the \n, inclusive from the message.
proc remove_prefix {message} {
	set index [string first \n $message]
	incr index
	return [string range $message $index end]
}

proc filter_glob {prefix args} {
	global uplevel_prefix
        if {[catch [lappend $uplevel_prefix glob -nocomplain -- $prefix*] result]} {set result ""}
	return "Possible files:\n$result"
}

proc filter_cmds {prefix args} {
	global uplevel_prefix
	return "Possible Tcl commands:\n[eval $uplevel_prefix \
		info commands [list [list $prefix*]]]"
}

# Completes either variable or array index.
proc filter_vars {prefix args} {
	set list [divide_string $prefix \(]
	global uplevel_prefix
	if {[lindex $list 1] == ""} {	return "Possible variables:\n[eval $uplevel_prefix info globals [list $prefix*]]"}
# Must complete array index
	set array [lindex $list 0] ; 	set index [lindex $list 2]
	if {[catch "$uplevel_prefix array names $array" names]} {return ""
	} else {set message [filter_completions $index $names]
		set indices [remove_prefix $message]
		set results ""
		foreach index $indices {
			set results [lappend results "$array\($index\)"]}
		return "Possible array indices:\n$results"
}}

# Returns each possibility that begins with prefix.
proc filter_completions {prefix possibilities} {
	set completions ""
	foreach completion $possibilities {
		if {[string match $prefix* $completion]} {
			lappend completions $completion}}
	return "Possible completions:\n$completions"
}

# Attempts completion of a prefix,
# First function is given prefix and possibilities, and should return a list
# of valid completions.
proc complete_string {function possibilities prefix} {
	set message [$function $prefix $possibilities]
	set completions [remove_prefix $message]
	if {$completions == ""} {return [list "" ""]}
	if {[llength $completions] == 1} {
		return [list [lindex $completions 0] ""]
	} else {set index 0
		set flag 1
		set string [lindex $completions 0]
		while 1 {
			set char [string index $string $index]
			if {$char == ""} break
			foreach completion $completions {
				if {[string index $completion $index] != $char} {
					set flag 0 ; break}}
			if $flag {incr index} else {break}}
		return [list [string range $string 0 [expr $index-1]] $message]
}}

# Like complete_string, but considers only the ]delimiter-end] in string, using
# function on this string range to yield the possibilities.
proc complete_substring {function delimiter s} {
	set list [divide_string $s $delimiter]
	if {([lindex $list 1] == "")} {return [list "" ""]}
	set completion [complete_string $function "" [lindex $list 2]]
	if {([lindex $completion 0] == "")} {return [list "" ""]
	} else {return [lreplace $completion 0 0 [join [lreplace $list 2 2 \
						[lindex $completion 0]] ""]]
}}


set default_completions {
  	{replace_substring string_tcl_result {[}}
	{replace_substring string_global_value {$}}
	{replace_substring string_glob_files { }}
	{complete_substring filter_cmds {[}}
	{complete_substring filter_vars {$}}
	{complete_substring filter_glob { }}
}

# Tries each completion in the completion list until one works.
# Returns {result message index} where index is the index in s where
# completion started.
proc multiple_completion {s {completion_list ""}} {
	if {($completion_list == "")} {
		global default_completions
		set completion_list $default_completions
	}
	set message ""
	foreach completion_fn $completion_list {
		set output [eval $completion_fn [list $s]]
		if {[lindex $output 1] != ""} {
			set message [lindex $output 1]}
		if {[lindex $output 0] != ""} {
			set result [lindex $output 0]
			if {[lindex $completion_fn 2] == ""} {
				return [list $result $message -1]}

			set sublist [divide_string $s [lindex $completion_fn 2]]
			if {[lindex $sublist 2] == ""} {
				set index [string length $s]
			} else {set index [string length [lindex $sublist 0]]}
			return [list $result $message $index]
		}}
	return [list "" $message [string length $s]]
}
