# Functions for keeping a list of all active BETH interpreters, and switching
# between them.


# Interpreter management

# Code remote interpreter must successfully execute to be added.
set shiboleth "$text configure -background"
# List of interpreters
set all_interps {}
# List of interpreter titles (same order)
set all_titles {}


# Ensures that interp is still active, removes it if it isn't
# Returns 1 if interp is OK, 0 if not.
proc check_interp {interp} {
	global shiboleth all_interps all_titles
	if {[catch {send $interp $shiboleth}]} {
		delete_interp $interp
		return 0
	}
	return 1
}

# Removes interpreter from all_interps list
proc delete_interp {interp} {
	global all_interps all_titles
	set place [lsearch $all_interps $interp]
	if {($place >= 0)} {
		set all_interps [lreplace $all_interps $place $place]
		set all_titles [lreplace $all_titles $place $place]
}}

# Finds out the list of all currently active interps (from another interp)
proc register_interps {} {
	global all_interps all_titles me
	foreach interp [winfo interps] {
		if {[check_interp $interp]} {
			set all_interps [send $interp {set all_interps}]
			set all_titles [send $interp {set all_titles}]
			lappend all_interps $me
			lappend all_titles [wm title .]
			return
}}}

# Figures out the list of all currently active interpreters.
# (w/o contacting another interp)
proc register_interps_2 {} {
	global all_interps all_titles
	set all_interps ""
	set all_titles ""
	foreach interp [winfo interps] {
		if {[check_interp $interp]} {
			lappend all_interps $interp
			lappend all_titles [send $interp {wm title .}]
}}}

# Sends my interp/title lists to every other interp
proc send_interp_updates {} {
	global all_interps all_titles me
	foreach interp $all_interps {
		if {($interp != $me) && [check_interp $interp]} {
			send $interp "set all_interps \{$all_interps\} ; \
				set all_titles \{$all_titles\} ; \
				set im_last 0"
}}}

# When I quit, notify the other interps
proc delete_me {f} {
	global me
	if {([grab status .] != "none")} {catch {give_control +1 0 $f}}
	delete_interp $me
	send_interp_updates
}


# Interpreter switching

# Given an interp, returns the next/prev one. (d is +1/-1 for next/prev)
# If deiconify_flag is 1, the interp returned is deiconifed first, if it's 0
# iconified windows are skipped.
proc another_interp {d deiconify_flag} {
	global all_interps me
	set index [lsearch $all_interps $me]
	set first_index $index
	if {($first_index < 0)} {set first_index 0}
	while {1} {	incr index $d
		if {($index >= [llength $all_interps])} {set index 0}
		if {($index == -1)} {set index [llength $all_interps] ; incr index -1}
		if {($index == $first_index)} {return ""}

		set next [lindex $all_interps $index]
		if {![check_interp $next]} {continue}
		if {[send $next {winfo ismapped .}]} {
			return $next
		} elseif {($deiconify_flag == 1)} {
			send $next {wm deiconify .}
			return $next
}}}

# Gives control to another interpreter (see another_interp for parm details)
proc give_control {d deiconify_flag f} {
	set next_interp [another_interp $d $deiconify_flag]
	relinquish_control $f
	if {($next_interp == "")} {beep
	} else {send $next_interp "take_control $f"
}}

# Given an interp's title, returns the interp
proc which_interp_is {int_title} {
	global all_interps all_titles
	set i [lsearch $all_titles $int_title*]
	if {($i == 0)} {beep ; return}
	set int [lindex $all_interps $i]
	if {([check_interp $int] == 1)} {return $int
	} else {return ""
}}

proc choose_beth_which {t f} {
	set which [$f.choosee get]
	destroy_f_entry $t $f.choosel $f.choosee

	set next_interp [which_interp_is $which]
	if {($next_interp == "")} {beep ; return}
	relinquish_control $f
	send $next_interp wm_raise $f
	send $next_interp take_control $f
}

proc choose_beth_interp {t f} {
	create_f_entry $t $f.choosel $f.choosee
	$f.choosel configure -text "Which:"
	$f.choosee insert 0 "Edit: "
	bind $f.choosee <Return> "choose_beth_which $t $f"
}


# Misc. multi-interpreter functions

# Notify other interps when I change my window title.
proc reset_my_title {new_title new_iconname} {
	wm title . $new_title
	wm iconname . $new_iconname
	global me
	check_interp $me
	send_interp_updates
}

proc wm_iconify {f} {
	if {([grab status .] != "none")} {
		give_control +1 0 $f
	} else {relinquish_control $f}
	wm iconify .
}


# Interpreter bindings. f is a frame widget to put messages in.
proc interpbind {f} {
	bind Text <Control-g> "+catch \{destroy_f_entry %W $f.choosel $f.choosee\}"
	bind Text <Meta-I> "wm_iconify $f"
	bind Text <Meta-Control-b> "choose_beth_interp %W $f"
	bind Text <Meta-N> "give_control +1 1 $f"
	bind Text <Meta-n> "give_control +1 0 $f"
	bind Text <Meta-P> "give_control -1 1 $f"
	bind Text <Meta-p> "give_control -1 0 $f"
}

interpbind $frame

# Get the list of interpreters
register_interps
send_interp_updates
bind . <Destroy> "delete_me $frame"

# If a bunch of beth interpreters were started at once, they may be confused
# about who all is out there. If I am the last one started, then I'll figure
# out who's out there & tell 'em.
set im_last 1
after 2000 {if {($im_last)} {register_interps_2 ; send_interp_updates}}
