/*
 *=============================================================================
 *                              tSippIStore.c
 *-----------------------------------------------------------------------------
 * Manage the table of image store.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippIStore.c,v 1.5 1995/08/15 08:38:59 markd Exp $
 *-----------------------------------------------------------------------------
 * Scenes are rendered to and images stored or displayed in image store.  Each
 * storage class is defined by a tSippStorageClass_t structure.  It contains
 * attributes an access functions for the class.  A table of these structures
 * is stored in tSippGlob.  An instance of the storage class is specified by a
 * handle.
 *
 * The following attributes are defined for each image storage class:
 *
 *   o handlePrefix - The unique prefixed characters in the handle used to
 *     distinguish image storage classes.
 *
 *   o prefixSize - Number of characters in the handlePrefix string.
 *
 *   o identFunc - A function that is called to determine if the handle belongs
 *     to the class.  If not NULL, this is used in place of the prefix for
 *     determining if a handle belongs to the class.  This is necessary when
 *     the handle is nondeterministic.  The prefix is still used for error
 *     messages.  The function should return:
 *        o TCL_OK - If this belongs to this class.
 *        o TCL_CONTINUE - If it doesn't belong to the class.
 *        o TCL_ERROR - If an error occured (message in result).
 *
 *   o scanDirection - The scanning direction:
 *       o TSIPP_SCAN_RANDOM - If the image store can be written top to
 *         bottom or bottom to top or randomly.
 *       o TSIPP_SCAN_TOP_DOWN - If the image store is written top to
 *         bottom.
 *       o TSIPP_SCAN_BOTTOM_UP - If the image store is written bottom to
 *         top.
 *
 *   o bitMapOptimal - TRUE if storage of bitmaps in this format is optimized,
 *     FALSE if there is no special optimization for bitmaps.
 *
 *   o outputStart - Function called at the beginning of output.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o outputParmsPtr (I) - The parameters describing the image to
 *           output.
 *         o handle (I) - The handle for the image store.
 *         o comments (I) - If the source image has comments associated
 *           with it, they are passed here.  If the destination image can
 *           store them, it possible.  NULL if no comments available.
 *       o Returns a generic client data pointer that is then passed in to the
 *         other routines, or NULL if an error occurs.
 *
 *   o outputLine - Function called to output each line.  If ODD or EVEN fields
 *     are being outputed, it is called for alternate lines.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o clientdata returned from outputStart.
 *         o y (I) - The scan line that was just rendered.
 *         o rowPtr (I) - The pixels for the scanline that was just rendered.
 *       o Returns TRUE if all is ok, FALSE and an error message in
 *         tSippGlobPtr->interp->result if an error occurs.
 *
 *   o outputBitMap - Function called to output a bitmap when doing LINE
 *     rendering or output.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o clientdata returned from outputStart.
 *         o bitMapPtr - Pointer to the SIPP bitmap structure to output. 
 *       o Returns TRUE if all is ok, FALSE and an error message in
 *         tSippGlobPtr->interp->result if an error occurs.
 *
 *   o outputEnd - Function called at the end of output.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o outputParmsPtr (I) - The parameters describing the image to
 *           output.
 *         o clientdata returned from outputStart.
 *       o Returns TRUE if all is ok, FALSE and an error message in
 *         tSippGlobPtr->interp->result if an error occurs.
 *
 *   o copyImage - Copy a single image from image store to another.
 *     This function is responsible for setting up the output parameters
 *     structure and calling the outputStart, outputEnd functions and using
 *     outputPixel or outputBitMap to output the image to the target image
 *     store.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o srcHandle (I) - The handle for the source image store.
 *         o destHandle (I) - The handle for the destination image store.
 *         o destClassPtr (I) - Pointer to the image storage class description
 *           for the destination class.
 *         o clear (I) - If TRUE, clear target image before copying, if FALSE,
 *           don't clear.  Ignored if the image store does not support this.
 *       o Returns TRUE is all is OK, or FALSE if an error occurs.
 *
 *   o  commentPut - Store a comment in the image.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o handle (I) - The handle of the image.
 *         o name (I) - Name of the comment to store.
 *         o value (I) - Value of the comment to store.  If NULL, just store
 *           the comment name.
 *       o Returns TRUE is all is OK, or FALSE if an error occurs.
 *
 *   o  commentGet - Get a comment from the image.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o handle (I) - The handle of the image.
 *         o name (I) - Name of the comment to get.
 *         o valuePtr (O) - Value of the comment is returned here, or NULL if
 *           its not found.
 *       o Returns TRUE is all is OK, or FALSE if an error occurs.
 *
 *   o  commentGetVec - Get the comment for the image.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o handle (I) - The handle of the image.
 *         o argvPtr (O) - The comment vector is returned here, it might be
 *           NULL or empty.
 *       o Returns TRUE is all is OK, or FALSE if an error occurs.
 *
 *   o  commentDelete - Delete a comment from the image or all image comments.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o handle (I) - The handle of the image.
 *         o name (I) - Name of the comment to delete.  Ignored if not
 *           found.  If NULL, delete all image comments.
 *       o Returns TRUE is all is OK, or FALSE if an error occurs.
 *
 *=============================================================================
 */

#include "tSippInt.h"

/*=============================================================================
 * TSippAddStorageClass --
 *   Add an image storage class to the image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o storageClassPtr (I) - A pointer to static structure describing the
 *     image storage class.  This structure MUST be static.
 *-----------------------------------------------------------------------------
 */
void
TSippAddStorageClass (tSippGlobPtr, storageClassPtr)
    tSippGlob_t          *tSippGlobPtr;
    tSippStorageClass_t  *storageClassPtr;
{
    int idx;

    /*
     * Find a spot for the output class entry.
     */
    for (idx = 0; idx < TSIPP_MAX_STORAGE_CLASSES; idx++) {
        if (tSippGlobPtr->storageClassPtrs [idx] == NULL)
            break;
    }
    if (idx >= TSIPP_MAX_STORAGE_CLASSES)
        panic ("TSIPP_MAX_STORAGE_CLASSES too small");

    tSippGlobPtr->storageClassPtrs [idx]     = storageClassPtr;
    tSippGlobPtr->storageClassPtrs [idx + 1] = NULL;
}

/*=============================================================================
 * TSippFindStorageClass --
 *   Given a image store handle, locate the image storage class in the
 * class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - The output handle.
 * Returns:
 *   A pointer to the storage class entry, or NULL if an error occurs.
 *-----------------------------------------------------------------------------
 */
tSippStorageClass_t *
TSippFindStorageClass (tSippGlobPtr, handle)
    tSippGlob_t  *tSippGlobPtr;
    char         *handle;
{
    int                  idx;
    tSippStorageClass_t *storageClassPtr;

    /*
     * Search the image storage class table by calling classes with ident
     * functions before comparing prefixes.  This assures us that a name
     * that conflicts with a prefined handle prefix will be found.
     */
    for (idx = 0; tSippGlobPtr->storageClassPtrs [idx] != NULL; idx++) {
        storageClassPtr = tSippGlobPtr->storageClassPtrs [idx];
        if (storageClassPtr->identFunc == NULL)
            continue;

        switch (storageClassPtr->identFunc (tSippGlobPtr, handle)) {
          case TCL_OK:
            return tSippGlobPtr->storageClassPtrs [idx];
          case TCL_CONTINUE:
            break;
          case TCL_ERROR:
          default:
            return NULL;
        }
    }


    /*
     * Search for images by comparing prefixes.
     */
    for (idx = 0; tSippGlobPtr->storageClassPtrs [idx] != NULL; idx++) {
        storageClassPtr = tSippGlobPtr->storageClassPtrs [idx];
        if (storageClassPtr->identFunc != NULL)
            continue;

        if (STRNEQU (handle, storageClassPtr->handlePrefix,
                     storageClassPtr->prefixSize))
            return tSippGlobPtr->storageClassPtrs [idx];
    }

    Tcl_AppendResult (tSippGlobPtr->interp,
                      "invalid image store handle \"",
                      handle, "\"", (char *) NULL);
    return NULL;
}

