/*
 *=============================================================================
 *                                  tSippPhoto.c
 *-----------------------------------------------------------------------------
 * Rendering callbacks for the Tk photo image.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippPhoto.c,v 5.12 1996/10/26 19:00:04 markd Exp $
 *=============================================================================
 */

#include <varargs.h>

#include "../tsipp/tSippInt.h"

#include "tk.h"

/*
 * Client-data for output callback used to hold a block of data.  The actual
 * set of scanlines to buffer follows this structure.  The structure is setup
 * so it can be rendered in either scan direction.
 *
 */
typedef struct {
    char               *imageName;
    Tk_PhotoHandle      photoHandle;
    int                 xSize;
    int                 ySize;
    scanDirection_t     scanDirection;
    Tk_PhotoImageBlock  block;
} renderData_t;

/*
 * Image command name.
 */
static char *imageCmd = "image";


/*
 * Internal prototypes.
 */
static Tk_PhotoHandle
FindPhotoImage _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                            char         *handle));

static int
GetImageCommand _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                             Tcl_CmdInfo  *cmdInfoPtr));

static void
SetBackGroundColor _ANSI_ARGS_((Tk_PhotoHandle  photoHandle,
                                Color           color,
                                int             xSize,
                                int             ySize));

static bool
ReditherPhoto _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                           char         *image));

static void *
PhotoOutputStart _ANSI_ARGS_((tSippGlob_t         *tSippGlobPtr,
                              tSippOutputParms_t  *renderParmsPtr,
                              char                *handle,
                              char               **comments));

static bool
PhotoOutputLine _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                             void_pt      clientData,
                             int          y,
                             u_char      *rowPtr));

static bool
PhotoOutputBitMap _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                               void_pt       clientData,
                               Sipp_bitmap  *bitMapPtr));

static bool
PhotoOutputEnd _ANSI_ARGS_((tSippGlob_t         *tSippGlobPtr,
                            tSippOutputParms_t  *renderParmsPtr,
                            void_pt              clientData));

static bool
PhotoToPhotoCopy _ANSI_ARGS_((tSippGlob_t        *tSippGlobPtr,
                              Tk_PhotoImageBlock *srcImagePtr,
                              char               *destHandle,
                              bool                clear));

static bool
PhotoToOtherImage _ANSI_ARGS_((tSippGlob_t          *tSippGlobPtr,
                               tSippOutputParms_t   *outputParmsPtr,
                               Tk_PhotoImageBlock   *srcImagePtr,
                               void                 *destClientData,
                               tSippStorageClass_t  *storageClassPtr));

static bool
PhotoCopyImage _ANSI_ARGS_((tSippGlob_t          *tSippGlobPtr,
                            char                 *srcHandle,
                            char                 *destHandle,
                            tSippStorageClass_t  *destClassPtr,
                            bool                  clear));

static bool
PhotoHandleIdent _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                              char         *handle));

/*=============================================================================
 * FindPhotoImage --
 *   Translate an photo image name (string handle) to its photo handle (C
 * reference to a photo image).
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (O) - String handle to the photo image.
 * Returns:
 *    The photo handle or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static Tk_PhotoHandle
FindPhotoImage (tSippGlobPtr, handle)
    tSippGlob_t  *tSippGlobPtr;
    char         *handle;
{
    Tk_PhotoHandle  photoHandle;

    photoHandle = Tk_FindPhoto (handle);
    if (photoHandle == NULL) {
        Tcl_AppendResult (tSippGlobPtr->interp, "invalid photo image name\"",
                          handle, "\"", (char *) NULL);
        return NULL;
    }
    return photoHandle;
}

/*=============================================================================
 * GetImageCommand --
 *   Get the command info for the "image" command.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o cmdInfoPtr (O) - Command information is returned here.
 * Returns:
 *    TCL_OK or TCL_ERROR if image command is not found.
 *-----------------------------------------------------------------------------
 */
static int
GetImageCommand (tSippGlobPtr, cmdInfoPtr)
    tSippGlob_t  *tSippGlobPtr;
    Tcl_CmdInfo  *cmdInfoPtr;
{

    if (!Tcl_GetCommandInfo (tSippGlobPtr->interp, imageCmd, cmdInfoPtr)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "tk command \"", imageCmd,
                          "\" not found, required for operations on photo ",
                          "images", (char *) NULL);
        return TCL_ERROR;
    }
    return TCL_OK;
}
    
/*=============================================================================
 * SetBackGroundColor --
 *
 * Set the entire photo image to the background color
 *
 * Parameters:
 *   o photoHandle (I) - Photo image to set to the background color.
 *   o color (I) - Color to set the background to.
 *   o xSize (I) - Width of the image.
 *   o ySize (I) - Height of the image.
 *-----------------------------------------------------------------------------
 */
static void
SetBackGroundColor (photoHandle, color, xSize, ySize)
    Tk_PhotoHandle  photoHandle;
    Color           color;
    int             xSize;
    int             ySize;
{
    Tk_PhotoImageBlock  block;
    u_char             *rowPtr;
    int                 x, y;

    block.pixelPtr = (u_char *) alloca (3 * xSize);
    block.width = xSize;
    block.height = 1;
    block.pitch = 3 * xSize;
    block.pixelSize  = 3;
    block.offset [0] = 0;
    block.offset [1] = 1;
    block.offset [2] = 2;

    rowPtr = block.pixelPtr;
    
    for (x = 0; x < xSize; x++) {
        rowPtr [TSIPP_RED]   = color.red * 255;
        rowPtr [TSIPP_GREEN] = color.grn * 255;
        rowPtr [TSIPP_BLUE]  = color.blu * 255;
        rowPtr += 3;
    }
    
    for (y = 0; y < ySize; y++) {
        Tk_PhotoPutBlock (photoHandle,
                          &block,
                          0, y,
                          xSize, 1);
    }
}

/*=============================================================================
 * ReditherPhoto --
 *
 * Redither an instance of a photo image.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o image (I) - The photo image name.
 * Returns:
 *   TRUE if all is OK, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
ReditherPhoto (tSippGlobPtr, image)
    tSippGlob_t *tSippGlobPtr;
    char        *image;
{

    if (Tcl_VarEval (tSippGlobPtr->interp, image, " redither",
                      (char *) NULL) == TCL_ERROR) {
        return FALSE;
    }
    
    Tcl_ResetResult (tSippGlobPtr->interp);
    return TRUE;
}

/*=============================================================================
 * PhotoOutputStart --
 *   Start output to a photo image.  Sets the image's blank color to be the
 * same as the background color and clears the image if requested. This
 * routine is pointed to by the photo image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o handle (I) - photo handle of the image to write to.
 *   o comments (I) - Comments are not useful with the photo image, so this
 *     argument is ignored.
 * Returns:
 *   A pointer to be passed back into the other Photo output routines or NULL
 *   an a message in tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static void *
PhotoOutputStart (tSippGlobPtr, renderParmsPtr, handle, comments)
    tSippGlob_t         *tSippGlobPtr;
    tSippOutputParms_t  *renderParmsPtr;
    char                *handle;
    char               **comments;
{
    Tk_PhotoHandle photoHandle;
    renderData_t  *renderDataPtr;

    photoHandle = FindPhotoImage (tSippGlobPtr, handle);
    if (photoHandle == NULL)
        return NULL;

    /*
     * Set up the client-data and fill in image block information.
     * The image block scanline is not actually allocated here.
     */
    renderDataPtr = (renderData_t *) smalloc (sizeof (renderData_t));

    renderDataPtr->imageName        = strdup (handle);
    renderDataPtr->photoHandle      = photoHandle;
    renderDataPtr->xSize            = renderParmsPtr->imgData.xSize;
    renderDataPtr->ySize            = renderParmsPtr->imgData.ySize;
    renderDataPtr->scanDirection    = renderParmsPtr->scanDirection;
    renderDataPtr->block.pixelPtr   = NULL;
    renderDataPtr->block.width      = renderParmsPtr->imgData.xSize;
    renderDataPtr->block.height     = 1;
    renderDataPtr->block.pitch      = 3 * renderParmsPtr->imgData.xSize;
    renderDataPtr->block.pixelSize  = 3;
    renderDataPtr->block.offset [0] = 0;
    renderDataPtr->block.offset [1] = 1;
    renderDataPtr->block.offset [2] = 2;

    if (renderParmsPtr->clear) {
        SetBackGroundColor (photoHandle,
                            tSippGlobPtr->backgroundColor,
                            renderDataPtr->xSize,
                            renderDataPtr->ySize);
    }

    return renderDataPtr;
}

/*=============================================================================
 * PhotoOutputLine --
 *   Output a rendered line to a photo image.  This routine is pointed to by
 * the photo image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o clientData (I) - Actually a pointer to the Tk Photo rendering data.
 *   o y (I) - The scan line that was just rendered.
 *   o rowPtr (I) - The pixels for the scanline that was just rendered.
 * Returns:
 *   Always returns TRUE.
 *-----------------------------------------------------------------------------
 */
static bool
PhotoOutputLine (tSippGlobPtr, clientData, y, rowPtr)
    tSippGlob_t *tSippGlobPtr;
    void_pt      clientData;
    int          y;
    u_char      *rowPtr;
{
    renderData_t  *renderDataPtr = (renderData_t *) clientData;

    renderDataPtr->block.pixelPtr = rowPtr;

    Tk_PhotoPutBlock (renderDataPtr->photoHandle,
                      &renderDataPtr->block,
                      0, y,
                      renderDataPtr->xSize, 1);
    return TRUE;
}

/*=============================================================================
 * PhotoOutputBitMap --
 *   Output a SIPP bit map to a photo image.  This routine is pointed to by
 * the photo image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o clientData (I) - Actually a pointer to the Tk Photo rendering data.
 *   o bitMapPtr (I) - Pointer to the SIPP bit map structure.
 * Returns:
 *   Always returns TRUE.
 *-----------------------------------------------------------------------------
 */
static bool
PhotoOutputBitMap (tSippGlobPtr, clientData, bitMapPtr)
    tSippGlob_t  *tSippGlobPtr;
    void_pt       clientData;
    Sipp_bitmap  *bitMapPtr;
{
    renderData_t *renderDataPtr = (renderData_t *) clientData;
    int           xSize        = renderDataPtr->xSize;
    int           yStart, yEnd, yIncr;
    u_char        lineColor [3], backgroundColor [3];
    int           x, y;
    u_char       *bitRowPtr;
    unsigned      bit;
    u_char       *rowPtr;

    lineColor [TSIPP_RED]   = tSippGlobPtr->lineColor.red * 255;
    lineColor [TSIPP_GREEN] = tSippGlobPtr->lineColor.grn * 255;
    lineColor [TSIPP_BLUE]  = tSippGlobPtr->lineColor.blu * 255;

    backgroundColor [TSIPP_RED]   = tSippGlobPtr->backgroundColor.red * 255;
    backgroundColor [TSIPP_GREEN] = tSippGlobPtr->backgroundColor.grn * 255;
    backgroundColor [TSIPP_BLUE]  = tSippGlobPtr->backgroundColor.blu * 255;


    if (renderDataPtr->scanDirection == TSIPP_SCAN_TOP_DOWN) {
        yStart = renderDataPtr->ySize - 1;
        yEnd   = 0;
        yIncr  = -1;
    } else {
        yStart = 0;
        yEnd   = renderDataPtr->ySize - 1;
        yIncr  = 1;
    }

    renderDataPtr->block.pixelPtr =
        (u_char *) alloca (renderDataPtr->block.pitch);
    /*
     * Loop setting pixels based on the bit map and outputting rows to 
     * the image.
     */

    for (y = yStart; y != yEnd; y += yIncr) {
        bitRowPtr = &bitMapPtr->buffer [y * bitMapPtr->width_bytes];
        rowPtr = renderDataPtr->block.pixelPtr;

        for (x = 0; x < xSize; x++) {
            bit = (bitRowPtr [x >> 3] >> (7-(x & 7))) & 1;
            if (bit) {
                rowPtr [TSIPP_RED]   = lineColor [TSIPP_RED];
                rowPtr [TSIPP_GREEN] = lineColor [TSIPP_GREEN];
                rowPtr [TSIPP_BLUE]  = lineColor [TSIPP_BLUE];
            } else {
                rowPtr [TSIPP_RED]   = backgroundColor [TSIPP_RED];
                rowPtr [TSIPP_GREEN] = backgroundColor [TSIPP_GREEN];
                rowPtr [TSIPP_BLUE]  = backgroundColor [TSIPP_BLUE];
            }
            rowPtr += 3;
        }
        Tk_PhotoPutBlock (renderDataPtr->photoHandle,
                          &renderDataPtr->block,
                          0, y,
                          renderDataPtr->xSize, 1);
    }
    return TRUE;
}

/*=============================================================================
 * PhotoOutputEnd --
 *   Finish up output to a photo image.  This routine is pointed to by the
 * photo image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o clientData (I) - Actually a pointer to the Tk Photo rendering data.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in
 *  tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
PhotoOutputEnd (tSippGlobPtr, renderParmsPtr, clientData)
    tSippGlob_t         *tSippGlobPtr;
    tSippOutputParms_t  *renderParmsPtr;
    void_pt              clientData;
{
    renderData_t *renderDataPtr = (renderData_t *) clientData;
    bool          status = TRUE;
    
    /*
     * Final updates are handled by the tSippRender code.  If we were scanning
     * from bottom-up, the photo is not dithered, so dither it here.
     */
    if (renderParmsPtr->scanDirection == TSIPP_SCAN_BOTTOM_UP)
        status = ReditherPhoto (tSippGlobPtr, renderDataPtr->imageName);

    sfree (renderDataPtr->imageName);
    sfree (renderDataPtr);

    return status;
}

/*=============================================================================
 * PhotoToPhotoCopy --
 *   Copy an image from one photo image to another photo image.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o srcImagePtr (I) - Pointer to the source image block.
 *   o destHandle (I) - The handle for the destination photo.
 *   o clear (I) - If TRUE, clear target image before copying.
 * Returns:
 *   TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
PhotoToPhotoCopy (tSippGlobPtr, srcImagePtr, destHandle, clear)
    tSippGlob_t        *tSippGlobPtr;
    Tk_PhotoImageBlock *srcImagePtr;
    char               *destHandle;
    bool                clear;
{
    Tk_PhotoHandle  destPhotoHandle;

    destPhotoHandle = FindPhotoImage (tSippGlobPtr,
                                      destHandle);
    if (destPhotoHandle == NULL)
        return FALSE;

    if (clear) {
        SetBackGroundColor (destPhotoHandle,
                            tSippGlobPtr->backgroundColor,
                            srcImagePtr->width,
                            srcImagePtr->height);
    }

    Tk_PhotoPutBlock (destPhotoHandle,
                      srcImagePtr,
                      0, 0,
                      srcImagePtr->width,
                      srcImagePtr->height);

    return TRUE;
}

/*=============================================================================
 * PhotoToOtherImage --
 *   Copy a photo image from photo  to some other image type.  This assumes
 * all setup has been done, it just copies the scanlines.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o srcImagePtr (I) - Pointer to the source image block.
 *   o destClientData (I) - The clientdata returned by the output start
 *     function for the image destination.
 *   o storageClassPtr (I) - Pointer to the image storage class description
 *     for the target class.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in tSippGlobPtr->interp->
 * result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
PhotoToOtherImage (tSippGlobPtr, outputParmsPtr, srcImagePtr, destClientData, 
                   storageClassPtr)
    tSippGlob_t          *tSippGlobPtr;
    tSippOutputParms_t   *outputParmsPtr;
    Tk_PhotoImageBlock   *srcImagePtr;
    void                 *destClientData;
    tSippStorageClass_t  *storageClassPtr;
{
    int           yStart, yLimit, yIncr, y;
    u_char       *rowPtr;
    outputLine_t *outputLine = storageClassPtr->outputLine;

    /*
     * Set the scanline bounds based on the target scan direction.
     */
    if (storageClassPtr->scanDirection == TSIPP_SCAN_BOTTOM_UP) {
        yStart = srcImagePtr->height - 1;
        yLimit = -1;
        yIncr  = -1;
    } else {
        yStart = 0;
        yLimit = srcImagePtr->height;
        yIncr  = 1;
    }

    /*
     * Copy the image.
     */
    rowPtr = srcImagePtr->pixelPtr;
    for (y = yStart; y != yLimit; y += yIncr) {
        if (!(*outputLine) (tSippGlobPtr,
                            destClientData,
                            y,
                            rowPtr))
            return FALSE;
        rowPtr += srcImagePtr->pitch;
    }
    return TRUE;
}

/*=============================================================================
 * PhotoCopyImage --
 *   Copy an image from a photo to another image store.
 * This routine is pointed to by the photo image store class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o srcHandle (I) - The handle of the photo image to read.
 *   o destHandle (I) - The handle for the destination image store.
 *   o destClassPtr (I) - Pointer to the image storage class description
 *     for the destination class.
 *   o clear (I) - If TRUE, clear target image before copying, if FALSE,
 *     don't clear.  Ignored if the ISO does not support this.
 * Returns:
 *   TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
PhotoCopyImage (tSippGlobPtr, srcHandle, destHandle, destClassPtr, clear)
    tSippGlob_t         *tSippGlobPtr;
    char                *srcHandle;
    char                *destHandle;
    tSippStorageClass_t *destClassPtr;
    bool                 clear;
{
    Tk_PhotoHandle       srcPhotoHandle;
    Tk_PhotoImageBlock   srcImage;
    tSippOutputParms_t   outputParms;
    void                *destClientData;
    bool                 status1, status2;

    srcPhotoHandle = FindPhotoImage (tSippGlobPtr,
                                     srcHandle);
    if (srcPhotoHandle == NULL)
        return FALSE;

    Tk_PhotoGetImage (srcPhotoHandle,
                      &srcImage);

    if (srcImage.pixelSize != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp,
                          "can't read from photo images where pixel size is ",
                          "not three bytes", (char *) NULL);
        return FALSE;
    }

    if ((srcImage.offset [0] != 0) ||
        (srcImage.offset [1] != 1) ||
        (srcImage.offset [2] != 2)) {
        Tcl_AppendResult (tSippGlobPtr->interp,
                          "can't read from photo images where colors are ",
                          "not ordered as red, green and blue bytes",
                          (char *) NULL);
        return FALSE;
    }

    /*
     * If the target is a photo, optimize the copy.
     */
    if (destClassPtr->copyImage == PhotoCopyImage)
        return PhotoToPhotoCopy (tSippGlobPtr, &srcImage, destHandle, clear);

    /*
     * Set up the general output parameters assoicated with this image.
     */
    outputParms.imgData.xSize = srcImage.width;
    outputParms.imgData.ySize = srcImage.height;
    outputParms.imgData.mode = MODE_UNKNOWN24;  /* ??? */
    outputParms.imgData.overSampling = 0;  /* ??? */
    outputParms.imgData.field = BOTH;
    outputParms.imgData.backgroundColor [TSIPP_RED] = 0;
    outputParms.imgData.backgroundColor [TSIPP_GREEN] = 0;
    outputParms.imgData.backgroundColor [TSIPP_BLUE] = 0;
    outputParms.imgData.lineColor [TSIPP_RED]   = 255;
    outputParms.imgData.lineColor [TSIPP_GREEN] = 255;
    outputParms.imgData.lineColor [TSIPP_BLUE]  = 255;

    /*
     * Initialize other output parms fields.
     */
    outputParms.tSippGlobPtr = tSippGlobPtr;
    outputParms.argv = NULL;   
    outputParms.update = 0;
    outputParms.clear = clear;

    outputParms.bitMapOutput  = FALSE;
    outputParms.scanDirection = destClassPtr->scanDirection;


    /*
     * Copy to other image source.
     */
    destClientData = (*destClassPtr->outputStart) (tSippGlobPtr,
                                                   &outputParms,
                                                   destHandle,
                                                   NULL);
    if (destClientData == NULL)
        return FALSE;

    status1 = PhotoToOtherImage (tSippGlobPtr,
                                 &outputParms,
                                 &srcImage,
                                 destClientData,
                                 destClassPtr);
    
    status2 = (*destClassPtr->outputEnd) (tSippGlobPtr,
                                          &outputParms,
                                          destClientData);
    return status1 && status2;
}

/*=============================================================================
 * PhotoHandleIdent --
 *   Determine if a supplied handle is a photo handle.  Stored in the class
 * table since the user defines the image name. 
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - Handle to identify.
 * Returns:
 *   o TCL_OK - If this belongs to this class.
 *   o TCL_CONTINUE - If it doesn't belong to the class.
 *   o TCL_ERROR - If an error occured (message in result).
 *-----------------------------------------------------------------------------
 */
static int
PhotoHandleIdent (tSippGlobPtr, handle)
   tSippGlob_t  *tSippGlobPtr;
   char         *handle;
{
#define IMAGE_ARGC  3
    Tcl_CmdInfo  cmdInfo;
    int          result;
    char        *imageArgv [IMAGE_ARGC + 1];

    /*
     * We must execute image command here, since the command not found is an
     * error, but other errors just indicate its not a photo image.
     */

    if (GetImageCommand (tSippGlobPtr, &cmdInfo) == TCL_ERROR)
        return TCL_ERROR;

    /*
     * Build up the arguments.
     */
    imageArgv [0] = imageCmd;
    imageArgv [1] = "type";
    imageArgv [2] = handle;
    imageArgv [3] = NULL;

    result = TCL_CONTINUE;
    if (cmdInfo.proc (cmdInfo.clientData,
                      tSippGlobPtr->interp,
                      IMAGE_ARGC,
                      imageArgv) == TCL_OK) {
        if (STREQU (tSippGlobPtr->interp->result, "photo")) {
            result = TCL_OK;
        }
    }
    Tcl_ResetResult (tSippGlobPtr->interp);
    return result;
}

/*=============================================================================
 * TSippPhotoInit --
 *   Initialized the photo image rendering commands.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippPhotoInit (tSippGlobPtr)
    tSippGlob_t  *tSippGlobPtr;
{
    static tSippStorageClass_t storageClass = {
        "photo",                    /* handlePrefix  */
        5,                          /* prefixSize    */
        PhotoHandleIdent,           /* identFunc     */
        TSIPP_SCAN_RANDOM,          /* scanDirection */
        FALSE,                      /* bitMapOptimal */
        PhotoOutputStart,
        PhotoOutputLine,
        PhotoOutputBitMap,
        PhotoOutputEnd,
        PhotoCopyImage,
        NULL,
        NULL,
        NULL,
        NULL
    };

    TSippAddStorageClass (tSippGlobPtr, &storageClass);
}


