/* useInstallTk.c --
 *   install Tk for a Tcl interpreter, part from tkMain.c
 *
 * R"udiger Franke, 2/7/94
 *
 * Copyright (c) 1990-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <stdio.h>
#include <stdlib.h>
#include <tcl.h>
#include <tk.h>

extern char *		strrchr _ANSI_ARGS_((CONST char *string, int c));

/*
 * Command-line options:
 */

static int synchronize;
static char *name;
static char *display;
static char *geometry;

static Tk_ArgvInfo argTable[] = {
  {"-geometry", TK_ARGV_STRING, (char *) NULL, (char *) &geometry,
   "Initial geometry for window"},
  {"-display", TK_ARGV_STRING, (char *) NULL, (char *) &display,
   "Display to use"},
  {"-name", TK_ARGV_STRING, (char *) NULL, (char *) &name,
   "Name to use for application"},
  {"-sync", TK_ARGV_CONSTANT, (char *) 1, (char *) &synchronize,
   "Use synchronous mode for display server"},
  {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL,
   (char *) NULL}
};

/*
 *--------------------------------------------------------------------
 */
Tk_Window Use_InstallTk(Tcl_Interp *interp, char *defaultName,
		        int *argcPtr, char *argv[])
{
  Tk_Window mainWindow;
  char *args, *p;
  int argc, code;
  char buf[20];

  /*
   * Parse command-line arguments.
   */

  synchronize = 0;
  name = NULL;
  display = NULL;
  geometry = NULL;

  if (Tk_ParseArgv(interp, (Tk_Window) NULL, argcPtr, argv, argTable, 0)
      != TCL_OK) {
    return NULL;
  }

  if (name == NULL) {
    if (defaultName != NULL) {
      p = defaultName;
    } else {
      p = argv[0];
    }
    name = strrchr(p, '/');
    if (name != NULL) {
      name++;
    } else {
      name = p;
    }
  }

  /*
   * If a display was specified, put it into the DISPLAY
   * environment variable so that it will be available for
   * any sub-processes created by us.
   */

  if (display != NULL) {
    Tcl_SetVar2(interp, "env", "DISPLAY", display, TCL_GLOBAL_ONLY);
  }

  /*
   * Initialize the Tk application.
   */

  mainWindow = Tk_CreateMainWindow(interp, display, name, "Tk");
  if (mainWindow == NULL) {
    return NULL;
  }
  if (synchronize) {
    XSynchronize(Tk_Display(mainWindow), True);
  }

  /*
   * Set the "geometry" variable from the geometry
   * specified on the command line.
   * Set the geometry of the main window.
   */

  if (geometry != NULL) {
    Tcl_SetVar(interp, "geometry", geometry, TCL_GLOBAL_ONLY);
    code = Tcl_VarEval(interp, "wm geometry . ", geometry, (char *) NULL);
    if (code != TCL_OK) {
      Tk_DestroyWindow(mainWindow);
      return NULL;
    }
  }
  else {
    Tk_GeometryRequest(mainWindow, 200, 200);
  }

  /*
   * Make command-line arguments available in the Tcl variables "argc"
   * and "argv".
   */

  argc = *argcPtr;
  args = Tcl_Merge(argc-1, argv+1);
  Tcl_SetVar(interp, "argv", args, TCL_GLOBAL_ONLY);
  ckfree(args);
  sprintf(buf, "%d", argc-1);
  Tcl_SetVar(interp, "argc", buf, TCL_GLOBAL_ONLY);
  Tcl_SetVar(interp, "argv0", argv[0], TCL_GLOBAL_ONLY);

  /*
   * Init the "tcl_interactive" variable.
   */

  Tcl_SetVar(interp, "tcl_interactive", "0", TCL_GLOBAL_ONLY);

  return mainWindow;
}

