# $Id: shql.tcl,v 1.11 1995/07/11 16:50:13 jfontain Exp $

set shqlDebugLevel 0

# see limits.h for limit values (always account for sign)
set shqlTypeLength(int) 11
set shqlTypeLength(float) 25

set shqlEngineInitialized 0
set shqlNextCursor 0

proc dbengine {type {debugLevel 0}} {
    global shqlDebugLevel shqlEngineInitialized

    if {[string compare $type shql]!=0} {
        puts stderr "dbengine ERROR: unknown engine: $type"
    }
    if {$shqlEngineInitialized} {
        return
    }
    if {($shqlDebugLevel>0)||($debugLevel>0)} {
        puts "initialize $type"
    }
    set shqlEngineInitialized 1
}

proc dbdebug {debugLevel} {
    global shqlDebugLevel

    set shqlDebugLevel $debugLevel
}

proc dbcreate {engineName databaseName {debugLevel 0}} {
    global shqlDebugLevel

    dbengine $engineName
    if {($shqlDebugLevel>0)||($debugLevel>0)} {
        puts "create database $databaseName"
    }
    # remove simple quotes
    regsub -all ' $databaseName {} databaseName
    # database is a directory for shql, so create it
    exec mkdir $databaseName
}

proc dbconnect {engineName databaseName {debugLevel 0}} {
    global shqlDebugLevel shqlDatabaseName

    dbengine $engineName
    if {($shqlDebugLevel>0)||($debugLevel>0)} {
        puts "connect to database $databaseName"
    }
    # remove simple quotes
    regsub -all ' $databaseName {} databaseName
    set pipe [open "| shql -q $databaseName 2>@ stdout" r+]
    if {[string compare [gets $pipe] "Database: $databaseName"]!=0} {
        error "shql ERROR: could not connect to \"$databaseName\""
    }
    # connection is pipe descriptor
    set shqlDatabaseName($pipe) $databaseName
    return $pipe
}

proc dbdisconnect {connection {debugLevel 0}} {
    global shqlDebugLevel shqlDatabaseName

    if {($shqlDebugLevel>0)||($debugLevel>0)} {
        puts "disconnect from database $shqlDatabaseName($connection)"
    }
    # catch close to avoid error caused by unread data at one end of the pipe
    catch {close $connection}
}

proc dbsql {connection statement {debugLevel 0}} {
    global shqlDebugLevel shqlTypeLength

    scan $statement "%s" firstWord
    if {[string compare $firstWord select]==0} {
        error "shql ERROR: use dbcursor for SQL statement \"$statement\""
    }

    if {($shqlDebugLevel>0)||($debugLevel>0)} {
        puts $statement
    }

    set statement [encodeSpecialCharactersInStatement $statement]

    # replace remaining tab spacers with spaces to simplify substitutions
    regsub -all "\011" $statement " " statement

    # replace NULL values with empty strings, handle where special case
    regsub -all {is +not +NULL} $statement {!= ''} statement
    regsub -all {is +NULL} $statement {= ''} statement
    regsub -all NULL $statement '' statement

    switch $firstWord {
        create {
            # replace SQL types in create statement with their length
            foreach type [array names shqlTypeLength] {
                regsub -all " +$type *\(\[,)\]\)" $statement " $shqlTypeLength($type)\\1" statement
            }
            # replace character arrays with their length
            regsub -all { +char\(([0-9]+)\)} $statement { \1} statement

            set expect {\* OK}
        }
        delete -
        insert -
        update {
            set expect {\* (*rows*)}
        }
        default {
            set expect {\* OK}
        }
    }

    puts $connection "$statement /g"
    flush $connection
    gets $connection result
    if {![string match $expect $result]} {
        error "shql ERROR: SQL statement \"$statement\" failed: $result"
    }
}

proc dbcursor {connection statement {debugLevel 0}} {
    global shqlDebugLevel shqlNextCursor

    scan $statement "%s" firstWord
    if {[string compare $firstWord select]!=0} {
        error "shql ERROR: SQL statement \"$statement\" must be a select"
    }

    if {($shqlDebugLevel>0)||($debugLevel>0)} {
        puts $statement
    }

    # encode special characters that might be in a where clause
    set statement [encodeSpecialCharactersInStatement $statement]

    # replace remaining tab spacers with spaces to simplify substitutions
    regsub -all "\011" $statement " " statement

    # replace NULL clauses in where clause with corresponding empty string comparisons
    regsub -all {is +not +NULL} $statement {!= ''} statement
    regsub -all {is +NULL} $statement {= ''} statement

    set cursor cursor$shqlNextCursor
    incr shqlNextCursor
    # save rows in array bearing the cursor name
    global shql$cursor
    # initialize index for latter continue operation
    set shql${cursor}(index) -1

    puts $connection "$statement /g"
    flush $connection

    # look for union in statement (we will handle only one union)
    set union [expr [string first " union " $statement]>=0]
    set rowsResult {( * rows)}

    for {set row 0} {[gets $connection line]>0} {} {
        if {[string first {* } $line]==0} {
            # filter out prompt residue
            set line [string range $line 2 end]
        }
        if {[string compare [string index $line 0] |]!=0} {
            if {$union} {
                # end of first select in union, continue
                if {![string match $rowsResult $line]} {
                    error "shql ERROR: SQL statement \"$statement\" failed: $line"
                }
                set union 0
                continue
            } else {
                # not a row, done
                break
            }
        }
        set shql${cursor}($row) ""
        foreach column [split [string trim $line |] |] {
            # remove peripheral spaces before decoding special characters
            lappend shql${cursor}($row) [decodeSpecialCharacters [string trim $column]]
        }
        incr row
    }
    if {![string match $rowsResult $line]} {
        # if successful, shql returns number of rows read
        unset shql$cursor
        error "shql ERROR: cursor SQL statement \"$statement\" failed: $line"
    }
    return $cursor
}

proc dbcontinue {cursor {debugLevel 0}} {
    global shqlDebugLevel shql$cursor

    set debug [expr ($shqlDebugLevel>0)||($debugLevel>0)]
    if {$debug} {
        puts -nonewline "with cursor $cursor continue: "
    }
    set index [incr shql${cursor}(index)]
    if {[info exists shql${cursor}($index)]} {
        if {$debug} {
            puts [set shql${cursor}($index)]
        }
        return [set shql${cursor}($index)]
    } else {
        if {$debug} {
            puts ""
        }
        # end of table: automatically free cursor data
        dbfree $cursor $debugLevel
        return ""
    }
}

proc dbfree {cursor {debugLevel 0}} {
    global shqlDebugLevel shql$cursor

    if {[catch {unset shql${cursor}}]} {
        # do nothing if cursor is already freed
        return
    }
    if {($shqlDebugLevel>0)||($debugLevel>0)} {
        puts "free $cursor"
    }
}

# encode special characters except '
array set shqlEncoding [list { } \002 \" \003 $ \004 ( \005 ) \006 * \007 , \010 < \013 > \014 \\ \016 ` \017 | \020]
# build decoding array using encoding one
foreach character [array names shqlEncoding] {
    set shqlDecoding($shqlEncoding($character)) $character
}

proc encodeSpecialCharactersInStatement {statement} {
    global shqlEncoding

    if {[string length $statement]==0} {
        return ""
    }
    set encoded ""
    set inString 0
    set specialSet [join [array names shqlEncoding] ""]
    set index 0
    set nextCharacter [string index $statement 0]
    while 1 {
        set character $nextCharacter
        set nextCharacter [string index $statement [incr index]]
        if {[string compare $character ']==0} {
            if {$inString&&([string compare $nextCharacter ']==0)} {
                # escaped quote within SQL string, replace with encoded character and go on to next character
                append encoded \001
                set nextCharacter [string index $statement [incr index]]
                continue
            }
            # beginning or end of SQL string
            set inString [expr $inString^1]
        }
        if {!$inString||([string first "$character" $specialSet]<0)} {        
            append encoded $character
        } else {
            append encoded $shqlEncoding($character)
        }
        if {[string length $nextCharacter]==0} {
            break
        }
    }
    return $encoded
}

proc decodeSpecialCharacters {string} {
    global shqlDecoding

    # handle ' special case
    regsub -all \001 $string ' string
    foreach character [array names shqlDecoding] {
        regsub -all $character $string $shqlDecoding($character) string
    }
    return $string
}
