#
# Super quick and dirty multi-user Tcl / ASCII database facility
#
# Copyright (c) 1995 Cirque Labs, Inc.
# 
# See the file "license.terms" for information on usage and
# redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
#
#========================================================================
# PROCEDURES AND THEIR USAGE:
#
#  tstodate ts 
#	convert numeric value in "ts" to a date in the form of "Mmm dd, yyyy"
#
#  datetots date 
#	convert a date string (form = "Mmm dd, yyyy") into a sortable
#	numeric value.
#
#  todaydate  
#	return today's date in the form of "Mmm dd, yyyy"
#
#  dbcreate path var tablename fieldnames 
#	define, create, and open a new database table.  
#	path = directory path to the database
#	var = global array associated with the database
#	      (does not need to exist with first table
#	      definition).
#	tablename = name of the table (no whitespace allowed)
#	fieldnames = list of names for the fields in the table
#		     (no field name may have embedded whitespace)
#
#  dbopen path var 
#	open an existing database at the specified directory, 
#	associate the database state and rows to the named
#	global array variable.
#
#  dbcleanup var tpath
#	resequence and compress out voids in a database (database
#	must be open, but have no other users prior to issuing
#	this command).
# 
#  dbclose var 
#	flush and close all database tables, remove "var" and free
#	up the resources it consumed.
#
#  dbgetrow var tablename seqno 
#	get the row associated with the provided sequence number for
#	the table; make it the current row of the table.
#
#  dbputrow var tablename 
#	store the current row for the table to file.
#
#  dbclearrow var tablename 
#	clear the data stored in all fields in the current row in the
#	table.
#
#  dbnewrow var tablename 
#	create a new row in the table, make it the current row.
#
#  dbdelrow var tablename 
#	delete the current row in the table
#
#  dbsearchstring var tablename fieldname searchstring 
#	return a list of sequence numbers of rows who have
#	a string in the given field that contains the
#	searchstring (case-insensitive) as a substring.
#
#  dbsearchdate var tablename fieldname date1 date2 
#	return a list of sequence numbers of rows who
#	have a date string (form:  "Mmm dd, yyyy") in the
#	indicated field that matches or falls between the
#	two date arguments (date1 <= date2).
#
#  dbsearchnum var tablename fieldname num1 num2 
#	return a list of sequence numbers of rows who have
#	a numeric value in the indicated field that matches
#	or falls between the two numeric value arguemnts
#	(num1 <= num2).
#
#  dbfirst var tablename 
#	set the first row for a table as the current row.
#
#  dblast var tablename 
#	set the last row for a table as the current row.
#
#  dbnext var tablename 
#	set next row in the table (from the current row) as the next.
#	row.
#
#  dbprev var tablename 
#	set the previous row in the table (relative to the current
#	row) as the current row.
#
#  dbreglockhitproc var proc  
#	register the name of a user-defined procedure to call if
#	a persistant lock on a row is encountered.
#
#  dbregmodnowriteproc var proc  
#	register the name of a user-defined procedure to call if
#	the current row has been modified and not written to
#	the database and a request has been made move on to a
#	different row.
#
#  dbmarkrowdirty var element operation 
#	marks the current row as modified (set via trace on
#	the fields of the current row).
#	(INTERNAL USE ONLY)
#
#  dbtracerowon var tablename 
#	activate modification trace on the fields of the current
#	row.
#	(INTERNAL USE ONLY)
#
#  dbtracerowoff var tablename 
#	deactivate modification trace on the fields of the
#	current row.
#	(INTERNAL USE ONLY)
#
#  dbsetrowlock var tablename seqno 
#	lock the row associated with the sequence number in the
#	indicated table.
#	(INTERNAL USE ONLY)
#
#  dbfreerowlock var tablename seqno rowpos 
#	free the lock on the row associated with the sequence number
#	in the indicated table.  ("rowpos" is the offset in the
#	index file, provided for a little efficiency).
#	(INTERNAL USE ONLY)
#
#  dbsettablelock var tablename 
#	lock the entire table.
#	(INTERNAL USE ONLY)
#
#  dbfreetablelock var tablename 
#	free the lock for the entire table.
#	(INTERNAL USE ONLY)
#
#=======================================================================
# DATABASE GLOBAL ARRAY ELEMENT DEFINITIONS
#
# $var					global array providing state 
#					information and current rows
#					for all tables in the database.
#
# ${var}(lockhitproc)			user defined procedure name to
#					call when a persistant locked
#					row is encountered.
#
# ${var}(modnowriteproc)		user defined procedure name to
#					call when the current row has been
#					modified but the row was not written
#					prior to moving on to a different
#					row.
#
# ${var}(path)				directory path to the database
#					files.
#
# ${var}($tablename,DUMMY)		special reserved field for row
#					padding allowing in-place minor
#					row modification (has no user 
#					significance).
#
# ${var}($tablename,currowlen)		number of characters in the
#					current row image after formatting, 
#					as read from the table file
#					(has no user significance).
#
# ${var}($tablename,currowpos)		table file offset to the current
#					row's storage location (has no
#					user significance).
#
# ${var}($tablename,curseqno)		sequence number of the current
#					row in the table.
#
# ${var}($tablename,dascount)		count of number of date search
#					full table scans made on the 
#					table.
#
# ${var}($tablename,dasmax)		maximum number of rows
#					matching date range search criteria
#					for table.
#
# ${var}($tablename,delcount)		count of the number of rows deleted
#					in the table.
#
# ${var}($tablename,$fieldname)		data for this field in this
#					row in the table.
#
# ${var}($tablename,fieldnames)		list of field names for this table.
#
# ${var}($tablename,getcount)		count of the number of rows read
#					in this table.
#
# ${var}($tablename,indexfilepath)	path to this table's index file
#
# ${var}($tablename,indexhandle)	file handle for this table's
#					index file
#
# ${var}($tablename,lastseqno)		sequence number of the last
#					row in this table.
#
#				
# ${var}($tablename,lckcount)		count of the number of times a
#					lock was encountered for I/O
#					operations against this table.
#
# ${var}($tablename,modcount)		number of rows written to this
#					table.
#
# ${var}($tablename,newcount)		number of new rows created in
#					this table.
#
# ${var}($tablename,nextidxpos)		file offset to the next
#					available index record for the
#					index file for this table.
#					(has no user significance).
#
# ${var}($tablename,nexttblpos)		file offset to the next
#					available table record for
#					the table file for this table.
#					(has no user significance).
#
# ${var}($tablename,nuscount)		count of the number of numeric
#					search full table scans made 
#					against this table.
#
# ${var}($tablename,nusmax)		maximum number of rows falling
#					within the numeric search range
#					for numeric searches made against
#					this table.
#
# ${var}($tablename,rowdirty)		flag to indicate that the current
#					row has been modified (has no
#					user significance).
#				
# ${var}($tablename,seqno)		reserved field for each row in the
#					table containing the row's
#					sequence number.
#
# ${var}($tablename,stscount)		count of the number of string
#					pattern search full table scans
#					made against this table.
#
# ${var}($tablename,stsmax)		maximum number of rows meeting
#					glob-style pattern string
#					search criteria for this table.
#
# ${var}($tablename,tablefilepath)	path to this table's data
#					file.
#
# ${var}($tablename,tablehandle)	file handle for this table's 
#					data file.
#
# ${var}($tablename,tablelock)		flag to indicate that this
#					table has a full table lock.
#			
# ${var}(tablenames)			list of table names in this
#					database.
#
#========================================================================
# ROW LEVEL LOCKING MECHANISM:
# 1) Each index file entry has two fields, [1] byte offset to data
#    area in table file encoded as 8 character hex number string
#    [2] a lock byte (0=unlocked, 1=locked) - index record = 11 bytes
# 2) The first record of the index file is for non-existant row 0.
#    Its lock byte indicates if the entire table is locked for
#    expansion (adding new row or relocating expanded row to end of file).
# 3) Table-level locking is only used for table file growth, it does not
#    affect row level access.
# 4) If a lock is encountered, we sleep for 1 second and try again.
#    If it is still locked, the "lockhitproc", if registered by the
#    application, is called.  If no procedure is registered, we try
#    the lock 5 more times then simply ignore it, writing a message to
#    stderr.
#========================================================================
#
#
# CODE SECTION
#
#
#========================================================================
# date utilities to move date string to timestamps and back and
# today's date

proc tstodate {ts} {
	set modays(norm) "0 31 59 90 120 151 181 212 243 273 304 334"
	set modays(leap) "0 31 60 91 121 152 182 213 244 274 305 335"
	set monthnames "Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec"
	set year [expr 1970 + ($ts / 31557600)]
	set yrtype norm
	if {[expr $year % 4] == 0} {set yrtype leap}
	set ts [expr $ts % 31557600]
	set day [expr 1 + ($ts / 86400)]
	for {set i 0} {$i < 12} {incr i} {
		if {$day < [lindex $modays($yrtype) $i]} {
			set month [lindex $monthnames [expr $i - 1 ]]
			set mostart [lindex $modays($yrtype) [expr $i - 1]]
			set day [expr $day - $mostart]	
			break
		}
	}
	return "$month $day, $year"
}

proc datetots {date} {
	if {[catch {scan $date "%3s %d, %d" mon day yr}]} {return 0}
	set ts [expr ($yr - 1970) * 31557600]
	set yrtype "norm"
	if {[catch {expr "$day + 0"}]} {return 0}
	if {[catch {expr "$yr + 0"}]} {return 0}
	if {$day < 1 || $day > 31} {return 0}
	if {$yr < 1970 || $yr > 2100} {return 0}
	if {[expr $yr%4] == 0} {set yrtype leap}
	set modays(norm) "0 31 59 90 120 151 181 212 243 273 304 334"
	set modays(leap) "0 31 60 91 121 152 182 213 244 274 305 335"
	set monthnames "Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec"
	set moidx [lsearch $monthnames $mon]
	if {$moidx == -1} {return 0}
	set yrday [expr [lindex $modays($yrtype) $moidx] + $day - 1]
	set yrday [expr $yrday * 86400]
	return [expr $ts + $yrday + 3600]
}

proc todaydate {} {
	scan [exec /bin/date "+%h %d, 19%y"] "%3s %d, %4s" mo da yr
	return "$mo $da, $yr"
}

#----------------
# dbcreate -- create, define, and open a new table for a database

proc dbcreate {path var tablename fieldnames} {
	global $var
	# -- basic sanity checking stuff
	if {![file isdirectory $path]} {
		error "dbcreate: database directory $path does not exist."
	}
	if {![file writable $path]} {
		error "dbcreate: database directory $path is not writeable by you."
	}
	# -- seed the index file and the table file
	exec /bin/echo "00000000 0" > $path/$tablename.idx
	set fldnames [concat seqno $fieldnames DUMMY]
	exec /bin/echo $fldnames > $path/$tablename.tbl
	# -- if the database variable is that of a real open database,
	# -- close it and use the dbopen to initialize everything nicely
	if {[info exists ${var}(lockhitproc)]} {
		dbclose $var
	}
	dbopen $path $var
}

#----------------
# dbopen -- open up a database based upon path to database directory
#           set all interesting information in global variable var

proc dbopen {path var} {
	global $var
	# -- basic sanity checking stuff
	if {![file isdirectory $path]} {
		error "dbopen: database directory $path does not exist."
	}
	if {![file writable $path]} {
		error "dbopen: database directory $path is not writeable by you."
	}
	# -- initialize the database global array, open table and
	# -- index files
	set ${var}(path) $path
	set ${var}(tablenames) {}
	set ${var}(lockhitproc) {}
	set ${var}(modnowriteproc) {}
	set tablepaths [glob $path/*.tbl]
	set ${var}(tablenames) {}
	foreach tablepath $tablepaths {
		set tablename [file tail [file rootname $tablepath]]
		lappend ${var}(tablenames) $tablename
		# -- initialize per table information
		set ${var}($tablename,indexhandle) {}
		set ${var}($tablename,tablehandle) {}
		set ${var}($tablename,fieldnames) {}
		set ${var}($tablename,tablefilepath) {}
		set ${var}($tablename,indexfilepath) {}
		set ${var}($tablename,lastseqno) 0
		set ${var}($tablename,nextidxpos) 0
		set ${var}($tablename,nexttblpos) 0
		set ${var}($tablename,curseqno) 0
		set ${var}($tablename,currowlen) 0
		set ${var}($tablename,currowpos) 0
		set ${var}($tablename,rowdirty) 0
		set ${var}($tablename,tablelock) 0
		set ${var}($tablename,getcount) 0
		set ${var}($tablename,delcount) 0
		set ${var}($tablename,newcount) 0
		set ${var}($tablename,modcount) 0
		set ${var}($tablename,lckcount) 0
		set ${var}($tablename,stscount) 0
		set ${var}($tablename,stsmax) 0
		set ${var}($tablename,dascount) 0
		set ${var}($tablename,dasmax) 0
		set ${var}($tablename,nuscount) 0
		set ${var}($tablename,nusmax) 0
		# -- now start filling in per table values
		set ${var}($tablename,tablefilepath) "[set ${var}(path)]/$tablename.tbl"
		set ${var}($tablename,indexfilepath) "[set ${var}(path)]/$tablename.idx"
		set ${var}($tablename,tablehandle) [open [set ${var}($tablename,tablefilepath)] r+]
		set ${var}($tablename,indexhandle) [open [set ${var}($tablename,indexfilepath)] r+]
		seek [set ${var}($tablename,tablehandle)] 0 start
		set ${var}($tablename,fieldnames) [gets [set ${var}($tablename,tablehandle)]]
		foreach fieldname [set ${var}($tablename,fieldnames)] {
			set ${var}($tablename,$fieldname) {}
		}
		set indexfilesize [file size [set ${var}($tablename,indexfilepath)]]
		set ${var}($tablename,lastseqno) \
			[expr ($indexfilesize / 11) - 1]
		set ${var}($tablename,nextidxpos) \
			[expr $indexfilesize - 1]
		set ${var}($tablename,nexttblpos) \
			[expr [file size [set ${var}($tablename,tablefilepath)]] - 0]
		dbtracerowoff $var $tablename
	}
	if {![info exists ${var}(modnowriteproc)]} {set ${var}(modnowriteproc) {}}
	if {![info exists ${var}(lockhitproc)]} {set ${var}(lockhitproc) {}}
}
		
#----------------
# dbcleanup -- compress voids and resequence table files

proc dbcleanup {var {tpath ""}}  {
	global $var
	# -- create a temporary  database variable
	set tvar "${var}_tmp"
	global $tvar 
	# -- define location of cleanup database (default is original location)
	set path [set ${var}(path)]
	if {$tpath == {}} {
		set ${tvar}(path) $path
		set tpath $path
	} else {
		set ${tvar}(path) $tpath
	}
	# -- capture the database table names
	set tablenames [set ${var}(tablenames)]
	# -- basic sanity checking stuff
	if {![file isdirectory [set ${tvar}(path)]]} {
		error "dbcleanup: database directory $tpath does not exist."
	}
	if {![file writable [set ${tvar}(path)]]} {
		error "dbcreate: database directory $tpath is not writeable by you."
	}
	# -- create a mirror database definition
	foreach tablename $tablenames {
		set tablenamet "${tablename}_tmp"
		set fieldnames ""
		# -- build a list of field names for the table
		foreach field [set ${var}($tablename,fieldnames)] {
			if {$field != "seqno" && $field != "DUMMY"} {
				lappend fieldnames $field
			}
		}
		# -- create the table files
		dbcreate $tpath $tvar $tablenamet $fieldnames
		# -- now walk through the original row of the table
		# -- and write them to the temporary database (if they exist)
		for {set seqno 1} {$seqno <= [set ${var}($tablename,lastseqno)]} {incr seqno} {
			dbnewrow $tvar $tablenamet
			dbgetrow $var $tablename $seqno
 			foreach fieldname $fieldnames {
				set ${tvar}($tablenamet,$fieldname) \
					[set ${var}($tablename,$fieldname)]
			}
			dbputrow $tvar $tablenamet
		}
	}
	dbclose $tvar
	dbclose $var
	# -- copy the temporary table files to the current database
	# -- location and name, then delete the temporary table files
	foreach tablename $tablenames {
		exec /bin/cp $tpath/${tablename}_tmp.idx $path/${tablename}.idx
		exec /bin/cp $tpath/${tablename}_tmp.tbl $path/${tablename}.tbl
		exec /bin/rm $tpath/${tablename}_tmp.idx
		exec /bin/rm $tpath/${tablename}_tmp.tbl
	}
	dbopen $path $var
}
		
#----------------
# dbclose -- close a database

proc dbclose {var} {
	global $var
	foreach tablename [set ${var}(tablenames)] {
		if {[set ${var}($tablename,rowdirty)]} {
			if {[string length [set ${var}(modnowriteproc)]] > 0} {
				if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
					dbputrow $var $tablename
				}
			 } else {
				# -- puts stderr "dbgetrow: dirty current row in $tablename."
				# -- puts stderr "          uncommitted mods ignored."
				set ${var}($tablename,rowdirty) 0
				dbclearrow $var $tablename
			}
		}
		close [set ${var}($tablename,tablehandle)]
		close [set ${var}($tablename,indexhandle)]
	}
	if {![info exists DEBUGDB]} {
		unset $var
	}
}

#----------------
# dbgetrow -- fetch a row from a table by its sequence number

proc dbgetrow {var tablename seqno} {
	global $var
	# -- puts stderr "dbgetrow -- var=$var tablename=$tablename seqno=$seqno"
	# -- test if we have uncommitted changes to current row
	if {[set ${var}($tablename,rowdirty)]} {
		if {[string length [set ${var}(modnowriteproc)]] > 0} {
			if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
				dbputrow $var $tablename
			}
		} else {
			set ${var}($tablename,rowdirty) 0
			dbclearrow $var $tablename
		}
	}
	# -- test validity of sequence number of row to fetch
	if {$seqno <= 0 || $seqno > [set ${var}($tablename,lastseqno)]} {
		error "dbgetrow: table=($tablename) seqno=($seqno) -  \
rowid outside of bounds ( 1 - [set ${var}($tablename,lastseqno)] )"
	}
	# -- turn off modification trace on fields
	dbtracerowoff $var $tablename
	# -- if this is not an empty row, read it in and store its fields
	# -- get index entry of row to fetch
	set tbloffset [dbsetrowlock $var $tablename $seqno]
	if {$tbloffset != 0} {
		seek [set ${var}($tablename,tablehandle)] $tbloffset start
		set irow [gets [set ${var}($tablename,tablehandle)]]
		set ${var}($tablename,currowlen) [string length $irow]
		set ${var}($tablename,curseqno) $seqno
		set ${var}($tablename,currowpos) $tbloffset
#		regsub -all <CR> $irow \n row
		set i 0
		foreach fieldname [set ${var}($tablename,fieldnames)] {
			set fld [lindex $irow $i]
			set ofld [string trimright $fld "<CR>"]
			regsub -all <CR> $ofld \n fld
			set  ${var}($tablename,$fieldname) $fld
			incr i
		}
	} else {
		# -- this is a previously deleted row, clear some stuff
		set ${var}($tablename,currowlen) 0
		set ${var}($tablename,currowpos) 0
		set ${var}($tablename,seqno) $seqno
		# -- clear the contents of the current row fields
		foreach fieldname [set ${var}($tablename,fieldnames)] {
			if {!($fieldname == "seqno" || $fieldname == "DUMMY")} {
				set ${var}($tablename,$fieldname) {}
			}
		}
		set ${var}($tablename,DUMMY) "                         "
	}
	# -- free the lock on this row
	dbfreerowlock $var $tablename $seqno $tbloffset
	# -- set trace on fields if they become modified
	set ${var}($tablename,rowdirty) 0
	dbtracerowon $var $tablename
	# -- increment the row fetch counter for session
	incr ${var}($tablename,getcount)
#	# -- puts stderr "dbgetrow exit"
}

#----------------
# dbputrow -- write a row back into database file

proc dbputrow {var tablename} {
	global $var
	# -- puts stderr "dbputrow -- var=$var tablename=$tablename"
	dbtracerowoff $var $tablename
	# -- if no changes to the fields in the row, simply return
	if {0 == [set ${var}($tablename,rowdirty)]} {
		dbtracerowon $var $tablename
		return
	}
	# -- create output row of all fields appended into a list
	set row ""
	foreach fieldname [set ${var}($tablename,fieldnames)] {
		regsub -all \n [set ${var}($tablename,$fieldname)] <CR> ofld
		set oofld [string trimright $ofld <CR>]
		lappend row $oofld
	}
	# -- get info on row length, and DUMMY field padding
	set dummylen [string length [set ${var}($tablename,DUMMY)]]
	set rowlen [string length $row]
	set rowdiff [expr $rowlen - [set ${var}($tablename,currowlen)]]
	# -- get a lock on the row and its offset in the table file
	set tbloffset [dbsetrowlock $var $tablename [set ${var}($tablename,curseqno)]]
	set ${var}($tablename,currowpos) $tbloffset
	# -- handle the special case that this is a new row or a
	# -- previously deleted row
	if {[set ${var}($tablename,currowpos)] == 0} {
		# -- get a table lock
		dbsettablelock $var $tablename
		set ${var}($tablename,currowpos) [set ${var}($tablename,nexttblpos)]
		set ${var}($tablename,currowlen) $rowlen
		incr ${var}($tablename,nexttblpos) [expr 1 + $rowlen] 
		# -- setup to fall through to actual table write of row
		set rowdiff 0
	} elseif {[expr [set ${var}($tablename,currowpos)] + \
		        [set ${var}($tablename,currowlen)] + 1] == \
		  [set ${var}($tablename,nexttblpos)]} {
		# -- special case that this is the last physical row in
		# -- table, always expand/contract in place.
		# -- index entry is ok at this point.
		# -- if contracting, expand dummy field to size.
		if {$rowdiff < 0} {
			set ${var}($tablename,DUMMY) \
				[format "%[expr $dummylen + abs($rowdiff)]s" " "]
			set lastelem [expr [llength $row] - 1]
			set row [lreplace $row $lastelem $lastelem [set ${var}($tablename,DUMMY)]]
			set rowlen [string length $row]
		} elseif {$rowdiff > 0} {
			# -- get a table lock
			dbsettablelock $var $tablename
			# -- ensure that padding field is maximum size
			set ${var}($tablename,DUMMY) [format "%25s" " "]
			set dummylen [string length [set ${var}($tablename,DUMMY)]]
			set lastelem [expr [llength $row] - 1]
			set row [lreplace $row $lastelem $lastelem [set ${var}($tablename,DUMMY)]]
			set rowlen [string length $row]
		}
		# -- set up to fall through to actual table write
		set rowdiff 0
	}
	if {$rowdiff < 0} {
		# -- row will fit in place with adjustment to dummy pad area
		set ${var}($tablename,DUMMY) [format "%[expr $dummylen + abs($rowdiff)]s" " "]
		set lastelem [expr [llength $row] - 1]
		set row [lreplace $row $lastelem $lastelem [set ${var}($tablename,DUMMY)]]
		set rowlen [string length $row]
	} elseif {$rowdiff > 0} {
		# -- see if the row will fit in place by reducing DUMMY
		if {$rowdiff < $dummylen} {
			set ${var}($tablename,DUMMY) [format "%[expr $dummylen - $rowdiff]s" " "]
			set lastelem [expr [llength $row] - 1]
			set row [lreplace $row $lastelem $lastelem [set ${var}($tablename,DUMMY)]]
			set rowlen [string length $row]
		} else {
			# -- not enough space in current table position, move row
			# -- to end of table file
			# -- first, clear the current entry in the table file.
			set blank [format "%[set ${var}($tablename,currowlen)]s" " "]
			seek [set ${var}($tablename,tablehandle)] [set ${var}($tablename,currowpos)] start
			puts [set ${var}($tablename,tablehandle)] $blank
			# -- reexpand DUMMY field, if necessary
			if {$dummylen < 25} {
				set ${var}($tablename,DUMMY) \
					"                         "
				set lastelem [expr [llength $row] - 1]
				set row [lreplace $row $lastelem $lastelem [set ${var}($tablename,DUMMY)]]
				set rowlen [string length $row]
			}
			# -- set a lock on the table
			dbsettablelock $var $tablename
			# -- now update the index file row offset (row still locked)
			set ${var}($tablename,currowpos) [set ${var}($tablename,nexttblpos)]
			# -- now setup to write row to end of table file
			set ${var}($tablename,currowlen) $rowlen
			set ${var}($tablename,currowpos) [set ${var}($tablename,nexttblpos)]
			incr ${var}($tablename,nexttblpos) [expr 1 + $rowlen]
		}
	}
	# -- write the row to the tablespace file
	seek [set ${var}($tablename,tablehandle)] [set ${var}($tablename,currowpos)] start
	puts [set ${var}($tablename,tablehandle)] $row
	flush [set ${var}($tablename,tablehandle)]
	# -- if the table was locked, free it
	if {[set ${var}($tablename,tablelock)]} {
		incr ${var}($tablename,nexttblpos) [expr $rowlen - [set ${var}($tablename,currowlen)]]
		dbfreetablelock $var $tablename
	}
	set ${var}($tablename,currowlen) $rowlen
	# -- update the index and free the row lock
	dbfreerowlock $var $tablename [set ${var}($tablename,curseqno)] [set ${var}($tablename,currowpos)]
	# -- reset row dirty flag and turn field mod trace back on
	set ${var}($tablename,rowdirty) 0
	dbtracerowon $var $tablename
	# -- increment the row write counter for session
	incr ${var}($tablename,modcount)
}

#----------------
# dbclearrow -- clear the contents of the current row

proc dbclearrow {var tablename} {
	global $var
	# -- puts stderr "dbclearrow -- var=$var tablename=$tablename"
	# -- test if we have uncommitted changes to current row
	if {[set ${var}($tablename,rowdirty)]} {
		if {[string length [set ${var}(modnowriteproc)]] > 0} {
			if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
				dbputrow $var $tablename
			}
		} else {
			# -- puts stderr "dbclearrow: dirty current row in $tablename."
			# -- puts stderr "            uncommitted mods ignored."
		}
	}
	foreach fieldname [set ${var}($tablename,fieldnames)] {
		set ${var}($tablename,$fieldname) {}
	}
	foreach fieldname [set ${var}($tablename,fieldnames)] {
		set ${var}($tablename,$fieldname) {}
	}
	set ${var}($tablename,DUMMY) "                         "
	set ${var}($tablename,currowlen) 0
	set ${var}($tablename,rowdirty) 0
	set ${var}($tablename,seqno) [set ${var}($tablename,curseqno)]
}

#----------------
# dbnewrow -- create a new row in a table

proc dbnewrow {var tablename} {
	global $var
	# -- puts stderr "dbnewrow -- var=$var tablename=$tablename"
	# -- test if we have uncommitted changes to current row
	if {[set ${var}($tablename,rowdirty)]} {
		if {[string length [set ${var}(modnowriteproc)]] > 0} {
			if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
				dbputrow $var $tablename
			}
		} else {
			# -- puts stderr "dbnewrow: dirty current row in $tablename."
			# -- puts stderr "          uncommitted mods ignored."
			set ${var}($tablename,rowdirty) 0
			dbclearrow $var $tablename
		}
	}
	dbtracerowoff $var $tablename
	# -- obtain a table lock
	dbsettablelock $var $tablename
	# -- grow the sequence numbers for the table by 1
	incr ${var}($tablename,lastseqno) 1
	set ${var}($tablename,curseqno) [set ${var}($tablename,lastseqno)]
	# -- clear the fields for the new row and show no data in table file
	dbclearrow $var $tablename	
	set ${var}($tablename,currowpos) 0
	set ${var}($tablename,seqno) [set ${var}($tablename,curseqno)]
	# -- update the index for the new record via a row unlock
	dbfreerowlock $var $tablename [set ${var}($tablename,curseqno)] [set ${var}($tablename,currowpos)]
	# -- free the table lock
	dbfreetablelock $var $tablename
	# -- reset to watch fields for modification
	set ${var}($tablename,rowdirty) 0
	dbtracerowon $var $tablename
	# -- increment the new row counter for the session
	incr ${var}($tablename,newcount)
}

#----------------
# dbdelrow -- delete a row from a table

proc dbdelrow {var tablename} {
	global $var
	# -- puts stderr "dbdelrow -- var=$var tablename=$tablename"
	# -- simply return if there is nothing to delete
	if {[set ${var}($tablename,currowlen)] == 0} {
		return
	}
	if {[set ${var}($tablename,currowpos)] == 0} {
		dbclearrow $var $tablename
		return
	}
	set blanks [format "%[set ${var}($tablename,currowlen)]s" " "]
	set pos [set ${var}($tablename,currowpos)]
	# -- turn off tracing modification to the fields in the table
	dbtracerowoff $var $tablename
	# -- clear the fields of the row
	dbclearrow $var $tablename
	# -- obtain a row lock
	set tbloffset [dbsetrowlock $var $tablename [set ${var}($tablename,curseqno)]]
	# -- if there was data in table file, then
	# -- overwrite the row data with a blank line of equal size
	if {$tbloffset != 0} {
		seek [set ${var}($tablename,tablehandle)] $pos start
		puts [set ${var}($tablename,tablehandle)] $blanks
		flush [set ${var}($tablename,tablehandle)]
	}
	# -- free the lock on the file and indicate the row is now empty
	dbfreerowlock $var $tablename [set ${var}($tablename,curseqno)] 0
	# -- turn field tracing back on
	set ${var}($tablename,rowdirty) 0
	dbtracerowon $var $tablename
	# -- increment the row delete count for the session
	incr ${var}($tablename,delcount)
}

#----------------
# return a list of seqno from a table who have a field matching glob
# search string

proc dbsearchstring {var tablename fieldname searchstring} {
	global $var
	if {[set ${var}($tablename,lastseqno)] == 0} {
		incr ${var}($tablename,stscount)
		return {}
	}
	if {[string length $searchstring] == 0} {return {}}
	set searchstring [string tolower "*${searchstring}*"]
	set result ""
	set fieldindex -1
	set i 0
	foreach field [set ${var}($tablename,fieldnames)] {
		if {$field == $fieldname} {
			set fieldindex $i
			break
		}
		incr i
	}
	if {$fieldindex == -1} {
		error "dbsearchstring: field name $fieldname does not \
exist in table $tablename."
	}
	for {set seqno 1} {$seqno <= [set ${var}($tablename,lastseqno)]} {incr seqno} {
		# -- get the row position in table file and lock row
		set rowpos [dbsetrowlock $var $tablename $seqno]
		# -- if an empty or deleted row, free lock and go on to next row
		if {$rowpos != 0} {
			# -- get the row from the table and check for string match
			seek [set ${var}($tablename,tablehandle)] $rowpos
			set field [lindex [gets [set ${var}($tablename,tablehandle)]] $fieldindex]
			if {[string match $searchstring [string tolower $field]]} {
				lappend result $seqno
			}
		}
		# -- free the lock on the row
		dbfreerowlock $var $tablename $seqno $rowpos
	}
	incr ${var}($tablename,stscount)
	set resultsize [llength $result]
	if {$resultsize > [set ${var}($tablename,stsmax)]} {
		set ${var}($tablename,stsmax) $resultsize
	}
	return $result
}

#----------------
# return a list of seqnos from a table who have a date field between
# or matching a date in the date range

proc dbsearchdate {var tablename fieldname date1 date2} {
	global $var
	if {[set ${var}($tablename,lastseqno)] == 0} {
		incr ${var}($tablename,dascount)
		return {}
	}
	if {[datetots $date1] == 0} {return {}}
	if {[datetots $date2] == 0} {return {}}
	set result {}
	set fieldindex -1
	set i 0
	foreach field [set ${var}($tablename,fieldnames)] {
		if {$field == $fieldname} {
			set fieldindex $i
			break
		}
		incr i
	}
	if {$fieldindex == -1} {
		error "dbsearchstring: field name $fieldname does not \
exist in table $tablename."
	}
	set ts1 [datetots $date1]
	set ts2 [datetots $date2]
	for {set seqno 1} {$seqno <= [set ${var}($tablename,lastseqno)]} {incr seqno} {
		# -- get the row position in table file and lock row
		set rowpos [dbsetrowlock $var $tablename $seqno]
		# -- if an empty or deleted row, free lock and go on to next row
		if {$rowpos != 0} {
			# -- get the row from the table and check for string match
			seek [set ${var}($tablename,tablehandle)] $rowpos
			set field [datetots [lindex \
				[gets [set ${var}($tablename,tablehandle)]] $fieldindex]]
			if {$field == 0} {return {}}
			if {$ts1 <= $field && $ts2 >= $field} {
				lappend result $seqno
			}
		}
		# -- free the lock on the row
		dbfreerowlock $var $tablename $seqno $rowpos
	}
	incr ${var}($tablename,dascount)
	set resultsize [llength $result]
	if {$resultsize > [set ${var}($tablename,dasmax)]} {
		set ${var}($tablename,dasmax) $resultsize
	}
	return $result
}

#----------------
# return a list of seqnos from a table who have a numeric value
# between or matching values in a numeric range

proc dbsearchnum {var tablename fieldname num1 num2} {
	global $var
	if {[set ${var}($tablename,lastseqno)] == 0} {
		incr ${var}($tablename,nuscount)
		return {}
	}
	if {[catch {expr "$num1 + $num2"}]} {
#		# -- puts "dbsearchnum $num1 or $num2 are not valid numbers."
		return {}
	}
	set result {}
	set fieldindex -1
	set i 0
	foreach field [set ${var}($tablename,fieldnames)] {
		if {$field == $fieldname} {
			set fieldindex $i
			break
		}
		incr i
	}
	if {$fieldindex == -1} {
		error "dbsearchstring: field name $fieldname does not \
exist in table $tablename."
	}
	for {set seqno 1} {$seqno <= [set ${var}($tablename,lastseqno)]} {incr seqno} {
		# -- get the row position in table file and lock row
		set rowpos [dbsetrowlock $var $tablename $seqno]
		# -- if an empty or deleted row, free lock and go on to next row
		if {$rowpos != 0} {
			# -- get the row from the table and check for string match
			seek [set ${var}($tablename,tablehandle)] $rowpos
			set field [lindex [gets [set ${var}($tablename,tablehandle)]] $fieldindex]
			if {[catch {expr "$field + 0"}]} {
#				# -- puts stderr "dbsearchnum: seqno=$seqno, field=$field, invalid numeric value"
				return {}
			}
			if {$num1 <= $field && $num2 >= $field} {
				lappend result $seqno
			}
		}
		# -- free the lock on the row
		dbfreerowlock $var $tablename $seqno $rowpos
	}
	incr ${var}($tablename,nuscount)
	set resultsize [llength $result]
	if {$resultsize > [set ${var}($tablename,nusmax)]} {
		set ${var}($tablename,nusmax) $resultsize
	}
	return $result
}

#----------------
# dbfirst -- return first row from table
proc dbfirst {var tablename} {
	global $var
	# -- puts stderr "dbfirst -- var=$var tablename=$tablename"
	if {[set ${var}($tablename,lastseqno)] == 0} {return}
	# -- test if we have uncommitted changes to current row
	if {[set ${var}($tablename,rowdirty)]} {
		if {[string length [set ${var}(modnowriteproc)]] > 0} {
			if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
				dbputrow $var $tablename
			}
		} else {
			# -- puts stderr "dbfirst: dirty current row in $tablename."
			# -- puts stderr "         uncommitted mods ignored."
			set ${var}($tablename,rowdirty) 0
			dbclearrow $var $tablename
		}
	}
	set seqno 1
	dbgetrow $var $tablename $seqno
	while {[set ${var}($tablename,currowpos)] == 0 && $seqno <= [set ${var}($tablename,lastseqno)]} {
		incr seqno
		dbgetrow $var $tablename $seqno
	}
}
#----------------
# dblast -- return last row from table
proc dblast {var tablename} {
	global $var 
	# -- puts stderr "dblast -- var=$var tablename=$tablename"
	if {[set ${var}($tablename,lastseqno)] == 0} {return}
	# -- test if we have uncommitted changes to current row
	if {[set ${var}($tablename,rowdirty)]} {
		if {[string length [set ${var}(modnowriteproc)]] > 0} {
			if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
				dbputrow $var $tablename
			}
		} else {
			# -- puts stderr "dblast: dirty current row in $tablename."
			# -- puts stderr "        uncommitted mods ignored."
			set ${var}($tablename,rowdirty) 0
			dbclearrow $var $tablename
		}
	}
	set seqno [set ${var}($tablename,lastseqno)]
	dbgetrow $var $tablename $seqno
	while {[set ${var}($tablename,currowpos)] == 0 && $seqno > 0} {
		incr seqno -1
		dbgetrow $var $tablename $seqno
	}
}
#----------------
# dbnext -- return next row from table
proc dbnext {var tablename} {
	global $var
	# -- puts stderr "dbnext -- var=$var tablename=$tablename"
	if {[set ${var}($tablename,lastseqno)] == 0} {return}
	# -- test if we have uncommitted changes to current row
	if {[set ${var}($tablename,rowdirty)]} {
		if {[string length [set ${var}(modnowriteproc)]] > 0} {
			if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
				dbputrow $var $tablename
			}
		} else {
			# -- puts stderr "dbnext: dirty current row in $tablename."
			# -- puts stderr "        uncommitted mods ignored."
			set ${var}($tablename,rowdirty) 0
			dbclearrow $var $tablename
		}
	}
	set seqno [set ${var}($tablename,curseqno)]
	incr seqno
	if {$seqno <= [set ${var}($tablename,lastseqno)]} {
		dbgetrow $var $tablename $seqno
	}
	while {[set ${var}($tablename,currowpos)] == 0 && $seqno <= [set ${var}($tablename,lastseqno)]} {
		incr seqno
		dbgetrow $var $tablename $seqno
	}
}
#----------------
# dbprev -- return previous row from table
proc dbprev {var tablename} {
	global $var 
	# -- puts stderr "dbprev -- var=$var tablename=$tablename"
	if {[set ${var}($tablename,lastseqno)] == 0} {return}
	# -- test if we have uncommitted changes to current row
	if {[set ${var}($tablename,rowdirty)]} {
		if {[string length [set ${var}(modnowriteproc)]] > 0} {
			if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
				dbputrow $var $tablename
			}
		} else {
			# -- puts stderr "dbprev: dirty current row in $tablename."
			# -- puts stderr "        uncommitted mods ignored."
			set ${var}($tablename,rowdirty) 0
			dbclearrow $var $tablename
		}
	}
	set seqno [set ${var}($tablename,curseqno)]
	incr seqno -1
	if {$seqno > 0} {
		dbgetrow $var $tablename $seqno
	}
	while {[set ${var}($tablename,currowpos)] == 0 && $seqno > 1} {
		incr seqno -1
		dbgetrow $var $tablename $seqno
	}
}

#---------------------
# dbreglockhitproc -- register procedure to call if a lock has been
#		      encountered.  The registered procedure needs to 
#		      have the argument list: {var, tablename,
#		      msgstring}.  The registered procedure needs to
#		      return 0 to cancel the operation, or 1 to spin
#	  	      and retry the operation for a while

proc dbreglockhitproc {var proced} {
	global $var
	set ${var}(lockhitproc) $proced
}

#---------------------
# dbregmodnowriteproc -- register procedure to call if the current
#			 row is modified and a write has not be made.  
#			 The registered procedure needs to have the 
#			 argument list: {var tablename msgstring}.  
#			 The registered procedure needs to return 0
#			 to ignore the dirty row and continue, or 
#			 1 to write the row and continue on.

proc dbregmodnowriteproc {var proced} {
	global $var
	set ${var}(modnowriteproc) $proced
}
	
#------------------
# dbmarkrowdirty -- mark the fields of the current row as modified

proc dbmarkrowdirty {var element operation} {
	global $var
	# -- puts stderr "dbmarkrowdirty -- var=$var element=$element operation=$operation"
	set tablename [lindex [split $element ","] 0]
	set ${var}($tablename,rowdirty) 1
#	# -- puts stderr "dbmarkrowdirty: $var $element "
}

#------------------
# dbtracerowon -- set current table row fields in watched state if
# 		  mods are made to any of them.

proc dbtracerowon {var tablename} {
	global $var
	# -- puts stderr "dbtracerowon -- var=$var tablename=$tablename"
	foreach fieldname [set ${var}($tablename,fieldnames)] {
		set t "${var}($tablename,$fieldname)"
		trace variable $t w dbmarkrowdirty
	}
#	# -- puts stderr "dbtracerowon: $var $tablename"
}

#-------------------
# dbtracerowoff -- put current table row fields into unwatched
#		   state so to ignore mods made to any of them.

proc dbtracerowoff {var tablename} {
	global $var
	# -- puts stderr "dbtracerowoff -- var=$var tablename=$tablename"
	foreach fieldname [set ${var}($tablename,fieldnames)] {
		set t "${var}($tablename,$fieldname)"
		trace vdelete $t w dbmarkrowdirty
	}
#	# -- puts stderr "dbtracerowoff:  $var $tablename"
}

#----------------------
# dbsetrowlock -- obtain a lock on a row in a table

proc dbsetrowlock {var tablename seqno} {
	global $var
	# -- puts stderr "dbsetrowlock -- var=$var tablename=$tablename seqno=$seqno"
	seek [set ${var}($tablename,indexhandle)] [expr $seqno * 11] start
	scan [gets [set ${var}($tablename,indexhandle)]] "%x %s" tbloffset lock
	# -- if the record is locked, wait for 1 second, try again, then
	# -- call the lock procedure if still locked, default - ignore lock
	while {$lock} {
		incr ${var}($tablename,lckcount)
		exec /bin/sleep 1
		seek [set ${var}($tablename,indexhandle)] [expr $seqno * 11] start
		scan [gets [set ${var}($tablename,indexhandle)]] "%x %s" tbloffset lock
		if {$lock} {
			if {[set ${var}(lockhitproc)] != {}} {
				set lock [eval [set ${var}(lockhitproc)] $var $tablename]
			} else {
				set spin 0
				while {$lock && $spin < 5} {
					exec sleep 1
					seek [set ${var}($tablename,indexhandle)] [expr $seqno * 11] start
					scan [gets [set ${var}($tablename,indexhandle)]] \
						"%x %d" tbloffset lock
					incr spin
				}
				if {$lock} {
					# -- puts stderr "dbputrow: lock hit on table: $tablename sequence number: $seqno"
					# -- puts stderr "          ignoring lock."
					set lock 0
				}
			}
		}
	}
	# -- set a lock on the record
	seek [set ${var}($tablename,indexhandle)] [expr $seqno * 11] start 
	puts [set ${var}($tablename,indexhandle)] [format "%08x 1" $tbloffset]
	flush [set ${var}($tablename,indexhandle)]
	return $tbloffset
}

#----------------------
# dbfreerowlock -- free a row lock on a row in a table

proc dbfreerowlock {var tablename seqno rowpos} {
	global $var
	# -- puts stderr "dbfreerowlock -- var=$var tablename=$tablename seqno=$seqno rowpos=$rowpos"
	seek [set ${var}($tablename,indexhandle)] [expr $seqno * 11] start
	puts [set ${var}($tablename,indexhandle)] [format "%08x 0" $rowpos]
	flush [set ${var}($tablename,indexhandle)]
}

#----------------------
# dbsettablelock -- obtain a lock on a table to grow it

proc dbsettablelock {var tablename} {
	global $var
	# -- puts stderr "dbsettablelock -- var=$var tablename=$tablename"
	set x [dbsetrowlock $var $tablename 0]
	set ${var}($tablename,nexttblpos) \
		[expr [file size [set ${var}($tablename,tablefilepath)]] - 0]
	set ${var}($tablename,tablelock) 1
	# -- in case of multiusers, make certain that we have current info
	# -- on nextpositions for this table and last sequence number
	set idxsize [file size [set ${var}($tablename,indexfilepath)]]
	set ${var}($tablename,lastseqno) [expr ($idxsize / 11) - 1]
	set ${var}($tablename,nextidxpos) [expr $idxsize - 1]
	set ${var}($tablename,nexttblpos) [file size [set ${var}($tablename,tablefilepath)]]
}

#----------------------
# dbfreetablelock -- free a table lock

proc dbfreetablelock {var tablename} {
	global $var
	# -- puts stderr "dbfreetablelock -- var=$var tablename=$tablename"
	dbfreerowlock $var $tablename 0 0
	set ${var}($tablename,tablelock) 0
}

