# TableListConfig.tcl - itcl popup widget for editing table layout
# for the TableList class.
#
# Copyright (C) 1994 Allan Brighton (allan@piano.sta.sub.org)



itcl_class TableListConfig {
    inherit TopLevelWidget


    method create_main_window {} {
	make_list_window
	make_options_window
	make_button_row
    }

    
    # create a window with 2 scrolling lists and arrows
    # between then to move items back and forth.
    
    method make_list_window {} {
	pack [DoubleTableList $this.dtab \
		-left_title {Show} \
		-right_title {Don't Show} \
		-headings "Heading Align $expr_type_ Separator Precision" \
		-exportselection 0 \
		-selectmode extended] \
		-side top -fill both -expand 1

	# enable setting options when the left list is active
	bind $this.dtab.left.listbox <ButtonRelease-1> "+$this left_selected"
	bind $this.dtab.right.listbox <ButtonRelease-1> "+$this right_selected"
    }


    # called when an item in the left listbox is selected

    method left_selected {} {
	$this show_option
	$this.dtab.right clear_selection
    }


    # called when an item in the right listbox is selected

    method right_selected {} {
	$this.dtab.left clear_selection
	foreach i {align wildcard sep} {
	    $this.opts.$i config -value {} -state disabled
	}
	$this.opts.$i config -value 0 -state disabled
    }

    
    # display the options for the item selected in the left list

    method show_option {} {
	foreach i {align wildcard sep} {
	    $this.opts.$i config -state normal
	}
	set name [lindex [lindex [$this.dtab.left get_selected] 0] 0]
	if {"$name" == ""} {
	    return
	}
	$this.opts.align config -value \
		$options_($name,Align)
	$this.opts.wildcard config -value \
		$options_($name,Wildcard)
	$this.opts.sep config -value \
		$options_($name,Separator)
	$this.opts.prec config -value \
		[set prec  $options_($name,Precision)]

	# 0 precision means its probably a text field
	if {$prec == 0} {
	    $this.opts.prec config -state disabled
	} else {
	    $this.opts.prec config -state normal
	}
	
    }

    
    # this method is called whenever the user types something into one of
    # the entries. Set the option on all selected items.
    
    method set_option_old {option value} {
	set index_list [$this.dtab.left.listbox curselection]
	set n 0
	foreach i [$this.dtab.left get_selected] {
	    set name [lindex $i 0]
	    set index [lindex $index_list $n]
	    set options_($name,$option) $value
	    set r [get_row_info $name]
	    $this.dtab.left set_row $index $r
	    incr n
	}
	$this.dtab.left select_rows \
	    [lindex $index_list 0] \
	    [lindex $index_list [expr [llength $index_list]-1]]
    }

    
    # this method is called whenever the user types something into one of
    # the entries. Set the option on all selected items.
    
    method set_option {option value} {
	foreach i [$this.dtab.left get_selected] {
	    set name [lindex $i 0]
	    set options_($name,$option) $value
	    set r [get_row_info $name]
	    $this.dtab.left set_row $i $r
	}
    }


    # create a window for setting column options

    method make_options_window {} {
	pack [frame $this.opts -bd 1m] \
		-side top -fill x

	pack [label $this.opts.title -text "Options"] \
		-side top -fill x -expand 1 -ipady 1m

	pack [set f [frame $this.opts.f -bd 3 -relief groove]] \
		-pady 1m -padx 1m -ipady 1m -ipadx 1m -fill both -expand 1

	pack [Choice $this.opts.align \
		-text {Align:} \
		-choice {Left Right} \
		-state disabled \
		-command "$this set_option Align"] \
		-side left -expand 1 \
		-ipadx 1m -ipady 1m -in $f
	
	pack [LabelEntry $this.opts.wildcard \
		-text "${expr_type_}:" \
		-value {} \
		-changecmd "$this set_option Wildcard" \
		-action "$this ok" \
		-valueWidth 8 \
		-state disabled] \
		-side left -fill x -expand 1 \
		-ipadx 1m -ipady 1m -in $f

	pack [LabelEntry $this.opts.sep \
		-text {Separator:} \
		-value {} \
		-changecmd "$this set_option Separator" \
		-action "$this ok" \
		-valueWidth 3 \
		-state disabled] \
		-side left -expand 1 \
		-ipadx 1m -ipady 1m -in $f

	pack [LabelNumber $this.opts.prec \
		-text {Precision:} \
		-value {0} \
		-valueWidth 1 \
		-min 1 \
		-max 20 \
		-changecmd "$this set_option Precision" \
		-state disabled] \
		-side left -expand 1 -in $f
    }


    # deselect all items in the lists and disable the entries and options

    method deselect_all {} {
	$this.dtab.left clear_selection
	$this.dtab.right clear_selection
	foreach i {align wildcard sep} {
	    $this.opts.$i config -value {} -state disabled
	}
	$this.opts.prec config -value 0 -state disabled
    }

    
    # make the row of buttons at bottom

    method make_button_row {} {
	pack [ButtonFrame $this.btns \
		-ok_cmd "$this ok" \
		-cancel_cmd "$this cancel"] \
		-side top -fill x -ipady 1m
	#$this.btns append {Reset} "$this reset"
    }


    # reset to default configuration

    method reset {} {
	deselect_all
	$table.config_file set_defaults 1
	$this.dtab clear
	load
    }


    # called for the cancel button - reset to original values and quit

    method cancel {} {
	deselect_all
	quit
	$this.dtab clear
	load
    }

    
    # accept the current information

    method ok {args} {
	# get/set the list of headings to show/hide
	set list {}
	foreach i [$this.dtab.left get_contents] {
	    set name [lindex $i 0]
	    set options_($name,Show)  1
	    lappend list $name
	}
	foreach i [$this.dtab.right get_contents] {
	    set name [lindex $i 0]
	    set options_($name,Show) 0
	    lappend list $name
	}
	busy {
	    $table.config_file set_options options_
	    $table.config_file config -order $list
	    $table match_info_
	    $table set_info_
	}

	# we're done, make window go away now
	deselect_all
	quit
    }

    
    # return info for the given name

    method get_row_info {name} {
	set align $options_($name,Align)
	set wildcard $options_($name,Wildcard)
	set sep $options_($name,Separator)
	set prec $options_($name,Precision)
	return [list $name $align $wildcard $sep $prec]
    }


    # The information was read by $table.config_file already, so we
    # just display it here.

    method load {} {
	$table.config_file get_options options_
	set left {}
	set right {}
	foreach name [$table.config_file info public order -value] {
	    if {$options_($name,Show)} {
		set side left
	    } else {
		set side right
	    }
	    lappend $side [get_row_info $name]
	}
	$this.dtab.left config -info $left
	$this.dtab.right config -info $right
    }

    
    # constructor - create the windows and load config file

    constructor {config} {
	TopLevelWidget::constructor
	if {"$table" == ""} {
	    error "table not specified"
	}

	if {[$table.config_file info public use_regexp -value]} {
	    set expr_type_ {Regexpr:}
	} else {
	    set expr_type_ {Wildcard:}
	}

	create_main_window
	wm title $this "Table Column Layout"
	load
	set initialized_ 1
   }


    # -- public member variables --

    # TableList whose layout is being edited
    public table {}


    # -- protected members --

    # set to 1 after initialization
    protected initialized_ 0

    # array(name,option) of option values, loaded from config_file
    protected options_

    # type of matching expressions used (Regexp, Wildcard)
    protected expr_type_ {Wildcard}
}

