# TableListSort.tcl - itcl popup widget for sorting contents of a TableList
#
# Copyright (C) 1994 Allan Brighton (allan@piano.sta.sub.org)



itcl_class TableListSort {
    inherit TopLevelWidget


    # do the main window layout

    method create_main_window {} {
	make_dlist_window
	make_options_window
	make_button_row
    }

    
    # create a window with 2 lists - one for the headings
    # and one for the sort keys.
    
    method make_dlist_window {} {
	pack [DoubleList $this.dlist \
		-left_title {Sort Keys} \
		-right_title {Table Headings}] \
		-side top -fill both -expand 1
    }


    # create a window for setting column options

    method make_options_window {} {
	pack [frame $this.opts -bd 1m] \
		-side top -fill x

	pack [label $this.opts.title -text "Options"] \
		-side top -fill x -expand 1 -ipady 1m

	pack [set f [frame $this.opts.f -bd 3 -relief groove]] \
		-pady 1m -padx 1m -ipady 1m -ipadx 1m -fill x -expand 1

	pack [Choice $this.order \
		-text {Order:} \
		-choice {increasing decreasing}] \
		-side top -expand 1 \
		-ipadx 1m -ipady 1m -in $f
    }


    # make the row of buttons at bottom

    method make_button_row {} {
	pack [ButtonFrame $this.btns \
		-ok_cmd "$this apply" \
		-cancel_cmd "$this cancel"] \
		-side bottom -fill x -expand 1 -ipady 1m
	$this.btns append {Reset} "$this reset"
    }


    # reset the sorting parameters to the default values

    method reset {} {
	$this.dlist clear
	foreach i [$table.config_file info public headings -value] {
	    $this.dlist.right append $i
	}
	$this.order config -value increasing
    }


    # called for the cancel button

    method cancel {} {
	$this quit
	$this.dlist clear
	load
    }


    # apply the changes (called for OK button)

    method apply {} {

	# map headings to col index
	set i -1
	set headings [$table.config_file info public headings -value]
	foreach name $headings {
	    set col($name) [incr i]
	}

	set sort_by {}
	foreach name [$this.dlist.left get_contents] {
	    lappend sort_by $col($name)
	}
	set sort_order [$this.order get]

	# configure table and config file
	set prev_sort_by [$table info public sort_by -value]
	set prev_sort_order [$table info public sort_order -value]
	$table config -sort_by $sort_by
	$table config -sort_order $sort_order
	$table.config_file config -sort_by $sort_by
	$table.config_file config -sort_order $sort_order

	# if the sort params have changed, redisplay the table
	if {"$prev_sort_by" != "$sort_by" || "$prev_sort_order" != "$sort_order"} {
	    busy {
		$table new_info
	    }
	}
	
	quit
    }


    # load the sort settings from the config file object

    method load {} {
	set headings [$table.config_file info public headings -value]
	set sort_by [$table.config_file info public sort_by -value]
	foreach i $sort_by {
	    set key([lindex $headings $i]) 1
	}

	foreach i $headings {
	    if {[info exists key($i)]} {
		$this.dlist.left append $i
	    } else {
		$this.dlist.right append $i
	    }
	}
    }

    
    # constructor - create the windows and load config file

    constructor {config} {
	TopLevelWidget::constructor
	if {"$table" == ""} {
	    error "table not specified"
	}
	create_main_window
	load
	wm minsize $this 10 10 
	wm title $this "Table Sort"
	
	set initialized_ 1
   }

   
    # -- public member variables --

    # TableList whose layout is being edited
    public table {}


    # -- protected members --

    # set to 1 after initialization
    protected initialized_ 0
}

