# safe-tk.tcl,v 1.1 1995/11/17 00:42:10 steve Exp
#
#	PASTIME Project
#	Cooperative Research Centre for Advanced Computational Systems
#	COPYRIGHT NOTICE AND DISCLAIMER.
#
#	Copyright (c) 1995 ANU and CSIRO
#	on behalf of the participants in
#	the CRC for Advanced Computational Systems (ACSys)
#
# This software and all associated data and documentation ("Software")
# was developed for research purposes and ACSys does not warrant that 
# it is error free or fit for any purpose.  ACSys disclaims all liability
# for all claims, expenses, losses, damages and costs any user may incur 
# as a result of using, copying or modifying the Software.
#
# You may make copies of the Software but you must include all of this
# notice on any copy.
###
# Utility module for creating Safe interpreters with Tk
#
# All procedures and global variables use the prefix "interp"

proc interp_create_safe_tk {name} {
    global interp_tk

    # Create a safe interpreter, w/o Tk
    set slave [eval interp create -safe $name]

    # Create a top level for this interpreter

    set top [toplevel .$slave -width 200 -height 200]
    wm title $top "Applet $name"
    bind interp$top <Destroy> "if {\"%W\" == \"$top\"} {interp_delete_slave $slave}"
    bindtags $top "interp$top [bindtags $top]"

    # Make sure any previous interp_tk array elements are removed
    foreach e [array get interp_tk $slave*] {catch {unset interp_tk($e)}}

    # Add in aliases for Tk commands

    # The initial window
    $slave alias . interp_tk_method $slave .

    interp_add_toplevel $slave $top

    # Widget creation commands
    foreach cmd {
	button 
	canvas checkbutton 
	entry 
	frame 
	image 
	label listbox 
	menubutton message 
	radiobutton 
	scale scrollbar 
	text } {
	$slave alias $cmd interp_tk_create $slave $cmd
    }
    # Exceptions: toplevels need some further restriction
    $slave alias toplevel interp_tk_create_toplevel $slave
    # menu items take a -command argument which must be trapped
    $slave alias menu interp_tk_create_menu $slave
    # these have different formal parameters
    $slave alias tk_dialog interp_tk_dialog $slave
    $slave alias tk_popup interp_tk_popup $slave

    # These commands will need restriction in some way
    foreach cmd {
	after 
	bell bind bindtags 
	destroy 
	focus tk_focusNext tk_focusPrev tk_focusFollowsMouse 
	grab 
	option 
	pack place 
	tkerror 
	update 
	winfo wm} {
	$slave alias $cmd interp_tk_$cmd $slave
    }
    # These two share the same implementation
    $slave alias lower interp_tk_order lower $slave
    $slave alias raise interp_tk_order raise $slave

    # Notable exceptions.  These commands would allow the system to be
    # compromised
    # clipboard fileevent palette selection send tk tkwait 
    # Note that it may be possible to implement the tkwait command
    # clipboard could be implemented in a restricted fashion

    # These commands are useless since they require access to variables
    # inside the slave interpreter
    # tk_optionMenu 

    return $slave
}

# Convenience routine for sending arguments for evaluation in the slave
# Just to get out of "quoting hell".  Courtesy Jacob Levy, Sun Labs.
proc interp_collect {args} {list $args}

# Add a toplevel widget to the list of exposed heirarchies

proc interp_add_toplevel {slave top} {
    global interp_tk
    lappend interp_tk($slave,toplevels) $top
}

proc interp_delete {slave} {
    destroy .$slave
}

proc interp_delete_slave {slave} {
    global interp_tk

    if {[array get interp_tk $slave,destroy] != {}} {
	return; # Already been here
    }

    foreach e [array get interp_tk $slave*] {catch {unset interp_tk($e)}}
    set interp_tk($slave,destroy) 1
    after idle interp_really_delete $slave
}

proc interp_really_delete {slave} {
    foreach a [interp aliases $slave] {$slave alias $a {}}
    interp delete $slave
}

###
### Map slave widget paths to master widget paths, and vice versa
###

# By convention, slave pathnames starting with "master.<top>" are
# mapped to the pathname ".<top>".

proc interp_get_win {slave win} {
    global interp_tk

    if {[regexp {^master(\..+)} $win all win]} {
	# Find matching toplevel
	if {[regsub "([join [lrange $interp_tk($slave,toplevels) 1 end] |])" $win {} dummy]} {
	    return $win
	} else {error "bad window path name for \"$win\""}
    } else {
	if {[string index $win 0] != "."} {error "bad window path name \"$win\""}
	if {$win == "."} {
	    return [lindex $interp_tk($slave,toplevels) 0]
	} else {return [lindex $interp_tk($slave,toplevels) 0]$win}
    }
}

# Ditto for a line full of references

proc interp_fix_pathnames {top cmd} {
    # Take care of pathnames
    regsub -all [format {(^|[ 	[\{"]+)master%s(\.[^ 	]+)?} $top] $cmd "\\1$top\\2 " cmd
    return $cmd
}

proc interp_fix_all_pathnames {slave cmd} {
    global interp_tk

    # Do it for the applet's heirarchy
    set top [lindex $interp_tk($slave,toplevels) 0]
    regsub -all {(^|[ 	[\{"]+)(\.[^ 	]+)} $cmd "\\1$top\\2 " cmd
    # Take care of "."
    regsub -all {(^|[ 	[\{"]+)\.([ 	]+|$)} $cmd "\\1$top " cmd

    foreach top $interp_tk($slave,toplevels) {
	set cmd [interp_fix_pathnames $top $cmd]
    }
    return [string trim $cmd]
}

proc interp_unfix_all_pathnames {slave cmd} {
    global interp_tk

    # Do it for the applet's heirarchy
    regsub -all "(^|\[ 	\]+)[lindex $interp_tk($slave,toplevels) 0](\\.\[^ 	\]*)" \
	$cmd { \2 } cmd
    # Take care of "."
    regsub -all "(^|\[  \]+)[lindex $interp_tk($slave,toplevels) 0](\[ 	\]+|\$)" \
	$cmd { . } cmd

    foreach top [lrange $interp_tk($slave,toplevels) 1 end] {
	regsub -all [format {(^|[ 	]+)%s(\.[^ 	]*)?} $top] \
	$cmd " master$top\\2 " cmd
    }
    return [string trim $cmd]
}

###
### Creation, method and configuration alias targets
###

proc interp_tk_create {slave cmd win args} {
    # Map into master pathname & also check validity of operation
    set path [interp_get_win $slave $win]
    # Create the widget
    $cmd $path
    if {[llength $args] > 0} {
	interp_tk_configure $slave $path $args
    }
    $slave alias $win interp_tk_method $slave $win
    return $win
}

# Same as above, only trap to menu-specific procedure

proc interp_tk_create_menu {slave win args} {
    # Map into master pathname & also check validity of operation
    set path [interp_get_win $slave $win]
    # Create the widget
    menu $path
    if {[llength $args] > 0} {
	interp_tk_configure $slave $path $args
    }
    $slave alias $win interp_tk_method_menu $slave $win
    return $win
}

proc interp_tk_create_toplevel {slave win args} {
    # Map into master pathname & also check validity of operation
    set path [interp_get_win $slave $win]
    # Create the widget
    toplevel $path
    if {[llength $args] > 0} {
	interp_tk_configure $slave $path $args
    }
    $slave alias $win interp_tk_method $slave $win
    wm title $path "Applet: $slave; [wm title $path]"
    return $win
}

proc interp_tk_dialog {slave win title text bitmap default string args} {
    interp_evaluate_list tk_dialog \
	"[interp_get_win $slave $win] {$title} {$text} {$bitmap} {$default} {$string} $args"
}

proc interp_tk_popup {slave menu x y {entry {}}} {
    tk_popup [interp_get_win $slave $menu] $x $y $entry
}

proc interp_tk_method {slave win args} {
    # Map into master pathname & also check validity of operation
    set path [interp_get_win $slave $win]
    if {[lindex $args 0] == "configure"} {
	return [interp_tk_configure $slave $path [lrange $args 1 end]]
    } else {
	interp_evaluate_list $path [interp_fix_all_pathnames $slave $args]
    }
}

# For menu items we need to trap the -command option
# and disallow the -variable option

proc interp_tk_method_menu {slave win args} {
    # Map into master pathname & also check validity of operation
    set path [interp_get_win $slave $win]
    set method [lindex $args 0]
    set command {}
    set entryidx {}
    switch $method {
	add {
	    if {[lsearch $args "-var*"] != -1} {
		error "cannot access variables inside a slave interpreter"
	    } elseif {[set cidx [lsearch $args "-com*"]] != -1} {
		set command [lindex $args [expr $cidx + 1]]
		set args [lreplace $args $cidx [expr $cidx + 1]]
	    }
	}
	insert {
	    if {[lsearch $args "-var*"] != -1} {
		error "cannot access variables inside a slave interpreter"
	    } elseif {[set cidx [lsearch $args "-com*"]] != -1} {
		set command [lindex $args [expr $cidx + 1]]
		set args [lreplace $args $cidx [expr $cidx + 1]]
		set entryidx [lindex $args 1]
	    }
	}
	entryconfigure {
	    if {[lsearch [lrange $args 2 end] "-var*"] != -1} {
		error "cannot access variables inside a slave interpreter"
	    } elseif {[set cidx [lsearch $args "-com*"]] != -1} {
		set command [lindex $args [expr $cidx + 1]]
		set args [lreplace $args $cidx [expr $cidx + 1]]
		set entryidx [lindex $args 1]
	    }
	}
    }

    set result {}
    # similar to interp_tk_method
    if {$args != ""} {
	if {[lindex $args 0] == "configure"} {
	    return [interp_tk_configure $slave $path [lrange $args 1 end]]
	} else {
	    set result [interp_evaluate_list $path [interp_fix_all_pathnames $slave $args]]
	}
	if {$entryidx == {}} {
	    set entryidx [$path index end]	;# New menu items are created on the end
	}
    }

    # correctly configure -command option
    if {$command != {}} {
	$path entryconfigure $entryidx -command "$slave eval \[interp_unfix_all_pathnames $slave \{$command\}\]"
    }
    return $result
}

# We need to be careful here and intercept any options that specify
# scripts.  These scripts should be evaluated in the slave interpreter.

# Also, -variable and -textvariable variables should be made to refer to 
# a variable in the slave interpreter.  This is not currently possible.

# We need to prevent the Applet from covering the entire screen.  This is
# difficult to do in general by examining the -width & -height options
# given to each widget.  Rather, we will allow widgets to size themelves
# but then check the final width & height of the toplevel once we are done.

proc interp_tk_configure {slave win cfg} {
    if {[llength $cfg] == 0} {
	return [$win configure]
    } elseif {[llength $cfg] == 1} {
	return [$win configure $cfg]
    }

    if {[expr [llength $cfg] % 2] == 1} {
	error "value for \"[lindex $cfg end]\" missing"
    }
    while {$cfg != {}} {
	set opt [lindex $cfg 0]
	set value [interp_fix_all_pathnames $slave [string trim [lindex $cfg 1]]]
	set cfg [lrange $cfg 2 end]
	if {[string first $opt "-xscrollcommand"] == 0 ||
	    [string first $opt "-yscrollcommand"] == 0 ||
	    [string first $opt "-postcommand"] == 0 ||
	    [string first $opt "-command"] == 0} {
	    $win configure $opt "$slave eval \[interp_unfix_all_pathnames $slave \{$value\}\]"
	} elseif {[string first $opt "-variable"] == 0 ||
	    ($opt != "-text" && [string first $opt "-textvariable"] == 0)} {
	    error "cannot access variables inside a slave interpreter"
	} else {
	    # Looks harmless enough
	    $win configure $opt $value
	}
    }

    # Check that the screen hasn't been covered (include a safety margin)
    set top [winfo toplevel $win]
    if {[winfo reqheight $top] > [expr [winfo screenheight .] - 10]} {
	$top configure -height [expr [winfo screenheight .] - 10]
    }
    if {[winfo reqwidth $top] > [expr [winfo screenwidth .] - 10]} {
	$top configure -width [expr [winfo screenwidth .] - 10]
    }

    return {}
}

###
### Tk commands
###

# Make sure that any scripts are evaluated in the slave interpreter

proc interp_tk_after {slave args} {
    if {$args == {}} {after} ;# Cause the error
    switch [lindex $args 0] {
	cancel {
	    if {[llength $args] == 2} {
		# Is arg #3 an id or a script?  Do gross hack...
		if {[regexp {^after#} [lindex $args 1]]} {
		    after cancel [lindex $args 1]
		    break
		}
	    }
	    set args "cancel $slave eval [lrange $args 1 end]"
	    interp_evaluate_list after $args
	}
	idle {
	    return [after idle catch \{$slave eval [lrange $args 1 end]\}]
	}
	default {
	    if {[llength $args] == 1} {error "safe interpreters may not sleep"}
	    return [after [lindex $args 0] catch \{$slave eval [lrange $args 1 end]\}]
	}
    }
    return {}
}

# Restrict to ringing the bell of the slave's toplevel window
# interp_bell_quota specifies how many times the bell may be rung at a time
# interp_bell_freq specifies how often, in milliseconds, the bell may be rung

# By default the bell may only be rung three times each second.
array set interp_tk {
	interp_bell_quota	3
	interp_bell_freq	1000
}
proc interp_tk_bell {slave args} {
    global interp_tk

    # If the handler routine hasn't been scheduled, 
    # then ring the bell immediately and schedule handler
    if {[array get interp_tk $slave,bell_handler] == {}} {
	# should check -displayof option...
	bell -displayof [lindex $interp_tk($slave,toplevels) 0]
	set interp_tk($slave,bell_handler) [after $interp_tk(interp_bell_freq) interp_bell_handler $slave]
    } else {
	# Add a bell request to the request queue
	if {[array get interp_tk $slave,bell_cnt] == {}} {
	    set interp_tk($slave,bell_cnt) 1
	} else {incr interp_tk($slave,bell_cnt) 1}
    }
}

proc interp_bell_handler {slave} {
    global interp_tk

    if {[array get interp_tk $slave,bell_cnt] != {}} {
	for {set i 0} {$i < $interp_tk($slave,bell_cnt) && $i < $interp_tk(interp_bell_quota)} {incr i} {
	    bell -displayof [lindex $interp_tk($slave,toplevels) 0]
	}
    }
    catch {unset interp_tk($slave,bell_handler)}
    catch {unset interp_tk($slave,bell_cnt)}
}

# The thing to watch here, of course, is to make sure that scripts get 
# evaluated in the slave interpreter.
# Also, bindings added to windows not created by the applet must be
# removed when the slave interpreter is deleted (this doesn't currently happen)

proc interp_tk_bind {slave tag args} {
    if {[regexp {^master\.} $tag] || [string index $tag 0] == "."} {
	set slavetag [interp_get_win $slave $tag]
    } else {set slavetag $tag}

    if {[llength $args] == 0} {
	return [bind $slavetag]
    } elseif {[llength $args] == 1} {
	return [bind $slavetag $args]
    } elseif {[llength $args] > 2} {
	# Cause an error message to be sent back
	return {[bind $slavetag <1> too many arguments]}
    } elseif {[string index [lindex $args 1] 0] == "+"} {
	bind $slavetag [lindex $args 0] \
		"+$slave eval \[interp_unfix_all_pathnames $slave \{[string range [lindex $args 1] 1 end]\}\]"
    } else {
	bind $slavetag [lindex $args 0] \
		"$slave eval \[interp_unfix_all_pathnames $slave \{[lindex $args 1]\}\]"
    }
    return {}
}

# We need to store the bindtags added to windows not created by the
# applet and remove these when the applet is deleted

proc interp_tk_bindtags {slave args} {
    if {[llength $args] == 1} {
	return [interp_unfix_all_pathnames $slave \
	    [bindtags [interp_get_win $slave [lindex $args 0]]]]
    } else {
	return [interp_evaluate_list bindtags [interp_fix_all_pathnames $slave $args]]
    }
}

proc interp_tk_destroy {slave args} {
    foreach win $args {
	set win [interp_get_win $slave $win]
	destroy $win
    }
}

# Disallow the "force" method - too dangerous

proc interp_tk_focus {slave args} {
    if {$args == {}} {
	return [focus displayof $top]
    } elseif {[llength $args] == 1} {
	return [focus [interp_get_win $slave $args]]
    } elseif {[llength $args] > 2} {
	# Cause the error message to be returned
	return [focus [lindex $args 0] [lindex $args 1] [lindex $args 2]]
    } elseif {[lindex $args 0] != "force"} {
	return [focus [lindex $args 0] [interp_get_win $slave [lindex $args 1]]]
    } else {
	error "method \"force\" is not permitted for safe interpreters"
    }
}

proc interp_tk_tk_focusNext {slave win} {
    return tk_focusNext [interp_get_win $slave $win]
}

proc interp_tk_tk_focusPrev {slave win} {
    return tk_focusPrev [interp_get_win $slave $win]
}

# Don't let the slave interpreter affect other windows

proc interp_tk_tk_focusFollowsMouse {slave} {
    error "safe interpreters may not change the focus model"
}

# Only allow local grabs
# Window pathnames returned should be normalized for the slave interpreter's
# context

proc interp_tk_grab {slave args} {
    if {$args == {}} {grab}; # Cause the error
    switch [lindex $args 0] {
	-global {error "global grabs not permitted for safe interpreters"}
	current {
	    if {[llength $args] == 1} {
		return [grab current]
	    } elseif {[lindex $args] > 2} {
		grab current [lindex $args 1] [lindex $args 2]; # Cause the error
	    } else {
		return [grab current [interp_get_win $slave [lindex $args 1]]]
	    }
	}
	release {
	    if {[llength $args] > 2} {
		grab release [lindex $args 1] [lindex $args 2]; # Cause the error
	    }
	    grab release [interp_get_win $slave [lindex $args 1]]
	}
	set {
	    if {[llength $args] == 1} {grab set}; # Cause the error
	    if {[lindex $args 1] == "-global"} {error "global grabs not permitted for safe interpreters"}
	    if {[[length $args] > 2} {
		grab set [lindex $args 1] [lindex $args 2]; # Cause the error
	    }
	    grab set [interp_get_win $slave [lindex $args 1]]
	}
	status {
	    if {[llength $args] == 1} {grab status}; # Cause the error
	    if {[[length $args] > 2} {
		grab status [lindex $args 1] [lindex $args 2]; # Cause the error
	    }
	    return [grab status [interp_get_win $slave [lindex $args 1]]]
	}
	default {grab [lindex $args 0] [lrange $args 1 end]}; # Cause the error
    }
    return {}
}

proc interp_tk_order {dir slave top win args} {
    set win [interp_get_win $slave $win]
    if {$args == {}} {
	$dir $win
    } elseif {[llength $args > 1]} {
	$dir $win [lindex $args 1] [lindex $args 2]; # Cause the error
    } else {
	$dir $win [interp_get_win $slave $args]
    }
}

# Only the get method is permitted

proc interp_tk_option {slave method args} {
    if {$method != "get"} {error "safe interpreters may not change options"}
    if {[llength $args] != 3} {error "wrong # arguments"}
    return [option get [interp_get_win $slave [lindex $args 0]] [lindex $args 1] [lindex $args 2]]
}

proc interp_tk_pack {slave args} {
    return [interp_evaluate_list pack [interp_fix_all_pathnames $slave $args]]
}

proc interp_tk_place {slave args} {
    return [interp_evaluate_list place [interp_fix_all_pathnames $slave $args]]
}

proc interp_tk_update {slave args} {
    if {$args == {}} {update} else {update $args}
}

# Disallow interps method.  Munge parent, pathname, toplevel return value

proc interp_tk_winfo {slave args} {
    if {$args == {}} {winfo}; # Cause the error
    set args [interp_fix_all_pathnames $slave $args]

    switch [lindex $args 0] {
	interps {error "\"interps\" method disallowed for safe interpreters"}
	parent -
	pathname -
	toplevel {
	    return [interp_unfix_all_pathnames $slave \
		[interp_evaluate_list winfo $args]]
	}
	default {
	    return [interp_evaluate_list winfo $args]
	}
    }
}

# Play it safe with the potentially dangerous methods for the moment
# Simply retrieving values is probably safe though.

proc interp_tk_wm {slave args} {
    if {$args == {}} {wm}; # Cause the error
    set args [interp_fix_all_pathnames $slave $args]

    switch [lindex $args 0] {
	client -
	colormapwindows -
	command -
	focusmodel -
	protocol -
	transient {error "method \"[lindex $args 0]\" disallowed for safe interpreters"}
	title {
	    if {[llength $args] == 3} {
		wm title [lindex $args 1] "Applet: $slave; [lindex $args 2]"
	    } else {
		return [wm title [lindex $args 1]]
	    }
	}
	default {
	    return [interp_evaluate_list wm $args]
	}
    }
}

# There must be a better way!

proc interp_evaluate_list {cmd args} {
    set args [lindex $args 0]
    switch [llength $args] {
	0	{return [$cmd]}
	1	{return [$cmd $args]}
	2	{return [$cmd [lindex $args 0] [lindex $args 1]]}
	3	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2]]}
	4	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3]]}
	5	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4]]}
	6	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5]]}
	7	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6]]}
	8	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7]]}
	9	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8]]}
	10	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9]]}
	11	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10]]}
	12	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11]]}
	13	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12]]}
	14	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13]]}
	15	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14]]}
	16	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15]]}
	17	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16]]}
	18	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17]]}
	19	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18]]}
	20	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19]]}
	21	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20]]}
	22	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21]]}
	23	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22]]}
	24	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23]]}
	25	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24]]}
	26	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25]]}
	27	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26]]}
	28	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27]]}
	29	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28]]}
	30	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28] [lindex $args 29]]}
	31	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28] [lindex $args 29] [lindex $args 30]]}
	31	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28] [lindex $args 29] [lindex $args 30] [lindex $args 31]]}
	33	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28] [lindex $args 29] [lindex $args 30] [lindex $args 31] [lindex $args 32]]}
	34	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28] [lindex $args 29] [lindex $args 30] [lindex $args 31] [lindex $args 32] [lindex $args 33]]}
	35	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28] [lindex $args 29] [lindex $args 30] [lindex $args 31] [lindex $args 32] [lindex $args 33] [lindex $args 34]]}
	36	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28] [lindex $args 29] [lindex $args 30] [lindex $args 31] [lindex $args 32] [lindex $args 33] [lindex $args 34] [lindex $args 35]]}
	37	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28] [lindex $args 29] [lindex $args 30] [lindex $args 31] [lindex $args 32] [lindex $args 33] [lindex $args 34] [lindex $args 35] [lindex $args 36]]}
	38	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28] [lindex $args 29] [lindex $args 30] [lindex $args 31] [lindex $args 32] [lindex $args 33] [lindex $args 34] [lindex $args 35] [lindex $args 36] [lindex $args 37]]}
	39	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28] [lindex $args 29] [lindex $args 30] [lindex $args 31] [lindex $args 32] [lindex $args 33] [lindex $args 34] [lindex $args 35] [lindex $args 36] [lindex $args 37] [lindex $args 38]]}
	40	{return [$cmd [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3] [lindex $args 4] [lindex $args 5] [lindex $args 6] [lindex $args 7] [lindex $args 8] [lindex $args 9] [lindex $args 10] [lindex $args 11] [lindex $args 12] [lindex $args 13] [lindex $args 14] [lindex $args 15] [lindex $args 16] [lindex $args 17] [lindex $args 18] [lindex $args 19] [lindex $args 20] [lindex $args 21] [lindex $args 22] [lindex $args 23] [lindex $args 24] [lindex $args 25] [lindex $args 26] [lindex $args 27] [lindex $args 28] [lindex $args 29] [lindex $args 30] [lindex $args 31] [lindex $args 32] [lindex $args 33] [lindex $args 34] [lindex $args 35] [lindex $args 36] [lindex $args 37] [lindex $args 38] [lindex $args 39]]}
	default {error "sorry, unable to process"}
    }
}
