#!/usr/local/bin/tclsh
set rcsId {$Id: browse.tcl,v 1.21 1995/12/07 16:18:35 jfontain Exp $}

proc echo {args} {
    foreach argument $args {
        puts -nonewline "$argument "
    }
    puts ""
}

source stooop.tcl
source class.tcl

proc ignore {args} {}

set memberPattern "\[^\]\[)(}{\$ \t\n\]+"

proc detectStaticDataMembers {args} {
    global arrayMember memberPattern

    set body [join $args]
    set pattern "(\[^ \t\n\]+)\\(($memberPattern)\\)"
    set start 0
    while {[regexp -indices $pattern [string range $body $start end] indices]>0} {
        regexp $pattern [string range $body $start end] dummy array member
        if {![info exists arrayMember($array)]||([lsearch -exact $arrayMember($array) $member]<0)} {
            lappend arrayMember($array) $member
        }
        incr start [expr [lindex $indices 1]+1]
    }
}

proc detectArrayDataMembers {className body} {
    global classId memberPattern

    # look for class${this}member, dollar sign is already escaped
    set pattern "$className\\\\\\\${this}($memberPattern)"
    set start 0
    while {[regexp -indices $pattern [string range $body $start end] indices]>0} {
        regexp $pattern [string range $body $start end] dummy member
        # mark array name with array like syntax (parentheses)
        class::addData $classId($className) $member\(\)
        incr start [expr [lindex $indices 1]+1]
    }
}

proc detectDataMembers {className body} {
    global classId memberPattern

    # look for class($this,member), dollar sign is already escaped
    set pattern $className\\(\\\\\\\$this,($memberPattern)\\)
    set start 0
    while {[regexp -indices $pattern [string range $body $start end] indices]>0} {
        regexp $pattern [string range $body $start end] dummy member
        class::addData $classId($className) $member
        incr start [expr [lindex $indices 1]+1]
    }
}

proc procedureDefinition {virtual name arguments args} {
    global classId

    if {![regexp {^(.+)::(.+)$} $name dummy className procedure]} {
        detectStaticDataMembers [lindex $args 0]
        return
    }
    if {[string compare $className $procedure]==0} {
        # constructor, gather base classes
        set bases {}
        set index 0
        if {[set number [expr [llength $args]-1]]>1} {
            for {} {$index<$number} {incr index 2} {
                lappend bases [lindex $args $index]
            }
        }
        set body [lindex $args $index]
        set classId($className) [new class $className $bases]
        foreach base $bases {
            class::addDerived $classId($base) $className
        }
        class::addProcedure $classId($className) $procedure
    } else {
        # not a constructor
        set body [lindex $args 0]
        if {$virtual} {
            if {[llength $args]==0} {
                # no body means pure virtual
                class::addProcedure $classId($className) $procedure pure
            } else {
                class::addProcedure $classId($className) $procedure virtual
            }
        } else {
            if {[string compare [lindex $arguments 0] this]==0} {
                class::addProcedure $classId($className) $procedure
            } else {
                class::addProcedure $classId($className) $procedure static
            }
        }
    }
    detectDataMembers $className $body
    detectArrayDataMembers $className $body
    detectStaticDataMembers $body
}

proc virtualProcedureDefinition {args} {
    eval procedureDefinition 1 [lrange $args 1 end]
}

proc processStaticData {} {
    global classId arrayMember

    foreach name [array names arrayMember] {
        # scan global array names
        if {![catch {set id $classId($name)}]} {
            # array is a class
            foreach member $arrayMember($name) {
                # add static members
                class::addData $id $member 1
            }
        }
    }
}

proc processCode {code} {
    set interpreter [interp create]
    # discard stooop renamed proc command
    $interpreter eval proc ::proc args \{\}
    $interpreter alias virtual virtualProcedureDefinition
    $interpreter alias proc procedureDefinition 0
    $interpreter alias unknown detectStaticDataMembers
    foreach command [$interpreter eval info commands] {
        if {![regexp virtual|proc|rename|unknown $command]} {
            $interpreter eval rename $command \{\}
        }
    }
    $interpreter eval rename rename \{\}
    $interpreter eval $code

    processStaticData
}

proc sourceCode {fileName} {
    global env

    upvar sourced sourced
    if {![info exists sourced]} {
        set sourced {}
    }
    if {[lsearch -exact $sourced $fileName]>=0} {
        # do not process the same file more than once
        return {}
    }
    lappend sourced $fileName

    set code ""
    if {[string length $fileName]==0} {
        set file stdin
    } else {
        set file [open $fileName]
    }
    while {[gets $file line]>=0} {
        if {[scan $line " source %s" name]==1} {
            append code [sourceCode [subst $name]]
        }
        regsub -all {\$} $line {\$} line
        append code $line\n
    }
    if {[string length $fileName]>0} {
        close $file
    }
    return $code
}

proc displayDerived {id} {
    global class output classId

    if {[llength $class($id,derived)]==0} {
        return
    }
    puts $output <UL>
    foreach name [lsort $class($id,derived)] {
        puts $output <LI>$name
        displayDerived $classId($name)
    }
    puts $output </UL>
}

proc generateDocument {} {
    global output classId class

    puts $output <P><B>hierarchy</B></P>
    puts $output <UL>
    foreach className [lsort [array names classId]] {
        set id $classId($className)
        if {[llength $class($id,bases)]>0} {
            continue
        }
        puts $output <LI>$className
        displayDerived $id
    }
    puts $output </UL>

    puts $output <P><B>classes</B></P>
    puts $output <UL>
    foreach className [lsort [array names classId]] {
        set id $classId($className)
        puts $output <LI>$className
        puts $output <UL>
        foreach procedure [lsort $class($id,procedures)] {
            set display "$procedure {}"
            switch $class($id,type,$procedure) {
                virtual {
                    set begin <I>
                    set end </I>
                }
                pure {
                    set begin <I>
                    set end </I>
                    set display $procedure
                }
                static {
                    set begin <U>
                    set end </U>
                }
                default {
                    set begin ""
                    set end ""
                }
            }
            puts $output <LI>$begin$display$end
        }
        foreach data [lsort $class($id,data)] {
            if {[info exists class($id,static,$data)]} {
                puts $output <LI><U>$data</U>
            } else {
                puts $output <LI>$data
            }
        }
        puts $output </UL>
    }
    puts $output </UL>
}

if {[string compare [lindex $argv 0] -o]==0} {
    set output [open [lindex $argv 1] w]
    set index 2
} else {
    set output stdout
    set index 0
}

processCode [sourceCode [lindex $argv $index]]
generateDocument
