
// Objectify -- turn C++ classes into Tcl objects.
//
// Wayne Christopher, faustus@cs.berkeley.edu
// Version 0.1, May 1, 1993
//
// Copyright (c) 1993 Wayne Christopher.  Permission is granted
// to copy, modify and distribute, provided this notice is
// preserved.  No warranty is provided or implied, etc, etc.

#include "tcl.h"
#include <assert.h>

static Tcl_HashTable* nametable = NULL;
static Tcl_HashTable* ptrtable = NULL;
static int namecount = 0;

static void
init_tables()
{
    nametable = new Tcl_HashTable;
    Tcl_InitHashTable(nametable, TCL_STRING_KEYS);
    
    ptrtable = new Tcl_HashTable;
    Tcl_InitHashTable(ptrtable, TCL_ONE_WORD_KEYS);
}

void
Objectify_NewName(char* obtype, char* ret, void* ob)
{
    if (!nametable) init_tables();
    
    namecount++;
    char* key = new char[strlen(obtype) + 32];
    Tcl_HashEntry* ent;
    int newp;
    while (1) {
	sprintf(key, "%s__%d", obtype, namecount);
	ent = Tcl_CreateHashEntry(nametable, key, &newp);
	if (newp) break;
	namecount++;
    }
    Tcl_SetHashValue(ent, (ClientData) ob);
    
    ent = Tcl_CreateHashEntry(ptrtable, (char *) ob, &newp);
    if (!newp) {
	fprintf(stderr, "Internal error: object \"%s\" in hash table twice\n",
		ret);
	return;
    }
    char* nbuf = new char[strlen(key) + 1];
    strcpy(nbuf, key);
    Tcl_SetHashValue(ent, (ClientData) nbuf);
    
    strcpy(ret, key);
    delete key;
}

void
Objectify_SetName(char* name, void* ob)
{
    if (!nametable) init_tables();
    
    int newp;
    Tcl_HashEntry* ent = Tcl_CreateHashEntry(nametable, name, &newp);
    Tcl_SetHashValue(ent, (ClientData) ob);
    
    ent = Tcl_CreateHashEntry(ptrtable, (char *) ob, &newp);
    char* nbuf = new char[strlen(name) + 1];
    strcpy(nbuf, name);
    if (!newp) {
	char* oval = (char *) Tcl_GetHashValue(ent);
	if (oval)
	    delete oval;
	else
	    fprintf(stderr, "Internal error: no name for object 0x%x\n",
		    (int) ob);
    }
    Tcl_SetHashValue(ent, nbuf);
}

void
Objectify_DeleteName(char* name)
{
    assert(nametable);
    
    Tcl_HashEntry* ent = Tcl_FindHashEntry(nametable, name);
    void* ob = (void *) Tcl_GetHashValue(ent);
    assert(ent);
    Tcl_DeleteHashEntry(ent);
    
    ent = Tcl_FindHashEntry(ptrtable, (char *) ob);
    assert(ent);
    char* n = (char *) Tcl_GetHashValue(ent);
    if (!n)
	fprintf(stderr, "Internal error: no name for object 0x%x\n", (int) ob);
    else
	delete n;
    Tcl_DeleteHashEntry(ent);
}

void*
Objectify_LookupName(char* name)
{
    if (!nametable) return (NULL);
    
    Tcl_HashEntry* ent = Tcl_FindHashEntry(nametable, name);
    if (ent)
	return ((void *) Tcl_GetHashValue(ent));
    else
	return (NULL);
}

char*
Objectify_LookupPtr(void* ptr)
{
    if (!ptrtable) return (NULL);
    
    Tcl_HashEntry* ent = Tcl_FindHashEntry(ptrtable, (char *) ptr);
    if (ent)
	return ((char *) Tcl_GetHashValue(ent));
    else
	return (NULL);
}
