/************************************************************************
 *  Copyright (c) 1993 by Charles A. Measday                            *
 *                                                                      *
 *  Permission to use, copy, modify, and distribute this software       *
 *  and its documentation for any purpose and without fee is hereby     *
 *  granted, provided that the above copyright notice appear in all     *
 *  copies.  The author makes no representations about the suitability  *
 *  of this software for any purpose.  It is provided "as is" without   *
 *  express or implied warranty.                                        *
 ************************************************************************/

/*
@(#)  FILE: nix_util.c  RELEASE: 1.1  DATE: 7/7/94, 18:40:46
*/
/*******************************************************************************

File:

    nix_util.c

    Network I/O Handler (Not Including X).


Author:    Alex Measday


Purpose:

    The functions in this file implement an X Toolkit-style I/O event
    dispatcher.  Applications that make use of the NIX dispatcher are
    generally structured as follows:

          Perform any application-specific initialization activities.
          Register event "callbacks" with the NIX event dispatcher.
          DO forever
              Wait for the next event.
              CALL the callback bound to the event.
          ENDDO

    The event processing loop is encapsulated in a NIX function, NXMAINLOOP().
    Other NIX functions are available to:

      - Register an I/O source with the dispatcher.  When an I/O event is
        detected at the source, the dispatcher automatically invokes an
        application-defined "callback" function to respond to the event
        (e.g., to read input from a network connection).

      - Register a timeout timer with the dispatcher.  When the
        specified time interval has elapsed, the dispatcher
        automatically invokes an application-defined callback
        function to react to the timeout.

      - Register a work procedure (an application-defined function) with
        the dispatcher.  When no I/O is active and no timers are ready
        to fire, the dispatcher will execute the next work procedure on
        its queue.  Work procedures are used to perform tasks in the
        "background".

    The NIX_UTIL functions are patterned after the following X Toolkit
    functions:

        NxAddInput ()		-	XtAppAddInput ()
        NxAddTimeOut ()		-	XtAppAddTimeOut ()
        NxAddWorkProc ()	-	XtAppAddWorkProc ()
        NxMainLoop ()		-	XtAppMainLoop ()
        NxRemoveInput ()	-	XtRemoveInput ()
        NxRemoveTimeOut ()	-	XtRemoveTimeOut ()
        NxRemoveWorkProc ()	-	XtRemoveWorkProc ()

    I/O sources are registered with the NIX dispatcher by NXADDINPUT(),
    timeout timers by NXADDTIMEOUT(), and work procedures by NXADDWORKPROC().
    The dispatcher itself is contained in NXMAINLOOP().  Once called,
    NXMAINLOOP(), like XtAppMainLoop(), never returns; it loops forever,
    waiting for I/O and timer events and invoking the application-specified
    callback functions bound to the events.

    The following program (working) shows a simple application of the NIX
    functions to reading standard input and echoing it to standard output:

        #include  <stdio.h>		-- Standard I/O definitions.
        #include  "nix_util.h"		-- Network I/O Handler definitions.

        static  int  read_input (...)	-- Standard input callback.
        {
            char  buffer[128] ;
            if (gets (buffer) == NULL)  exit (0) ;
            printf ("read_input: %s\n", buffer) ;
        }

        main (...)			-- Main routine.
        {
            NxAddInput (NULL, fileno (stdin), NxInputReadMask,
                        read_input, NULL) ;
            NxMainLoop (NULL) ;
        }

    The NIX main loop is implemented using a UNIX SELECT(2) call and it
    supports read, write, and exceptional I/O events (much like the X
    Toolkit does).  The NIX functions have been tested on UNIX and VMS
    machines and, if you get the header files straight, they should work
    under operating systems like VxWorks that support SELECT(2).

    Under VMS, you can use the SELECT(2) implementation or, by calling
    NXMAINLOOPEF(), an event flag-based implementation a la DECWindows.
    The VMS UCX implementation of SELECT(2) only supports socket I/O and
    not arbitrary device I/O as in UNIX; event flags provide this missing
    capability.


Notes:

    The original NIX functions were patterned after the corresponding
    X Toolkit functions, but they didn't mimic the X Toolkit names and
    signatures like they do now.  With the thought of making it easier
    to reuse X- and non-X-based event-related code, I redid the NIX
    interface to closely parallel the X Toolkit interface.  Despite
    the new NX names, I still speak of the "NIX" utilities.

    The NIX utilities were developed under VMS and subsequently ported
    to SunOS and VxWorks.  In porting the code, I rearranged the header
    files somewhat, so I hope the VMS version still compiles.  I briefly
    had access to an HP/UX machine, on which I attempted to compile the
    utilities using HP's ANSI C compiler.  This attempt failed because
    of some problem with the "time.h" definitions; I didn't have time
    to dig into the problem further.  I believe I did compile and
    successfully test the code using HP's non-ANSI C compiler.

    Under VxWorks, the NIX utilities are reentrant (except for the
    global debug flag).  A pointer to the statically-allocated, default
    application context is automatically registered as a task-specific
    variable, so the operating system will save and restore it when
    a task context switch occurs.


Procedures:

    NXADDINPUT - registers an I/O source with the NIX dispatcher.
    NXADDTIMEOUT - registers a timeout callback with the NIX dispatcher.
    NXADDWORKPROC - registers a background work procedure with the NIX
        dispatcher.
    NXCREATECONTEXT - creates an application context.
    NXMAINLOOP - monitors and responds to I/O events and timeouts.
    NXMAINLOOPEF - monitors and responds to I/O events and timeouts using
        event flags (VMS only).
    NXREMOVEINPUT - removes the registration of an I/O source.
    NXREMOVETIMEOUT - removes the registration of a timeout callback.
    NXREMOVEWORKPROC - removes the registration of a work procedure.
    NXSETDEBUG - enables/disables debug output.

*******************************************************************************/


#include  <errno.h>			/* System error definitions. */
#include  <stdio.h>			/* Standard I/O definitions. */
#include  <stdlib.h>			/* Standard C library definitions. */
#if defined(VMS)
#    include  <libdtdef.h>		/* VMS date/time definitions. */
#    include  <socket.h>		/* Socket-related definitions. */
#    include  <stsdef.h>		/* VMS status code structure. */
#    include  "fd.h"			/* File descriptor set definitions. */
#elif defined(VXWORKS)
#    include  <selectLib.h>		/* SELECT(2) definitions. */
#    include  <taskVarLib.h>		/* Task variable definitions. */
#else
#    include  <sys/time.h>		/* System time definitions. */
#    include  <sys/types.h>		/* System type definitions. */
#endif
#include  "nix_util.h"			/* Network I/O Handler definitions. */
#include  "tv_util.h"			/* "timeval" manipulation functions. */


/*******************************************************************************
    I/O Source - contains information about an input/output source.
*******************************************************************************/

typedef  struct  _NxIOSource {
    int  source ;			/* File descriptor (UNIX). */
    NxInputMask  condition ;		/* Read, write, or exception (UNIX). */
    NxInputCallback  callback ;		/* Function to call when event occurs. */
    void  *client_data ;		/* Client-specified data passed to callback. */
    struct  _NxIOSource  *next ;
}  _NxIOSource ;


/*******************************************************************************
    Timeout Timer - contains information about a timeout timer.
*******************************************************************************/

typedef  struct  _NxTimer {
    double  interval ;			/* Timeout in seconds. */
    struct  timeval  expiration ;	/* Absolute time of expiration. */
    NxTimerCallback  callback ;		/* Function to call upon timeout. */
    void  *client_data ;		/* Client-specified data passed to callback. */
    struct  _NxTimer  *next ;
}  _NxTimer ;


/*******************************************************************************
    Background Task - contains information about a work procedure to be
        executed in the "background".
*******************************************************************************/

typedef  struct  _NxBackgroundTask {
    NxWorkProc  workproc ;		/* Function to call in "background". */
    void  *client_data ;		/* Client-specified data passed to callback. */
    struct  _NxBackgroundTask  *next ;
}  _NxBackgroundTask ;


/*******************************************************************************
    Application Context - gathers the information about a particular, NIX
        event loop.
*******************************************************************************/

typedef  struct  _NxAppContext {
    int  debug ;			/* Debug switch (1/0 = yes/no). */
    _NxIOSource  *IO_source_list ;	/* List of registered I/O sources. */
    _NxTimer  *timeout_list ;		/* List of registered timeout timers. */
    _NxBackgroundTask  *workproc_queue ;/* Queue of registered work procedures. */
}  _NxAppContext ;

					/* Default application context if
					   the caller doesn't specify one.
					   (Under VxWorks, this pointer is
					   automatically registered with
					   the operating system as a task
					   variable.) */
static  _NxAppContext  *default_context = NULL ;


int  nix_util_debug = 0 ;		/* Global debug switch (1/0 = yes/no). */

/*******************************************************************************

Procedure:

    NxAddInput ()

    Register an I/O Source with the NIX Dispatcher.


Purpose:

    Function NXADDINPUT registers an I/O source with the NIX dispatcher;
    when an I/O event is detected at the source, a caller-specified "callback"
    function is invoked to respond to the event (e.g., to read data from an
    input source).  NXADDINPUT() is modeled after the X Toolkit function,
    "XtAppAddInput()".


    Invocation:

        source_ID = NxAddInput (&context, source, type, callback, client_data) ;

    where:

        <context>	- I/O
            is the address of a variable containing the client's reference
            (an opaque pointer) to the application context.  If the address
            passed in for CONTEXT is NULL, then the default application
            context is used.  If the value of CONTEXT is NULL, then a new
            application context is created and an opaque pointer to that
            context is returned in CONTEXT.  If the value of CONTEXT is not
            NULL, then the previously-created application context is used.
        <source>	- I
            is an operating system-dependent object used to monitor the
            I/O source.  Under UNIX, this is simply the I/O source's file
            descriptor.
        <type>		- I
            is an operating system-dependent entity.  Under UNIX, this
            argument is a bit mask specifying the type of I/O event being
            monitored; the value of this mask is the bit-wise ORing of
            one or more of the following: NxInputReadMask (input events),
            NxInputWriteMask (output events), and NxInputExceptMask
            (exceptional events).  (An "exceptional" I/O event, as I
            only just found out, is the detection of out-of-band input
            on a network connection.)
        <callback>	- I
            is the function that is to be called when a monitored event is
            detected at the I/O source.  The callback function should be
            declared as follows:
                int  callback_function (NxAppContext context,
                                        NxInputId source_ID,
                                        int source,
                                        void *client_data) ;
            where "context" is the application context as used in or created
            by this call to NXADDINPUT(); "source_ID" is the value returned
            by this call to NXADDINPUT(); "source" and "client_data" are the
            arguments that were passed in to NXADDINPUT().  The return value
            of the callback function is ignored by the NIX dispatcher.
        <client_data>	- I
            is an arbitrary data value, cast as a (VOID *) pointer, that will
            be passed to the callback function if and when it is invoked.
        <source_ID>	- O
            returns an ID for the registered I/O source; NULL is returned in
            the event of an error.  This ID is used by NXREMOVEINPUT() to
            identify an I/O source being "unregistered".

*******************************************************************************/


NxInputId  NxAddInput (

#    if __STDC__ || defined(vaxc)
        NxAppContext  *context,
        int  source,
        NxInputMask  condition,
        NxInputCallback  callbackF,
        void  *client_data)
#    else
        context, source, condition, callbackF, client_data)

        NxAppContext  *context ;
        int  source ;
        NxInputMask  condition ;
        NxInputCallback  callbackF ;
        void  *client_data ;
#    endif

{    /* Local variables. */
    _NxAppContext  *app ;
    _NxIOSource  *ios ;




/* Use the desired application context. */

    if (context == NULL)  context = &default_context ;
    if ((*context == NULL) && NxCreateContext (context)) {
        vperror ("(NxAddInput) Error creating application context.\nNxCreateContext: ") ;
        return (NULL) ;
    }
    app = *context ;

/* Allocate an I/O source structure for the I/O source. */

    ios = (_NxIOSource *) malloc (sizeof (_NxIOSource)) ;
    if (ios == NULL) {
        vperror ("(NxAddInput) Error allocating I/O source structure.\nmalloc: ") ;
        return (NULL) ;
    }

    ios->source = source ;
    ios->condition = condition ;
    ios->callback = callbackF ;
    ios->client_data = client_data ;

/* Add the I/O source to the list of registered sources. */

    ios->next = app->IO_source_list ;
    app->IO_source_list = ios ;

    if (app->debug)
        printf ("(NxAddInput) I/O source %d registered.\n", source) ;

    return (ios) ;

}

/*******************************************************************************

Procedure:

    NxAddTimeOut ()

    Register a Timeout Timer with the NIX Dispatcher.


Purpose:

    Function NXADDTIMEOUT registers a timeout timer with the NIX dispatcher;
    when the timer expires, a caller-specified "callback" function is invoked
    to respond to the timeout.  At a minimum, the specified time interval will
    elapse before the callback function is called; there is no guarantee on how
    soon after the timer fires that the callback will be called.  The timer is
    automatically deleted after the timeout interval expires, so periodic timers
    must be explicitly re-registered.  NXADDTIMEOUT() is modeled after the X
    Toolkit function, "XtAppAddTimeOut()".


    Invocation:

        timer_ID = NxAddTimeOut (&context, interval, callback, client_data) ;

    where:

        <context>	- I/O
            is the address of a variable containing the client's reference
            (an opaque pointer) to the application context.  If the address
            passed in for CONTEXT is NULL, then the default application
            context is used.  If the value of CONTEXT is NULL, then a new
            application context is created and an opaque pointer to that
            context is returned in CONTEXT.  If the value of CONTEXT is not
            NULL, then the previously-created application context is used.
        <interval>	- I
            specifies the timeout interval in seconds.  (This is a real
            number, so fractions of a second can be specified.)
        <callback>	- I
            is the function that is to be called when the timeout interval
            expires.  The callback function should be declared as follows:
                int  callback_function (NxAppContext context,
                                        NxIntervalId timer_ID,
                                        void *client_data) ;
            where "context" is the application context as used in or created
            by this call to NXADDTIMEOUT(); "timer_ID" is the value returned
            by this call to NXADDTIMEOUT(); "client_data" is the argument
            that was passed in to NXADDTIMEOUT().  The return value of the
            callback function is ignored by the NIX dispatcher.
        <client_data>	- I
            is an arbitrary data value, cast as a (VOID *) pointer, that will
            be passed to the callback function when it is invoked.
        <timer_ID>	- O
            returns an ID for the registered timer; NULL is returned if
            an error occurs.  This handle is used by NXREMOVETIMEOUT() to
            identify a timer being "unregistered".

*******************************************************************************/


NxIntervalId  NxAddTimeOut (

#    if __STDC__ || defined(vaxc)
        NxAppContext  *context,
        double  interval,
        NxTimerCallback  callbackF,
        void  *client_data)
#    else
        context, interval, callbackF, client_data)

        NxAppContext  *context ;
        double  interval ;
        NxTimerCallback  callbackF ;
        void  *client_data ;
#    endif

{    /* Local variables. */
    _NxAppContext  *app ;
    _NxTimer  *next, *prev, *tot ;




/* Use the desired application context. */

    if (context == NULL)  context = &default_context ;
    if ((*context == NULL) && NxCreateContext (context)) {
        vperror ("(NxAddTimeOut) Error creating application context.\nNxCreateContext: ") ;
        return (NULL) ;
    }
    app = *context ;

/* Allocate a timeout structure for the timer. */

    tot = (_NxTimer *) malloc (sizeof (_NxTimer)) ;
    if (tot == NULL) {
        vperror ("(NxAddTimeOut) Error allocating timeout structure.\nmalloc: ") ;
        return (NULL) ;
    }
    tot->interval = interval ;
		/* Expiration time = current time-of-day + timeout interval. */
    tot->expiration = tv_add (tv_tod (), tv_create_f (interval)) ;
    tot->callback = callbackF ;
    tot->client_data = client_data ;

/* Add the timer to the list of registered timers.  The list is sorted
   by expiration time. */

    prev = NULL ;  next = app->timeout_list ;
    while (next != NULL) {
        if (tv_compare (tot->expiration, next->expiration) < 0)  break ;
        prev = next ;  next = next->next ;
    }

    if (prev == NULL) {			/* Beginning of list? */
        tot->next = app->timeout_list ;
        app->timeout_list = tot ;
    } else {				/* Middle or end of list. */
        tot->next = next ;
        prev->next = tot ;
    }

    if (app->debug) {
        char  *s = tv_display (tot->expiration) ;
        printf ("(NxAddTimeOut) %g-second timeout; expiration time: %s\n",
                interval, (s == NULL) ? "<nil>" : s) ;
#ifdef VXWORKS
        if (s != NULL)  free (s) ;
#endif
    }

    return (tot) ;

}

/*******************************************************************************

Procedure:

    NxAddWorkProc ()

    Register a Background Work Procedure with the NIX Dispatcher.


Purpose:

    Function NXADDWORKPROC registers a work procedure with the NIX
    dispatcher.  When no other event sources are active (e.g., I/O
    ready or timer expired), the NIX dispatcher will remove the next
    work procedure from the queue and call it; the work procedure
    effectively executes in "background" mode.  Work procedures are
    responsible for returning control to the NIX dispatcher in a
    timely fashion and for re-registering themselves if need be.
    NXADDWORKPROC() is modeled after the X Toolkit function,
    "XtAppAddWorkProc()".


    Invocation:

        workproc_ID = NxAddWorkProc (&context, callback, client_data) ;

    where:

        <context>	- I/O
            is the address of a variable containing the client's reference
            (an opaque pointer) to the application context.  If the address
            passed in for CONTEXT is NULL, then the default application
            context is used.  If the value of CONTEXT is NULL, then a new
            application context is created and an opaque pointer to that
            context is returned in CONTEXT.  If the value of CONTEXT is not
            NULL, then the previously-created application context is used.
        <callback>	- I
            is the function ("work procedure") that is to be called when the
            NIX dispatcher has nothing else to do.  The callback function
            should be declared as follows:
                int  callback_function (NxAppContext context,
                                        NxWorkProcId workproc_ID,
                                        void *client_data) ;
            where "context" is the application context as used in or created
            by this call to NXADDWORKPROC(); "workproc_ID" is the value
            returned by this call to NXADDWORKPROC(); "client_data" is the
            argument that was passed in to NXADDWORKPROC().  The return
            value of the callback function is ignored by the NIX dispatcher.
        <client_data>	- I
            is an arbitrary data value, cast as a (VOID *) pointer, that will
            be passed to the callback function when it is invoked.
        <workproc_ID>	- O
            returns an ID for the registered work procedure; NULL is returned
            if an error occurs.  This ID is used by NXREMOVEWORKPROC() to
            identify a work procedure being "unregistered".

*******************************************************************************/


NxWorkProcId  NxAddWorkProc (

#    if __STDC__ || defined(vaxc)
        NxAppContext  *context,
        NxWorkProc  workprocF,
        void  *client_data)
#    else
        context, workprocF, client_data)

        NxAppContext  *context ;
        NxWorkProc  workprocF ;
        void  *client_data ;
#    endif

{    /* Local variables. */
    _NxAppContext  *app ;
    _NxBackgroundTask  *bat, *rear ;




/* Use the desired application context. */

    if (context == NULL)  context = &default_context ;
    if ((*context == NULL) && NxCreateContext (context)) {
        vperror ("(NxAddWorkProc) Error creating application context.\nNxCreateContext: ") ;
        return (NULL) ;
    }
    app = *context ;

/* Allocate a background task structure for the work procedure. */

    bat = (_NxBackgroundTask *) malloc (sizeof (_NxBackgroundTask)) ;
    if (bat == NULL) {
        vperror ("(NxAddWorkProc) Error allocating background task structure.\nmalloc: ") ;
        return (NULL) ;
    }
    bat->workproc = workprocF ;
    bat->client_data = client_data ;

/* Add the work procedure to the queue of registered work procedures. */

    for (rear = app->workproc_queue ;  rear != NULL ;  rear = rear->next)
        if (rear->next == NULL)  break ;

    if (rear == NULL)			/* Add to empty queue? */
        app->workproc_queue = bat ;
    else				/* Append to non-empty queue. */
        rear->next = bat ;

    if (app->debug)
        printf ("(NxAddWorkProc) Workproc: 0x%lX, Data: 0x%lX\n",
                (long) workprocF, (long) client_data) ;

    return (bat) ;

}

/*******************************************************************************

Procedure:

    NxCreateContext ()

    Create an Application Context.


Purpose:

    Function NXCREATECONTEXT creates a new application context for the
    NIX dispatcher.


    Invocation:

        status = NxCreateContext (&context) ;

    where:

        <context>	- O
            returns the address of the new application context structure.
        <status>	- O
            returns the status of creating the application context, zero if
            no errors occurred and ERRNO otherwise.

*******************************************************************************/


int  NxCreateContext (

#    if __STDC__ || defined(vaxc)
        NxAppContext  *context)
#    else
        context)

        NxAppContext  *context ;
#    endif

{
    if (context == NULL) {		/* Creating the default context? */
#ifdef VXWORKS
        if (taskVarAdd (0, (int *) &default_context) == ERROR) {
            vperror ("(NxCreateContext) Error declaring the default context as a task variable.\ntaskVarAdd: ") ;
            return (errno) ;
        }
#endif
        context = &default_context ;
    }

    *context = (_NxAppContext *) malloc (sizeof (_NxAppContext)) ;
    if (*context == NULL) {
        vperror ("(NxCreateContext) Error allocating application context structure.\nmalloc: ") ;
        return (errno) ;
    }

    (*context)->debug = nix_util_debug ;
    (*context)->IO_source_list = NULL ;
    (*context)->timeout_list = NULL ;
    (*context)->workproc_queue = NULL ;

    if (nix_util_debug)  printf ("(NxCreateContext) Created context 0x%lX.\n",
                                 (long) *context) ;

    return (0) ;

}

/*******************************************************************************

Procedure:

    NxMainLoop ()

    Handle I/O Events.


Purpose:

    Function NXMAINLOOP monitors I/O events and, when they occur, invokes the
    event handlers ("callback" functions) defined for the events.  The types
    of events handled by NXMAINLOOP include I/O events registered by calls to
    NXADDINPUT() and timeout events registered by calls to NXADDTIMEOUT().
    NXMAINLOOP() is modeled after - you guessed it - the X Toolkit function,
    "XtAppMainLoop()".

    Once called, NXMAINLOOP() never returns.  Well, almost never.  NXMAINLOOP()
    only returns if:

      - there are no more I/O events to monitor and no timeouts to be timed,
        or if

      - an error occurs while polling an I/O source.  This might occur if a
        closed I/O source is not removed from the NIX dispatcher's I/O source
        list.

    In either case, NXMAINLOOP() returns an error indication to the calling
    program.


    Invocation:

        status = NxMainLoop (&context) ;

    where:

        <context>	- I/O
            is the address of a variable containing the client's reference
            (an opaque pointer) to the application context.  This should be
            the same context used to register the I/O sources, timers, and
            work procedures.  If the address passed in for CONTEXT is NULL,
            then the default application context is used.  If the value of
            CONTEXT is NULL, then a new application context is created and
            an opaque pointer to that context is returned in CONTEXT.  If
            the value of CONTEXT is not NULL, then the previously-created
            application context is used.
        <status>	- O
            returns the status (ERRNO) of monitoring the registered events.
            NXMAINLOOP() only returns in the event of an unrecoverable
            error or if there are no more I/O sources or timers to monitor.

*******************************************************************************/


int  NxMainLoop (

#    if __STDC__ || defined(vaxc)
        NxAppContext  *context)
#    else
        context)

        NxAppContext  *context ;
#    endif

{    /* Local variables. */
    _NxAppContext  *app ;
    fd_set  except_mask, except_mask_save ;
    fd_set  read_mask, read_mask_save ;
    fd_set  write_mask, write_mask_save ;
#ifdef VMS
    float  f_timeout ;
#endif
    int  num_active ;
    _NxIOSource  *ios ;
    struct  timeval  timeout ;
    _NxTimer  *tot ;




/* Use the desired application context. */

    if (context == NULL)  context = &default_context ;
    if ((*context == NULL) && NxCreateContext (context)) {
        vperror ("(NxMainLoop) Error creating application context.\nNxCreateContext: ") ;
        return (errno) ;
    }
    app = *context ;


/*******************************************************************************
    Loop forever, "listening" for and responding to I/O events and timeouts.
    When a monitored I/O event is detected, invoke the callback function
    bound by NXADDINPUT() to the source of the event.  When a timeout
    interval expires, invoke the callback function bound by NXADDTIMEOUT()
    to the timer.
*******************************************************************************/


    for ( ; ; ) {


/* Construct the SELECT(2) masks for the I/O sources being monitored. */

        FD_ZERO (&read_mask_save) ;
        FD_ZERO (&write_mask_save) ;
        FD_ZERO (&except_mask_save) ;
        num_active = 0 ;

        for (ios = app->IO_source_list ;  ios != NULL ;  ios = ios->next) {
            if (ios->condition & NxInputReadMask) {
                FD_SET (ios->source, &read_mask_save) ;  num_active++ ;
            }
            if (ios->condition & NxInputWriteMask) {
                FD_SET (ios->source, &write_mask_save) ;  num_active++ ;
            }
            if (ios->condition & NxInputExceptMask) {
                FD_SET (ios->source, &except_mask_save) ;  num_active++ ;
            }
        }

        if ((num_active == 0) && (app->timeout_list == NULL)) {
            errno = EINVAL ;
            vperror ("(NxMainLoop) No I/O sources or timeouts to monitor.\n") ;
            return (errno) ;
        }


/* Wait for an I/O event to occur or for the timeout interval to expire. */

        for ( ; ; ) {
            read_mask = read_mask_save ;
            write_mask = write_mask_save ;
            except_mask = except_mask_save ;
            if (app->timeout_list == NULL) {	/* Wait forever. */
                num_active = select (FD_SETSIZE, &read_mask, &write_mask,
                                     &except_mask, NULL) ;
            } else {		/* Wait for I/O or until timeout expires. */
                timeout = tv_subtract ((app->timeout_list)->expiration,
                                        tv_tod ()) ;
#ifdef VMS
                if (num_active > 0) {
#endif
                num_active = select (FD_SETSIZE, &read_mask, &write_mask,
                                     &except_mask, &timeout) ;
#ifdef VMS
                } else {
			/* VMS doesn't allow SELECT(2)ing when no bits are set
			   in the masks, so LIB$WAIT() is used for timeouts. */
                    f_timeout = (float) timeout.tv_sec +
                                (timeout.tv_usec / 1000000.0) ;
                    LIB$WAIT (&f_timeout) ;
                    num_active = 0 ;
                }
#endif
            }
            if (num_active >= 0)  break ;
            if (errno == EINTR)  continue ;	/* SELECT interrupted by signal - try again. */
            vperror ("(NxMainLoop) Error monitoring I/O sources.\nselect: ") ;
            return (errno) ;
        }


/* Scan the SELECT(2) bit masks.  For each I/O event detected, invoke the
   callback function bound to that event and its source. */

        for (ios = app->IO_source_list ;  ios != NULL ;  ios = ios->next) {
            if (((ios->condition & NxInputReadMask) &&
                 FD_ISSET (ios->source, &read_mask)) ||
                ((ios->condition & NxInputWriteMask) &&
                 FD_ISSET (ios->source, &write_mask)) ||
                ((ios->condition & NxInputExceptMask) &&
                 FD_ISSET (ios->source, &except_mask))) {
                ios->callback ((void *) app, (void *) ios,
                               ios->source, ios->client_data) ;
            }
        }


/* If any timeout timers have fired, invoke the callback functions bound to
   the timeout events.  Since the timer list is sorted by expiration time,
   the entire timer list does not need to be scanned. */

        while (app->timeout_list != NULL) {
            tot = app->timeout_list ;
            if (tv_compare (tv_tod (), tot->expiration) < 0)  break ;
            app->timeout_list = tot->next ;
            (void) tot->callback ((void *) app, (void *) tot,
                                  tot->client_data) ;
            free (tot) ;
        }


    }     /* Loop forever */

}

#ifdef VMS
/*******************************************************************************

Procedure:

    NxMainLoopEF ()

    Handle I/O Events using Event Flags.  (VMS Only)


Purpose:

    Function NXMAINLOOPEF, like NXMAINLOOP(), monitors I/O events and, when
    they occur, invokes the event handlers ("callback" functions) defined for
    the events.  Unlike NXMAINLOOP(), which uses the UNIX SELECT(2) mechanism
    to monitor I/O sources, NXMAINLOOPEF() uses VMS event flags to monitor the
    I/O sources.  Under UNIX, SELECT(2) works for any type of I/O source; the
    VMS version of SELECT(2), however, only works for network connections.
    One way of surmounting this limitation is to monitor event flags triggered
    by I/O activity instead of calling SELECT(2).  The event flag approach is
    used by the DECWindows implementation of "XtAddInput()" and "XtMainLoop()";
    NXMAINLOOPEF() makes this approach available to NIX-based programs.


    Invocation:

        status = NxMainLoopEF (&context, timer_ef) ;

    where:

         <context>	- I/O
            is the address of a variable containing the client's reference
            (an opaque pointer) to the application context.  This should be
            the same context used to register the I/O sources, timers, and
            work procedures.  If the address passed in for CONTEXT is NULL,
            then the default application context is used.  If the value of
            CONTEXT is NULL, then a new application context is created and
            an opaque pointer to that context is returned in CONTEXT.  If
            the value of CONTEXT is not NULL, then the previously-created
            application context is used.
       <timer_ef>	- I
            specifies an event flag (0-23) to be used by NXMAINLOOPEF() for
            monitoring timeout timers.  The calling routine is responsible for
            reserving the event flag, if necessary, and ensuring that the flag
            is not used elsewhere in the program.  If timers are not used (i.e.,
            NXADDTIMEOUT() is not called), then this argument is also not used.
        <status>	- O
            returns the status (ERRNO) of monitoring the registered events.
            NXMAINLOOPEF() only returns in the event of an unrecoverable
            error or if there are no more I/O sources or timers to monitor.

*******************************************************************************/


int  NxMainLoopEF (

#    if __STDC__ || defined(vaxc)
        NxAppContext  *context,
        int  timer_ef)
#    else
        context, timer_ef)

        NxAppContext  *context ;
        int  timer_ef ;
#    endif

{    /* Local variables. */
    _NxAppContext  *app ;
    float  f_timeout ;
    int  status ;
    _NxIOSource  *ios ;
    long  delta_time[2], event_flag_mask, operation ;
    struct  timeval  timeout ;
    _NxTimer  *tot ;




/* Use the desired application context. */

    if (context == NULL)  context = &default_context ;
    if ((*context == NULL) && NxCreateContext (context)) {
        vperror ("(NxMainLoopEF) Error creating application context.\nNxCreateContext: ") ;
        return (errno) ;
    }
    app = *context ;


/*******************************************************************************
    Loop forever, "listening" for and responding to I/O events and timeouts.
    When a monitored I/O event is detected, invoke the callback function
    bound by NXADDINPUT() to the source of the event.  When a timeout
    interval expires, invoke the callback function bound by NXADDTIMEOUT()
    to the timer.
*******************************************************************************/


    for ( ; ; ) {


/* Construct the event flag mask for the I/O sources being monitored. */

        event_flag_mask = 0 ;

        for (ios = app->IO_source_list ;  ios != NULL ;  ios = ios->next)
            event_flag_mask = event_flag_mask | (1 << ios->source) ;

        if ((event_flag_mask == 0) && (app->timeout_list == NULL)) {
            errno = EINVAL ;
            vperror ("(NxMainLoopEF) No I/O sources or timeouts to monitor.\n") ;
            return (errno) ;
        }


/* If the calling routine has set a timer via NXADDTIMEOUT(), then set a
   VMS timer to go off at that time.  When the VMS timer fires, it sets the
   caller-specified timer event flag. */

        if (app->timeout_list != NULL) {

				/* Add timer event flag to event flag mask. */
            event_flag_mask = event_flag_mask | (1 << timer_ef) ;

				/* Compute time until timer expiration. */
            timeout = tv_subtract ((app->timeout_list)->expiration, tv_tod ()) ;
            f_timeout = (float) timeout.tv_sec +
                        (timeout.tv_usec / 1000000.0) ;

				/* Convert to internal VMS time format. */
            operation = LIB$K_DELTA_SECONDS_F ;
            status = LIB$CVTF_TO_INTERNAL_TIME (&operation, &f_timeout,
                                                delta_time) ;
            if (f_timeout <= 0.0) {	/* 0.0 is an invalid delta time. */
                delta_time[0] = 0 ;
                delta_time[1] = 0 ;
            } else if (!(status & STS$M_SUCCESS)) {
                errno = EVMSERR ;  vaxc$errno = status ;
                vperror ("(NxMainLoopEF) Error converting time (%d secs, %d usecs).\nLIB$CVTF_TO_INTERNAL_TIME: ",
                         timeout.tv_sec, timeout.tv_usec) ;
                return (errno) ;
            }
				/* Set up a VMS timer. */
            status = SYS$SETIMR (timer_ef, delta_time, NULL, 0, 0) ;
            if (!(status & STS$M_SUCCESS)) {
                errno = EVMSERR ;  vaxc$errno = status ;
                vperror ("(NxMainLoopEF) Error setting timer.\nSYS$SETIMR: ") ;
                return (errno) ;
            }

        }


/* Wait for an I/O source's event flag to be set (indicating I/O activity) or
   for the timeout timer's event flag to be set. */

        status = SYS$WFLOR (0, event_flag_mask) ;
        if (!(status & STS$M_SUCCESS)) {
            errno = EVMSERR ;  vaxc$errno = status ;
            vperror ("(NxMainLoopEF) Error monitoring event flags (0x%08.8lX).\nSYS$WFLOR: ",
                     event_flag_mask) ;
            return (errno) ;
        }


/* Get the current state of the event flags being monitored. */

        status = SYS$READEF (0, &event_flag_mask) ;
        if (!(status & STS$M_SUCCESS)) {
            errno = EVMSERR ;  vaxc$errno = status ;
            vperror ("(NxMainLoopEF) Error reading event flags.\nSYS$READEF: ") ;
            return (errno) ;
        }


/* Scan the event flag mask.  For each flag set (indicating an I/O event),
   invoke the callback function bound to that event flag's source. */

        for (ios = app->IO_source_list ;  ios != NULL ;  ios = ios->next) {
            if (event_flag_mask & (1 << ios->source))
                ios->callback ((void *) app, (void *) ios,
                               ios->source, ios->client_data) ;
        }


/* If any timeout timers have fired, invoke the callback functions bound to
   the timeout events.  Since the timer list is sorted by expiration time,
   the entire timer list does not need to be scanned. */

        while (app->timeout_list != NULL) {
            tot = app->timeout_list ;
            if (tv_compare (tv_tod (), tot->expiration) < 0)  break ;
            app->timeout_list = tot->next ;
            (void) tot->callback ((void *) app, (void *) tot,
                                  tot->client_data) ;
            free (tot) ;
        }


    }     /* Loop forever */

}
#endif /* VMS */

/*******************************************************************************

Procedure:

    NxRemoveInput ()

    Remove the Registration of an I/O Source with the NIX Dispatcher.


Purpose:

    Function NXREMOVEINPUT "unregisters" an I/O source previously registered
    with the NIX dispatcher.  NXREMOVEINPUT() is modeled after the X Toolkit
    function, "XtRemoveInput()".


    Invocation:

        status = NxRemoveInput (&context, source_ID) ;

    where:

        <context>	- I/O
            is the address of a variable containing the client's reference
            (an opaque pointer) to the application context.  This should be
            the same context used in the call to NXADDINPUT() that registered
            the I/O source.  If the address passed in for CONTEXT is NULL,
            then the default application context is used.  If the value of
            CONTEXT is NULL, then a new application context is created and
            an opaque pointer to that context is returned in CONTEXT.  If
            the value of CONTEXT is not NULL, then the previously-created
            application context is used.
        <source_ID>	- I
            is the ID assigned to the I/O source by NXADDINPUT().
        <status>	- O
            returns the status of "unregistering" the I/O source, zero if
            no errors occurred and ERRNO otherwise.

*******************************************************************************/


int  NxRemoveInput (

#    if __STDC__ || defined(vaxc)
        NxAppContext  *context,
        NxInputId  source_ID)
#    else
        context, source_ID)

        NxAppContext  *context ;
        NxInputId  source_ID ;
#    endif

{    /* Local variables. */
    _NxAppContext  *app ;
    _NxIOSource  *ios, *prev ;




/* Use the desired application context. */

    if (context == NULL)  context = &default_context ;
    if ((*context == NULL) && NxCreateContext (context)) {
        vperror ("(NxRemoveInput) Error creating application context.\nNxCreateContext: ") ;
        return (errno) ;
    }
    app = *context ;

/* Locate the target I/O source in the list of registered sources. */

    for (prev = NULL, ios = app->IO_source_list ;
         ios != NULL ;  ios = ios->next) {
        if (ios == source_ID)  break ;
        prev = ios ;
    }

    if (ios == NULL) {
        errno = EINVAL ;
        vperror ("(NxRemoveInput) Source ID 0x%X not found.\n",
                 (long) source_ID) ;
        return (errno) ;
    }

    if (app->debug)
        printf ("(NxRemoveInput) I/O source %d unregistered.\n",
                ios->source) ;

/* Unlink the I/O source structure from the list and deallocate it. */

    if (prev == NULL)			/* Beginning of list? */
        app->IO_source_list = ios->next ;
    else				/* Middle or end of list. */
        prev->next = ios->next ;

    free (ios) ;

    return (0) ;

}

/*******************************************************************************

Procedure:

    NxRemoveTimeOut ()

    Remove the Registration of a Timeout Timer with the NIX Dispatcher.


Purpose:

    Function NXREMOVETIMEOUT "unregisters" a timeout timer previously
    registered with the NIX dispatcher.  NXREMOVETIMEOUT() is modeled
    after the X Toolkit function, "XtRemoveTimeOut()".


    Invocation:

        status = NxRemoveTimeOut (&context, timer_ID) ;

    where:

        <context>	- I/O
            is the address of a variable containing the client's reference
            (an opaque pointer) to the application context.  This should
            be the same context used in the call to NXADDTIMEOUT() that
            registered the timer.  If the address passed in for CONTEXT
            is NULL, then the default application context is used.  If
            the value of CONTEXT is NULL, then a new application context
            is created and an opaque pointer to that context is returned
            in CONTEXT.  If the value of CONTEXT is not NULL, then the
            previously-created application context is used.
        <timer_ID>	- I
            is the ID assigned to the timeout timer by NXADDTIMEOUT().
        <status>	- O
            returns the status of "unregistering" the timer, zero if
            no errors occurred and ERRNO otherwise.

*******************************************************************************/


int  NxRemoveTimeOut (

#    if __STDC__ || defined(vaxc)
        NxAppContext  *context,
        NxIntervalId  timer_ID)
#    else
        context, timer_ID)

        NxAppContext  *context ;
        NxIntervalId  timer_ID ;
#    endif

{    /* Local variables. */
    _NxAppContext  *app ;
    _NxTimer  *prev, *tot ;




/* Use the desired application context. */

    if (context == NULL)  context = &default_context ;
    if ((*context == NULL) && NxCreateContext (context)) {
        vperror ("(NxRemoveTimeOut) Error creating application context.\nNxCreateContext: ") ;
        return (errno) ;
    }
    app = *context ;

/* Locate the target timer in the list of registered timers. */

    for (prev = NULL, tot = app->timeout_list ;
         tot != NULL ;  tot = tot->next) {
        if (tot == timer_ID)  break ;
        prev = tot ;
    }

    if (tot == NULL) {
        errno = EINVAL ;
        vperror ("(NxRemoveTimeOut) Timer ID 0x%X not found.\n",
                 (long) timer_ID) ;
        return (errno) ;
    }

    if (app->debug)
        printf ("(NxRemoveTimeOut) %g-second timeout unregistered.\n",
                tot->interval) ;

/* Unlink the I/O timer structure from the list and deallocate it. */

    if (prev == NULL)			/* Beginning of list? */
        app->timeout_list = tot->next ;
    else				/* Middle or end of list. */
        prev->next = tot->next ;

    free (tot) ;

    return (0) ;

}

/*******************************************************************************

Procedure:

    NxRemoveWorkProc ()

    Remove the Registration of a Work Procedure with the NIX Dispatcher.


Purpose:

    Function NXREMOVEWORKPROC "unregisters" a work procedure previously
    registered with the NIX dispatcher.  NXREMOVEWORKPROC() is modeled
    after the X Toolkit function, "XtRemoveWorkProc()".


    Invocation:

        status = NxRemoveWorkProc (&context, workproc_ID) ;

    where:

        <context>	- I/O
            is the address of a variable containing the client's reference
            (an opaque pointer) to the application context.  This should
            be the same context used in the call to NXADDWORKPROC() that
            registered the work procedure.  If the address passed in for
            CONTEXT is NULL, then the default application context is used.
            If the value of CONTEXT is NULL, then a new application context
            is created and an opaque pointer to that context is returned
            in CONTEXT.  If the value of CONTEXT is not NULL, then the
            previously-created application context is used.
        <workproc_ID>	- I
            is the ID assigned to the work procedure by NXADDWORKPROC().
        <status>	- O
            returns the status of "unregistering" the work procedure, zero
            if no errors occurred and ERRNO otherwise.

*******************************************************************************/


int  NxRemoveWorkProc (

#    if __STDC__ || defined(vaxc)
        NxAppContext  *context,
        NxWorkProcId  workproc_ID)
#    else
        context, workproc_ID)

        NxAppContext  *context ;
        NxWorkProcId  workproc_ID ;
#    endif

{    /* Local variables. */
    _NxAppContext  *app ;
    _NxBackgroundTask  *bat, *prev ;




/* Use the desired application context. */

    if (context == NULL)  context = &default_context ;
    if ((*context == NULL) && NxCreateContext (context)) {
        vperror ("(NxRemoveWorkProc) Error creating application context.\nNxCreateContext: ") ;
        return (errno) ;
    }
    app = *context ;

/* Locate the target procedure in the queue of registered work procedures. */

    for (prev = NULL, bat = app->workproc_queue ;
         bat != NULL ;  bat = bat->next) {
        if (bat == workproc_ID)  break ;
        prev = bat ;
    }

    if (bat == NULL) {
        errno = EINVAL ;
        vperror ("(NxRemoveWorkProc) Workproc ID 0x%X not found.\n",
                 (long) workproc_ID) ;
        return (errno) ;
    }

    if (app->debug)
        printf ("(NxRemoveWorkProc) Workproc 0x%lX (Data 0x%lX) unregistered.\n",
                (long) bat->workproc, (long) bat->client_data) ;

/* Unlink the background task structure from the queue and deallocate it. */

    if (prev == NULL)			/* Beginning of queue? */
        app->workproc_queue = bat->next ;
    else				/* Middle or end of queue. */
        prev->next = bat->next ;

    free (bat) ;

    return (0) ;

}

/*******************************************************************************

Procedure:

    NxSetDebug ()

    Enable/Disable Debug Output.


Purpose:

    Function NXSETDEBUG enabled/disables debug output (to standard output)
    for an application context.


    Invocation:

        status = NxSetDebug (&context, enable) ;

    where:

        <context>	- I/O
            is the address of a variable containing the client's reference
            (an opaque pointer) to the application context.  If the address
            passed in for CONTEXT is NULL, then the default application
            context is used.  If the value of CONTEXT is NULL, then a new
            application context is created and an opaque pointer to that
            context is returned in CONTEXT.  If the value of CONTEXT is not
            NULL, then the previously-created application context is used.
        <enable>	- I
            if true (a non-zero value), enables debug output for the
            application context; false (zero) disables debug output for
            the application context.
         <status>	- O
            returns the status of enabling or disabling debug output, zero
            if no errors occurred and ERRNO otherwise.

*******************************************************************************/


int  NxSetDebug (

#    if __STDC__ || defined(vaxc)
        NxAppContext  *context,
        int  enable)
#    else
        context, enable)

        NxAppContext  *context ;
        int  enable ;
#    endif

{    /* Local variables. */
    _NxAppContext  *app ;



/* Use the desired application context. */

    if (context == NULL)  context = &default_context ;
    if ((*context == NULL) && NxCreateContext (context)) {
        vperror ("(NxSetDebug) Error creating application context.\nNxCreateContext: ") ;
        return (errno) ;
    }
    app = *context ;

/* Enable or disable debug output for this application context. */

    app->debug = enable ;

    return (0) ;

}
