/*
 *	$Source: /user/nlfm/Working/Frink/RCS/token.c,v $
 *	$Date: 1995/11/22 08:43:03 $
 *	$Revision: 1.2.1.11 $
 *
 *------------------------------------------------------------------------
 *   AUTHOR:  Lindsay Marshall <lindsay.marshall@newcastle.ac.uk>
 *------------------------------------------------------------------------
 *    Copyright 1994 The University of Newcastle upon Tyne (see COPYRIGHT)
 *========================================================================
 *
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "frink.h"

struct tbuff_s
{
    char *buff;
    int   size;
    int   length;
    int   unit;
};

typedef struct tbuff_s TBuff;

static TBuff *newTB(int size)
{
    TBuff *tp = (TBuff *) malloc(sizeof(TBuff));
    tp->buff = malloc(size);
    tp->length = 0;
    tp->unit = tp->size = size;
    return tp;
}

static void freeTB(TBuff *tp)
{
    free(tp->buff);
    free(tp);
}

static void addTB(TBuff *tp, char ch)
{
    if (tp->length == tp->size)
    {
        tp->size += tp->unit;
        tp->buff = realloc(tp->buff, tp->size);
    }
    tp->buff[tp->length] = ch;
    tp->length += 1;
}

static void capTB(TBuff *tp)
{
    if (tp->length == tp->size)
    {
        tp->size += tp->unit;
        tp->buff = realloc(tp->buff, tp->size);
    }
    tp->buff[tp->length] ='\0';
}

static void catTB(TBuff *tp, char *str)
{
    while (*str)
    {
        addTB(tp, *str++);
    }
    capTB(tp);
}

static char *tokenName[] =
{
    "ENDF",
    "ENDLINE",
    "ENDC",
    "LIST",
    "STRING",
    "CONC",
    "CALL",
    "VAR",
    "CPART",
    "SPART",
    "CONST",
    "ARRAY",
    "NL",
    "SP",
    "COMMENT",
    "HEAD",
    "LBRACK",
    "LBRACE",
    "LPAREN",
    "DOLLAR",
    "VNAME",
    "OSTART",
    "START",
    "XCONT",
    "CONT",
    "RBRACK",
    "RBRACE",
    "BLANK",
    "DQSTART",
    "DQEND",
    "SEMI",
    "EM",
    "EN",
    "ECONT",
    "SLIST",
    "SCOMMENT",
    "NOSP",
    "SRBRACE"
};


static Token *concToken(char, char, Input *);
static Token *callToken(Input *);
static Token *varToken(Input *);

void warn(Token *tp, char *msg)
{
    fprintf(stderr, "*** Warning : %s", msg);
    if (tp != 0)
    {
	if (tp->lineNo != 0)
	{
	    fprintf(stderr, ", line %d", tp->lineNo);
	}
	fprintf(stderr, "\nToken Information : %s \"%s\"", tokenName[tp->type],
	  (tp->text == (char *) 0) ? "(NULL)" : tp->text);
    }
    fprintf(stderr, "\n");
}

void fail(Token *tp, char *msg)
{
    fprintf(stderr, "*** Error : %s", msg);
    if (tp != 0)
    {
	if (tp->lineNo != 0)
	{
	    fprintf(stderr, ", line %d", tp->lineNo);
	}
	fprintf(stderr, "\nToken Information : %s \"%s\"", tokenName[tp->type],
	  (tp->text == (char *) 0) ? "(NULL)" : tp->text);
    }
    fprintf(stderr, "\n");
    exit(1);
}

void dumpToken(Token *tk)
{
    Token *tp;
    static int depth = -1;
    if (tk == 0)
    {
	fprintf(stderr, "Null Token Pointer\n");
    }
    else
    {
	depth += 1;
	fprintf(stderr, "{%s {", tokenName[tk->type]);
	if (tk->text)
	{
	    fprintf(stderr, "%s", tk->text);
	}
	fprintf(stderr, "} {");
	tp = tk->sequence;
	while (tp != (Token *) 0)
	{
	    dumpToken(tp);
	    tp = tp->next;
	}
	fprintf(stderr,"} }");
	if ((depth -= 1) < 0) { fprintf(stderr, "\n"); }
    }
}

static void makeTB(Token *hd, TBuff *tb)
{
    if (hd == (Token *) 0) { return; }

    switch (hd->type)
    {
    case VAR :
	catTB(tb, "$");
	makeTB(hd->sequence, tb);
	break;
    case ARRAY :
        catTB(tb, "(");
	makeTB(hd->sequence, tb);
	catTB(tb, ")");
	break;
    case SEMI :
	catTB(tb, ";");
	break;
    case NL :
	catTB(tb, "\n");
	break;
    case CONC :
	makeTB(hd->sequence, tb);
	break;
    case STRING :
	catTB(tb, "\"");
	makeTB(hd->sequence, tb);
	catTB(tb, "\"");
	break;
    case SLIST :
	catTB(tb, "{");
	makeTB(hd->sequence, tb);
	catTB(tb, "}");
	break;
    case CALL :
	catTB(tb, "[");
	makeTB(hd->sequence, tb);
	catTB(tb, "]");
	break;
    case LIST :
	catTB(tb, "{");
	if (hd->text != (char *) 0) { catTB(tb, hd->text); }
	catTB(tb, "}");
	break;
    default :
	if (hd->text != (char *) 0) { catTB(tb, hd->text); }
	break;
    }
    makeTB(hd->next, tb);
}

void streamMore(Input *file)
{
    file->pushed = 0;
    file->position = file->text;
    if (file->stream != NULL)
    {
	file->remaining = fread(file->text, 1, 64*1024, file->stream);
    }
}

static int isBlank(char ch)
{
    return ch == ' ' || ch == '\t';
}

static char reallyGet(Input *file)
{
    char ch;
    if (file->remaining <= 0) {	streamMore(file); }
    if (file->remaining <= 0) { return '\0'; }
    file->remaining -= 1;
    if ((ch = *file->position++) == '\n') { file->lineNumber += 1; }
    return ch;
}

static void push(Input *file, char ch)
{
    if (ch != '\0')
    {
	file->back[file->pushed] = ch;
	file->pushed += 1;
    }
}

static char chGet(Input *file)
{
    char ch;
    if (file->pushed)
    {
	file->pushed -= 1;
	return file->back[file->pushed];
    }
    else if ((ch = reallyGet(file)) == '\\')
    {
	if ((ch = reallyGet(file)) == '\n')
	{
	    while (isBlank(ch = reallyGet(file)))
	    {
	    }
	    push(file, ch);
	    ch = ' ';
	}
        else
	{
	    push(file, ch);
	    ch = '\\';
	}
    }

    return ch;
}

static char chCGet(Input *file)
{
    if (file->pushed)
    {
	file->pushed -= 1;
	return file->back[file->pushed];
    }
    return reallyGet(file);
}

static Token *newToken(TokenType t)
{
    Token *tp = malloc(sizeof(Token));
    tp->lineNo = 0;
    tp->type = t;
    tp->text = (char *) 0;
    tp->length = 0;
    tp->sequence = tp->next = (Token *) 0;
    return tp;
}

void freeToken(Token *tp)
{
    if (tp != (Token *) 0)
    {
	if (tp->text != (char *) 0 && tp->text != tp->little)
	{
	    free(tp->text);
	}
	if (tp->sequence) freeToken(tp->sequence);
	if (tp->next) freeToken(tp->next);
	free(tp);
    }
}

Input *tokenise(char *string, int length)
{
    Input *file = (Input *) malloc(sizeof(Input));
    file->position = string;
    file->remaining = length;
    file->pushed = 0;
    file->text = string;
    file->stream = NULL;
    file->tcall = 0;
    file->lineNumber = 0;
    return file;
}

void untokenise(Input *file)
{
    free(file);
}

static Token *tokenPop(Token **tp)
{
    Token *sp;
    if ((sp = *tp) != (Token *) 0)
    {
	*tp = sp->next;
        sp->next = (Token *) 0;
    }
    return sp;
}

void tokenPush(Token **tp, Token *v)
{
    Token * ptr;
    v->next = (Token *) 0;
    if ((ptr = *tp) == (Token *) 0)
    {
	*tp = v;
    }
    else
    {
	while (ptr->next != (Token *) 0)
	{
	    ptr = ptr->next;
	}
	ptr->next = v;
    }
}

static Token *makeToken(TokenType t, TBuff *tb)
{
    Token *tp = newToken(t);
    if ((tp->length = tb->length) < sizeof(tp->little))
    {
	strcpy(tp->little, tb->buff);
	freeTB(tb);
	tp->text = tp->little;
    }
    else
    {
	tp->text = tb->buff;
	free(tb);
    }
    return tp;
}

static Token *spaceToken(char ch, Input *file)
{
    TBuff *tb = newTB(128);
    do
    {
        addTB(tb, ch);
    }
    while ((ch = chGet(file)) == ' ' || ch == '\t');
    push(file, ch);
    capTB(tb);
    return makeToken(SP, tb); 
}

static Token *listToken(Input *file)
{
    int bufSize = 16*1024;
    char ch;
    TBuff *tb = newTB(bufSize);
    int bcount = 0, inString = 0;

    for (;;)
    {
	switch (ch = chGet(file))
	{
	case '\\' :
	    addTB(tb, ch);
	    ch = chGet(file);
	    break;
/* I cant remember why this was here and it seems to be broken!!!
	case '"' :
	    inString = !inString;
	    break;
*/
	case '{' :
	    if (!inString) { bcount += 1; };
	    break;
	case '}' :
	    if (!inString) { if ((bcount -= 1) < 0) { goto done; } }
	    break;
	case '\0' :
	    warn((Token *) 0, "Missing }");
	    goto done;
	}
	addTB(tb, ch);
    }
done:
    capTB(tb);
    return makeToken(LIST, tb);
}

Token *stringToken(Input *file, char lst, TokenType res)
{
    int bufSize = 16*1024;
    Token *tp = newToken(res), *nt;
    char ch;
    TBuff *tb = newTB(bufSize);
    static char errmsg [] ="Missing X";

    for(;;)
    {
	if ((ch = chGet(file)) == lst) { goto done; }
	switch (ch)
	{
	case '\n' :
	    nt =  newToken(NL);
	    break;
	case ';' :
	    nt = newToken(SEMI);
	    break;
	case ' ' :
	case '\t':
	    nt = spaceToken(ch, file);
	    break;
	case '[' :
	    nt = callToken(file);
	    break;
	case '$' :
	    if ((nt = varToken(file)) == (Token *) 0)
	    {
		addTB(tb, '$');
		continue;
	    }
	    break;
	case '\0' :
	    errmsg[8] = lst;
	    warn((Token *) 0, errmsg);
	    goto done;
	case '{' :
	    nt = newToken(LBRACE);
	    break;
	case '}' :
	    nt = newToken(RBRACE);
	    break;
	case '\\' :
	    addTB(tb, ch);
	    ch = chGet(file);
	default :
	    addTB(tb, ch);
	    continue;
	}
        if (tb->length != 0)
	{
	    capTB(tb);
	    tokenPush(&tp->sequence, makeToken(SPART, tb));
	    tb = newTB(bufSize);
	}
	tokenPush(&tp->sequence, nt);
    }
done:
    capTB(tb);
    if (tb->length != 0)
    {
	tokenPush(&tp->sequence, makeToken(SPART, tb));
    }
    else
    {
	freeTB(tb);
    }
    return tp;
}

Token *getToken(Input *file)
{
    char ch;
    Token *tp;
    extern int trace;

    switch (ch = chGet(file))
    {
    case '\0' : tp = newToken(ENDF); break;
    case ';' : tp = newToken(SEMI); break;
    case '\n' : tp = newToken(NL); break;
    case ' ' :
    case '\t': tp = spaceToken(ch, file); break;
    case ']':
	tp = (file->tcall) ? newToken(ENDC) : concToken(ch, '\0', file);
	break;
    case '{' : tp = listToken(file); break;
    case '"' : tp = stringToken(file, '"', STRING); break;
    default  : tp = concToken(ch, '\0', file); break;
    }
    if (trace) dumpToken(tp);
    return tp;
}

static Token *callToken(Input *file)
{
    Token *tp = newToken(CALL), *stp;
    file->tcall += 1;
    for(;;)
    {
	stp = getToken(file);
	switch (stp->type)
	{
	case SP :
	    freeToken(stp);
	    continue;
	case ENDF :
	    warn((Token *) 0, "Missing ]");
	case ENDC :
	    freeToken(stp);
	    file->tcall -= 1;
	    return tp;
	default :
	    break;
	}
	tokenPush(&tp->sequence, stp);
    }
}

/*
 * varToken processes the name following a $. It can be:
 *
 * 1) a sequence of alphanumerics or _
 * 2) a list enclosed in {}
 * 3) a name followed by an array reference
 * 
 */
static Token *varToken(Input *file)
{
    Token *tp = newToken(VAR);
    int bufSize = 512, leng;
    char ch;
    TBuff *tb;
    
    if ((ch = chGet(file)) == '{')	/* it's a name in {} */
    {
	tokenPush(&tp->sequence, listToken(file));
	return tp;
    }

    tb = newTB(bufSize);
    while (isalnum(ch) || ch == '_')	/* alphanumeric name */
    {
        addTB(tb, ch);
  	ch = chGet(file);
    }
    capTB(tb);
    if ((leng = tb->length) == 0)
    {
	push(file, ch);
	freeToken(tp);
	freeTB(tb);
	return (Token *) 0;
    }
    tokenPush(&tp->sequence, makeToken(VNAME, tb));
    if (ch == '(')
    {
	if (leng == 0) { fail((Token *) 0, "Variable error"); }
	tokenPush(&tp->sequence, stringToken(file, ')', ARRAY));
    }
    else
    {
	push(file, ch);
    }
    return tp;
}

static Token *concToken(char fst, char lst, Input *file)
{
    Token *tp = newToken(CONC), *vp;
    int bufSize = 512;
    TBuff *tb = newTB(bufSize);
    char ch;

    ch = fst;
    for(;;)
    {
	if (ch == lst) { goto done; }
	switch (ch)
	{
	case ';' :
	case ' ' :
	case '\t' :
	case '\n' :
	case '\0' :
	    goto done;
	case ']' :
	    if (file->tcall) { goto done; }
	    break;
	case '\\' :
	    addTB(tb, ch);
	    ch = chGet(file);
	    break;
	case '[' :
	    if (tb->length != 0)
	    {
		capTB(tb);
		tokenPush(&tp->sequence, makeToken(CPART, tb));
		tb = newTB(bufSize);
	    }
	    tokenPush(&tp->sequence, callToken(file));
	    ch = chGet(file);
	    continue;
	case '$' :
	    if ((vp = varToken(file)) != (Token *) 0)
	    {
		if (tb->length != 0)
		{
		    capTB(tb);
		    tokenPush(&tp->sequence, makeToken(CPART, tb));
		    tb = newTB(bufSize);
		}
		tokenPush(&tp->sequence, vp);
		ch = chGet(file);
		continue;
	    }
	}
	addTB(tb, ch);
	ch = chGet(file);
    }
done :
    push(file, ch);
    if (tb->length != 0)
    {
        capTB(tb);
	if (tp->sequence == (Token *) 0)
	{
	    freeToken(tp);
	    return makeToken(CONST, tb);
	}
        tokenPush(&tp->sequence, makeToken(CPART, tb));
    }
    else
    {
	freeTB(tb);
    }
    return tp;
}

int handle(Token *line)
{
    Token *hd;

    extern int tclop(Token*, Token*);
    extern void comment(Token *);

    if (line == (Token *) 0)
    {
	if (!minimise) { blank(); }
	return 1;
    }
    hd = tokenPop(&line);
    switch (hd->type)
    {
    case ENDF :
        freeToken(hd);
        return 0;
    case CONST :
	if (tclop(hd, line)) { break; }
    case VAR :
    case CALL :
    case CONC :
	makeCall(hd, line);
	break;
    case SCOMMENT :
    case COMMENT :
	comment(hd);
	break;
    case SEMI :
    case NL:
	blank();
	break;
    default :
	fail(hd, "Invalid tcl program");
	/*NOT REACHED*/
    }
    freeToken(hd);
    freeToken(line);
    return 1;
}

static Token *getComment(Input *file, TBuff *tb)
{
    char ch;

    for(;;)
    {
	switch (ch = chCGet(file))
	{
	case '\n' :
	case '\0' :
	    push(file, ch);
	    capTB(tb);
	    return makeToken(COMMENT, tb);

	case '\\' :
	    addTB(tb, '\\');
	    ch = chCGet(file);
	default:
	    addTB(tb, ch);
	}
    }
}

Token *collect(Input *file)
{
    Token *line = (Token *) 0, *tp;
    TBuff *tb;
    for(;;)
    {
	tp = getToken(file);
	switch (tp->type)
	{
	case SEMI:
	case NL :
	case ENDF :
	    if (line == (Token *) 0) { return tp; }
	    freeToken(tp);
	    return line;
	case CONST :
	    if (line == (Token *) 0 && tp->text[0] == '#')
	    {
	        tb = newTB(128);
		catTB(tb, tp->text);
		freeToken(tp);
		tp = getComment(file, tb);
	    }
	    break;
	case CONC :
	    if (line == (Token *) 0 && tp->sequence->type == CPART &&
		tp->sequence->text[0] == '#')
	    {
		tb = newTB(128);
		makeTB(tp, tb);
		freeToken(tp);
		tp = getComment(file, tb);
	    }
	    break;
	case SP :
	    freeToken(tp);
	    continue;
	default :
	    break;
	}
	tokenPush(&line, tp);
    }
}

static void sconv(Token **conc, Token **line)
{
    Token *tp;
    if (*conc != (Token *) 0)
    {
	tp = (*conc)->sequence;
	if (tp->type == CPART && tp->next == (Token *) 0)
	{
	    tp->type = CONST;
	    (*conc)->sequence = (Token *) 0;
	    freeToken(*conc);
	    *conc = tp;
	}
	tokenPush(line, *conc);
	*conc = (Token *) 0;
    }
}

static Token *listconv(Token **lst)
{
    Token *tp = newToken(SLIST);

    freeToken(tokenPop(lst));	/* remove the LBRACE token */

    for(;;)
    {
	switch ((*lst)->type)
	{
	case ENDF :
	    fail((Token *) 0, "String body too complex");
	    /*NOT REACHED*/
	case RBRACE :
	    freeToken(tokenPop(lst));
	    return tp;
	case SPART :
	    (*lst)->type = CPART;
	default :
	    tokenPush(&tp->sequence, tokenPop(lst));
	}
    }
}

void sprocess(Token *lst, int nls)
{
    Token *line = (Token *) 0, *conc = (Token *) 0;
    while (lst != (Token *) 0)
    {
	switch (lst->type)
	{
	case LBRACE :
	    sconv(&conc, &line);
	    tokenPush(&line, listconv(&lst));
	    continue;
	case SPART :
	    lst->type = CPART;
	    break;
	case SEMI:
	case NL :
	    sconv(&conc, &line);
	    if (nls || line != (Token *) 0) { handle(line); }
	    line = (Token *) 0;
	    freeToken(tokenPop(&lst));
	    continue;
	case SP :
	    sconv(&conc, &line);
	    freeToken(tokenPop(&lst));
	    continue;
	default :
	    break;
	}
	if (conc == (Token *) 0) { conc = newToken(CONC); }
	tokenPush(&conc->sequence, tokenPop(&lst));
    }
    sconv(&conc, &line);
    if (line != (Token *) 0) { handle(line); }
}

void lprocess(Token *lst, int nls)
{
    Token *line = (Token *) 0;

    while (lst != (Token *) 0)
    {
	switch (lst->type)
	{
	case ENDF :
	    freeToken(lst);
	    lst = (Token *) 0;
	    break;

	case SEMI :
	case NL :
	    if (nls || line != (Token *) 0) { handle(line); }
	    line = (Token *) 0;
	case SP :
	    freeToken(tokenPop(&lst));
	    break;

	default :
	    tokenPush(&line, tokenPop(&lst));
	    break;
	}
    }
    if (line != (Token *) 0) { handle(line); }
}

Token *accumulate(Input *file, int nl)
{
    Token *line = (Token *) 0, *hd;
    TokenType last = NL;
    TBuff *tb;
    for(;;)
    {
	hd = getToken(file);
	switch (hd->type)
	{
	case ENDF :
	    if (line == (Token *) 0)
	    {
	        line = hd;
	    }
	    else
	    {
	        freeToken(hd);
	    }
	    return line;
	case SEMI:
	case NL :
	    if (!nl) { freeToken(hd) ; continue; }
	    break;
	case CONST :
	    if ((last == NL || last == SEMI) && hd->text[0] == '#')
	    {
		tb = newTB(128);
		catTB(tb, hd->text);
		freeToken(hd);
		hd = getComment(file, tb);
		if (last == SEMI)
		{
		    hd->type = SCOMMENT;
		}
	    }
	    break;
	case CONC :
	    if ((last == NL  || last == SEMI) && hd->sequence->type == CPART
		&& hd->sequence->text[0] == '#')
	    {
		tb = newTB(128);
		makeTB(hd, tb);
		freeToken(hd);
		hd = getComment(file, tb);
		if (last == SEMI)
		{
		    hd->type = SCOMMENT;
		}
	    }
	    break;
	case SP :
	    freeToken(hd);
	    continue;
	default :
	    break;
	}
	tokenPush(&line, hd);
	last = hd->type;
    }
}
