#--------------------dv-Diagrammdarstellungsroutinen-------------------------

#-----------------------------------------------------------------------------
# drawundefmarks {} zeichnet Marken fuer undefinierte Stellen einer Kurve
#                   in ein Diagramm ein 
#                   path : Widgetpfad des Diagramms
#                   curve: Index der Kurve
#-----------------------------------------------------------------------------
#
proc drawundefmarks {path curve} {

   # Diagrammdatensatz sichtbar machen
   #
   global diagdata

   if {$diagdata($path.undefmarks) == "on"} {
   # undefinierte Marken sollen geteilt werden =>
   # Durchmesser der Marken fuer undefinierte Stellen berechnen
   #
   set l [expr [min $diagdata($path.sizex) $diagdata($path.sizey)] / 100]
 
      # Alle Marken fuer undefinierte Stellen dieser Kurve als Striche zeichnen 
      #
      set i 0
      foreach mark $diagdata($path.undefmarks$curve) {
        if {[expr $i%2] == 0} {
            set x $mark
        } {
           if {$x >= $diagdata($path.originx) && 
               $x <= $diagdata($path.xaxis)} {
   
               # Marke liegt im Darstellungsbereich =>
               # Koordinaten fuer Kreismarke berechnen 
               #
               set x1 [expr double($x)-$l]
               set y1 [expr double($mark)-2.0*$l]
               set x2 [expr double($x)+0.5*$l]
               set y2 [expr double($mark)-0.5*$l]
            
               # Marke zeichnen
               #
               $path.report.surface create oval $x1 $y1 $x2 $y2
           }
        }
        incr i
      }
   }
}

#-----------------------------------------------------------------------------
# drawpoints {} Hilfsroutine, die eine Liste von Koordinaten als Punkte
#               (genauer: Kreuze)  in ein Diagramm einzeichnet
#               path  : Pfad des Diagramms
#               coords: Liste von Punkten
#               color : Farbe der Punkte
#----------------------------------------------------------------------------
#
proc drawpoints {path coords color} { 

  # Diagrammvariablen sichtbar machen
  #
  global diagdata 

  # Groesse der Kreuze berechnen
  #
  set s [expr [min $diagdata($path.sizex) $diagdata($path.sizey)] / 100]

  # Kreuze zeichnen
  #
  set i 0
  foreach val $coords {
      if {[expr $i%2] == 0} {
          set x $val
          } {
          # Kreuz zeichnen
          #
          set x1 [expr $x-$s]
          set x2 [expr $x+$s]
          set y1 [expr $val-$s]
          set y2 [expr $val+$s] 
          $path.report.surface.clipbox create line $x1 $y1 $x2 $y2 -fill $color
          $path.report.surface.clipbox create line $x2 $y1 $x1 $y2 -fill $color
          }
      incr i
  }
}

#-----------------------------------------------------------------------------
# drawcurve {} zeichnet eine schon scalierte Kurve in ein Diagramm in den 
#              gewaehlten Representationsarten ein
#              path : Widgetpfad des Diagramms
#              col  : Index der zu zeichnenden Kurve
#              color: Farbe der zu zeichnenden Kurve
#-----------------------------------------------------------------------------
#
proc drawcurve {path col color} {

  # Diagrammvariablen sichtbar machen
  #
  global diagdata

  # alle Kurvenstuecke ausgeben
  #
  foreach coords $diagdata($path.colscaled$col) {
    if {$diagdata($path.points) == "on"} {

        # Representationsart "points" eingeschaltet =>
        # Kurven als Punkte ausgeben
        #
        drawpoints $path "$coords" $color
    }
    if {[llength $coords] >= 4 } {

       # mindestens zwei Punkte in der Liste =>
       # Kurve eventuell als Polygon oder Bspline darstellen
       #
       if {$diagdata($path.polygon) == "on"} {

           # Representationsart "polygon" eingeschaltet=>
           # Kurve als Polygonzug ausgeben
           #
           eval $path.report.surface.clipbox create line $coords -fill $color
       }
       if {$diagdata($path.bspline) == "on"} {

           # Representationsart "bspline" eingeschaltet =>
           # Kurve als Bspline ausgeben
           #
           eval $path.report.surface.clipbox create line $coords -smooth true \
                   -fill $color
       }
    } ; # if mindestens zwei Punkte
  } ; # foreach
}

#------------------------------------------------------------------------------
# drawindex {} zeichnet Kurven im Modus "domainindex" in ein Diagramm ein
#              und setzt die Titel in die Legende
#              path : Pfad des zugehoerigen Diagramms
#------------------------------------------------------------------------------
#
proc drawindex {path} {
 
  # Diagrammvariablen sichtbar machen
  #
  global diagdata
 
  # Spalten, die in der Kurvenliste stehen ausgeben
  #
  foreach col $diagdata($path.curvelist) {

     # ermittle Kurvenfarbe
     #
     set color [getcolor $path $col]
 
     # Kurve skalieren
     #
     scalecurveindex $path $col

     # alle Kurvenstuecke ausgeben
     #
     drawcurve $path $col $color 
  
     # alle Marken fuer undefinierte Werte ausgeben
     #
     drawundefmarks $path $col
  
     # trage Legendenstring mit zugehoeriger Farbe in Legende ein
     #
     label $path.report.legend.titles.$diagdata($path.curvenum) \
           -text $diagdata($path.coltitle$col) -foreground $color \
           -font $diagdata($path.scalefont)
     pack $path.report.legend.titles.$diagdata($path.curvenum) -expand true
   
     # erhoehe aktuelle Kurvenanzahl
     #
     incr diagdata($path.curvenum)
     update
  }
}

#-----------------------------------------------------------------------------
# drawreal {} zeichnet Kurven im Modus "domainreal" in ein Diagramm ein und
#             traegt die zuegehoerigen Titel in die Legende ein
#             path : Pfad des zugehoerigen Diagramms
#-----------------------------------------------------------------------------
#
proc drawreal {path} {

   # Diagrammvariablen sichtbar machen
   #
   global diagdata
 
   # Spalten in der Kurvenliste als Kurven ausgeben
   #
   set i 0
   foreach col $diagdata($path.curvelist) {

       if {[expr $i%2] == 0} {
          # Urbildspalte sichern
          #
          set cold $col
       } {
          # Bildspalte sichern
          #
          set colc $col

          # Kurvenfarbe ermitteln
          #
          set color [getcolor2 $path $cold $colc]

          # Kurve skalieren
          #
          scalecurvereal $path $cold $colc

          # alle Kurvenstuecke ausgeben
          #
          drawcurve $path $cold $color
       
          # alle Marken fuer undefinierte Werte ausgeben
          #
          drawundefmarks $path $cold
       
          # trage Legendenstring mit zugehoeriger Farbe in Legende ein
          #
          label $path.report.legend.titles.$diagdata($path.curvenum) -text\
              "$diagdata($path.coltitle$cold)->$diagdata($path.coltitle$col)"\
               -foreground $color -font $diagdata($path.scalefont)
          pack $path.report.legend.titles.$diagdata($path.curvenum) -expand true
       
          # erhoehe aktuelle Kurvenanzahl
          #
          incr diagdata($path.curvenum)
          update
       }
       incr i
   }
} 

#-----------------------------------------------------------------------------
# scalesxindex {} gibt den Skalenwert (/ string) fuer den Modus
#                     "domaininteger" zurueck
#                     path : Pfad des Diagramms
#                     i    : aktuelle Skalierung
#                     return Skalenwert (/string)
#-----------------------------------------------------------------------------
#
proc scalesxindex {path i} {
 
  # Diagrammvariablen sichtbar machen und umschreiben
  #
  global diagdata
 
  if {$diagdata($path.scxcol) == "on"} {
 
     # Option "Skalierung aus Spalte" eingeschaltet =>
      
     # zugehoerige Spalte ermitteln
     #
     set index $diagdata($path.scxcolind)

     # aus Spalte zugehoeriges Feld ermitteln
     #
     set out [lindex $diagdata($path.col$index) [int $i]]
  } {
     # Option "Skalierung aus Spalte" ausgeschaltet =>
     
     # aktuelle Skalierung in eckigen klammern als ganze Zahl ausgeben 
     #
     set out "\[[int $i]\]"

  } ; # if
  return $out
}

#-----------------------------------------------------------------------------
# scalesxreal {} gibt den Skalenwert fuer den Modus
#                     "domaininreal" zurueck
#                     path : Pfad des Diagramms
#                     i    : aktuelle Skalierung
#                     return Skalenwert
#-----------------------------------------------------------------------------
#
proc scalesxreal {path i} {

  # Skalierungswert auf drei Stellen hinter dem Komma formatieren
  #
  set out [format "%.3lf" $i] 
  return $out
}

#-----------------------------------------------------------------------------
# scalesxdate {} gibt den Skalenwert fuer den Modus "domaindate" zurueck;
#                kovertiert eine Unixzeit in das Format MM:DD:YY
#                (siehe auch tabdate2real)
#                path : Pfad des Diagramms
#                i    : aktuelle Skalierung (wird als Unixzeit ausgewertet)
#                return Skalenwert
#-----------------------------------------------------------------------------
#
proc scalesxdate {path i} {

  # Wert in Integer umwandeln
  #
  set unixtime [expr round($i)]

  # Datumsstring berechnen 
  #
  set time [fmtclock $unixtime "%d:%m:%y" GMT]
  return $time
}

#-----------------------------------------------------------------------------
# scalesxtime {} gibt den Skalenwert fuer den Modus "domaintime" zurueck;
#                konvertiert eine Unixzeit in das Format HH:MM
#                (siehe auch tabtime2real)
#                path : Pfad des Diagramms
#                i    : aktuelle Skalierung (wird als Unixzeit ausgewertet)
#                return Skalenwert
#-----------------------------------------------------------------------------
#
proc scalesxtime {path i} {

  # Wert in zugehoerigen Zeitwert konvertieren
  #
  set unixtime [expr round($i)]
  set time [fmtclock $unixtime "%H:%M" GMT]

  # 24.00 Uhr wurde als 00:00 des naechsen Tages uebersetzt (coltime2real);
  # deshalb hier 00:00 auf 24:00 setzen
  #
  if {$time == "00:00"} {
     set time "24:00"
  }

  return $time 
}
 
# ----------------------------------------------------------------------------
# drawdiag {} zeichnet das Diagramm relativ zur gesetzten Zeichenflaeche und
#             schreibt den Titel in das Grafikfenster;
#             path : identifiziert das zugehoerige Diagramm
# ----------------------------------------------------------------------------
#
proc drawdiag {path} {

   # Diagrammvariablen sichtbar machen und umschreiben
   #
   global diagdata
   set fromx [double $diagdata($path.fromx)]
   set tox   [double $diagdata($path.tox)]
   set stepx [double $diagdata($path.stepx)]
   set fromy [double $diagdata($path.fromy)]
   set toy   [double $diagdata($path.toy)]
   set stepy [double $diagdata($path.stepy)]
   set x $diagdata($path.sizex)
   set y $diagdata($path.sizey)

   # bestimme zu verwendene Fontgroessen und Verschubweite (Abstand der
   # Skalierungen von der Yachse)
   #
   set minimum [min $x $y]
   if {$minimum > $diagdata($path.fontbig)} {
       set font1 $diagdata($path.scalefont2)
       set font2 $diagdata($path.titlefont2)
       set dist $diagdata($path.scaledist2) 
   } {

      if {$minimum < $diagdata($path.fontsmall) } {
          set font1 $diagdata($path.scalefont0) 
          set font2 $diagdata($path.titlefont0) 
          set dist $diagdata($path.scaledist0) 
      } {
          set font1 $diagdata($path.scalefont1) 
          set font2 $diagdata($path.titlefont1) 
          set dist $diagdata($path.scaledist1) 
      }
   } 
   
   # Skalierungsfonts und Distanzmasszahl speichern
   #
   set diagdata($path.scalefont) $font1
   set diagdata($path.titlefont) $font2
   set diagdata($path.scaledist) $dist

   # berechne absolute Koordinaten fuer das Diagramm 
   #
   set originx [expr 10*$dist]
   set yaxis   [expr 10*$dist] 
   set originy [expr $y-$yaxis]
   set xaxis   [expr $x-$originx]

   # sichere Koordinaten fuer das Diagramm
   #
   set diagdata($path.xaxis)   $xaxis
   set diagdata($path.yaxis)   $yaxis
   set diagdata($path.originx) $originx
   set diagdata($path.originy) $originy

   # setze clipping-box (Abmessung vom Diagramm)
   #
   set height [expr $originy-$yaxis] 
   set width  [expr $xaxis-$originx]
   place $path.report.surface.clipbox -anchor sw -x $originx -y $originy\
         -width  $width -height $height   

   # setze DiagrammKoordinaten innerhalb der Clipping-box 
   #
   set cliporiginx 1
   set cliporiginy [expr $height-1]
   set clipxaxis   [expr $width-1 ] 
   set clipyaxis   1

   # speichere relative Diagrammkoordinaten
   #
   set diagdata($path.cliporiginx)  $cliporiginx
   set diagdata($path.cliporiginy)  $cliporiginy
   set diagdata($path.clipxaxis)    $clipxaxis
   set diagdata($path.clipyaxis)    $clipyaxis

   # setze Titel ins Grafikfenster
   #
   $path.report.surface create text [expr $x/2] [expr 3*$dist]\
        -text $diagdata($path.title) -font $font2

   # setze Untertitel ins Grafikfenster
   #
   $path.report.surface create text [expr $x/2] [expr 6*$dist]\
        -text $diagdata($path.subtitle) -font $font1

   # setze Xachsenbezeichnung ins Fenster
   #
   $path.report.surface create text \
         [expr $xaxis+8*$dist] [expr 7*$dist+$originy] -anchor e \
        -text $diagdata($path.labelx) -font $font1

   # setze Yachsenbezeichnung ins Fenster
   #
   $path.report.surface create text \
         [expr $originx-8*$dist] [expr $yaxis-3*$dist] -anchor w \
        -text $diagdata($path.labely) -font $font1

   # zeichne Koordinatenkreuz; der Ursprung des Diagramms liegt unten links,
   #                           der Ursprung des canvas-widget oben links...
   #
   $path.report.surface.clipbox create line [expr $cliporiginx-1] $cliporiginy\
        $clipxaxis   $cliporiginy  -width 2 

   # Anmerkung: cliporiginx-1 wegen -width 2, da sonst ein kleines Dreieck
   #            im Ursprung nicht gezeichnet wuerde

   $path.report.surface.clipbox create line $cliporiginx $cliporiginy\
        $cliporiginx $clipyaxis    -width 2


   # schreibe Skalenanzeigen und zeichne Aufteilungen fuer xachse
   #
   if {$stepx != 0 && [expr $fromx+$stepx] != $fromx} {
 
     # stepx !=0 => Skalenanzeigen werden gezeichnet
     #
     set lauf 1
     if {$diagdata($path.domain) == "index"} {
        set toxdeltha $tox
     } {
        set toxdeltha [expr $tox+($stepx/2)]
     }
     for {set i $fromx} {$i <= $toxdeltha } {set i [expr $i+$stepx]} {

        # Skaliere xw so, dass 0<= xw <= 1
        #
        set xw [expr ([double $i]-$fromx)/($tox-$fromx)]
 
        # skaliere xw so, dass cliporiginx <= xw <= clipxaxis
        #
        set xw [expr $xw*($clipxaxis-$cliporiginx)+$cliporiginx]
 
        # zeichne Linie, falls Grid Option gesetzt
        #
        if {$diagdata($path.grid) == "on"} {
            $path.report.surface.clipbox create line $xw $cliporiginy\
            $xw $clipyaxis
        }

        # Skalenanzeigen je nach gewaehlter Urbildmenge setzen
        #
        set out [eval $diagdata($path.scalesx) $path $i] 
 
        # versetzt ausgeben
        #
        set ypos [expr $originy+$lauf%2*2*$dist]
        incr lauf
        $path.report.surface create text [expr $xw+$originx] [expr $ypos+$dist]\
              -font $font1 -anchor n -text $out
      } ;# loop
  } ; # if

   # bei den folgenden Berechnungen werden die gegensaetzlichen Lagen der
   # Urspruenge beruecksichtigt :
   # schreibe Skalenanzeigen und zeichne Aufteilungen fuer yachse
   #
   if {$stepy != 0 && [expr $fromy+$stepy] != $fromy} {
      # stepy != 0 => Skalenanzeigen werden gezeichnet
      #
      set toydeltha [expr $toy+($stepy/2)]
      for {set i $fromy} {$i <= $toydeltha} {set i [expr $i+$stepy]} {
 
          # Skaliere yw so, dass 0<= yw <= 1
          #
          set yw [expr ($i-$toy)/($fromy-$toy)]
 
          # skaliere yw so, dass clipyaxis <= yw <= cliporiginy
          #
          set yw [expr $yw*($cliporiginy-$clipyaxis)+$clipyaxis]
 
          # zeichne Linie, falls Grid Option gesetzt
          #
          if {$diagdata($path.grid) == "on"} {
             $path.report.surface.clipbox create line $cliporiginx $yw\
                $clipxaxis $yw
          }
 
          # gebe Skalenwert formatiert aus
          # maximal drei Stellen hinter dem Komma
          #
          set out [format "%.3lf" $i]       

          # Zahl ausgeben
          # 
          $path.report.surface create text [expr $originx-$dist]\
                [expr  $yw+$yaxis] -font $font1 -anchor e -text $out 
      }
   } 
}

#----------------------------------------------------------------------------
# resizediagram {} passt die widgets eines Diagramms an die veraenderte
#                  Groesse des Diagramm-toplevels an.
#                  path : widgetpfad des aktuellen Diagramms
#                  xnew : neue Toplevelbreite
#                  ynew : neue Toplevelhoehe
#----------------------------------------------------------------------------
#
proc resizediagram {path xnew ynew} {

  # Diagrammvariablen sichtbar machen
  #
  global diagdata

  if {$xnew != $diagdata($path.winsizex) ||\
      $ynew != $diagdata($path.winsizey)} {
        # Abmessungen vom Diagramm sind geaendert => Canvas anpassen 
        #

        # Auf Groessenaenderung innerhalb der resizeroutine nicht reagieren
        #
        bind $path <Configure> ""

        # Dieses Toplevel waehrend Resize fixieren 
        #
        wm geometry $path [set xnew]x$ynew

        # passende Groesse des Canvas-widgets ermitteln 
        #
        set x [expr $xnew-$diagdata($path.winsizex)+$diagdata($path.sizex)]
        set y [expr $ynew-$diagdata($path.winsizey)+$diagdata($path.sizey)] 

        # x und y werte aktualisieren
        #
        set diagdata($path.sizex) $x
        set diagdata($path.sizey) $y

        # Refresh durchfuehren
        #
        busy {refreshwindows $path .}
   } ; # if
}

#----------------------------------------------------------------------------
# showzoomdiag {} erstellt ein Zoomdiagramm
#                 cp : Pfad des ausloesenden Canvaswidgets
#----------------------------------------------------------------------------
#
proc showzoomdiag {cp} {

   # Status und Diagrammvariablen sichtbar machen
   #
   global status diagdata

   # WidgetPfad des Ausloesers feststellen
   #
   set path .[lindex [translit . { } $cp] 0]

   # Diagrammabmessungen umschreiben
   #
   set cliporiginx [double $diagdata($path.cliporiginx)]
   set cliporiginy [double $diagdata($path.cliporiginy)]
   set clipxaxis   [double $diagdata($path.clipxaxis)]
   set clipyaxis   [double $diagdata($path.clipyaxis)]
   set fromx       [double $diagdata($path.fromx)]
   set tox         [double $diagdata($path.tox)]
   set fromy       [double $diagdata($path.fromy)]
   set toy         [double $diagdata($path.toy)]

   # umschriebenen Auschnitt des Zoomrahmens ermitteln 
   # dabei gegensaetzliche Urspruenge von Canvas und Diagramm beruecksichtigen
   #
   set x1 [double [min $status(zoomx1) $status(zoomx2)]]
   set x2 [double [max $status(zoomx1) $status(zoomx2)]]
   set y1 [double [max $status(zoomy1) $status(zoomy2)]]
   set y2 [double [min $status(zoomy1) $status(zoomy2)]]

   if { $x1 != $x2 && $y1 != $y2} {
      # es wurde ein Ausschnitt ausgewaehlt=> 
      # Koordinaten des Zoomrahmens auf Bild bzw Urbildbereich zurueckscalieren,
      # dabei gegensaetzliche Ursprungs Lagen von Canvas und Diagramm
      # beruecksichtigen
      # 
      set x1 [expr ($x1-$cliporiginx)/($clipxaxis-$cliporiginx)]
      set x2 [expr ($x2-$cliporiginx)/($clipxaxis-$cliporiginx)]
      set y1 [expr ($cliporiginy-$y1)/($cliporiginy-$clipyaxis)]
      set y2 [expr ($cliporiginy-$y2)/($cliporiginy-$clipyaxis)] 
   
      set x1 [expr ($x1*($tox-$fromx)+$fromx)]
      set x2 [expr ($x2*($tox-$fromx)+$fromx)]
      set y1 [expr ($y1*($toy-$fromy)+$fromy)]
      set y2 [expr ($y2*($toy-$fromy)+$fromy)]
   
      # stepx berechnen : so, dass #gridx+1 Skalierungen gezeichnet werden
      #
      set stepx [expr ($x2-$x1)/double($diagdata($path.gridx))]
   
      # stepy berechnen : so, dass #gridy+1 Skalierungen gezeichnet werden
      #
      set stepy  [expr ($y2-$y1)/double($diagdata($path.gridy))]
   
      if {![winfo exists .zoomdiag]} { 
         # das Zoomdiagramm existiert noch nicht => 
         # Diagrammdatensatz in Zoomdatensatz kopieren
         #
         diagdatacopy $path .zoomdiag 
      } 
 
      if {$diagdata($path.domain) == "index"} {
   
         # Programm ist im Modus : domainint
         # => x1,x2 und stepx entsprechend runden
         #
         set x1 [int $x1]
         set x2 [int [ceil $x2]]
         set stepx [int [ceil $stepx]]
      }
   
      if {[expr $x2-$x1 > 0.001] && [expr $y2-$y1 > 0.001]} {
         # Vergroesserungsfaktor noch gross genug => 
         # Zoomdatensatz anpassen
         #
         set diagdata(.zoomdiag.frametitle) "Zoom"
         set diagdata(.zoomdiag.autoscale)  "off"
         set diagdata(.zoomdiag.fromy)       $y1
         set diagdata(.zoomdiag.toy)         $y2
         set diagdata(.zoomdiag.stepy)       $stepy  
         set diagdata(.zoomdiag.fromx)       $x1
         set diagdata(.zoomdiag.tox)         $x2
         set diagdata(.zoomdiag.stepx)       $stepx
      
         # Refresh fuer .zoomdiag-Menues ausloesen
         #
         refreshwindows .zoomdiag .
 
         if {![winfo exists .zoomdiag]} {
            # Zoomdiagramm existiert nicht => ZoomDiagramm erzeugen
            #
            showdiagram .zoomdiag
         }  
      } ; # Vergroesserungsfaktor gross genug
  }; # Ausschnitt gewaehlt
} 

#----------------------------------------------------------------------------
# initzoom {} realisiert die bindings zum oeffnen eines Zoomdiagramms;
#             Folgender Ablauf wird realisiert
#             -Maustaste 1 druecken: setzen von Punkt1 des Zoomrahmens
#             -Maustaste 1 ziehen  : Rahmen mit Diagonal-Eckpunkten 
#                                    (Punkt1,aktuelle Mausposition) darstellen 
#             -Maustaste2  oder ESCAPE druecken:
#                                    loeschen des aktuellen Rahmens
#             -Maustaste1 loslassen: setzen des zweiten Punktes
#                                    und erzeugen des Zoomdiagramms
#             
#             path : widgetpfad des aktuellen canvas
#----------------------------------------------------------------------------
#
proc initzoom {path} {

  proc initzoomframe {path x y} {
 
    # Statusvariablen sichtbar machen
    #
    global status
 
    # alten Focus speichern und Focus auf das Canvas-Fenster
    #
    savefocus
    focus $path 

    # erste Ecke des Zoomrahmens merken
    #
    set status(zoomx1) $x 
    set status(zoomy1) $y

    # "Maustaste1 ziehen" und "Maustaste1 loslassen" aktivieren 
    #
    bind $path <ButtonRelease-1> "fixzoomframe %W %x %y"
    bind $path <B1-Motion> "pullzoomframe %W %x %y"
  }


  proc pullzoomframe {path x y } {

    # Statusvariablen initialisieren
    #
    global status 

    # letzten Rahmen loeschen
    #
    $path delete zoomframe 

    # neuen Rahmen setzen
    #
    $path create rectangle $status(zoomx1) $status(zoomy1) $x $y \
       -tag zoomframe -width 2
  }


  proc fixzoomframe {path x y} {

    # Statusvariablen sichtbar machen
    #
    global status

    # Zoomrahmen loeschen
    #
    $path delete zoomframe 

    # zweiten Punkt des Zoomrahmens merken
    #
    set status(zoomx2) $x
    set status(zoomy2) $y 

    # alten focus wieder setzen
    #
    setoldfocus

    # Zoomdiagramm erzeugen (bzw refreshen)
    #
    busy {showzoomdiag $path}
  }


  proc destroyzoomframe {path x y} {
   
     # Statusvariablen sichtbar machen
     #
     global status

     # "Maustaste1 loslassen" und "Maustaste ziehen" Binding ausschalten
     #
     bind $path <ButtonRelease-1> ""
     bind $path <B1-Motion> "" 

     # Zoomrahmen loeschen
     #
     $path delete zoomframe

     # alten focus setzen
     #
     setoldfocus

   } ; # destroyzoomframe


  # Maustaste1 gedrueckt
  #
  bind $path <Button-1> "initzoomframe %W %x %y"

  # Maustaste2 gedrueckt
  #
  bind $path <Any-Button-2> "destroyzoomframe %W %x %y"

  # Escape gedrueckt
  #
  bind $path <Any-Key-Escape> "destroyzoomframe %W %x %y"

} ; # initzoom 

#----------------------------------------------------------------------------
# nozoom {} loescht die Zoom-bindings (macht initzoom wieder rueckgaengig>
#           path : widgetpfad des canvas
#----------------------------------------------------------------------------
#
proc nozoom {path} {

  bind $path <Button-1> ""
  bind $path <B1-Motion> ""
  bind $path <ButtonRelease-1> ""
  bind $path <Any-Button-2> ""
  bind $path <Any-Key-Escape> ""
}

#------------------------------------------------------------------------------
# builddiagram {} Hilfroutine; erzeugt Zeichenflaeche fuer ein Diagramm
#                 path : Pfadname des Diagramms
#------------------------------------------------------------------------------
#
proc builddiagram {path} {

   # erzeuge KontainerWidget fuer Zeichenflaeche und Legende
   #
   frame $path.report 
   pack $path.report -anchor nw 

     # erzeuge Zeichenflaeche
     #
     canvas $path.report.surface -relief raised
     pack $path.report.surface -side left 
   
     # erzeuge innere Zeichenflaeche (Klipping-box Ersatz) 
     #
     canvas $path.report.surface.clipbox
   
     # erzeuge KontainerWidget fuer Legende
     #
     frame $path.report.legend -relief raised
     pack $path.report.legend -fill y -expand true 

     # Zoom-bindings setzen
     #
     initzoom $path.report.surface.clipbox
}
  

#------------------------------------------------------------------------------
# builddiagbuttons {}  Hilfsroutine; erzeugt Schnellknopfleiste fuer ein
#                      Diagramm
#                      path : Pfadname des Diagramms
#------------------------------------------------------------------------------
#  
proc builddiagbuttons {path} {

   # erzeuge frame fuer Schnelltastenleiste
   #
   frame $path.buttons
   pack $path.buttons -fill x

     # erzeuge button fuer Layout-Aufruf
     #
     button $path.buttons.layout -text "layout..."  \
            -command "diagprefacemenu $path\_layout $path"
     pack $path.buttons.layout -expand true -fill x -side left

     # erzeuge Button fuer Kurvenauswahl 
     #
     button $path.buttons.selectdata -text "curves..." -width 10 \
            -command "curvechoose $path"
     pack $path.buttons.selectdata -expand true -fill x -side left

     # erzeuge Button fuer Postscript/Druckerausgabe
     #
     button $path.buttons.output -text "output..." -width 10 \
            -command "diagoutputmenu $path.output $path"
     pack $path.buttons.output -expand true -fill x -side left

     # erzeuge Cancel Button (Betaetigung zerstoert das Diagramm)
     #
     button $path.buttons.cancel -text "Cancel" \
            -command "setoldfocus; destroy $path"
     pack $path.buttons.cancel  -expand true -fill x -side left

   # binde return-Taste an den ok-Knopf 
   #
   bind $path <Return> "$path.buttons.cancel invoke"
}


#---------------------------------------------------------------------------- 
# showdiagram {} stellt ein Diagramm mit seinen Daten dar;
#                "path"  aktueller Datensatz
#----------------------------------------------------------------------------
#
proc showdiagram {path} {
 
# Aufbau des Diagramms:
#
# +----------$path-----------+ 
# |    +-----report------+   |
# |    | surface| legend |   |
# |    +-----------------+   |
# |          buttons         |
# +--------------------------+
#

   # falls toplevel schon existiert, nur Fenster hervorheben
   #
   if {[winfo exist $path]} {
       myraise $path
       return
   }

   # erzeuge eigenes toplevel-fenster
   #
   toplevel $path

   # Bitmap fuer icon setzen
   #
   global env
   wm iconbitmap $path @$env(DV_HOME)/bitmaps/statist.xbm
   wm iconname $path Diagram

   # erzeuge dummy-frame, um grab setzen zu koennen
   #
   frame $path.grab
   pack $path.grab

   # alten focus speichern und focus auf Diagramm
   #
   savefocus
   focus $path 

   # Diagramm zusammenbauen
   #
   builddiagram $path
   builddiagbuttons $path

   # definiere RefreshProzedur
   #
   proc $path.refresh {path dp} {

      # Beginn des geschuetzten Bereichs
      #
      catch {grab $path.grab}

      # waehrend des Refreshs keine Groessenaenderungen zulassen
      #
      wm minsize $path "" ""

      # Auf Groessenaenderung innerhalb der refreshroutine nicht reagieren
      #
      bind $path <Configure> ""

      # Diagrammvariablen sichtbar machen
      #
      global diagdata

      # Rahmentitel setzen
      #
      wm title $path $diagdata($path.frametitle)

      # eventuell vorhandene Canvas-Eintraege loeschen
      #
      $path.report.surface delete all
      $path.report.surface.clipbox delete all

      # eventuell vorhandenen Legenden-Eintraege, sowie Ueberschrift loeschen
      #
      catch {destroy $path.report.legend.titles}
      catch {destroy $path.report.legend.headline}
 
      # Breite und Hoehe der Grafikflaeche setzen
      #
      $path.report.surface config -width  $diagdata($path.sizex)\
                                  -height $diagdata($path.sizey)

      # Autoscalierungen setzen, falls autoscale auf "on"
      #
      if {$diagdata($path.autoscale) == "on"} {
          set auto [eval $diagdata($path.calcautoscale) $path]
          foreach i {fromx tox stepx fromy toy stepy} {
              set diagdata($path.$i) [lindex $auto 0]
              lvarpop auto
          } ; # foreach
      } ; # if
 
      # Diagrammachsen/Skalierungen eintragen
      #
      drawdiag $path

      # erzeuge Ueberschrift fuer die Legende
      #
      label $path.report.legend.headline -width 15 -text "Legende :"\
            -font $diagdata($path.scalefont)\
            -foreground #000000

      pack $path.report.legend.headline 

      # erzeuge KontainderWidget fuer Legendeneintraege
      #
      frame $path.report.legend.titles
      pack $path.report.legend.titles -fill y -expand true

      update

      # Kurvenanzahl sowie Farbe initialisieren 
      #
      set diagdata($path.curvenum) 0 

      # falls Daten definiert; Kurven ins Diagramm je nach Modus eintragen
      #
      if {$diagdata($path.status) == "defined"} {
         eval $diagdata($path.drawcurves) $path
      }

      # Geometrie des Fensters loeschen, damit sich die Legende expandieren
      # kann
      #
      wm geometry $path ""
      update

      # Breite und Hoehe des Diagrammfensters ermitteln und abspeichern
      #
      regexp {([0-9]*)x([0-9]*)} [wm geometry $path] all x y
      set diagdata($path.winsizex) $x
      set diagdata($path.winsizey) $y

      # Minimale Groesse des Diagrammfensters festlegen und damit
      # Groessenveraenderung wieder zulassen
      #
      wm minsize $path 256 172

      # Bindung fuer Groessenaenderung setzen
      #
      bind $path <Configure> "resizediagram $path %w %h"

      # Ende des geschuetzten Bereichs
      #
      grab release $path.grab

   } ; # Refreshprozedur
 
   # zum Initialisieren Refresh einmal Aufrufen
   #
   $path.refresh $path $path
}
