/*
 *=============================================================================
 *                                  tSippShader.c
 *-----------------------------------------------------------------------------
 * Tcl commands to set SIPP shader type and parameters.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippShader.c,v 5.0 1994/09/05 01:23:34 markd Rel $
 *=============================================================================
 */

#include "tSippInt.h"

/*
 * A shader definition.  The surface descriptor is allocated as part of this
 * structure.  The reference count is increment one for a handle being
 * associated with it and once for each surface it is a part of.
 */
typedef struct {
    Shader        *proc;           /* Shader function.                    */
    void          *surfDescPtr;    /* Surface desc.  Follows this struct. */
    int            surfDescSize;   /* Size of desc. Header not included.  */
    /*
     * SURFACE DESCRIPTOR HEADER MUST BE LAST!!!  The descriptor will follow
     * this structure.
     */
    Surf_desc_hdr  surfDescHdr;

} shaderDesc_t, *shaderDesc_pt;

/*
 * Default opacity is completely opaque.
 */
static Color defaultOpacity = {1.0, 1.0, 1.0};

/*
 * Internal prototypes.
 */
static void
SurfaceDescFreeProc _ANSI_ARGS_((Surf_desc_hdr *surfDescHdrPtr));

static shaderDesc_pt
BindShaderToHandle _ANSI_ARGS_((tSippGlob_pt     tSippGlobPtr,
                                Shader          *shaderProc,
                                void            *surfDescPtr,
                                int              surfDescSize));

static void
ShaderHandleFree _ANSI_ARGS_((tSippGlob_pt   tSippGlobPtr,
                              shaderDesc_pt *descEntryPtr));

static void
ShaderHandleCleanup  _ANSI_ARGS_((tSippGlob_pt   tSippGlobPtr));

/*=============================================================================
 * SurfaceDescFreeProc --
 *   Function called to handle a surface reference count going to zero.
 *
 * Parameters:
 *   o surfDescHdrPtr (I) - Pointer to the surface descriptor header.  This is
 *     contained within our shader descriptor, but the client data points
 *     back to top.
 *-----------------------------------------------------------------------------
 */
static void
SurfaceDescFreeProc (surfDescHdrPtr)
     Surf_desc_hdr  *surfDescHdrPtr;
{
    /*
     * Also deletes what surfDescHdrPtr is pointing to.  MUST BE LAST.
     */
    sfree (surfDescHdrPtr->client_data);
}

/*=============================================================================
 * BindShaderToHandle --
 *   Bind a shader and surface description to a handle, setting up the table
 * entry.  A single memory block is allocated to hold the header and surface
 * descriptor.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the Tcl SIPP globals.  The handle is
 *     returned in interp->result.
 *   o shaderProc (I) - A pointer to the shader procedure.
 *   o surfDescPtr (I) - A pointer to the surface descriptor.
 *   o surfDescSize (I) - The size of the surface descriptor.
 * Returns:
 *   A pointer to the surface descriptor entry that was set up.  This is
 * normally ignored, but can be used to assign a different free callback
 * function.
 *-----------------------------------------------------------------------------
 */
static shaderDesc_pt
BindShaderToHandle (tSippGlobPtr, shaderProc, surfDescPtr, surfDescSize)
    tSippGlob_pt     tSippGlobPtr;
    Shader          *shaderProc;
    void            *surfDescPtr;
    int              surfDescSize;
{
    shaderDesc_pt  descPtr;
    shaderDesc_pt *descEntryPtr;

    descPtr = (shaderDesc_pt) smalloc (sizeof (shaderDesc_t) + surfDescSize);

    descPtr->surfDescPtr = PTR_ADD (descPtr, sizeof (shaderDesc_t));
    memcpy (descPtr->surfDescPtr, surfDescPtr, surfDescSize);

    descPtr->proc = shaderProc;
    descPtr->surfDescSize = surfDescSize;

    descPtr->surfDescHdr.ref_count   = 1;
    descPtr->surfDescHdr.free_func   = SurfaceDescFreeProc;
    descPtr->surfDescHdr.client_data = descPtr;

    descEntryPtr = (shaderDesc_pt *)
        Tcl_HandleAlloc (tSippGlobPtr->shaderTblPtr, 
                         tSippGlobPtr->interp->result);
    *descEntryPtr = descPtr;

    return descPtr;
}

/*=============================================================================
 * TSippShaderHandleToPtr --
 *   Utility procedure to convert an shader handle to an shader pointer.
 *   For use by functions outside of this module.
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A shader handle.
 *   o surfDescHdrPtrPtr (O) - A pointer to the surface descritor header is
 *     returned here, the descriptor follows it.
 * Returns:
 *   A pointer to the shader procedure, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
Shader *
TSippShaderHandleToPtr (tSippGlobPtr, handle, surfDescHdrPtrPtr)
    tSippGlob_pt    tSippGlobPtr;
    char           *handle;
    void          **surfDescHdrPtrPtr;
{
    shaderDesc_pt *descEntryPtr;
    shaderDesc_pt  descPtr;

    descEntryPtr = (shaderDesc_pt *)
        Tcl_HandleXlate (tSippGlobPtr->interp, 
                         tSippGlobPtr->shaderTblPtr, handle);
    if (descEntryPtr == NULL)
        return NULL;

    descPtr = *descEntryPtr;
    *surfDescHdrPtrPtr = &descPtr->surfDescHdr;
    return descPtr->proc;

}

/*=============================================================================
 * ShaderHandleFree --
 *    Free a handle associated with a shader and decrementing the reference
 * count on the surface descriptor and possibly cleaning up if it goes to zero.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o descEntryPtr (I) - Entry in the shader handle table.
 *-----------------------------------------------------------------------------
 */
static void
ShaderHandleFree (tSippGlobPtr, descEntryPtr)
    tSippGlob_pt    tSippGlobPtr;
    shaderDesc_pt  *descEntryPtr;
{
    shaderDesc_pt  descPtr = *descEntryPtr;

    Tcl_HandleFree (tSippGlobPtr->shaderTblPtr, descEntryPtr);

    surface_desc_unref (&descPtr->surfDescHdr);
}

/*=============================================================================
 * ShaderHandleCleanup --
 *    Delete all shader handles that are defined. Shaders themselves are not
 * currently freed.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static void
ShaderHandleCleanup (tSippGlobPtr)
    tSippGlob_pt   tSippGlobPtr;
{
    int            walkKey = -1;
    shaderDesc_pt *descEntryPtr;

    while (TRUE) {
        descEntryPtr = Tcl_HandleWalk (tSippGlobPtr->shaderTblPtr, &walkKey);
        if (descEntryPtr == NULL)
            break;

        ShaderHandleFree (tSippGlobPtr, descEntryPtr);
    }

}

/*=============================================================================
 * SippShaderBasic --
 *   Process the basic shader command:
 *     SippShaderBasic ambient specular c3 color [opacity]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShaderBasic (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt tSippGlobPtr = (tSippGlob_pt) clientData;
    Surf_desc    surfDesc;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 5) || (argc > 6)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " ambient specular c3 color [opacity]",
                          (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertFraction (tSippGlobPtr, argv [1], &surfDesc.ambient))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [2], &surfDesc.specular))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [3], &surfDesc.c3))
        return TCL_ERROR;
    if (!TSippConvertColor (tSippGlobPtr, argv [4], &surfDesc.color))
        return TCL_ERROR;
    if (argc == 6) {
        if (!TSippConvertOpacity (tSippGlobPtr, argv [5], &surfDesc.opacity))
            return TCL_ERROR;
    } else
        surfDesc.opacity = defaultOpacity;

    BindShaderToHandle (tSippGlobPtr, basic_shader, 
                        &surfDesc, sizeof (surfDesc));
    return TCL_OK;

}

/*=============================================================================
 * SippShaderPhong --
 *   Process the Phong shader command:
 *     SippShaderPhong ambient diffuse specular spec_exp color [opacity]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShaderPhong (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt tSippGlobPtr = (tSippGlob_pt) clientData;
    Phong_desc   surfDesc;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 6) || (argc > 7)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " ambient diffuse specular spec_exp color [opacity]",
                          (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertFraction (tSippGlobPtr, argv [1], &surfDesc.ambient))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [2], &surfDesc.diffuse))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [3], &surfDesc.specular))
        return TCL_ERROR;
    if (Tcl_GetInt (interp, argv [4], &surfDesc.spec_exp) != TCL_OK)
        return TCL_ERROR;
    if (!TSippConvertColor (tSippGlobPtr, argv [5], &surfDesc.color))
        return TCL_ERROR;
    if (argc == 7) {
        if (!TSippConvertOpacity (tSippGlobPtr, argv [6], &surfDesc.opacity))
            return TCL_ERROR;
    } else
        surfDesc.opacity = defaultOpacity;

    BindShaderToHandle (tSippGlobPtr, phong_shader,
                        &surfDesc, sizeof (surfDesc));
    return TCL_OK;

}

/*=============================================================================
 * SippShaderStrauss --
 *   Process the strauss shader command:
 *     SippShaderStrauss ambient smoothness metalness color [opacity]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShaderStrauss (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt tSippGlobPtr = (tSippGlob_pt) clientData;
    Strauss_desc surfDesc;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 5) || (argc > 6)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " ambient smoothness metalness color [opacity]",
                          (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertFraction (tSippGlobPtr, argv [1], &surfDesc.ambient))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [2], &surfDesc.smoothness))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [3], &surfDesc.metalness))
        return TCL_ERROR;
    if (!TSippConvertColor (tSippGlobPtr, argv [4], &surfDesc.color))
        return TCL_ERROR;
    if (argc == 6) {
        if (!TSippConvertOpacity (tSippGlobPtr, argv [5], &surfDesc.opacity))
            return TCL_ERROR;
    } else
        surfDesc.opacity = defaultOpacity;

    BindShaderToHandle (tSippGlobPtr, strauss_shader, 
                        &surfDesc, sizeof (surfDesc));
    return TCL_OK;

}

/*=============================================================================
 * SippShaderWood --
 *   Process the wood shader command:
 *     SippShaderWood ambient specular c3 scale basecolor ringcolor [opacity]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShaderWood (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt tSippGlobPtr = (tSippGlob_pt) clientData;
    Wood_desc    surfDesc;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 7) || (argc > 8)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " ambient",
                          " specular c3 scale basecolor ringcolor [opacity]",
                          (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertFraction (tSippGlobPtr, argv [1], &surfDesc.ambient))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [2], &surfDesc.specular))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [3], &surfDesc.c3))
        return TCL_ERROR;
    if (Tcl_GetDouble (interp, argv [4], &surfDesc.scale) != TCL_OK)
        return TCL_ERROR;
    if (!TSippConvertColor (tSippGlobPtr, argv [5], &surfDesc.base))
        return TCL_ERROR;
    if (!TSippConvertColor (tSippGlobPtr, argv [6], &surfDesc.ring))
        return TCL_ERROR;
    if (argc == 8) {
        if (!TSippConvertOpacity (tSippGlobPtr, argv [7], &surfDesc.opacity))
            return TCL_ERROR;
    } else
        surfDesc.opacity = defaultOpacity;

    BindShaderToHandle (tSippGlobPtr, wood_shader, 
                        &surfDesc, sizeof (surfDesc));
    return TCL_OK;

}

/*=============================================================================
 * SippShaderMarble --
 *   Process the marble shader command:
 *     SippShaderMarble ambient specular c3 scale basecolor stripcolor
 *                      [opacity]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShaderMarble (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt tSippGlobPtr = (tSippGlob_pt) clientData;
    Marble_desc  surfDesc;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 7) || (argc > 8)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " ambient specular c3 scale basecolor stripcolor",
                          " [opacity]", (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertFraction (tSippGlobPtr, argv [1], &surfDesc.ambient))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [2], &surfDesc.specular))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [3], &surfDesc.c3))
        return TCL_ERROR;
    if (Tcl_GetDouble (interp, argv [4], &surfDesc.scale) != TCL_OK)
        return TCL_ERROR;
    if (!TSippConvertColor (tSippGlobPtr, argv [5], &surfDesc.base))
        return TCL_ERROR;
    if (!TSippConvertColor (tSippGlobPtr, argv [6], &surfDesc.strip))
        return TCL_ERROR;
    if (argc == 8) {
        if (!TSippConvertOpacity (tSippGlobPtr, argv [7], &surfDesc.opacity))
            return TCL_ERROR;
    } else
        surfDesc.opacity = defaultOpacity;

    BindShaderToHandle (tSippGlobPtr, marble_shader, 
                        &surfDesc, sizeof (surfDesc));
    return TCL_OK;

}

/*=============================================================================
 * SippShaderGranite --
 *   Process the granite shader command:
 *     SippShaderGranite ambient specular c3 scale color1 color2 [opacity]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShaderGranite (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Granite_desc  surfDesc;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 7) || (argc > 8)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " ambient specular c3 scale color1 color2 [opacity]",
                          (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertFraction (tSippGlobPtr, argv [1], &surfDesc.ambient))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [2], &surfDesc.specular))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [3], &surfDesc.c3))
        return TCL_ERROR;
    if (Tcl_GetDouble (interp, argv [4], &surfDesc.scale) != TCL_OK)
        return TCL_ERROR;
    if (!TSippConvertColor (tSippGlobPtr, argv [5], &surfDesc.col1))
        return TCL_ERROR;
    if (!TSippConvertColor (tSippGlobPtr, argv [6], &surfDesc.col2))
        return TCL_ERROR;
    if (argc == 8) {
        if (!TSippConvertOpacity (tSippGlobPtr, argv [7], &surfDesc.opacity))
            return TCL_ERROR;
    } else
        surfDesc.opacity = defaultOpacity;

    BindShaderToHandle (tSippGlobPtr, granite_shader, 
                        &surfDesc, sizeof (surfDesc));
    return TCL_OK;

}

/*=============================================================================
 * SippShaderBozo --
 *   Process the bozo shader command:
 *     SippShaderBozo colorlist ambient specular c3 scale [opacity]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShaderBozo (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt tSippGlobPtr = (tSippGlob_pt) clientData;
    Bozo_desc   *surfDescPtr;
    int          descSize;
    char       **colorsArgv;
    int          colorsArgc, idx;
    Color       *colorsPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 6) || (argc > 7)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " colorlist ambient specular c3 scale [opacity]",
                          (char *) NULL);
        return TCL_ERROR;
    }
    /*
     * Convert the color list, the array is part of the allocated surface
     * descriptor.
     */
    if (Tcl_SplitList (tSippGlobPtr->interp, argv [1], &colorsArgc,
                       &colorsArgv) != TCL_OK)
        return TCL_ERROR;

    descSize = sizeof (Bozo_desc) + (colorsArgc * sizeof (Color));
    surfDescPtr = (Bozo_desc *) alloca (descSize);
    colorsPtr = (Color *) (((char *) surfDescPtr) + sizeof (Bozo_desc));
    surfDescPtr->colors = colorsPtr;
    surfDescPtr->no_of_cols = colorsArgc;

    for (idx = 0; idx < colorsArgc; idx++) {
        if (!TSippConvertColor (tSippGlobPtr, colorsArgv [idx], 
                                &colorsPtr [idx])) {
            goto errorExit;
        }
    }
    if (!TSippConvertFraction (tSippGlobPtr, argv [2], &surfDescPtr->ambient))
        goto errorExit;
    if (!TSippConvertFraction (tSippGlobPtr, argv [3], &surfDescPtr->specular))
        goto errorExit;
    if (!TSippConvertFraction (tSippGlobPtr, argv [4], &surfDescPtr->c3))
        goto errorExit;
    if (Tcl_GetDouble (interp, argv [5], &surfDescPtr->scale) != TCL_OK)
        goto errorExit;
    if (argc == 7) {
        if (!TSippConvertOpacity (tSippGlobPtr, argv [6],
                                  &surfDescPtr->opacity))
            return TCL_ERROR;
    } else
        surfDescPtr->opacity = defaultOpacity;
    
    BindShaderToHandle (tSippGlobPtr, bozo_shader, surfDescPtr, descSize);

    sfree (colorsArgv);
    return TCL_OK;

  errorExit:
    sfree (colorsArgv);
    return TCL_ERROR;

}

/*=============================================================================
 * BumpyDescFreeProc --
 *   Function called to handle a surface reference count going to zero.
 *
 * Parameters:
 *   o surfDescHdrPtr (I) - Pointer to the surface descriptor header.  This is
 *     contained within our shader descriptor, but the client data points
 *     back to top.
 *-----------------------------------------------------------------------------
 */
static void
BumpyDescFreeProc (surfDescHdrPtr)
     Surf_desc_hdr  *surfDescHdrPtr;
{
    Bumpy_desc     *surfDescPtr;
    Surf_desc_hdr  *otherHdrPtr;

    /*
     * Get a pointer to  the surface descriptor header for the other surface
     * and decrement its reference count.  If it goes to zero, call its
     * free function.
     */
    surfDescPtr =  PTR_ADD (surfDescHdrPtr, sizeof (Surf_desc_hdr));
    otherHdrPtr = PTR_ADD (surfDescPtr->surface, -sizeof (Surf_desc_hdr));

    surface_desc_unref (otherHdrPtr);

    /*
     * Also deletes what surfDescHdrPtr is pointing to.  MUST BE LAST.
     */
    sfree (surfDescHdrPtr->client_data);

}

/*=============================================================================
 * SippShaderBumpy --
 *   Process the bumpy shader command:
 *     SippShaderBumpy shaderhandle scale [BUMPS] [HOLES]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShaderBumpy (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt   tSippGlobPtr = (tSippGlob_pt) clientData;
    Bumpy_desc     surfDesc;
    int            idx;
    shaderDesc_pt *otherDescPtrPtr, otherDescPtr, shaderDescPtr;
    double         scale;
    bool           bumpflag, holeflag;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 3) || (argc > 5)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " shaderhandle scale [BUMPS] [HOLES]",
                          (char *) NULL);
        return TCL_ERROR;
    }
    otherDescPtrPtr = (shaderDesc_pt *)
        Tcl_HandleXlate (tSippGlobPtr->interp, 
                         tSippGlobPtr->shaderTblPtr, argv [1]);
    if (otherDescPtrPtr == NULL)
        return TCL_ERROR;
    otherDescPtr = *otherDescPtrPtr;

    if (Tcl_GetDouble (interp, argv [2], &scale) != TCL_OK)
        return TCL_ERROR;

    bumpflag = holeflag = FALSE;

    for (idx = 3 ; idx < argc; idx ++) {
        if (STREQU (argv [idx], "BUMPS"))
            bumpflag = TRUE;
        else if (STREQU (argv [idx], "HOLES"))
            holeflag = TRUE;
        else {
            Tcl_AppendResult (tSippGlobPtr->interp, "expected one of `BUMPS'",
                              " or `HOLES', got `", argv [idx], "'",
                              (char *) NULL);
            return TCL_ERROR;
        }
    }
    if ((!bumpflag) && (!holeflag))
        bumpflag = holeflag = TRUE;

    /*
     * Set up the surface descriptor.  The reference count on the other
     * surface descriptor is incremented.  We use a special free procedure
     * for this surface descriptor.
     */
    surfDesc.shader  = otherDescPtr->proc;
    surfDesc.surface = otherDescPtr->surfDescPtr;
    otherDescPtr->surfDescHdr.ref_count++;

    surfDesc.bumpflag = bumpflag; 
    surfDesc.holeflag = holeflag;
    surfDesc.scale    = scale;

    shaderDescPtr = BindShaderToHandle (tSippGlobPtr, bumpy_shader,
                                        &surfDesc, sizeof (Bumpy_desc));
    shaderDescPtr->surfDescHdr.free_func = BumpyDescFreeProc;

    return TCL_OK;

}

/*=============================================================================
 * SippShaderPlanet --
 *   Process the planet shader command:
 *     SippShaderPlanet ambient specular c3 [opacity]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShaderPlanet (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt tSippGlobPtr = (tSippGlob_pt) clientData;
    Surf_desc    surfDesc;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 4) || (argc > 5)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " ambient specular c3 [opacity]", (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertFraction (tSippGlobPtr, argv [1], &surfDesc.ambient))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [2], &surfDesc.specular))
        return TCL_ERROR;
    if (!TSippConvertFraction (tSippGlobPtr, argv [3], &surfDesc.c3))
        return TCL_ERROR;

    surfDesc.color.red = 0.0;  /* Ignored */
    surfDesc.color.grn = 0.0;
    surfDesc.color.blu = 0.0;

    if (argc == 5) {
        if (!TSippConvertOpacity (tSippGlobPtr, argv [4], &surfDesc.opacity))
            return TCL_ERROR;
    } else
        surfDesc.opacity = defaultOpacity;

    BindShaderToHandle (tSippGlobPtr, planet_shader, 
                        &surfDesc, sizeof (surfDesc));
    return TCL_OK;

}

/*=============================================================================
 * SippShaderUnref --
 *   Implements the command:
 *     SippShaderUnref shaderhandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShaderUnref (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    int             idx;
    handleList_t    shaderEntryList;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " shaderlist|ALL", (char *) NULL);
        return TCL_ERROR;
    }                     
    if (STREQU (argv [1], "ALL")) {
        ShaderHandleCleanup (tSippGlobPtr);
        return TCL_OK;
    }

    if (!TSippHandleListConvert (tSippGlobPtr, tSippGlobPtr->shaderTblPtr,
                                 argv [1], NULL, &shaderEntryList,
                                 NULL))
        return TCL_ERROR;

    /*
     * Frees the entry and handle, but not the surface description.
     */
    for (idx = 0; idx < shaderEntryList.len; idx++) {
        ShaderHandleFree (tSippGlobPtr, shaderEntryList.ptr [idx]);
    }

    TSippHandleListFree (&shaderEntryList);
    return TCL_OK;

}

/*=============================================================================
 * TSippShaderInit --
 *   Initialized the shader commands.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the top level global data structure.
 *-----------------------------------------------------------------------------
 */
void
TSippShaderInit (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippShaderBasic",   (Tcl_CmdProc *) SippShaderBasic},
        {"SippShaderPhong",   (Tcl_CmdProc *) SippShaderPhong},
        {"SippShaderStrauss", (Tcl_CmdProc *) SippShaderStrauss},
        {"SippShaderWood",    (Tcl_CmdProc *) SippShaderWood},
        {"SippShaderMarble",  (Tcl_CmdProc *) SippShaderMarble},
        {"SippShaderGranite", (Tcl_CmdProc *) SippShaderGranite},
        {"SippShaderBozo",    (Tcl_CmdProc *) SippShaderBozo},
        {"SippShaderBumpy",   (Tcl_CmdProc *) SippShaderBumpy},
        {"SippShaderPlanet",  (Tcl_CmdProc *) SippShaderPlanet},
        {"SippShaderUnref",   (Tcl_CmdProc *) SippShaderUnref},
        {NULL,                NULL}
    };

    tSippGlobPtr->shaderTblPtr = 
        Tcl_HandleTblInit ("shader", sizeof (shaderDesc_t), 8);

    TSippInitCmds (tSippGlobPtr, cmdTable);

} 

/*=============================================================================
 * TSippShaderCleanUp --
 *   Cleanup the shader table and release all associated resources.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippShaderCleanUp (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    ShaderHandleCleanup (tSippGlobPtr);

    Tcl_HandleTblRelease (tSippGlobPtr->shaderTblPtr);
    tSippGlobPtr->shaderTblPtr = NULL;

}

