/*
 *=============================================================================
 *                                  tSippMisc.c
 *-----------------------------------------------------------------------------
 * Miscellaneous commands.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippMisc.c,v 4.0 1993/11/27 22:11:13 markd Rel $
 *============================================================================
 */

#include "tSippInt.h"
#include "patchlevel.h"

/*
 * Version string containing full version (SIPP version, TSIPP suffix and maybe
 * patchlevel.
 */
char *TSIPP_VERSION;

/*
 * Master directory.  Set here rather than passed directly to Tcl_VarEval to
 * help detect non-string values.
 */
static char  *masterPath = TSIPP_MASTER;


/*
 * Internal prototypes.
 */
static void
ReturnColor _ANSI_ARGS_((tSippGlob_pt   tSippGlobPtr,
                         Color         *colorPtr));

static void
UpdateSignalHandler _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

static void
TSippCleanUp _ANSI_ARGS_((ClientData   clientData,
                          Tcl_Interp  *interp));

/*=============================================================================
 * ReturnColor --
 *   Return a color into the interp->result with no lose of precision.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o colorPtr (I) - Pointer to the color to format.
 *-----------------------------------------------------------------------------
 */
static void
ReturnColor (tSippGlobPtr, colorPtr)
    tSippGlob_pt   tSippGlobPtr;
    Color         *colorPtr;
{
    char  *bufPtr;

    /*
     * Allocate a dynamic buffer only if there is a chance of the result not
     * fitting into the standard Tcl result.
     */
    if (TCL_RESULT_SIZE < (TSIPP_DOUBLE_STR_SIZE * 3)) {
        Tcl_SetResult (tSippGlobPtr->interp,
                       smalloc (TSIPP_DOUBLE_STR_SIZE * 3),
                       TCL_DYNAMIC);
    }

    bufPtr = TSippFormatDouble (colorPtr->red, tSippGlobPtr->interp->result);
    *bufPtr = ' ';
    bufPtr = TSippFormatDouble (colorPtr->grn, bufPtr + 1);
    *bufPtr = ' ';
    TSippFormatDouble (colorPtr->blu, bufPtr + 1);

}

/*=============================================================================
 * UpdateSignalHandler -- 
 *   This routine is called during rendering at the specified update period
 * to handle signals (any other async events) if they have occured.  When
 * linked with Tk, another routine is used that handles events as well as
 * signals.  If an error occurs in signal handling, then it is noted in the
 * tSippGlob and the rendering is aborted.  The error will be reported once
 * the rendering terminates.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static void
UpdateSignalHandler (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    if (tcl_AsyncReady) {
        if (Tcl_AsyncInvoke (tSippGlobPtr->interp, TCL_OK) == TCL_ERROR) {
            if (tSippGlobPtr->delayedError == NULL)
                tSippGlobPtr->delayedError =
                    strdup (tSippGlobPtr->interp->result);
            sipp_render_terminate ();
        }
        Tcl_ResetResult (tSippGlobPtr->interp);
    }

}

/*=============================================================================
 * SippShowBackFaces --
 *   Implements the command:
 *     SippShowBackFaces [flag]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShowBackFaces (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    int           flag;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc > 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " [flag]", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (argc == 1) {
        interp->result = tSippGlobPtr->showBackFaces ? "1" : "0";
        return TCL_OK;
    }

    if (Tcl_GetBoolean (interp, argv [1], &flag) != TCL_OK)
        return TCL_ERROR;
    tSippGlobPtr->showBackFaces = flag;
    sipp_show_backfaces (flag);
    return TCL_OK;

}

/*=============================================================================
 * SippBackground --
 *   Implements the command:
 *     SippBackground [color]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippBackground (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Color         color;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc > 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " [color]", (char *) NULL);
        return TCL_ERROR;
    }
    if (argc == 1) {
        ReturnColor (tSippGlobPtr, &tSippGlobPtr->backgroundColor);
        return TCL_OK;
    }

    if (!TSippConvertColor (tSippGlobPtr, argv [1], &color))
        return TCL_ERROR;
    sipp_background (color.red, color.grn, color.blu);
    tSippGlobPtr->backgroundColor = color;

    return TCL_OK;

}

/*=============================================================================
 * SippLineColor --
 *   Implements the command:
 *     SippLineColor [color]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippLineColor (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Color         color;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc > 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " [color]", (char *) NULL);
        return TCL_ERROR;
    }
    if (argc == 1) {
        ReturnColor (tSippGlobPtr, &tSippGlobPtr->lineColor);
        return TCL_OK;
    }

    if (!TSippConvertColor (tSippGlobPtr, argv [1], &color))
        return TCL_ERROR;
    tSippGlobPtr->lineColor = color;
    return TCL_OK;

}

/*=============================================================================
 * SippShadows --
 *   Implements the command:
 *     SippShadows flag [size]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShadows (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    int           flag;
    unsigned      size;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 2) || (argc > 3)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " flag [size]", (char *) NULL);
        return TCL_ERROR;
    }                     
    if (Tcl_GetBoolean (interp, argv [1], &flag) != TCL_OK)
        return TCL_ERROR;
    if (flag) {
        if (argc != 3) {
            Tcl_AppendResult (interp, "size argument must be specified when ",
                              "enabling shadows", (char *) NULL);
            return TCL_ERROR;
        }
        if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [2], &size))
            return TCL_ERROR;
    } else {
        if (argc != 2) {
            Tcl_AppendResult (interp, "size argument must be omitted when ",
                              "disabling shadows", (char *) NULL);
            return TCL_ERROR;
        }
        size = 0;
    }
    sipp_shadows (flag, size);
    return TCL_OK;

}

/*=============================================================================
 * SippInfo --
 *   Implements the command:
 *     SippInfo attribute
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippInfo (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " attribute", (char *) NULL);
        return TCL_ERROR;
    }                     
    if (STREQU ("VERSION", argv [1])) {
        interp->result = TSIPP_VERSION;
        return TCL_OK;
    }
    if (STREQU ("SIPPVERSION", argv [1])) {
        interp->result = SIPP_VERSION;
        return TCL_OK;
    }
    if (STREQU ("TSIPPVERSION", argv [1])) {
        interp->result = TSIPP_VERSION_SUFFIX;
        return TCL_OK;
    }
    if (STREQU ("TSIPPPATCHLEVEL", argv [1])) {
        sprintf (interp->result, "%d", PATCHLEVEL);
        return TCL_OK;
    }
    if (STREQU ("RLE", argv [1])) {
#ifdef TSIPP_HAVE_RLE
        interp->result = "1";
#else
        interp->result = "0";
#endif
        return TCL_OK;
    }
    if (STREQU ("PHOTO", argv [1])) {
        /*
         * A different update function is used under Tk.
         */
        interp->result =
            (tSippGlobPtr->updateProc == UpdateSignalHandler) ? "0" : "1";
        return TCL_OK;
    }
    if (STREQU ("RENDERING", argv [1])) {
        interp->result = tSippGlobPtr->rendering ? "1" : "0";
        return TCL_OK;
    }

    Tcl_AppendResult (interp, "expected an attribute of \"VERSION\", ",
                      "\"SIPPVERSION\", \"TSIPPVERSION\", ",
                      "\"TSIPPPATCHLEVEL\", \"RLE\", \"RENDERING\", or ",
                      "\"PHOTO\", got \"", argv [1], "\"", (char *) NULL);
    return TCL_ERROR;

}

/*=============================================================================
 * SippAbortRender --
 *   Implements the command:
 *     SippAbortRender
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippAbortRender (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (argc != 1) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], (char *) NULL);
        return TCL_ERROR;
    }                     

    sipp_render_terminate ();
    return TCL_OK;

}

/*=============================================================================
 * TSippInitialize -- 
 *   Initialize the Tcl/Sipp environment.  This also sets the auto_path to
 * point to the master directory.
 *
 * Parameters:
 *   o interp (I) - Pointer to the Tcl interpreter.
 *
 * Returns:
 *   A generic pointer to the tSipp globals.  This value is only useful for
 * future tsipp initialization, like the Tk commands, and is normally ignored.
 * NULL is returned if an error occured.
 *-----------------------------------------------------------------------------
 */
void *
TSippInitialize (interp)
    Tcl_Interp *interp;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippShowBackFaces", (Tcl_CmdProc *) SippShowBackFaces},
        {"SippBackground",    (Tcl_CmdProc *) SippBackground},
        {"SippLineColor",     (Tcl_CmdProc *) SippLineColor},
        {"SippShadows",       (Tcl_CmdProc *) SippShadows},
        {"SippInfo",          (Tcl_CmdProc *) SippInfo},
        {"SippAbortRender",   (Tcl_CmdProc *) SippAbortRender},
        {NULL,                NULL}
    };
    tSippGlob_pt tSippGlobPtr;
    char         patchBuf [10];
    char         dblTestBuf [512];

   /*
    * Validate that the string representation of a double will fix inside
    * of the specified buffer size.
    */
    TSippFormatDouble (MINDOUBLE, dblTestBuf);
    if (strlen (dblTestBuf) > TSIPP_DOUBLE_STR_SIZE - 1)
        panic ("TSIPP_DOUBLE_STR_SIZE defined in tSippInt.h is too %s %d",
               "small to hold a formated double.  Change it to at least",
               strlen (dblTestBuf));

   /*
    * Define strings to be returned by the SippInfo command.  Include patch
    * level if its not zero.
    */

    tclAppName = "tSIPP";
    tclAppLongname = "Tcl-SIPP";
    TSIPP_VERSION = smalloc (strlen (SIPP_VERSION) +
                             strlen (TSIPP_VERSION_SUFFIX) + 10);
    tclAppVersion = TSIPP_VERSION; 
    strcpy (TSIPP_VERSION, SIPP_VERSION);
    strcat (TSIPP_VERSION, TSIPP_VERSION_SUFFIX);

#if PATCHLEVEL
    sprintf (patchBuf, "-P%d", PATCHLEVEL);
    strcat (TSIPP_VERSION, patchBuf);
#endif

    /*
     * Initialize SIPP.  We associate headers with surface descriptors.
     */
    sipp_init ();
    sipp_surface_desc_headers (TRUE);

    /*
     * Set up the Tcl-SIPP global structure.
     */
    tSippGlobPtr = (tSippGlob_pt) smalloc (sizeof (tSippGlob_t));

    tSippGlobPtr->interp        = interp;
    tSippGlobPtr->showBackFaces = FALSE;
    tSippGlobPtr->rendering     = FALSE;
    tSippGlobPtr->tkMainWindow  = NULL;

    /*
     * Set update signal handler.  This is replaced if linked with Tk.
     */
    tSippGlobPtr->updateProc   = UpdateSignalHandler;
    tSippGlobPtr->delayedError = NULL;

    tSippGlobPtr->storageClassPtrs [0] = NULL;

    tSippGlobPtr->backgroundColor.red = 0.0;
    tSippGlobPtr->backgroundColor.grn = 0.0;
    tSippGlobPtr->backgroundColor.blu = 0.0;

    tSippGlobPtr->lineColor.red = 1.0;
    tSippGlobPtr->lineColor.grn = 1.0;
    tSippGlobPtr->lineColor.blu = 1.0;

    TSippInitCmds (tSippGlobPtr, cmdTable);

    TSippBezierInit (tSippGlobPtr);
    TSippCameraInit (tSippGlobPtr);
    TSippGeomInit   (tSippGlobPtr);
    TSippPixMapInit (tSippGlobPtr);
    TSippLightInit  (tSippGlobPtr);
    TSippObjectInit (tSippGlobPtr);
    TSippPBMInit    (tSippGlobPtr);
    TSippPolyInit   (tSippGlobPtr);
    TSippPrimInit   (tSippGlobPtr);
    TSippRenderInit (tSippGlobPtr);
    TSippRLEInit    (tSippGlobPtr);
    TSippShaderInit (tSippGlobPtr);

    Tcl_CallWhenDeleted (interp,
                         TSippCleanUp,
                         (ClientData) tSippGlobPtr);

    if (Tcl_VarEval (interp, "global auto_path; lvarpush auto_path ",
                     masterPath, (char *) NULL) == TCL_ERROR)
        return NULL;

    Tcl_ResetResult (interp);
    return tSippGlobPtr;

}
/*=============================================================================
 * TSipp_Init -- 
 *   Initialize the Tcl/Sipp environment.  This follows the Tcl init function
 * calling conventions.
 *
 * Parameters:
 *   o interp (I) - Pointer to the Tcl interpreter.
 *
 * Returns:
 *   TCL_OK.
 *-----------------------------------------------------------------------------
 */
int
TSipp_Init (interp)
    Tcl_Interp *interp;
{
    TSippInitialize (interp);
    return TCL_OK;
}

/*=============================================================================
 * TSippCleanUp -- 
 *   Called a interpreter deletion to clean up Tcl SIPP and release all
 * resources.
 *
 * Parameters:
 *   o clientData (I) - A pointer to the Tcl SIPP global structure.
 *   o interp (I) - Pointer to the Tcl interpreter.
 *-----------------------------------------------------------------------------
 */
static void
TSippCleanUp (clientData, interp)
    ClientData   clientData;
    Tcl_Interp  *interp;
{
    tSippGlob_pt tSippGlobPtr = (tSippGlob_pt) clientData;

    TSippPolyCleanUp (tSippGlobPtr);

    TSippObjectCleanUp (tSippGlobPtr);

    TSippShaderCleanUp (tSippGlobPtr);

    TSippLightCleanUp (tSippGlobPtr);

    TSippCameraCleanUp (tSippGlobPtr);

    if (tSippGlobPtr->rleCleanUpProc != NULL)
        (*tSippGlobPtr->rleCleanUpProc) (tSippGlobPtr);

    TSippPBMCleanUp (tSippGlobPtr);

    TSippPixMapCleanUp (tSippGlobPtr);

    sfree (tSippGlobPtr);

}
