/* 
 * tkInput.c --
 *
 *	This module implements "input" widgets for the Tk
 *	toolkit.  Inputs are windows with a transparent background.
 *
 * Copyright (c) 1990-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

/*
 *	Copyright (c) 1993, Mark G. Christenson, All rights reserved.
 */

#ifndef lint
static char rcsid[] = "$Header: /home/pecan7/mgc/Src-mgc/tk3.6/RCS/tkInput.c,v 1.2 1993/12/04 00:37:09 mgc Exp $ SPRITE (Berkeley)";
#endif

#include "default.h"
#include "tkConfig.h"
#include "tkInt.h"

#define DEF_INPUT_CURSOR		""
#define DEF_INPUT_GEOMETRY		""
#define DEF_INPUT_HEIGHT		"0"
#define DEF_INPUT_WIDTH			"0"

void Tk_MakeInputWindowExist _ANSI_ARGS_((Tk_Window tkwin));

/*
 * A data structure of the following type is kept for each
 * input that currently exists for this process:
 */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the input.  NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up.*/
    Display *display;		/* Display containing widget.  Used, among
				 * other things, so that resources can be
				 * freed even after tkwin has gone away. */
    Tcl_Interp *interp;		/* Interpreter associated with
				 * widget.  Used to delete widget
				 * command.  */
    int width;			/* Width to request for window.  <= 0 means
				 * don't request any size. */
    int height;			/* Height to request for window.  <= 0 means
				 * don't request any size. */
    char *geometry;		/* Geometry that user requested.  NULL
				 * means use width and height instead. 
				 * Malloc'ed. */
    Cursor cursor;		/* Current cursor for window, or None. */
    int flags;			/* Various flags;  see below for
				 * definitions. */
} Input;

/*
 * Flag bits for inputs:
 *
 *	NONE.
 */

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_ACTIVE_CURSOR, "-cursor", "cursor", "Cursor",
	DEF_INPUT_CURSOR, Tk_Offset(Input, cursor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-geometry", "geometry", "Geometry",
	DEF_INPUT_GEOMETRY, Tk_Offset(Input, geometry), TK_CONFIG_NULL_OK},
    {TK_CONFIG_PIXELS, "-height", "height", "Height",
	DEF_INPUT_HEIGHT, Tk_Offset(Input, height), 0},
    {TK_CONFIG_PIXELS, "-width", "width", "Width",
	DEF_INPUT_WIDTH, Tk_Offset(Input, width), 0},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static int	ConfigureInput _ANSI_ARGS_((Tcl_Interp *interp,
		    Input *inputPtr, int argc, char **argv, int flags));
static void	DestroyInput _ANSI_ARGS_((ClientData clientData));
static void	InputEventProc _ANSI_ARGS_((ClientData clientData,
		    XEvent *eventPtr));
static int	InputWidgetCmd _ANSI_ARGS_((ClientData clientData,
		    Tcl_Interp *interp, int argc, char **argv));
static void	MapInput _ANSI_ARGS_((ClientData clientData));

/*
 *--------------------------------------------------------------
 *
 * Tk_InputCmd --
 *
 *	This procedure is invoked to process the "input"
 *	Tcl command.  See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
Tk_InputCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    Tk_Window tkwin = (Tk_Window) clientData;
    Tk_Window new;
    Tk_Uid screenUid;
    char *className, *screen;
    int src, dst;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " pathName ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * The code below is a special hack that extracts a few key
     * options from the argument list now, rather than letting
     * ConfigureInput do it.  This is necessary because we have
     * to know the window's screen (if it's top-level) and its
     * class before creating the window.
     */

    screen = NULL;
    className = "Input";
    for (src = 2, dst = 2; src < argc;  src += 2) {
	char c;

	c = argv[src][1];
	if ((c == 'c')
		&& (strncmp(argv[src], "-class", strlen(argv[src])) == 0)) {
	    className = argv[src+1];
	} else if ((argv[0][0] == 't') && (c == 's')
		&& (strncmp(argv[src], "-screen", strlen(argv[src])) == 0)) {
	    screen = argv[src+1];
	} else {
	    argv[dst] = argv[src];
	    argv[dst+1] = argv[src+1];
	    dst += 2;
	}
    }
    argc -= src-dst;

    if (screen != NULL) {
	screenUid = Tk_GetUid(screen);
    } else {
	screenUid = NULL;
    }

    /*
     * Create the window and initialize our structures and event handlers.
     */

    new = Tk_CreateWindowFromPath(interp, tkwin, argv[1], screenUid);
    if (new == NULL) {
	return TCL_ERROR;
    }
    Tk_SetClass(new, className);
    return TkInitInput(interp, new, (screenUid != NULL), argc-2, argv+2);
}

/*
 *----------------------------------------------------------------------
 *
 * TkInitInput --
 *
 *	This procedure initializes a input widget.  It's
 *	separate from Tk_InputCmd so that it can be used for the
 *	main window, which has already been created elsewhere.
 *
 * Results:
 *	A standard Tcl completion code.
 *
 * Side effects:
 *	A widget record gets allocated, handlers get set up, etc..
 *
 *----------------------------------------------------------------------
 */

int
TkInitInput(interp, tkwin, toplevel, argc, argv)
    Tcl_Interp *interp;			/* Interpreter associated with the
					 * application. */
    Tk_Window tkwin;			/* Window to use for input or
					 * top-level.   Caller must already
					 * have set window's class. */
    int toplevel;			/* Non-zero means that this is a
					 * top-level window, 0 means it's a
					 * input. */
    int argc;				/* Number of configuration arguments
					 * (not including class command and
					 * window name). */
    char *argv[];			/* Configuration arguments. */
{
    register Input *inputPtr;

    inputPtr = (Input *) ckalloc(sizeof(Input));
    inputPtr->tkwin = tkwin;
    inputPtr->display = Tk_Display(tkwin);
    inputPtr->interp = interp;
    inputPtr->width = 0;
    inputPtr->height = 0;
    inputPtr->geometry = NULL;
    inputPtr->cursor = None;
    inputPtr->flags = 0;
    Tk_CreateEventHandler(inputPtr->tkwin, StructureNotifyMask,
	    InputEventProc, (ClientData) inputPtr);
    Tcl_CreateCommand(interp, Tk_PathName(inputPtr->tkwin),
	    InputWidgetCmd, (ClientData) inputPtr, (void (*)()) NULL);

    if (ConfigureInput(interp, inputPtr, argc, argv, 0) != TCL_OK) {
	Tk_DestroyWindow(inputPtr->tkwin);
	return TCL_ERROR;
    }
    Tk_MakeInputWindowExist(inputPtr->tkwin);
    interp->result = Tk_PathName(inputPtr->tkwin);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * InputWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a input widget.  See the user
 *	documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
InputWidgetCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Information about input widget. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    register Input *inputPtr = (Input *) clientData;
    int result = TCL_OK;
    int length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }
    Tk_Preserve((ClientData) inputPtr);
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)) {
	if (argc == 2) {
	    result = Tk_ConfigureInfo(interp, inputPtr->tkwin, configSpecs,
		    (char *) inputPtr, (char *) NULL, 0);
	} else if (argc == 3) {
	    result = Tk_ConfigureInfo(interp, inputPtr->tkwin, configSpecs,
		    (char *) inputPtr, argv[2], 0);
	} else {
	    result = ConfigureInput(interp, inputPtr, argc-2, argv+2,
		    TK_CONFIG_ARGV_ONLY);
	}
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\":  must be configure", (char *) NULL);
	result = TCL_ERROR;
    }
    Tk_Release((ClientData) inputPtr);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyInput --
 *
 *	This procedure is invoked by Tk_EventuallyFree or Tk_Release
 *	to clean up the internal structure of a input at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the input is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
DestroyInput(clientData)
    ClientData clientData;	/* Info about input widget. */
{
    register Input *inputPtr = (Input *) clientData;

    Tk_FreeOptions(configSpecs, (char *) inputPtr, inputPtr->display, 0);
    ckfree((char *) inputPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureInput --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a input widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for inputPtr;  old resources get freed, if there
 *	were any.
 *
 *----------------------------------------------------------------------
 */

static int
ConfigureInput(interp, inputPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    register Input *inputPtr;	/* Information about widget;  may or may
				 * not already have values for some fields. */
    int argc;			/* Number of valid entries in argv. */
    char **argv;		/* Arguments. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    if (Tk_ConfigureWidget(interp, inputPtr->tkwin, configSpecs,
	    argc, argv, (char *) inputPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    if (inputPtr->geometry != NULL) {
	int height, width;

	if (sscanf(inputPtr->geometry, "%dx%d", &width, &height) != 2) {
	    Tcl_AppendResult(interp, "bad geometry \"", inputPtr->geometry,
		    "\": expected widthxheight", (char *) NULL);
	    return TCL_ERROR;
	}
	Tk_GeometryRequest(inputPtr->tkwin, width, height);
    } else if ((inputPtr->width > 0) && (inputPtr->height > 0)) {
	Tk_GeometryRequest(inputPtr->tkwin, inputPtr->width,
		inputPtr->height);
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * InputEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher on
 *	structure changes to a input.  For inputs with 3D
 *	borders, this procedure is also invoked for exposures.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
InputEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    register XEvent *eventPtr;	/* Information about event. */
{
    register Input *inputPtr = (Input *) clientData;

    if (eventPtr->type == DestroyNotify) {
	Tcl_DeleteCommand(inputPtr->interp, Tk_PathName(inputPtr->tkwin));
	inputPtr->tkwin = NULL;
	Tk_EventuallyFree((ClientData) inputPtr, DestroyInput);
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tk_MakeInputWindowExist --
 *
 *	Ensure that a particular window actually exists.  This
 *	procedure shouldn't normally need to be invoked from
 *	outside the Tk package, but may be needed if someone
 *	wants to manipulate a window before mapping it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the procedure returns, the X window associated with
 *	tkwin is guaranteed to exist.  This may require the
 *	window's ancestors to be created also.
 *
 *--------------------------------------------------------------
 */

void
Tk_MakeInputWindowExist(tkwin)
    Tk_Window tkwin;		/* Token for window. */
{
    register TkWindow *winPtr = (TkWindow *) tkwin;
    TkWindow *winPtr2;
    Window parent;

    if (winPtr->window != None) {
	return;
    }

    if (winPtr->parentPtr->window == None) {
	Tk_MakeWindowExist((Tk_Window) winPtr->parentPtr);
    }
    parent = winPtr->parentPtr->window;

    winPtr->atts.event_mask = KeyPressMask | KeyReleaseMask | ButtonPressMask |
	ButtonReleaseMask | EnterWindowMask | LeaveWindowMask |
	PointerMotionMask | ExposureMask | VisibilityChangeMask |
	SubstructureNotifyMask | FocusChangeMask | PropertyChangeMask;

    winPtr->window = XCreateWindow(winPtr->display, parent,
	    winPtr->changes.x, winPtr->changes.y,
	    winPtr->changes.width, winPtr->changes.height,
	    CopyFromParent, CopyFromParent,
	    InputOnly, CopyFromParent, CWEventMask | CWCursor,
	    &winPtr->atts);
    XSaveContext(winPtr->display, winPtr->window, tkWindowContext,
	    (caddr_t) winPtr);
    winPtr->dirtyAtts = 0;
    winPtr->dirtyChanges = 0;

    /*
     * If any siblings higher up in the stacking order have already
     * been created then move this window to its rightful position
     * in the stacking order.
     *
     * NOTE: this code ignores any changes anyone might have made
     * to the sibling and stack_mode field of the window's attributes,
     * so it really isn't safe for these to be manipulated except
     * by calling Tk_RestackWindow.
     */

    for (winPtr2 = winPtr->nextPtr; winPtr2 != NULL;
	 winPtr2 = winPtr2->nextPtr) {
	if ((winPtr2->window != None) && !(winPtr2->flags & TK_TOP_LEVEL)) {
	    XWindowChanges changes;
	    changes.sibling = winPtr2->window;
	    changes.stack_mode = Below;
	    XConfigureWindow(winPtr->display, winPtr->window,
			     CWSibling|CWStackMode, &changes);
	    break;
	}
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tk_Input_Init --
 *
 *	Add the "input" command to an interpreter.
 *
 *--------------------------------------------------------------
 */

int
Tk_Input_Init(interp)
    Tcl_Interp *interp;

{
    Tcl_CreateCommand(interp, "input", Tk_InputCmd,
		      (ClientData)Tk_MainWindow(interp), (void (*)()) NULL);
    return TCL_OK;
}
