/*-
 * Copyright (c) 1993, 1994 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <linux/types.h>
#include <linux/major.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/fcntl.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/malloc.h>
#include <linux/sys.h>
#include <linux/termios.h>
#include <linux/midi.h>

#include <asm/io.h>
#include <asm/segment.h>
#include <asm/system.h>

#ifdef LOADABLE
#include "release.h"
#endif

extern struct midi_softc midi_sc[];
extern struct midi_config midi_conf[];
extern int NumMidi;

extern volatile u_long jiffies;
extern struct task_struct *current;

#if defined(__STDC__) || defined(__cplusplus)
#define __P(protos)	protos
#else
#define __P(protos)	()
#endif

#ifndef MIN
#define MIN(a, b)	(((a) < (b)) ? (a) : (b))
#endif

/* I've been burned too many times by different byte ordering */
#define OUTB(x, y) outb((y), (x))
#define INB(x) inb((x))


/* now in midi.h so mem.c can see it
 * long midiinit __P((long kmem_start));
 */
int midiopen __P((struct inode *inode, struct file *file));
void midirelease __P((struct inode *inode, struct file *file));
int midiread __P((struct inode *inode, struct file *file, char *buf, int len));
int midiwrite __P((struct inode *inode, struct file *file, char *buf, int len));
int midiioctl __P((struct inode *inode, struct file *file, unsigned int cmd,
    unsigned long arg));
int midiselect __P((struct inode *inode, struct file *file, int type,
    select_table *wait));
void midiintr __P((int irq));

static struct file_operations midi_fops = {
	NULL,		/* lseek */
	midiread,	/* read */
	midiwrite,	/* write */
	NULL,		/* readdir */
	midiselect,	/* select */
	midiioctl,	/* ioctl */
	NULL,		/* mmap */
	midiopen,	/* open */
	midirelease,	/* release */
	NULL		/* fsync */
};

static void midi_copy_event __P((struct event *, struct event *));
static void midi_initq __P((struct event_queue *));
static int midi_deq __P((struct event_queue *, struct event **));
static int midi_peekq __P((struct event_queue *, struct event **));
static int midi_enq __P((struct event_queue *, struct event *));
static int midi_fullreset __P((struct midi_softc *softc));
static int midi_wait_rdy_rcv __P((struct midi_softc *softc));
static int midi_send_command __P((struct midi_softc *softc,
    /* u_char */ int comm));
static int midi_reset __P((struct midi_softc *));
static int midi_uart __P((struct midi_softc *));
static int midi_event2smf __P((struct midi_softc *, struct event *event,
    struct stynamic *smf));
static int midi_next_byte __P((struct midi_softc *, char **buf, char *buf_end));
static int midi_str2event __P((struct midi_softc *softc, char **buf,
    char *buf_end, struct event *event));
static int midi_fix2var __P((u_long, u_char *));
static int midi_var2fix __P((u_char *, u_long *));
static u_long midi_smf2kernel_tick __P((struct midi_softc *, long));
static u_long midi_kernel2smf_tick __P((struct midi_softc *, long, long));
static void midi_timeout __P((unsigned long arg));
static void midi_add_complete_event __P((struct midi_softc *));
static void midi_schedule_timeout __P((struct midi_softc *));
static void midi_write_event __P((struct midi_softc *, struct event *));
static void midi_reset_devices __P((struct midi_softc *));
static int midi_get_bytes __P((char **buf, char *buf_end, char *dst, int len));
static int midi_get_byte __P((char **buf, char *buf_end));
static void stynamic_add_byte __P((struct stynamic *, int /* u_char */));
static void stynamic_add_bytes __P((struct stynamic *, u_char *, int));
static u_char stynamic_get_byte __P((struct stynamic *, int));
static void stynamic_copy __P((struct stynamic *, void *, int));
static void stynamic_copy_from __P((struct stynamic *, int, void *, int));
static void stynamic_append __P((struct stynamic *, struct stynamic *));
static void stynamic_release __P((struct stynamic *));
static void stynamic_shift __P((struct stynamic *, int));
static void stynamic_init __P((struct stynamic *));
static void stynamic_print __P((struct stynamic *));
static int midi_killable __P((struct task_struct *owner, int pid));

/* slightly modified copies of functions found in kernel/exit.c */
static int midi_sys_kill __P((struct task_struct *owner, int pid, int sig));
static int midi_kill_pg __P((struct task_struct *owner, int pgrp, int sig,
    int priv));
static int midi_kill_proc __P((struct task_struct *owner, int pid, int sig,
    int priv));
static int midi_send_sig __P((struct task_struct *owner, unsigned long sig,
    struct task_struct * p,int priv));
static int midi_generate __P((unsigned long sig, struct task_struct *p));


#ifdef LOADABLE
int
init_module(void)
{

	printk("midi.c: init_module called\n");
	return (midiinit(0));
}

void
cleanup_module(void)
{
	struct midi_softc *softc;
	int i;

	/* don't free anything until all devices are closed */
	for (i = 0; i < NumMidi; i++) {
		softc = &midi_sc[i];
		if (softc->status & MIDI_OPEN) {
			printk("midi device #%d is in use.  Try again later\n",
			    i);
			return;
		}
	}
	for (i = 0; i < NumMidi; i++) {
		softc = &midi_sc[i];
		kfree_s(softc->wqueue, sizeof(struct event_queue));
		kfree_s(softc->rqueue, sizeof(struct event_queue));
		midi_fullreset(softc);
	}
	if (unregister_chrdev(MIDI_MAJOR, "midi") != 0)
		printk("cleanup_module failed\n");
	else
		printk("cleanup_module succeeded\n");
}
#endif

long
midiinit(kmem_start)
	long kmem_start;
{
	struct midi_softc *softc;
	struct sigaction sa;
	int i;

	if (register_chrdev(MIDI_MAJOR, "midi", &midi_fops)) {
		printk("unable to get major %d for MPU401 MIDI\n",
		    MIDI_MAJOR);
		return (kmem_start);
	}
	for (i = 0; i < NumMidi; i++) {
		softc = &midi_sc[i];
		softc->addr = midi_conf[i].addr;
		softc->intr = midi_conf[i].intr;
		softc->status = MIDI_RD_BLOCK;
		if (!midi_reset(softc))
			if (!midi_reset(softc)) {
				printk("Couldn't reset MPU401 #%d\n", i);
				softc->status |= MIDI_UNAVAILABLE;
				continue;
			}
		if (!midi_uart(softc)) {
			printk("Couldn't put MPU401 #%d into UART mode!\n", i);
			softc->status |= MIDI_UNAVAILABLE;
			continue;
		}

		/* allocate memory for event queues */
#ifndef LOADABLE
		softc->rqueue = (struct event_queue *)kmem_start;
		kmem_start += sizeof(struct event_queue);
		softc->wqueue = (struct event_queue *)kmem_start;
		kmem_start += sizeof(struct event_queue);
#else
		softc->rqueue = kmalloc(sizeof(struct event_queue),
		    GFP_KERNEL);
		if (softc->rqueue == 0) {
			printk("Out of memory for read queue\n");
			return (-1);
		}
		softc->wqueue = kmalloc(sizeof(struct event_queue),
		    GFP_KERNEL);
		if (softc->wqueue == 0) {
			printk("Out of memory for write queue\n");
			kfree_s(softc->rqueue, sizeof(struct event_queue));
			return (-1);
		}
#endif
		memset(softc->rqueue, 0, sizeof(struct event_queue));
		memset(softc->wqueue, 0, sizeof(struct event_queue));
		stynamic_init(&softc->rpartial);
		stynamic_init(&softc->wpartial);
		softc->wpartialpos = 0;

		/* register the intr */
		sa.sa_handler = midiintr;
		sa.sa_flags = SA_INTERRUPT;
		sa.sa_mask = 0;
		sa.sa_restorer = NULL;
		if (irqaction(softc->intr, &sa)) {
#ifndef LOADABLE
			kmem_start -= 2 * sizeof(struct event_queue);
#else
			kfree_s(softc->rqueue, sizeof(struct event_queue));
			kfree_s(softc->wqueue, sizeof(struct event_queue));
#endif
			softc->status |= MIDI_UNAVAILABLE;
			printk("MPU401 #%d unable to use interrupt %d\n",
			    i, softc->intr);
			continue;
		}
		printk("Found MPU401 #%d at 0x%02x irq %d\n", i, softc->addr,
		    softc->intr);
	}
	return (kmem_start);
}

int
midiopen(inode, file)
	struct inode *inode;
	struct file *file;
{
	register struct midi_softc *softc;
	register int unit;
 
	unit = MINOR(inode->i_rdev);
	if (unit >= NumMidi)
		return (-ENXIO);
	softc = &midi_sc[unit];
	if (softc->status & MIDI_UNAVAILABLE)
		return (-EIO);
	
	if (softc->status & MIDI_OPEN)
		return (-EBUSY);
	else
		softc->status = MIDI_OPEN;

	/*
	 * reads will block until something appears
	 */
	softc->status |= MIDI_RD_BLOCK;

	/* initialize the queues */
	midi_initq(softc->rqueue);
	midi_initq(softc->wqueue);

	softc->partial_event.event.len = 0;
	softc->partial_event.event.datad = NULL;

	/* make sure we are in UART mode */
	if (!midi_fullreset(softc)) {
		printk("Couldn't put MPU401 into UART mode!\n");
		softc->status |= MIDI_UNAVAILABLE;
		return (-EIO);
	}

	softc->pgid = sys_getpid();
	softc->owner = current;

	/* are we going to read, write or both? */
	switch (file->f_flags & O_ACCMODE) {
	case O_RDONLY:
		softc->status &= ~MIDI_WRITING;
		softc->status |= MIDI_READING;
		break;
	case O_WRONLY:
		softc->status |= MIDI_WRITING;
		softc->status &= ~MIDI_READING;
		break;
	case O_RDWR:
		softc->status |= MIDI_READING;
		softc->status |= MIDI_WRITING;
		break;
	}
	if (file->f_flags & O_NONBLOCK)
		softc->status |= MIDI_NONBLOCK;
	else
		softc->status &= ~MIDI_NONBLOCK;

	return (0);
}

void
midirelease(inode, file)
	struct inode *inode;
	struct file *file;
{
	register struct midi_softc *softc;
	register int unit;

	unit = MINOR(inode->i_rdev);
	softc = &midi_sc[unit];

	/*
	 * we're not going to finish closing until everything has
	 * been played.
	 */
	if (softc->status & MIDI_WRITING) {
		/* same as MFLUSHQ ioctl */
		if (softc->wqueue->count != 0) {
			softc->status |= MIDI_FLUSH_SLEEP;
			interruptible_sleep_on(&softc->flush_waitq);
			/* if the sleep was interrupted, clear the flag */
			softc->status &= ~MIDI_FLUSH_SLEEP;
		}
	}

	/* turn off any notes that might be stuck on */
	midi_reset_devices(softc);
	midi_fullreset(softc);

	softc->status &= ~MIDI_OPEN;
	return;
}

int
midiread(inode, file, buf, len)
	struct inode *inode;
	struct file *file;
	char *buf;
	int len;
{
	struct event *event;
	register struct midi_softc *softc;
	int init_count, unit, num_to_move;

	unit = MINOR(inode->i_rdev);
	softc = &midi_sc[unit];
	init_count = len;

	if (softc->rqueue->count == 0 && softc->rpartial.len == 0) {
		if (softc->status & MIDI_NONBLOCK)
			return (-EWOULDBLOCK);
		else {
			softc->status |= MIDI_RD_SLEEP;
			interruptible_sleep_on(&softc->read_waitq);
			/* XXX maybe check for abort here */
		}
	}
	while (len) {
		/*
		 * dequeue an event if partial is empty
		 */
		if (softc->rpartial.len == 0) {
			if (!midi_deq(softc->rqueue, &event)) {
				softc->status |= MIDI_RD_BLOCK;
				return (init_count - len);
			}
			midi_event2smf(softc, event, &softc->rpartial);
			stynamic_release(&event->data);
		}
		/* read out as much of rpartial as possible */
		num_to_move = MIN(softc->rpartial.len, len);
		if (softc->rpartial.len <= STYNAMIC_SIZE)
			memcpy_tofs(buf, softc->rpartial.datas, num_to_move);
		else
			memcpy_tofs(buf, softc->rpartial.datad, num_to_move);
		stynamic_shift(&softc->rpartial, num_to_move);
		buf += num_to_move;
		len -= num_to_move;
	}
	return (init_count - len);
}

int
midiwrite(inode, file, buf, len)
	struct inode *inode;
	struct file *file;
	char *buf;
	int len;
{
	register struct midi_softc *softc;
	struct event event;
	int convert, init_count, unit, empty_queue;
	char *buf_end;

	unit = MINOR(inode->i_rdev);
	softc = &midi_sc[unit];
	init_count = len;

	if (softc->wqueue->count == 0)
		empty_queue = 1;
	else
		empty_queue = 0;

	/* check to see if we'll block */
	if (softc->status & MIDI_WR_BLOCK) {
		if (softc->status & MIDI_NONBLOCK)
			return (-EWOULDBLOCK);
		else {
			softc->status |= MIDI_WR_SLEEP;
			interruptible_sleep_on(&softc->write_waitq);
		}
	}
	/* if returns from sleep and should abort because of DRAIN */
	if (softc->status & MIDI_WR_ABORT) {
		softc->status &= ~MIDI_WR_ABORT;
		return (init_count - len);
	}

	buf_end = buf + len;
	stynamic_init(&event.data);
	while (len) {
		/* block if queue is full */
		if (softc->wqueue->count == MIDI_Q_SIZE) {
			softc->status |= MIDI_WR_BLOCK;
			if (empty_queue) {
				midi_schedule_timeout(softc);
				empty_queue = 0;
			}
			if (softc->status & MIDI_NONBLOCK) {
				return (init_count - len);
			} else {
				softc->status |= MIDI_WR_SLEEP;
				interruptible_sleep_on(&softc->write_waitq);
			}
		}

		/*
		 * 1) get a complete event off queue
		 * 2) convert it from SMF to board format
		 * 3) deal with it
		 */
		convert = midi_str2event(softc, &buf, buf_end, &event);
		len = buf_end - buf;
		switch (convert) {
		case 0:
			break;
		case -1:
			/* not a complete event - we'll get it next time */
			if (empty_queue)
				midi_schedule_timeout(softc);
			return (init_count - len);
		case -2:
			/* it was a noop event */
			break;
		default:
			return (convert);
		}
		if (convert == -2)
			continue;
		if (midi_enq(softc->wqueue, &event) == -1)
			return (-EIO);
		stynamic_release(&event.data);
		/* set flag so next time we cross LOW water we will SIGIO */
		if (softc->wqueue->count >= MIDI_LOW_WATER)
			softc->status |= MIDI_SENDIO;
	}

	if (empty_queue)
		midi_schedule_timeout(softc);

	return (init_count - len);
}

void
midiintr(irq)
	int irq;
{
	register struct midi_softc *softc;
	register u_char code;
	register struct partial_event *pe;
	int i;

	for (i = 0; i < NumMidi && midi_sc[i].intr != irq; i++);
	if (i == NumMidi) {
		printk("midi: ivec(%d) is not a valid MPU401 intr number\n",
		    irq);
		return;
	}
	softc = &midi_sc[i];

	code = INB(softc->addr + MIDI_DATA);
	if (code == 0xfe && (softc->status & MIDI_NEED_ACK)) {
		softc->status &= ~MIDI_NEED_ACK;
		return;
	}
	/* throw away data if no one has the device open */
	if (!(softc->status & MIDI_READING))
		return;
 
 	/* pass input data to out port if necessary */
 	if (softc->status & MIDI_THRU) {
 		if (!midi_wait_rdy_rcv(softc))
 			printk("Couldn't pass data thru\n");
		else
			OUTB(softc->addr + MIDI_DATA, code);
 	}
	pe = &softc->partial_event;

	/* check for realtime events */
	if ((code & 0xf8) == 0xf8) {
		switch (code) {
		case 0xfa:	/* start */
			/* reset prev_incoming */
			softc->prev_incoming = jiffies;
			break;
		case 0xff:	/* reset */
			/*
			 * I don't really want to do a full reset
			 * I'll just clear any event in progress
			 */
			stynamic_release(&pe->event);
			pe->state = START;
			break;
		case 0xf8:	/* timing clock */
		case 0xf9:	/* undefined */
		case 0xfb:	/* continue */
		case 0xfc:	/* stop */
		case 0xfd:	/* undefined */
		case 0xfe:	/* active sensing */
			break;
		}
		return;
	}
INTR_SWITCH:
	switch (pe->state) {
	case START:
		/* record when the time when the event started */
		pe->jiffs = jiffies;
		pe->tempo = softc->rtempo;
		/* start collecting the input */
		stynamic_release(&pe->event);
		/* decide what state is next */
		if (!(code & 0x80)) {
			stynamic_add_byte(&pe->event, code);
			switch (pe->rs) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				/*
				 * code is the first data byte, but
				 * we still need to get the second
				 */
				pe->state = NEEDDATA2;
				break;
			case 0xc0:
			case 0xd0:
				/* code is the only data byte */
				pe->state = START;
				midi_add_complete_event(softc);
				break;
			default:
				break;
			}
		} else {
			switch (code & 0xf0) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				pe->rs = code & 0xf0;
				stynamic_add_byte(&pe->event, code);
				pe->state = NEEDDATA1;
				break;
			case 0xc0:
			case 0xd0:
				pe->rs = code & 0xf0;
				stynamic_add_byte(&pe->event, code);
				pe->state = NEEDDATA2;
				break;
			default:
				switch (code) {
				case 0xf0: /* sysex */
					stynamic_add_byte(&pe->event, code);
					pe->state = SYSEX;
					break;
				case 0xf1: /* undefined */
				case 0xf4: /* undefined */
				case 0xf5: /* undefined */
				case 0xf6: /* tune request */
				case 0xf7: /* EOX (terminator) */
					/* ignore these */
					break;
				case 0xf2: /* song position */
					pe->state = SYSTEM2;
					break;
				case 0xf3: /* song select */
					pe->state = SYSTEM1;
					break;
				}
				break;
			}
		}
		break;
	case NEEDDATA1:
		stynamic_add_byte(&pe->event, code);
		pe->state = NEEDDATA2;
		break;
	case NEEDDATA2:
		stynamic_add_byte(&pe->event, code);
		pe->state = START;
		midi_add_complete_event(softc);
		break;
	case SYSEX:
		/* any non-data byte ends sysex */
		if (!(code & 0x80))
			stynamic_add_byte(&pe->event, code);
		else {
			stynamic_add_byte(&pe->event, 0xf7);
			midi_add_complete_event(softc);
			pe->state = START;
			if (code != 0xf7)
				goto INTR_SWITCH;
		}
		break;
	case SYSTEM1:
		/* throw away one data byte of a system message */
		pe->state = START;
		break;
	case SYSTEM2:
		/* throw away two data bytes of a system message */
		pe->state = SYSTEM1;
		break;
	}
	return;
}

int
midiioctl(inode, file, cmd, arg)
	struct inode *inode;
	struct file *file;
	unsigned int cmd;
	unsigned long arg;
{
	struct event *event;
	struct midi_softc *softc;
	register int unit;
	int ret, val;

	unit = MINOR(inode->i_rdev);
	softc = &midi_sc[unit];

	switch (cmd) {
	case FIOASYNC:
	case MASYNC:
		/*
		 * Linux doesn't pass FIOASYNC to the driver, so
		 * we need a separate ioctl entry point to do
		 * exactly what FIOASYNC does.  Thus MASYNC.
		 */
		memcpy_fromfs(&val, (int *)arg, sizeof(int));
		if (!val)
			softc->status &= ~MIDI_ASYNC;
		else {
			softc->status |= MIDI_ASYNC;
			if (softc->wqueue->count < MIDI_LOW_WATER
			    && softc->status & MIDI_WRITING)
				midi_sys_kill(softc->owner, softc->pgid, SIGIO);
		}
		break;
	case TIOCSPGRP:
		memcpy_fromfs(&val, (int *)arg, sizeof(int));
		/* verify that val is something we're allowed to kill */
		if ((ret = midi_killable(softc->owner, val)) != 0)
			return (-ret);
		softc->pgid = val;
		break;
	case TIOCGPGRP:
		memcpy_tofs((int *)arg, &softc->pgid, sizeof(int));
		break;
	case MRESET:
		if (!midi_fullreset(softc))
			return (-EIO);
		break;
	case MSDIVISION:
		memcpy_fromfs(&val, (int *)arg, sizeof(int));
 		/* must recalculate play remainder */
 		softc->premainder = softc->premainder * val / softc->division;
 		softc->division = val;
		break;
	case MGDIVISION:
		memcpy_tofs((int *)arg, &softc->division, sizeof(int));
		break;
	case MDRAIN:
		/* dequeue all everything */
		while (midi_deq(softc->rqueue, &event))
			stynamic_release(&event->data);
		while (midi_deq(softc->wqueue, &event))
			stynamic_release(&event->data);
		/* remove any events already being timed */
		del_timer(&softc->timer);
		softc->status &= ~MIDI_WR_BLOCK;
		softc->status |= MIDI_RD_BLOCK;
		if (softc->status & MIDI_WR_SLEEP) {
			softc->status &= ~MIDI_WR_SLEEP;
			softc->status |= MIDI_WR_ABORT;
			wake_up_interruptible(&softc->write_waitq);
		}
		midi_reset_devices(softc);
		break;
	case MFLUSH:
		if (softc->wqueue->count != 0) {
			softc->status |= MIDI_FLUSH_SLEEP;
			interruptible_sleep_on(&softc->flush_waitq);
		}
		break;
	case MGPLAYQ:
		memcpy_tofs((int *)arg, &softc->wqueue->count, sizeof(int));
		break;
	case MGRECQ:
		memcpy_tofs((int *)arg, &softc->rqueue->count, sizeof(int));
		break;
	case MGQAVAIL:
		val = MIDI_Q_SIZE - softc->wqueue->count;
		memcpy_tofs((int *)arg, &val, sizeof(int));
		break;
 	case MTHRU:
		memcpy_fromfs(&val, (int *)arg, sizeof(int));
 		if (val)
 			softc->status |= MIDI_THRU;
 		else
 			softc->status &= ~MIDI_THRU;
  		break;
 	case MRECONPLAY:
		memcpy_fromfs(&val, (int *)arg, sizeof(int));
 		if (val)
 			softc->status |= MIDI_RECONPLAY;
 		else
 			softc->status &= ~MIDI_RECONPLAY;
  		break;
	default:
		return (-ENOTTY);
	}
	return (0);
}

int
midiselect(inode, file, type, wait)
	struct inode *inode;
	struct file *file;
	int type;
	select_table *wait;
{
	register struct midi_softc *softc;
	int unit;

	unit = MINOR(inode->i_rdev);
	softc = &midi_sc[unit];

	switch (type) {
	case SEL_IN:
		if (!(softc->status & MIDI_RD_BLOCK))
			return (1);
		else {
			softc->status |= MIDI_SELIN;
			select_wait(&softc->selin_waitq, wait);
			return (0);
		}
		break;
	case SEL_OUT:
		if (!(softc->status & MIDI_WR_BLOCK))
			return (1);
		else {
			softc->status |= MIDI_SELOUT;
			select_wait(&softc->selout_waitq, wait);
			return (0);
		}
		break;
	case SEL_EX:
		select_wait(&softc->forever_waitq, wait);
		return (0);
	default:
		return (0);
	}
}

int
midi_fullreset(softc)
	struct midi_softc *softc;
{
	u_char pitch;
	struct event *event;

	/* dequeue all everything */
	while (midi_deq(softc->rqueue, &event))
		stynamic_release(&event->data);
	while (midi_deq(softc->wqueue, &event))
		stynamic_release(&event->data);
	/* remove any events already being timed */
	del_timer(&softc->timer);
	/* mark start time */
	softc->prev_incoming = jiffies;
	/* reset fractional count */
	softc->premainder = 0;
	softc->rremainder = 0;
	/* initialize some variables */
	/* this is 120 bpm when a quarter note == 1 beat */
	softc->ptempo = 500000;
	softc->rtempo = 500000;
 	softc->prev_rtempo = 500000;

	/* clear noteon */
	for (pitch = 0; pitch <= 0x7f; pitch++)
		softc->noteon[pitch] = 0;
	softc->noteonrs = 0;

	/* clear running play state */
	softc->writers = 0;

	/* reset partial event stuff */
	stynamic_release(&softc->partial_event.event);
	softc->partial_event.state = START;
	stynamic_release(&softc->rpartial);
	stynamic_release(&softc->wpartial);
	softc->wpartialpos = 0;

	/* defaults to 120 clocks per beat */
	softc->division = 120;
	/* reset and enter uart mode */
	return (midi_uart(softc));
}

int
midi_wait_rdy_rcv(softc)
	struct midi_softc *softc;
{
	int flag;
	register int i;

	for (i = 0; i < MIDI_TRIES; i++) {
		flag = INB(softc->addr + MIDI_STATUS) & MIDI_RDY_RCV;
		if (flag == 0)
			break;
	}
	if (i == MIDI_TRIES)
		return (0);
	return(1);
}

int
midi_send_command(softc, comm)
	struct midi_softc *softc;
	u_char comm;
{
	int flag;
	register int i;
	unsigned char ack;

	/* writing command and setting flag must be atomic */
	cli();
	OUTB(softc->addr + MIDI_COMMAND, comm);
	softc->status |= MIDI_NEED_ACK;
	sti();

	/* time out after MIDI_TRIES times */
	for (i = 0; i < MIDI_TRIES; i++) {
		/* did we pick up the ack via midiintr? */
		if (!(softc->status & MIDI_NEED_ACK))
			break;
		/* can we read a data byte? */
		flag = INB(softc->addr + MIDI_STATUS) & MIDI_DATA_AVL;
		if (flag == 0)
			break;
	}
	if (i == MIDI_TRIES)
		return (0);

	if (softc->status & MIDI_NEED_ACK) {
		ack = INB(softc->addr + MIDI_DATA);
		if (ack != MIDI_ACK)
			return (0);
	}
	softc->status &= ~MIDI_NEED_ACK;
	return (1);
}

int
midi_reset(softc)
	struct midi_softc *softc;
{

	if (!midi_wait_rdy_rcv(softc))
		return (0);

	if (!midi_send_command(softc, MIDI_RESET))
		return (0);

	return (1);
}

int
midi_uart(softc)
	struct midi_softc *softc;
{

	/*
	 * first reset.  We can't issue the UART command when already
	 * in UART mode.
	 */
	(void)midi_reset(softc);

	if (!midi_wait_rdy_rcv(softc))
		return (0);

	if (!midi_send_command(softc, MIDI_UART))
		return (0);
	return (1);
}

int
midi_event2smf(softc, event, smf)
	struct midi_softc *softc;
	struct event *event;
	struct stynamic *smf;
{
	long tempo, smf_ticks;
	int tmp_len;
	u_char tmp_buf[4];

	/* convert from kernel ticks to SMF ticks */
	smf_ticks = midi_kernel2smf_tick(softc, event->tempo, event->time);
	tmp_len = midi_fix2var(smf_ticks, tmp_buf);
	stynamic_add_bytes(smf, tmp_buf, tmp_len);

	switch (event->type) {
	case NORMAL:
		stynamic_append(smf, &event->data);
		break;
	case TEMPO:
		/* this case just won't occur */
		stynamic_copy(&event->data, (void *)&tempo, sizeof(tempo));
		stynamic_add_byte(smf, 0xff);
		stynamic_add_byte(smf, 0x51);
		stynamic_add_byte(smf, 0x03);
		stynamic_add_byte(smf, (tempo & 0xff000000) >> 16);
		stynamic_add_byte(smf, (tempo & 0xff00) >> 8);
		stynamic_add_byte(smf, tempo & 0xff);
		break;
	case SYSX:
		stynamic_add_byte(smf, 0xf0);
		/* skip over the leading 0xf0 */
		stynamic_shift(&event->data, 1);
		tmp_len = midi_fix2var(event->data.len, tmp_buf);
		stynamic_add_bytes(smf, tmp_buf, tmp_len);
		stynamic_append(smf, &event->data);
		break;
	default:
		break;
	}
	return (1);
}

int
midi_get_byte(buf, buf_end)
	char **buf, *buf_end;
{
	u_char byte;

	if (*buf >= buf_end)
		return (-1);
	else {
		memcpy_fromfs(&byte, *buf, 1);
		*buf += 1;
		return (byte);
	}
}

int
midi_get_bytes(buf, buf_end, dst, len)
	char **buf, *buf_end, *dst;
	int len;
{

	if (*buf + len >= buf_end)
		return (0);
	memcpy_fromfs(dst, *buf, len);
	*buf += len;
	return (1);
}

int
midi_next_byte(softc, buf, buf_end)
	struct midi_softc *softc;
	char **buf, *buf_end;
{
	int byte;

	/* if we're not at the end of a partial event, read from it */
	if (softc->wpartialpos < softc->wpartial.len) {
		byte = stynamic_get_byte(&softc->wpartial,
		    softc->wpartialpos++);
		return (byte);
	} else {
		/* read from str and copy uio onto partial event */
		if ((byte = midi_get_byte(buf, buf_end)) == -1) {
			/*
			 * reset partialpos so next time through
			 * we'll read from the partial event if
			 * it is non-zero in length
			 */
			softc->wpartialpos = 0;
			return (-1);
		}
		stynamic_add_byte(&softc->wpartial, byte);
		softc->wpartialpos = softc->wpartial.len;
		return (byte);
	}
}

int
midi_str2event(softc, buf, buf_end, event)
	struct midi_softc *softc;
	char **buf, *buf_end;
	struct event *event;
{
	u_long smf_ticks, ulen;
	long tempo;
	int byte, extra_byte, i, len, num_data_bytes;
	int rs_change;
	u_char meta_type, tmp_buf[256];

	/* copy in timing portion */
	len = 0;
	do {
		if ((byte = midi_next_byte(softc, buf, buf_end)) == -1) {
			stynamic_release(&event->data);
			return (-1);
		}
		tmp_buf[len++] = byte;
	} while (byte & 0x80);

	/* compute time in smf ticks */
	midi_var2fix(tmp_buf, &smf_ticks);
	smf_ticks += softc->write_noop_time;
	softc->write_noop_time = 0;

	/* now convert from smf to kernel */
	event->time = midi_smf2kernel_tick(softc, smf_ticks);

	/* get first byte of event data */
	if ((byte = midi_next_byte(softc, buf, buf_end)) == -1) {
		stynamic_release(&event->data);
		return (-1);
	}
	switch (byte) {
	case 0xf0:
		/* basic sysex type */
		event->type = SYSX;
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(tmp_buf, &ulen);
		stynamic_add_byte(&event->data, 0xf0);
		for (; ulen > 0; ulen--) {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			stynamic_add_byte(&event->data, byte);
		}
		break;
	case 0xf7:
		/* continued sysex type */
		event->type = SYSX;
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(tmp_buf, &ulen);
		for (; ulen > 0; ulen--) {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			stynamic_add_byte(&event->data, byte);
		}
		break;
	case 0xff:
		/* meta events */
		if ((byte = midi_next_byte(softc, buf, buf_end)) == -1) {
			stynamic_release(&event->data);
			return (-1);
		}
		meta_type = byte;
		/* get length of meta data */
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(tmp_buf, &ulen);
		/* read it in - meta events are not over 256 in size*/
		for (i = 0; i < ulen; i++) {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[i] = byte;
		}
		switch (meta_type) {
		default:
			/*
			 * we'll skip these events, but we need to
			 * save the timing info
			 */
			softc->write_noop_time = smf_ticks;
			stynamic_release(&softc->wpartial);
			softc->wpartialpos = 0;
			return (-2);
		case 0x51:
			/* tempo event */
			event->type = TEMPO;
			tempo = 0;
			for (i = 0; i < 3; i++) {
				tempo = tempo << 8;
				tempo |= tmp_buf[i];
			}
			stynamic_add_bytes(&event->data, (u_char *)&tempo,
			    sizeof(tempo));
			/*
			 * change ptempo now, rtempo will change when
			 * the event's time comes up
			 */
			softc->ptempo = tempo;
			break;
		}
		break;
	default:
		if ((byte & 0xf0) == 0x80) {
			u_char rs_chan, rs_type;

			/* check note off events separately */
			tmp_buf[0] = byte;
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[1] = byte;
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[2] = byte;
			len = 3;
			/*
			 * check to see if we can collapse and use
			 * running state
			 */
			rs_type = softc->writers & 0xf0;
			rs_chan = softc->writers & 0x0f;
			/*
			 * collapse to running state if time is 0
			 * and the running state is the same
			 * or the running state is note on for the
			 * same channel and the note off velocity is
			 * zero.
			 */
			if (event->time == 0 && (softc->writers == tmp_buf[0]
			    || (tmp_buf[2] == 0 && rs_type == 0x90
			    && rs_chan == (tmp_buf[0] & 0x0f)))) {
				tmp_buf[0] = tmp_buf[1];
				tmp_buf[1] = tmp_buf[2];
				len = 2;
			} else {
				softc->writers = tmp_buf[0];
			}
		} else {
			extra_byte = 0;
			rs_change = 0;
			if ((byte & 0x80) && (byte != softc->writers)) {
				softc->writers = byte;
				rs_change = 1;
			}
			len = 0;
			if (event->time != 0 || rs_change) {
				/*
				 * stick in a mode byte if time is non-zero
				 * This is so we don't confuse hardware that
				 * is turned on while we're playing
				 * also add it if the running state changes
				 */
				tmp_buf[0] = softc->writers;
				len = 1;
			}
			if (byte & 0x80)
				extra_byte = 1;
			else {
				tmp_buf[len] = byte;
				len++;
			}

			switch (softc->writers & 0xf0) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				num_data_bytes = 1;
				break;
			default:
				num_data_bytes = 0;
			}
			for (i = 0; i < num_data_bytes + extra_byte; i++) {
				if ((byte = midi_next_byte(softc, buf, buf_end))
				    == -1) {
					stynamic_release(&event->data);
					return (-1);
				}
				tmp_buf[len++] = byte;
			}
		}
		event->type = NORMAL;
		stynamic_add_bytes(&event->data, tmp_buf, len);
	}
	stynamic_release(&softc->wpartial);
	softc->wpartialpos = 0;
	return (0);
}

int
midi_fix2var(fix, var)
	u_long fix;
	u_char *var;
{
	int i;
	unsigned char buf[4], *bptr;

	buf[0] = buf[1] = buf[2] = buf[3] = 0;
	bptr = buf;
	*bptr++ = fix & 0x7f;
	while ((fix >>= 7) > 0) {
		*bptr |= 0x80;
		*bptr++ += (fix & 0x7f);
        }

	i = 0;
	do {
		*var++ = *--bptr;
		i++;
	} while (bptr != buf);

	return (i);
}

int
midi_var2fix(var, fix)
	u_char *var;
	u_long *fix;
{
	int delta;

	*fix = 0;
	delta = 0;
	do {
		*fix = (*fix << 7) + (*var & 0x7f);
		delta++;
	} while (*var++ & 0x80);

	return (delta);
}

u_long
midi_smf2kernel_tick(softc, smf)
	struct midi_softc *softc;
	long smf;
{
 	long long denominator, numerator;
 	u_long kernel;
  
	numerator = (long long)softc->ptempo * HZ * smf + softc->premainder;
 	denominator = 1000000 * (long long)softc->division;
 	kernel = numerator / denominator;
 	softc->premainder = numerator % denominator;
  	return (kernel);
}

u_long
midi_kernel2smf_tick(softc, tempo, kernel)
	struct midi_softc *softc;
	long tempo, kernel;
{
 	long long numerator, denominator;
 	u_long smf;
  
 	if (softc->prev_rtempo != tempo) {
 		/*
 		 * also update the rremainder to reflect tempo
 		 * change
 		 */
 		softc->rremainder = softc->rremainder * tempo
 		    / softc->prev_rtempo;
 		softc->prev_rtempo = tempo;
 	}
	numerator = (long long)softc->division * 1000000 * kernel
	    + softc->rremainder;
 	denominator = (long long)tempo * HZ;
 	smf = numerator / denominator;
 	softc->rremainder = numerator % denominator;
  	return (smf);
}

void
midi_add_complete_event(softc)
	struct midi_softc *softc;
{
	struct event event;
	struct partial_event *pe;
	long ticks;

	stynamic_init(&event.data);
	pe = &softc->partial_event;

	ticks = pe->jiffs - softc->prev_incoming;
	if (ticks < 0) {
		ticks = 0;
		pe->jiffs = softc->prev_incoming;
	}
	event.time = ticks;
	event.tempo = pe->tempo;
	switch (stynamic_get_byte(&pe->event, 0)) {
	case 0xf0:
		/* sysex */
		event.type = SYSX;
		break;
	default:
		event.type = NORMAL;
		break;
	}
	stynamic_append(&event.data, &softc->partial_event.event);
	/* enqueue new event */
	midi_enq(softc->rqueue, &event);
	/* readjust previous event time */
	softc->prev_incoming = pe->jiffs;

	softc->status &= ~MIDI_RD_BLOCK;
	if (softc->status & MIDI_RD_SLEEP) {
		softc->status &= ~MIDI_RD_SLEEP;
		wake_up_interruptible(&softc->read_waitq);
	}
	if (softc->status & MIDI_ASYNC)
		midi_sys_kill(softc->owner, softc->pgid, SIGIO);

	/* notify select that there is data to be read */
	if (softc->status & MIDI_SELIN) {
		softc->status &= ~MIDI_SELIN;
		wake_up_interruptible(&softc->selin_waitq);
	}

	return;
}

void
midi_schedule_timeout(softc)
	struct midi_softc *softc;
{
	struct event *event;

	if (!midi_peekq(softc->wqueue, &event))
		return;
	softc->timer.expires = event->time;
	softc->timer.data = (unsigned long)softc;
	softc->timer.function = midi_timeout;
	add_timer(&softc->timer);

	/* clear record timer? */
	if (softc->status & MIDI_RECONPLAY) {
		softc->prev_incoming = jiffies;
		/* clear flag */
		softc->status &= ~MIDI_RECONPLAY;
	}
}

void
midi_timeout(arg)
	unsigned long arg;
{
	struct event *event;
	struct midi_softc *softc = (struct midi_softc *)arg;

	/* send first event since we know it is ready */
	midi_deq(softc->wqueue, &event);
	midi_write_event(softc, event);
	stynamic_release(&event->data);

	/* process all events that also occur at this time (event->time == 0) */
	for(;;) {
		if (!midi_peekq(softc->wqueue, &event))
			return;
		if (event->time != 0)
			break;	
		midi_deq(softc->wqueue, NULL);
		midi_write_event(softc, event);
		stynamic_release(&event->data);
	}

	/* setup timeout for next event */
	midi_schedule_timeout(softc);
}

void
midi_write_event(softc, event)
	struct midi_softc *softc;
	struct event *event;
{
	int i;
	u_char bytes[4], channel, command, *dataptr;

	switch (event->type) {
	case TEMPO:
		stynamic_copy(&event->data, &softc->rtempo,
		    sizeof(softc->rtempo));
		break;
	case NORMAL:
		/*
		 * fourth byte might not be valid, but who cares,
		 * we're only reading and in the kernel.  We'll
		 * ignore it if it isn't.
		 */
		stynamic_copy(&event->data, &bytes, 4);
		if (!(bytes[0] & 0x80))
			dataptr = &bytes[0];
		else {
			softc->noteonrs = bytes[0];
			dataptr = &bytes[1];
		}
		command = softc->noteonrs & 0xf0;
		channel = softc->noteonrs & 0x0f;
		if (command == 0x90) {
			/*
			 * set of clear appropriate bit in noteon
			 * array depending on velocity value
			 */
			if (dataptr[1] != 0)
				softc->noteon[dataptr[0]] |= 1 << channel;
			else
				softc->noteon[dataptr[0]] &= ~(1 << channel);
		}
		if (command == 0x80)
			/* clear bit */
			softc->noteon[dataptr[0]] &= ~(1 << channel);
		/* FALLTHRU */
	default:
		for (i = 0; i < event->data.len; i++) {
			if (!midi_wait_rdy_rcv(softc))
				break;
			OUTB(softc->addr + MIDI_DATA,
			    stynamic_get_byte(&event->data, i));
		}
		break;
	}
	if (softc->wqueue->count < MIDI_LOW_WATER) {
		softc->status &= ~MIDI_WR_BLOCK;
		if (softc->status & MIDI_WR_SLEEP) {
			softc->status &= ~MIDI_WR_SLEEP;
			wake_up_interruptible(&softc->write_waitq);
		}
		if (softc->status & MIDI_ASYNC &&
		    (softc->status & MIDI_SENDIO || softc->wqueue->count
		    == 0))
			midi_sys_kill(softc->owner, softc->pgid, SIGIO);
		softc->status &= ~MIDI_SENDIO;
		/* notify select that writes will succeed */
		if (softc->status & MIDI_SELOUT) {
			softc->status &= ~MIDI_SELOUT;
			wake_up_interruptible(&softc->selout_waitq);
		}
	}
	if (softc->status & MIDI_FLUSH_SLEEP && softc->wqueue->count == 0) {
		softc->status &= ~MIDI_FLUSH_SLEEP;
		wake_up_interruptible(&softc->flush_waitq);
	}
}

/*
 * try to reset the midi devices as best we can
 */
void
midi_reset_devices(softc)
	struct midi_softc *softc;
{
	u_char channel, pitch;

	/* manual note off calls - turn off any note that is on */
	for (pitch = 0; pitch <= 0x7f; pitch++) {
		for (channel = 0; channel <= 0x0f; channel++) {
			if ((softc->noteon[pitch] >> (int)channel) & 0x01) {
				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, channel | 0x90);

				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, pitch);

				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, 0);
			}
		}
		softc->noteon[pitch] = 0;
	}
	for (channel = 0; channel <= 0x0f; channel++) {
		/*
		 * send paramter event for all notes off for redundancy
		 * some older synths don't support this
		 */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, channel | 0xb0);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x7b);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* modulation controller to zero */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x01);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* reset all controllers */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x79);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* lift sustain pedal */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x40);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* center pitch wheel */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0xe0 | channel);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x40);
	}
	softc->noteonrs = 0;
}

void
midi_initq(eq)
	struct event_queue *eq;
{

	eq->count = 0;
	eq->end = &eq->events[MIDI_Q_SIZE - 1];
	eq->head = eq->events;
	eq->tail = eq->events;
	/* zero events to clear stynamic stuctures */
	memset(eq->events, 0, MIDI_Q_SIZE * sizeof(struct event));
}

void
midi_copy_event(e1, e2)
	struct event *e1, *e2;
{

	e1->time = e2->time;
	e1->type = e2->type;
	e1->tempo = e2->tempo;
	stynamic_release(&e1->data);
	stynamic_append(&e1->data, &e2->data);
}

int
midi_deq(eq, event)
	struct event_queue *eq;
	struct event **event;
{

	cli();
	if (eq->count == 0) {
		sti();
		return (0);
	}
	if (event == NULL)
		eq->head++;
	else
		*event = eq->head++;
	if (eq->head > eq->end)
		eq->head = eq->events;
	eq->count--;
	sti();
	return (1);
}

int
midi_peekq(eq, event)
	struct event_queue *eq;
	struct event **event;
{

	cli();
	if (eq->count == 0) {
		sti();
		return (0);
	}
	*event = eq->head;
	sti();
	return (1);
}

int
midi_enq(eq, event)
	struct event_queue *eq;
	struct event *event;
{

	cli();
	if (eq->count == MIDI_Q_SIZE) {
		sti();
		return (0);
	}
	midi_copy_event(eq->tail++, event);
	if (eq->tail > eq->end)
		eq->tail = eq->events;
	eq->count++;
	sti();
	return (1);
}

void
stynamic_add_byte(sty, byte)
	struct stynamic *sty;
	u_char byte;
{
	u_char *new_ptr;

	cli();
	if (sty->len < STYNAMIC_SIZE)
		sty->datas[sty->len++] = byte;
	else {
		if (sty->len < sty->allocated) {
			sty->datad[sty->len++] = byte;
		} else {
			new_ptr = kmalloc(sty->allocated + STYNAMIC_ALLOC,
			    GFP_ATOMIC);
			if (new_ptr == NULL) {
				printk("midi: out of memory all hell"
				   " will break loose\n");
				return;
			}
			if (sty->allocated == 0)
				memcpy(new_ptr, sty->datas, sty->len);
			else {
				memcpy(new_ptr, sty->datad, sty->len);
				kfree_s(sty->datad, sty->allocated);
			}
			sty->datad = new_ptr;
			sty->datad[sty->len++] = byte;
			sty->allocated += STYNAMIC_ALLOC;
		}
	}
	sti();
}

void
stynamic_add_bytes(sty, bytes, num)
	struct stynamic *sty;
	u_char *bytes;
	int num;
{
	int size_inc;
	u_char *new_ptr;

	cli();
	if (sty->len + num <= STYNAMIC_SIZE) {
		memcpy(&sty->datas[sty->len], bytes, num);
		sty->len += num;
	} else {
		if (sty->len + num <= sty->allocated) {
			memcpy(&sty->datad[sty->len], bytes, num);
			sty->len += num;
		} else {
			size_inc = (num / STYNAMIC_ALLOC + 1) * STYNAMIC_ALLOC;
			new_ptr = kmalloc(sty->allocated + size_inc,
			    GFP_ATOMIC);
			if (new_ptr == NULL) {
				printk("midi: out of memory all hell"
				   " will break loose\n");
				return;
			}
			if (sty->allocated == 0)
				memcpy(new_ptr, sty->datas, sty->len);
			else {
				memcpy(new_ptr, sty->datad, sty->len);
				kfree_s(sty->datad, sty->allocated);
			}
			sty->datad = new_ptr;
			memcpy(&sty->datad[sty->len], bytes, num);
			sty->allocated += size_inc;
			sty->len += num;
		}
	}
	sti();
}

u_char
stynamic_get_byte(sty, index)
	struct stynamic *sty;
	int index;
{

	cli();
	if (sty->len <= 4)
		return (sty->datas[index]);
	else
		return (sty->datad[index]);
	sti();
}

void
stynamic_copy(sty, dest, len)
	struct stynamic *sty;
	void *dest;
	int len;
{

	cli();
	if (sty->len <= 4)
		memcpy(dest, sty->datas, len);
	else
		memcpy(dest, sty->datad, len);
	sti();
}

void
stynamic_append(dest, src)
	struct stynamic *dest;
	struct stynamic *src;
{

	if (src->len <= 4)
		stynamic_add_bytes(dest, src->datas, src->len);
	else
		stynamic_add_bytes(dest, src->datad, src->len);
}

void
stynamic_copy_from(sty, index, dest, len)
	struct stynamic *sty;
	int index, len;
	void *dest;
{

	cli();
	if (sty->len <= 4)
		memcpy(dest, &sty->datas[index], len);
	else
		memcpy(dest, &sty->datad[index], len);
	sti();
}


void
stynamic_release(sty)
	struct stynamic *sty;
{

	cli();
	if (sty->len > STYNAMIC_SIZE)
		kfree_s(sty->datad, sty->allocated);
	sty->len = 0;
	sty->allocated = 0;
	sty->datad = NULL;
	memset(sty->datas, 0, STYNAMIC_SIZE);
	sti();
}

void
stynamic_shift(sty, num)
	struct stynamic *sty;
	int num;
{
	int rem;
	u_char *ptr;

	if (sty->len <= num) {
		stynamic_release(sty);
		return;
	}
	cli();
	if (sty->len > STYNAMIC_SIZE)
		ptr = &sty->datad[num];
	else
		ptr = &sty->datas[num];
	rem = sty->len - num;
	if (rem > STYNAMIC_SIZE)
		memcpy(sty->datad, ptr, rem);
	else {
		memcpy(sty->datas, ptr, rem);
		if (sty->datad != NULL) {
			kfree_s(sty->datad, sty->allocated);
			sty->datad = NULL;
			sty->allocated = 0;
		}
	}
	sty->len = rem;
	sti();
}

void
stynamic_init(sty)
	struct stynamic *sty;
{

	cli();
	sty->len = 0;
	sty->allocated = 0;
	sty->datad = NULL;
	memset(sty->datas, 0, STYNAMIC_SIZE);
	sti();
}

void
stynamic_print(sty)
	struct stynamic *sty;
{

	printk("\t\tlen = %d\n", sty->len);
	printk("\t\tallocated = %d\n", sty->allocated);
	printk("\t\tdatad = 0x%x\n", (int)(sty->datad));
}

#define CANSIGNAL(p, q) \
	((p)->session == (q)->session || \
	    (p)->euid == (q)->euid || \
	    (p)->uid == (q)->uid || \
	    suser())

int
midi_killable(struct task_struct *owner, int pid)
{
	int found;
	struct task_struct *p;

	if (pid == 0)
		return (0);
	if (pid == -1)
		return (EPERM);
	if (pid < 0) {
		found = 0;
		for_each_task(p) {
			if (p && p->pgrp == -pid) {
				found = 1;
				if (!CANSIGNAL(owner, p))
					return (EPERM);
			}
		}
		if (!found)
			return (ESRCH);
		return (1);
	}
	/* Normal kill */
	for_each_task(p) {
		if (p && p->pid == pid) {
			if (!CANSIGNAL(owner, p))
				return (EPERM);
			else
				return (0);
		}
	}
	return (ESRCH);
}

/*
 * I really wish I didn't have to do this, but Linux is causing
 * trouble again.  sys_kill and the associated functions assume
 * that the process doing the killing is the current process.
 * This just isn't always the case.  My midi driver needs to send
 * signals from a function called from add_timer.  In this case
 * the current function might not always be the function doing the
 * killing.
 * Perhaps the Linux powers that be will see this and make a change.
 */
asmlinkage int
midi_sys_kill(struct task_struct *owner, int pid, int sig)
{
	int err, retval = 0, count = 0;

	if (!pid)
		return(midi_kill_pg(owner,owner->pgrp,sig,0));
	if (pid == -1) {
		struct task_struct * p;
		for_each_task(p) {
			if (p->pid > 1 && p != owner) {
				++count;
				if ((err = midi_send_sig(owner,sig,p,0))
				    != -EPERM)
					retval = err;
			}
		}
		return(count ? retval : -ESRCH);
	}
	if (pid < 0) 
		return(midi_kill_pg(owner,-pid,sig,0));
	/* Normal kill */
	return(midi_kill_proc(owner,pid,sig,0));
}

int
midi_kill_pg(struct task_struct *owner, int pgrp, int sig, int priv)
{
	struct task_struct *p;
	int err,retval = -ESRCH;
	int found = 0;

	if (sig<0 || sig>32 || pgrp<=0)
		return -EINVAL;
	for_each_task(p) {
		if (p->pgrp == pgrp) {
			if ((err = midi_send_sig(owner,sig,p,priv)) != 0)
				retval = err;
			else
				found++;
		}
	}
	return(found ? 0 : retval);
}

int
midi_kill_proc(struct task_struct *owner, int pid, int sig, int priv)
{
 	struct task_struct *p;

	if (sig<0 || sig>32)
		return -EINVAL;
	for_each_task(p) {
		if (p && p->pid == pid)
			return midi_send_sig(owner,sig,p,priv);
	}
	return(-ESRCH);
}

int
midi_send_sig(struct task_struct *owner, unsigned long sig,
    struct task_struct * p,int priv)
{
	if (!p || sig > 32)
		return -EINVAL;
	if (!priv && ((sig != SIGCONT) || (owner->session != p->session)) &&
	    (owner->euid != p->euid) && (owner->uid != p->uid) && !suser())
		return -EPERM;
	if (!sig)
		return 0;
	if ((sig == SIGKILL) || (sig == SIGCONT)) {
		if (p->state == TASK_STOPPED)
			p->state = TASK_RUNNING;
		p->exit_code = 0;
		p->signal &= ~( (1<<(SIGSTOP-1)) | (1<<(SIGTSTP-1)) |
				(1<<(SIGTTIN-1)) | (1<<(SIGTTOU-1)) );
	}
	/* Depends on order SIGSTOP, SIGTSTP, SIGTTIN, SIGTTOU */
	if ((sig >= SIGSTOP) && (sig <= SIGTTOU)) 
		p->signal &= ~(1<<(SIGCONT-1));
	/* Actually generate the signal */
	midi_generate(sig,p);
	return 0;
}

int
midi_generate(unsigned long sig, struct task_struct * p)
{
	unsigned long mask = 1 << (sig-1);
	struct sigaction * sa = sig + p->sigaction - 1;

	/* always generate signals for traced processes ??? */
	if (p->flags & PF_PTRACED) {
		p->signal |= mask;
		return 1;
	}
	/* don't bother with ignored signals (but SIGCHLD is special) */
	if (sa->sa_handler == SIG_IGN && sig != SIGCHLD)
		return 0;
	/* some signals are ignored by default.. (but SIGCONT already did its deed) */
	if ((sa->sa_handler == SIG_DFL) &&
	    (sig == SIGCONT || sig == SIGCHLD || sig == SIGWINCH))
		return 0;
	p->signal |= mask;
	return 1;
}
