/* 
 *
 *	Test driver for TCL.
 *
 * Copyright 1987-1991 Regents of the University of California
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appears in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include "tcl.h"
#include "tclipc.h"

#define STANDALONE 1

#define TRUE 1
#define FALSE 0

/* Something I use to reduce length of arg lists */
typedef struct _tclInfo {
  Tcl_Interp *interp;
  Tcl_CmdBuf buffer;
} tclinfo, *tclInfo;
static tclInfo info1, info2, info3;

/* In file function prototypes */
static int cmdEcho _ANSI_ARGS_((ClientData *clientData, Tcl_Interp *interp, 
		int argc, char *argv[]));
static int cmdQuit _ANSI_ARGS_((ClientData *clientData, Tcl_Interp *interp, 
		int argc, char *argv[]));
static tclInfo tclInit _ANSI_ARGS_((void));
static void tclEventLoop _ANSI_ARGS_((tclInfo info));
static evalcmd_res* tclTestCmdReceive _ANSI_ARGS_((Tcl_Interp *interp, 
                  cmdText cmd));

/* Used for setting time for message checks */
#define TIMEOUTSECS      0
#define TIMEOUTUSECS     200000

#define MYBUFSIZ        64
/* Tcp variables */
static int quitFlag = FALSE;
static char prompt[MYBUFSIZ];

char *initCmd =
 "if [file exists [info library]/init.tcl] {source [info library]/init.tcl}";

#ifdef STANDALONE
int main(argc, argv)
int argc;
char *argv[];
{
  char tmpbuff[MYBUFSIZ];

  if (argc != 2) {
    fprintf(stderr, "usage: %s base-name\n", argv[0]);
    exit(1);
  }

  /* Set the prompt to show us a reminder */
  sprintf(prompt, "%s> ", argv[1]);
  /* Initialize IPC once */
  ipcSendInit(TIMEOUTSECS, TIMEOUTUSECS, TIMEOUTSECS, TIMEOUTSECS,
              NULL, NULL, NULL, NULL);

  info1 = tclInit();
  ipcInterpInit(info1->interp);
  sprintf(tmpbuff, "%s1", argv[1]);
  if (ipcNewReceiver(info1->interp, tmpbuff, "localhost") == (Receiver)NULL) {
    fprintf(stderr, "The PortManager could not register its own receiver.\n");
    exit(-1);
  }
  printf("Registered \"mipc\" receiver: %s\n", tmpbuff); 

  info2 = tclInit();
  ipcInterpInit(info2->interp);
  sprintf(tmpbuff, "%s2", argv[1]);
  if (ipcNewReceiver(info2->interp, tmpbuff, "localhost") == (Receiver)NULL) {
    fprintf(stderr, "The PortManager could not register its own receiver.\n");
    exit(-1);
  }
  printf("Registered \"mipc\" receiver: %s\n", tmpbuff); 

  info3 = tclInit();
  ipcInterpInit(info3->interp);
  sprintf(tmpbuff, "%s3", argv[1]);
  if (ipcNewReceiver(info3->interp, tmpbuff, "localhost") == (Receiver)NULL) {
    fprintf(stderr, "The PortManager could not register its own receiver.\n");
    exit(-1);
  }
  printf("Registered \"mipc\" receiver: %s\n", tmpbuff); 

  tclEventLoop(info1);

  ipcDestroyAll(info1->interp);
  Tcl_DeleteInterp(info1->interp);
  Tcl_DeleteCmdBuf(info1->buffer);
  Tcl_DeleteInterp(info2->interp);
  Tcl_DeleteCmdBuf(info2->buffer);
  Tcl_DeleteInterp(info3->interp);
  Tcl_DeleteCmdBuf(info3->buffer);
  return 0;
}
#endif

static tclInfo tclInit()
{
  int result;
  tclInfo info;

  info = (tclInfo)malloc(sizeof(tclinfo));
  assert(info != NULL);

  info->interp = Tcl_CreateInterp();

  /* Some standard tcl built-ins */
  Tcl_CreateCommand(info->interp, "echo", (Tcl_CmdProc *)cmdEcho,
                    (ClientData) "echo",
                    (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(info->interp, "quit", (Tcl_CmdProc *)cmdQuit,
                    (ClientData) 0,
                    (Tcl_CmdDeleteProc *) NULL);

  info->buffer = Tcl_CreateCmdBuf();

  /* Source the tcl library */
  result = Tcl_Eval(info->interp, initCmd, 0, (char **) NULL);
  assert(result == TCL_OK);

  return info;
}

static evalcmd_res* tclTestCmdReceive(interp, argp)
Tcl_Interp *interp;
cmdText argp;
{
  static evalcmd_res result = { 0, NULL };

  /* Only free after the first time */
  if (result.text != NULL) {
    free((char *)result.text);
  }
  printf(">>Test\n");
  result.errno = Tcl_Eval(interp, argp, 0, (char **)NULL);
  result.text = strdup(interp->result);
  return(&result);
}

static void tclEventLoop(info)
tclInfo info;
{
  char line[1000], *cmd;
  int result, gotPartial;

  gotPartial = 0;
  while (1) {
    clearerr(stdin);
    if (!gotPartial) {
      fputs(prompt, stdout);
      fflush(stdout);
    }
    if (fgets(line, 1000, stdin) == NULL) {
      if (!gotPartial) {
	exit(0);
      }
      line[0] = 0;
    }
    cmd = Tcl_AssembleCmd(info->buffer, line);
    if (cmd == NULL) {
      gotPartial = 1;
      continue;
    }

    gotPartial = 0;
    result = Tcl_RecordAndEval(info->interp, cmd, 0);
    if (result == TCL_OK) {
      if (*(info->interp->result) != 0) {
	printf("%s\n", info->interp->result);
      }
      if (quitFlag) {
	return;
      }
    } else {
      if (result == TCL_ERROR) {
	printf("Error");
      } else {
	printf("Error %d", result);
      }
      if (*(info->interp->result) != 0) {
	printf(": %s\n", info->interp->result);
      } else {
	printf("\n");
      }
    }
  }
}



/* Echo command arguments */
static int cmdEcho(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  int i;

  for (i = 1; ; i++) {
    if (argv[i] == NULL) {
      if (i != argc) {
        echoError:
        sprintf(interp->result,
          "argument list wasn't properly NULL-terminated in \"%s\" command",
          argv[0]);
      }
      break;
    }
    if (i >= argc) {
      goto echoError;
    }
    fputs(argv[i], stdout);
    if (i < (argc-1)) {
      printf(" ");
    }
  }
  printf("\n");
  return TCL_OK;
}

/* A formal quit command */
static int cmdQuit(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
  quitFlag = 1;
  return TCL_OK;
}

