/* 
 * tmResult.c --
 *
 *	Allows a tcl result to be saved elsewhere
 *
 * Copyright (c) 1993 J.D. Newmarch
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE AUTHOR HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#ifndef lint
static char rcsid[] = "$Header";
#endif /* not lint */

#include <tcl.h>
#include "tmFuncs.h"

/*
 * A callback returns the result to the interpreter as usual. If an
 * error occurs Tm_CallbackHandler reports it, otherwise the result
 * is discarded - fine, no-one usually wants it anyway.
 *
 * When an action occurs, several callbacks may be invoked, as in
 * PushButton's ArmAndActivate. Only the last result is then available.
 * For batch mode testing of such behaviour, where we use callActionProc
 * we do need to keep all callback results.
 *
 * This file allows saves of results from one command to another
 */

static char empty[] = "";

/*
 *----------------------------------------------------------------------
 * Tm_SaveResult -
 *	Test if the vbl _Tm_SaveResult is True or otherwise
 *
 * Result
 *	True if vbl set to true, false otherwise
 *
 * Side effects
 *	none
 *----------------------------------------------------------------------
 */

int
Tm_SaveResult(interp)
    Tcl_Interp *interp;
{
    char *value;
    int bool;

    if ((value = Tcl_GetVar(interp, TM_SAVE_RESULT, TCL_GLOBAL_ONLY)) == NULL) {
	return False;
    }
    if (Tcl_GetBoolean(interp, value, &bool) != TCL_OK) {
	return False;
    }
    return bool;
}

/*
 *----------------------------------------------------------------------
 * Tm_ClearResult -
 *	Empty the result buffer
 *
 * Result
 *	None
 *
 * Side effects
 *	frees the result buf
 *----------------------------------------------------------------------
 */

void
Tm_ClearResult(interp)
    Tcl_Interp *interp;
{
    Tcl_SetVar(interp, TM_RESULT_BUF, "", TCL_GLOBAL_ONLY);
}

/*
 *----------------------------------------------------------------------
 * Tm_AppendResult -
 *	Add to the Tm result buffer
 *
 * Result
 *	none
 *
 * Side effects
 *	alters Tm_ResultBuf
 *----------------------------------------------------------------------
 */

void
Tm_AppendResult(interp, str)
    Tcl_Interp *interp;
    char *str;
{
    if (Tcl_GetVar(interp, TM_RESULT_BUF, TCL_GLOBAL_ONLY) == NULL) {
	Tcl_SetVar(interp, TM_RESULT_BUF, str, TCL_GLOBAL_ONLY);
    } else {
	Tcl_SetVar(interp, TM_RESULT_BUF, str, 
		(TCL_APPEND_VALUE | TCL_GLOBAL_ONLY));
    }
}

/*
 *----------------------------------------------------------------------
 * Tm_StartSavingResult -
 *	record that a save is going to start
 *
 * Result
 *	none
 *
 * Side effects
 *	alters interp
 *----------------------------------------------------------------------
 */
void
Tm_StartSavingResult(interp)
    Tcl_Interp *interp;
{
    Tcl_SetVar(interp, TM_SAVE_RESULT, "true", TCL_GLOBAL_ONLY);
}

/*
 *----------------------------------------------------------------------
 * Tm_StopSavingResult -
 *	record that saving is stopping
 *
 * Result
 *	none
 *
 * Side effects
 *	alters interp
 *----------------------------------------------------------------------
 */

void
Tm_StopSavingResult(interp)
    Tcl_Interp *interp;
{
    Tcl_SetVar(interp, TM_SAVE_RESULT, "false", TCL_GLOBAL_ONLY);
}


/*
 *----------------------------------------------------------------------
 * Tm_Result -
 *	return the Tm result buffer
 *
 * Result
 *	the Tm result buffer
 *
 * Side effects
 *	none
 *----------------------------------------------------------------------
 */

char *
Tm_Result(interp)
    Tcl_Interp *interp;
{
    return Tcl_GetVar(interp, TM_RESULT_BUF, TCL_GLOBAL_ONLY);
}
