
/*
 * TmResources.c --
 *	This module contains the functions to implement resource
 *	handling such as getValues, setValues, and resources
 *
 * Copyright 1993 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include "tm.h"
#include "tmFuncs.h"
#include <Xm/Xm.h>
#include <Xm/List.h>

/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertValue --
 *
 *	convert a value from one format to another
 *
 * Results:
 *
 *	new value (of whatever type) stored in "new_value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
Tm_ConvertValue(w, fromType, fromValue, fromSize,
		   toType, toValue, toSize)
    Widget w;
    char *fromType;
    char *fromValue;
    unsigned int fromSize;
    char *toType;
    XtArgVal *toValue;
    unsigned int toSize;
{
    XrmValue from, to;

    if (strcmp(fromType, toType) == 0) {
	*toValue = (XtArgVal) fromValue;
	return TRUE;
    }

    from.size = fromSize;
    from.addr = fromValue;
    to.size = toSize;
    to.addr = XtMalloc(toSize);

    if (!XtConvertAndStore(w, fromType, &from, toType, &to))  {
	fprintf(stderr, "failed conversion %s to %s\n",
				fromType, toType);
	return FALSE;
    }
    *toValue = 0;
    _XtCopyToArg(to.addr, toValue, to.size);
/*
    XtFree(to.addr);
*/

   return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertValueFromString --
 *
 *	convert a value from its Tcl string format to the one
 *	the widget wants.
 *
 * Results:
 *
 *	new value (of whatever type) stored in "new_value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
Tm_ConvertValueFromString(w, resources, num_resources,
		resource, orig_value, new_value)
    Widget w;
    XtResourceList resources;
    int num_resources;
    char *resource;
    char *orig_value;
    XtArgVal *new_value;
{
    int n;
    XrmValue from, converted;
    XtArgVal tmpval;

#   ifdef DEBUG
    fprintf(stderr, "converting from string \"%s\" of type %s\n", orig_value, resource);
#   endif
    for (n = 0; n < num_resources; n++) {
	if (strcmp(resource, resources->resource_name) == 0) {
	    break;
	}
	resources++;
    }
    if (n == num_resources) {
	return FALSE;
    }

    /* we have a match - convert and install the resource */
    /* special case - same type */
    return Tm_ConvertValue(w, XtRString, orig_value,
				strlen(orig_value)+1,
				resources->resource_type,
				new_value,
				resources->resource_size);
/*
    if (strcmp(resources->resource_type, XtRString) == 0) {
	*new_value =  orig_value;
	return TRUE;
    }
    from.size = strlen(orig_value) + 1;
    from.addr = orig_value;
    converted.size = resources->resource_size;
    converted.addr = XtMalloc(converted.size);
    if (!XtConvertAndStore(w, XtRString, &from,
		resources->resource_type, &converted)) {
	return FALSE;
    }
    *new_value = 0;
    _XtCopyToArg(converted.addr, new_value, converted.size);
*/
/*
    bzero(new_value, sizeof(XtArgVal));
    bcopy(converted.addr, new_value, converted.size);
    *new_value = *(XtArgVal *) (converted.addr);
*/
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertValueFromStringQuark --
 *
 *	convert a value from its Tcl string format to the one
 *	the widget wants. In this form the resources have been
 *	quark'ed so we don't have string compares but int ones
 *
 * Results:
 *
 *	new value (of whatever type) stored in "new_value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
Tm_ConvertValueFromStringQuark(w, resources, num_resources,
		resource, orig_value, new_value)
    Widget w;
    XtResourceList resources;
    int num_resources;
    char *resource;
    char *orig_value;
    XtArgVal *new_value;
{
    int n;
    XrmValue from, converted;
    XtArgVal tmpval;
    int resourceQ;

    resourceQ = XrmStringToQuark(resource);
#   ifdef DEBUG
    fprintf(stderr, "converting from string \"%s\" of type %s\n",
			orig_value, resource);
#   endif
    for (n = 0; n < num_resources; n++) {
	if (resourceQ == (int) resources->resource_name) {
	    break;
	}
	resources++;
    }
    if (n == num_resources) {
	return FALSE;
    }

    /* we have a match - convert and install the resource */
    /* special case - same type */
    if (strcmp(XrmQuarkToString((int) resources->resource_type), XtRString) == 0) {
	*new_value =  (XtArgVal) orig_value;
	return TRUE;
    }
    from.size = strlen(orig_value) + 1;
    from.addr = orig_value;
    converted.size = resources->resource_size;
    converted.addr = XtMalloc(converted.size);
    if (!XtConvertAndStore(w, XtRString, &from,
		XrmQuarkToString((int) resources->resource_type), &converted)) {
	return FALSE;
    }
    *new_value = 0;
    _XtCopyToArg(converted.addr, new_value, converted.size);
/*
    bzero(new_value, sizeof(XtArgVal));
    bcopy(converted.addr, new_value, converted.size);
    *new_value = *(XtArgVal *) (converted.addr);
*/
    
    return TRUE;
}
		

/*
 *--------------------------------------------------------------
 *
 * Tm_MaybeSetStringTableSize --
 *
 *	Hack for Motif XmStringTables.
 *	Motif *sometimes* NULL terminates its XmStringTables
 *	most times doesnt.
 *	XmStringGet...() sometimes returns False on running off
 *	the end of a table, sometimes crashes
 *	_XmStringIsXmString() sometimes returns False on running off
 *	the end of a table, sometimes crashes
 *	i.e. there is no internal way of finding the size of one
 *	of these things. So we have to check if we have one and
 *	then look up the resource that tells us the size :-(
 *
 * Results:
 *
 * Side effects:
 *
 *	Sets global vbl Tm_XmStringTableSize
 *--------------------------------------------------------------
 */

static void
Tm_MaybeSetStringTableSize(w, resource)
    Widget w;
    String resource;
{
    extern int Tm_XmStringTableSize;
    WidgetClass class;
    Arg arg;
    char *size_resource;

    class = XtClass(w);
    if (class == xmListWidgetClass) {
	if (strcmp (resource, XmNitems) == 0) {
	    size_resource = XmNitemCount;
	} else
	if (strcmp (resource, XmNselectedItems) == 0) {
	    size_resource = XmNselectedItemCount;
	} else
	return;
    } else
    return;

    /* Sigh, we have a StringTable, find its size */
    XtSetArg(arg, size_resource, &Tm_XmStringTableSize);
    XtGetValues(w, &arg, 1);
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CovertResourceToString --
 *	convert a single resource to a string
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
Tm_ConvertResourceToString(w, resource, resource_size, resource_type, value)
    Widget w;
    char *resource;
    Cardinal resource_size;
    String resource_type;
    char **value;
{
    XrmValue from, converted;
    Arg args[1];

    Tm_MaybeSetStringTableSize(w, resource);

    from.size = resource_size;
    from.addr = (XtPointer) XtMalloc(from.size);
    XtSetArg(args[0], resource, from.addr);
    XtGetValues(w, args, 1);
    
    /* special case - same type */
    if (strcmp(resource_type, XtRString) == 0) {
	*value =  *(char **) from.addr;
	return TRUE;
    }
    converted.addr = NULL;
    if (!XtConvertAndStore(w, resource_type, &from,
			XtRString, &converted)) {
	return FALSE;
    }
    bzero(value, sizeof(XtArgVal));
    bcopy(converted.addr, value, converted.size);
    
    XtFree(from.addr);
    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertValueToString --
 *
 *	converts from the internal widget form to a string for Tcl
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
Tm_ConvertValueToString(w, resources, num_resources,
		resource, new_value)
    Widget w;
    XtResourceList resources;
    int num_resources;
    char *resource;
    char **new_value;
{
    int n;

    for (n = 0; n < num_resources; n++) {
	if (strcmp(resource, resources->resource_name) == 0) {
	    break;
	}
	resources++;
    }
    if (n == num_resources) {
	return FALSE;
    }

    return Tm_ConvertResourceToString(w, resource,
		resources->resource_size, resources->resource_type, new_value);

}
		
/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertSubValueFromString --
 *
 *	convert a value from its Tcl string format to the one
 *	the widget wants.
 *
 * Results:
 *
 *	new value (of whatever type) stored in "new_value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static Boolean
Tm_ConvertSubValueFromString(w, sec_resources, num_sec_resources,
		resource, orig_value, new_value)
    Widget w;
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
    char *resource;
    char *orig_value;
    XtArgVal *new_value;
{
    int n, length;

    for (n = 0; n < num_sec_resources; n++) {
       	if (Tm_ConvertValueFromString(w, sec_resources[n]->resources, 
            sec_resources[n]->num_resources, resource,
       	    orig_value, new_value)) {
	    return True;
	}
    }
    return False;
}
		
/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertSubValueToString --
 *
 *	converts from the internal widget form to a string for Tcl
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static Boolean
Tm_ConvertSubValueToString(w, sec_resources, num_sec_resources,
		resource, new_value)
    Widget w;
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
    char *resource;
    char **new_value;
{
    int n, length;

    for (n = 0; n < num_sec_resources; n++) {
       	if (Tm_ConvertValueToString(w, sec_resources[n]->resources, 
            sec_resources[n]->num_resources, resource,
       	    new_value)) {
	    return True;
	}
    }
    return False;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_FreeSecondaryResources --
 *
 *	Motif gives us a copy of these, so we have to tidy up
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_FreeSecondaryResources(sec_resources, num_sec_resources)
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
{
    int n;

    for (n = 0; n < num_sec_resources; n++) {
	XtFree((char *) sec_resources[n]->resources);
	XtFree((char *) sec_resources[n]);
    }

    if (num_sec_resources > 0) {
        XtFree((char *) sec_resources);
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tm_SetValues --
 *
 *	set resource values on a widget
 *
 * Results:
 * 	a tcl result
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

int
Tm_SetValues(pathName, interp, w, parent, class, argv, argc, args, num_args)
    char *pathName;
    Tcl_Interp *interp;
    Widget w;
    Widget parent;
    WidgetClass class;
    char **argv;
    int argc;
    Arg args[];
    int *num_args;
{
    XtResourceList resources, constraint_resources;
#ifndef MOTIF11
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
#endif
    Cardinal num_resources;
    Cardinal num_constraint_resources;
    int num_values = 0;
    XrmValue from, converted;
    XtArgVal new_value;
    char *resource;
    char *value;
    Tm_ClientData *client_data;

    *num_args = 0;

    if (argc > TM_MAXARGS*2) {
	return TCL_ERROR;
    }

    XtGetResourceList(class, &resources, &num_resources);
#ifndef MOTIF11
    num_sec_resources = XmGetSecondaryResourceData(class, &sec_resources);
#endif
    if (parent != NULL && XtIsConstraint(parent)) {
	XtGetConstraintResourceList(XtClass(parent), 
	    &constraint_resources, &num_constraint_resources);
    } else {
	num_constraint_resources = 0;
    }

    while (argc >= 2) {
	if (argv[0][0] != '-') {
	    sprintf(interp->result, "illegal setValues option \"%50s\"", argv[0]);
	    return TCL_ERROR;
	}
	resource = argv[0]+1;
        value = argv[1];

        if (Tm_ConvertValueFromString(w, resources, num_resources,
		resource, value, &new_value)) {
            XtSetArg(args[num_values], resource, new_value);
	    num_values++;
	} else
        if (Tm_ConvertValueFromString(w, constraint_resources, 
	            num_constraint_resources, resource,
        	    value, &new_value)) {
            XtSetArg(args[num_values], resource, new_value);
	    num_values++;
	} else 
#ifndef MOTIF11
        if (Tm_ConvertSubValueFromString(w, sec_resources, 
	            num_sec_resources, resource,
        	    value, &new_value)) {
            XtSetArg(args[num_values], resource, new_value);
	    num_values++;
	} else
#endif
	{
	    sprintf(interp->result, "Conversion from String to \"%50s\" failed\n",
                                resource);
	    return TCL_ERROR;
	}
	argc -= 2;
	argv += 2;
    }

    *num_args = num_values;
    return TCL_OK;

/*
    Tm_FreeSecondaryResources(sec_resources, num_sec_resources);
*/
    /* memory leak: should XtFree all new_values stored in args[] */
}

/*
 *--------------------------------------------------------------
 *
 * Tm_GetValues --
 *
 *	get the resource values out of a widget
 *
 * Results:
 *	a tcl result
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

int
Tm_GetValues(pathName, interp, w, class, argv, argc)
    char *pathName;
    Tcl_Interp *interp;
    Widget w;
    WidgetClass class;
    char **argv;
    int argc;
{
    XtResourceList resources, constraint_resources;
#ifndef MOTIF11
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
#endif
    Cardinal num_resources;
    Cardinal num_constraint_resources;
    int num_values = 0;
    XrmValue from, converted;
    char *new_value;
#   define MAXARGS 100
    char *resource;
    Tm_ClientData *client_data;
    Widget parent;

    if (argc > MAXARGS*2) {
	return TCL_ERROR;
    }

    parent = XtParent(w);	/* this is NULL for w == "." */
    XtGetResourceList(class, &resources, &num_resources);

#ifndef MOTIF11
    num_sec_resources = XmGetSecondaryResourceData(class, &sec_resources);
#endif
  
    if (parent != NULL && XtIsConstraint(parent)) {
	XtGetConstraintResourceList(XtClass(parent), 
	    &constraint_resources, &num_constraint_resources);
    } else {
	num_constraint_resources = 0;
    }

    while (argc >= 2) {
	if (argv[0][0] != '-') {
	    sprintf(interp->result, "illegal setValues option \"%50s\"", argv[0]);
	    return TCL_ERROR;
	}
	resource = argv[0]+1;

        if (Tm_ConvertValueToString(w, resources, num_resources,
		resource, &new_value)) {
	    /* store new_value in variable in argv[1] now */
#	    ifdef DEBUG
	    fprintf(stderr, "Got value: %s\n", (char *) new_value);
#	    endif
	    if (Tcl_SetVar(interp, argv[1], (char *) new_value,
			TCL_LEAVE_ERR_MSG) == NULL)
		fprintf(stderr, "%s\n", interp->result);
	} else
        if (Tm_ConvertValueToString(w, constraint_resources, 
	            num_constraint_resources, resource,
        	    &new_value)) {
	    /* store in a variable now */
	    if (Tcl_SetVar(interp, argv[1], (char *) new_value,
			TCL_LEAVE_ERR_MSG) == NULL)
		fprintf(stderr, "%s\n", interp->result);
	} else 
#ifndef MOTIF11
        if (Tm_ConvertSubValueToString(w, sec_resources, 
	            num_sec_resources, resource,
        	    &new_value)) {
	    /* store in a variable now */
	    if (Tcl_SetVar(interp, argv[1], (char *) new_value,
			TCL_LEAVE_ERR_MSG) == NULL)
		fprintf(stderr, "%s\n", interp->result);
	} else
#endif
	{
	    sprintf(interp->result, "Conversion from %s to String failed\n",
					resource);
	    return TCL_ERROR;
	}
	argc -= 2;
	argv += 2;
    }
#ifndef MOTIF11
    Tm_FreeSecondaryResources(sec_resources, num_sec_resources);
#endif
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ResourceInfo --
 *
 *	get the resource info out of a widget
 *
 * Results:
 *	a tcl result
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_GetResourceInfo(resourceInfo, w, resource)
    Tcl_DString *resourceInfo;
    Widget w;
    XtResource *resource;
{
    char *resource_value;

    Tcl_DStringAppend(resourceInfo, "-", 1);
    Tcl_DStringAppend(resourceInfo, resource->resource_name, -1);
    Tcl_DStringAppendElement(resourceInfo, resource->resource_name);
    Tcl_DStringAppendElement(resourceInfo, resource->resource_class);
    Tcl_DStringAppendElement(resourceInfo, resource->resource_type);

    if (Tm_ConvertResourceToString(w, resource->resource_name,
		resource->resource_size, resource->resource_type,
		&resource_value)) {;
        Tcl_DStringAppendElement(resourceInfo, resource_value);
    } else {
        Tcl_DStringAppendElement(resourceInfo, "");
    }
}
/*
 *--------------------------------------------------------------
 *
 * Tm_NoErrorMsg --
 *
 * Results:
 * 	none
 *
 * Side effects:
 *	does nothing
 *
 *--------------------------------------------------------------
 */

void
Tm_NoErrorMsg(name, type, Class, defalt, params, num_params)
    String name;
    String type;
    String Class;
    String defalt;
    String *params;
    Cardinal *num_params;
{
    /* void */
}


/*
 *--------------------------------------------------------------
 *
 * Tm_ResourceList --
 *
 *	get the resource list out of a widget
 *
 * Results:
 *	a tcl result
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

int
Tm_ResourceList(interp, w, class)
    Tcl_Interp *interp;
    Widget w;
    WidgetClass class;
{
    XtResourceList resources, constraint_resources;
#ifndef MOTIF11
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
#endif
    Cardinal num_resources;
    Cardinal num_constraint_resources;
    int num_values = 0;
    char *new_value;
    char *resource;
    Tm_ClientData *client_data;
    Widget parent;
    int m, n;
    Tcl_DString resourceList, resourceInfo;
    XtErrorHandler oldWarningHandler;

    parent = XtParent(w);	/* this is NULL for w == "." */
    XtGetResourceList(class, &resources, &num_resources);

#ifndef MOTIF11
    num_sec_resources = XmGetSecondaryResourceData(class, &sec_resources);
#endif
  
    if (parent != NULL && XtIsConstraint(parent)) {
	XtGetConstraintResourceList(XtClass(parent), 
	    &constraint_resources, &num_constraint_resources);
    } else {
	num_constraint_resources = 0;
    }

    /* turn off error messages about converters that we don't support */
    oldWarningHandler = XtAppSetWarningHandler(XtWidgetToApplicationContext(w), 
					Tm_NoErrorMsg);

    Tcl_DStringInit(&resourceList);

    Tcl_DStringInit(&resourceInfo);
    for (n = 0; n < num_resources; n++) {
	Tm_GetResourceInfo(&resourceInfo, w, resources+n);
	/* hack to get string out of D-string */
	Tcl_DStringResult(interp, &resourceInfo);
	Tcl_DStringAppendElement(&resourceList, interp->result);
    }

#   ifndef MOTIF11
    Tcl_DStringInit(&resourceInfo);
    for (n = 0; n < num_sec_resources; n++) {
	for (m = 0; m < sec_resources[n]->num_resources; m++) {
	    Tm_GetResourceInfo(&resourceInfo, w, &(sec_resources[n]->resources[m]));
	    /* hack to get string out of D-string */
	    Tcl_DStringResult(interp, &resourceInfo);
	    Tcl_DStringAppendElement(&resourceList, interp->result);
	}
    }
#   endif

    Tcl_DStringInit(&resourceInfo);
    for (n = 0; n < num_constraint_resources; n++) {
	Tm_GetResourceInfo(&resourceInfo, w, constraint_resources+n);
	/* hack to get string out of D-string */
	Tcl_DStringResult(interp, &resourceInfo);
	Tcl_DStringAppendElement(&resourceList, interp->result);
    }

    /* turn error reporting back on */
    XtAppSetWarningHandler(XtWidgetToApplicationContext(w), oldWarningHandler);

    Tcl_DStringResult(interp, &resourceList);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_GetAppResources -- get the resources specific to the
 *	application level
 *
 * Results:
 * 	standard tcl result
 *
 * Side effects:
 *	sets tcl vbls with the application resources
 *
 *--------------------------------------------------------------
 */

int
Tm_GetAppResources(interp, w, resource_str)
    Tcl_Interp *interp;
    Widget w;
    String resource_str;
{
    char **resource_list;
    int num_resources;
    char **res_elmts;
    int num_elmts;
    XtResourceList resources;
    String *vbls;
    String *values;
    int n;

    Tcl_SplitList(interp, resource_str, &num_resources, &resource_list);
    resources = (XtResource *) XtMalloc(num_resources * sizeof(XtResource));
    vbls = (String *) XtMalloc(num_resources * sizeof(String));
    values = (String *) XtMalloc(num_resources * sizeof(String));

    for (n = 0; n < num_resources; n++) {
	Tcl_SplitList(interp, resource_list[n], &num_elmts, &res_elmts);
	if (num_elmts != 4) {
	    sprintf(interp->result, "wrong getAppResources \"%50s\"",
			resource_list[n]);
	   return TCL_ERROR;
	}

	resources[n].resource_name = res_elmts[0];
	resources[n].resource_class = res_elmts[1];
	resources[n].resource_type = XtRString;
	resources[n].resource_offset = n * sizeof(String);
	resources[n].default_type = XtRString;
	resources[n].default_addr = res_elmts[2];

	vbls[n] = res_elmts[3];
    }

    XtGetApplicationResources(w, values, resources, num_resources, NULL, 0);

    for (n = 0; n < num_resources; n++) {
	Tcl_SetVar(interp, vbls[n], values[n], 0);
    }

    return TCL_OK;
}
