#include <tk.h>
#include <malloc.h>
#include <X11/Xutil.h>
#include "tkRaster.h"
#include "tkRasterImage.h"
#include "ppm.h"

/*
 *  Loading images with arbitrary colors requires that we keep
 *  some data structure to find/store similar colors. Here,
 *  we implement a simple array of lists of XColors. 
 */

#define Distance(pix1,pix2) /* Estimates how different pix1 and pix2 are */ \
  (((pix1).r > (pix2).r ? (pix1).r - (pix2).r : (pix2).r - (pix1).r) + \
   ((pix1).g > (pix2).g ? (pix1).g - (pix2).g : (pix2).g - (pix1).g) + \
   ((pix1).b > (pix2).b ? (pix1).b - (pix2).b : (pix2).b - (pix1).b))

#define MaxDistance 255*3 /* max value Distance can return */
#define MaxBucket 8 /* max number of different colors per bucket */

typedef struct colorStruct {
   XColor * xcolor;
   struct colorStruct * closest;
   pixel pix;
   int freq;
} colorNode;

typedef struct colorListNode {
   colorNode * colorptr;
   struct colorListNode * next;
} * colorList;

/*
 *  Structure describing an image:
 */

typedef struct {
   XImage* ximage;
   char* name;
   int width, height;
   int x, y;
   colorList neededColors;
} tkImage;


static void DestroyTkImage (imagetable, tkimage)
/*          --------------
 *
 *  Deallocates tkimage and removes it from the hash table
 */
     tkImage* tkimage;
     Tcl_HashTable *imagetable;
{	
   Tcl_HashEntry *entryPtr;
   colorList tmp;
   
   entryPtr = Tcl_FindHashEntry (imagetable, tkimage->name);
   if (entryPtr != NULL) {
      Tcl_DeleteHashEntry (entryPtr);
   }
   free (tkimage->name);
   if (tkimage->ximage != NULL) XDestroyImage (tkimage->ximage);
   while (tkimage->neededColors != NULL) {
      tmp = tkimage->neededColors;
      Tk_FreeColor (tmp->colorptr->xcolor);
      free (tmp->colorptr);
      tkimage->neededColors = tmp->next;
      free (tmp);
   }
   free (tkimage);
}
   
static int RasterGetImage (interp, imagetable, tkimageptr, imagename)
/*         --------------
 *
 *  Gets the image pointer corresponding to the image named 'imagename'
 *  and stores it in *tkimageptr. If such an image is not defined in
 *  the 'imagetable', a standart tcl error is returned.
 *  
 */
     Tcl_Interp * interp;
     Tcl_HashTable* imagetable;
     tkImage ** tkimageptr;
     char * imagename;
{
   Tcl_HashEntry * entryPtr;

   if ((entryPtr = Tcl_FindHashEntry (imagetable, imagename)) == NULL) {
      Tcl_AppendResult (interp, "image ", imagename, " not defined",
			"\n", (char*)NULL);
      return TCL_ERROR;
   }

   *tkimageptr = (tkImage*) Tcl_GetHashValue (entryPtr);

   return TCL_OK;
}
   
static int RasterGetNewImage (interp, imagetable, tkimageptr, imagename)
/*         -----------------
 *
 *  Creates a skeleton for a tkImage named 'imagename', stores
 *  a pointer to it in *tkimageptr and registers it under the given
 *  name in the hashtable 'imagetable'. If image already exists,
 *  a standard tcl error is returned.
 */
     Tcl_Interp * interp;
     Tcl_HashTable* imagetable;
     tkImage ** tkimageptr;
     char * imagename;
{
   Tcl_HashEntry * entryPtr;
   tkImage * tkimage;
   int new;

   entryPtr = Tcl_CreateHashEntry (imagetable, imagename, &new);
   if (!new) {
      Tcl_AppendResult (interp, "image ", imagename, " already defined",
			"\n", (char*)NULL);
      return TCL_ERROR;
   }

   tkimage = (tkImage*) malloc (sizeof (tkImage));
   tkimage->name = (char*) malloc (strlen (imagename) + 1);
   strcpy (tkimage->name, imagename);
   tkimage->ximage = NULL;
   tkimage->width = 0;
   tkimage->height = 0;
   tkimage->neededColors = (colorList) NULL;

   Tcl_SetHashValue (entryPtr, tkimage);

   *tkimageptr =  tkimage;

   return TCL_OK;
}

static int RasterImageCreate (interp, raster, imagetable, argc, argv)
/*         ----------------- 
 *
 *  Creates an image of given width and height for displaying on 
 *  tkwin. Command:
 *
 *  <path> image create <img_name> <width> <height>
 *
 */
     Tcl_Interp * interp;
     Tk_Raster *raster;
     Tcl_HashTable *imagetable;
     int argc;
     char * argv [];
{  
   int wid, hgt, bytesperline;
   char * imagedata;
   tkImage* tkimage;
   Tk_Window tkwin = GetRasterTkWin (raster);
   
   if (argc < 3) { 
      interp->result = "usage: pathname image create <name> <wid> <hgt>";
      return TCL_ERROR;
   }

   if (RasterGetNewImage (interp, imagetable, &tkimage, argv [0]) != TCL_OK) {
      return TCL_ERROR;
   }
   
   argc--;
   argv++;
  
   /* 
    * Parse <wid> and <hgt>
    */

   if (Tcl_GetInt (interp, argv [0], &wid) != TCL_OK ||
       wid <= 0 || wid > 4096) {
      Tcl_AppendResult (interp, "wrong width: ", argv [0], 
			" should be a number between 1 and 4096",
			(char*)NULL);
      return TCL_ERROR;
   }
   if (Tcl_GetInt (interp, argv [1], &hgt) != TCL_OK || 
       hgt <= 0 || hgt > 4096) {
      Tcl_AppendResult (interp, "wrong height ", argv [1],
			" should be a number between 1 and 4096",
			(char*)NULL);
      return TCL_ERROR;
   }

   /*
    *  Try to alloc memory to store image pixels
    */
   bytesperline = (Tk_Depth (tkwin)+1)/8*wid;
   imagedata = malloc (bytesperline*hgt);
   if (imagedata == NULL) {
      Tcl_AppendResult (interp, "no memory for an image this big",
			(char*)NULL);
      DestroyTkImage (imagetable, tkimage);
      return TCL_ERROR;
   }

   tkimage->width = wid;
   tkimage->height = hgt;
   tkimage->ximage = XCreateImage (Tk_Display (tkwin),
				   Tk_Visual (tkwin),
				   Tk_Depth (tkwin),
				   ZPixmap,
				   0,
				   imagedata,
				   (unsigned)wid,
				   (unsigned)hgt,
				   8,
				   bytesperline);

   return TCL_OK;
}

static int RasterImageDelete (interp, raster, imagetable, argc, argv)
/*         ----------------- 
 *
 *  Deletes an image previously created or loaded. Command:
 *
 *  <path> image delete <img_name>
 *
 */
     Tcl_Interp * interp;
     Tk_Raster* raster;
     Tcl_HashTable* imagetable;
     int argc;
     char * argv [];
{
   tkImage *tkimage;

   if (argc < 1) {
      interp->result = "usage: pathname image delete <name>";
      return TCL_ERROR;
   }

   if (RasterGetImage (interp, imagetable, &tkimage, argv [0]) != TCL_OK) {
      return TCL_ERROR;
   }

   DestroyTkImage (imagetable, tkimage);
   return TCL_OK;
}   

static int RasterImageInfo (interp, raster, imagetable, argc, argv)
/*         ---------------
 *
 *  Returns a string with the width and height of the given image. 
 *  Command:
 *
 *  <path> image info <img_name>
 */
     Tcl_Interp * interp;
     Tk_Raster* raster;
     Tcl_HashTable* imagetable;
     int argc;
     char * argv [];    
{
   tkImage *tkimage;

   if (argc < 1) {
      interp->result = "usage: pathname image info <name>";
      return TCL_ERROR;
   }

   if (RasterGetImage (interp, imagetable, &tkimage, argv [0]) != TCL_OK) {
      return TCL_ERROR;
   }

   sprintf (interp->result, "%d %d", tkimage->width, tkimage->height);
   return TCL_OK;
}   

static int RasterImageGet (interp, raster, imagetable, argc, argv)
/*         -------------- 
 *
 *  Parses a command for moving pixels between stored images
 *  and windows/pixmaps. Command:
 *
 *    <path> image get <img_name> <x> <y> 
 */
     Tcl_Interp * interp;
     Tk_Raster* raster;
     Tcl_HashTable* imagetable;
     int argc;
     char * argv [];  
{
   tkImage* tkimage;
   int x, y, xsrc, xdst, ysrc, ydst;
   unsigned int wid, hgt, wid_return, hgt_return;
   Window root_return;
   int x_return, y_return;
   int border_width_return, depth_return;
   GC gc = GetRasterGC (raster);
   Drawable d = GetRasterDrawable (raster);
   Tk_Window tkwin = GetRasterTkWin (raster);

   /* 
    * Parse image name
    */

   if (argc < 3) {
      interp->result = "usage: pathname image get <name> <x> <y>";
      return TCL_ERROR;
   }

   if (RasterGetImage (interp, imagetable, &tkimage, argv [0]) != TCL_OK) {
      return TCL_ERROR;
   }

   argc--;
   argv++;

   /* 
    * Parse <x> and <y>
    */

   if (Tcl_GetInt (interp, argv [0], &x) != TCL_OK ||
       Tcl_GetInt (interp, argv [1], &y) != TCL_OK) {
      Tcl_AppendResult (interp, "wrong pixel position: ", argv [0], " ",
			argv[1], (char*)NULL);
      return TCL_ERROR;
   }

   /* 
    * Get Image 
    */
   xsrc = 0;
   ysrc = 0;
   xdst = x;
   ydst = y;
   wid = tkimage->width;
   hgt = tkimage->height;

   XGetGeometry (Tk_Display (tkwin), d, &root_return, &x_return, &y_return, 
	 &wid_return, &hgt_return, &border_width_return, &depth_return);
   if (x < 0) { xsrc = -x; xdst = 0; }
   if (y < 0) { ysrc = -y; ydst = 0; }
   if (xdst >= wid_return || ydst >= hgt_return) return TCL_OK;
   if (x+wid > wid_return) { wid = wid_return-x; }
   if (y+hgt > hgt_return) { hgt = hgt_return-y; }
   XGetSubImage (Tk_Display(tkwin), d, xdst, ydst, wid, hgt, 
		 ~0L, ZPixmap, tkimage->ximage, xsrc, ysrc);

   SetRasterModifiedArea (raster, 0, 0, 0, 0);

   return TCL_OK;
}


static int RasterImagePut (interp, raster, imagetable, argc, argv)
/*         -------------- 
 *
 *  Parses a command for moving pixels between stored images
 *  and windows/pixmaps. Command:
 *
 *    <path> image put <img_name> <x> <y> 
 */
     Tcl_Interp * interp;
     Tk_Raster* raster;
     Tcl_HashTable* imagetable;
     int argc;
     char * argv [];  
{
   tkImage* tkimage;
   int x, y, xsrc, xdst, ysrc, ydst;
   unsigned int wid, hgt, wid_return, hgt_return;
   Window root_return;
   int x_return, y_return;
   int border_width_return, depth_return;
   GC gc = GetRasterGC (raster);
   Drawable d = GetRasterDrawable (raster);
   Tk_Window tkwin = GetRasterTkWin (raster);

   /* 
    * Parse image name
    */

   if (argc < 3) {
      interp->result = "usage: pathname image put <name> <x> <y>";
      return TCL_ERROR;
   }

   if (RasterGetImage (interp, imagetable, &tkimage, argv [0]) != TCL_OK) {
      return TCL_ERROR;
   }

   argc--;
   argv++;

   /* 
    * Parse <x> and <y>
    */

   if (Tcl_GetInt (interp, argv [0], &x) != TCL_OK ||
       Tcl_GetInt (interp, argv [1], &y) != TCL_OK) {
      Tcl_AppendResult (interp, "wrong pixel position: ", argv [0], " ",
			argv[1], (char*)NULL);
      return TCL_ERROR;
   }

   /*
    * Put Image
    */

   xsrc = 0;
   ysrc = 0;
   xdst = x;
   ydst = y;
   wid = tkimage->width;
   hgt = tkimage->height;

   XGetGeometry (Tk_Display (tkwin), d, &root_return, &x_return, &y_return, 
	        &wid_return, &hgt_return, &border_width_return, &depth_return);
   if (x < 0) { xsrc = -x; xdst = 0; }
   if (y < 0) { ysrc = -y; ydst = 0; }
   if (xdst >= wid_return || ydst >= hgt_return) return TCL_OK;
   if (x+wid > wid_return) { wid = wid_return-x; }
   if (y+hgt > hgt_return) { hgt = hgt_return-y; }
   XPutImage (Tk_Display(tkwin), d, gc, tkimage->ximage, 
	      xsrc, ysrc, xdst, ydst, wid, hgt);

   SetRasterModifiedArea (raster, xdst, ydst, wid, hgt);

   return TCL_OK;
}


#ifdef DEBUG
#include <stdio.h>
#include <sys/time.h>
#include <sys/resource.h>

static struct {
   struct timeval utime;
   struct timeval stime;
} timemark;

static void clockreset () 
{
   struct rusage usage;
   getrusage (RUSAGE_SELF, &usage);
   timemark.utime = usage.ru_utime;
   timemark.stime = usage.ru_stime;
}

static void clockmark ()
{
   struct rusage usage;
   long dsec, dmsec;
   getrusage (RUSAGE_SELF, &usage);
   dsec = usage.ru_utime.tv_sec - timemark.utime.tv_sec;
   dmsec = (usage.ru_utime.tv_usec - timemark.utime.tv_usec)/1000;
   if (dmsec<0) { dsec--; dmsec+=1000; }
   printf ("user time: %ld.%03ld ", dsec, dmsec);
   dsec = usage.ru_stime.tv_sec - timemark.stime.tv_sec;
   dmsec = (usage.ru_stime.tv_usec - timemark.stime.tv_usec)/1000;
   if (dmsec<0) { dsec--; dmsec+=1000; }
   printf ("system time: %ld.%03ld \n", dsec, dmsec);
   clockreset ();
}
#endif 

static void ColorMerge (destptr, src)
     colorList * destptr;
     colorList  src;
{
   colorList tmp;
   while (*destptr != NULL && src != NULL) {
      if ((*destptr)->colorptr->freq < src->colorptr->freq) {
	 tmp = src;
	 src = src->next;
	 tmp->next = (*destptr);
	 (*destptr) = tmp;
      }
      destptr = & ((*destptr)->next);
   }
   if (src != NULL) *destptr = src;
}
      
static colorList SortedPrefix (colorlistptr) 
     colorList * colorlistptr;
{
   colorList result = *colorlistptr;
   colorList prev = result;

   if (*colorlistptr == NULL) return result;
   for (;;) {
      *colorlistptr = (*colorlistptr)->next; 
      if (*colorlistptr == NULL) break;
      if (prev->colorptr->freq < (*colorlistptr)->colorptr->freq) {
	 prev->next = NULL;
	 break;
      }
      prev = *colorlistptr;
   }
   return result;
}
      
static void ColorSort (colorlistptr)
     colorList * colorlistptr;
{
   colorList result;
   result = SortedPrefix (colorlistptr);
   while (*colorlistptr != NULL) {
      ColorMerge (&result, SortedPrefix (colorlistptr));
   }
   *colorlistptr = result;
}

static colorList Prefix (colorlistptr, n)
     colorList * colorlistptr;
     int n;
{
   colorList result = *colorlistptr;
   colorList prev = result;

   if (*colorlistptr == NULL) return result;
   for (;;) {
      *colorlistptr = (*colorlistptr)->next; 
      if (*colorlistptr == NULL) break;
      if (--n <= 0) {
	 prev->next = NULL;
	 break;
      }
      prev = *colorlistptr;
   }
   return result;
}

static void ColorSortLong (colorlistptr, n) 
     colorList * colorlistptr;
     int n;
{
   if (n < 256) {
      ColorSort (colorlistptr);
   }
   else {
      int n1 = n/2;
      int n2 = n-n1;
      colorList result = Prefix (colorlistptr, n1);
      ColorSortLong (&result, n1);
      ColorSortLong (colorlistptr, n2);
      ColorMerge (colorlistptr, result);
   }
}

static int CloseColor (nodeptr, list)
     colorNode* nodeptr;
     colorList list;
{
   int dist, mindist;
   mindist = MaxDistance;
   while (list != NULL) {
      if (list->colorptr->xcolor != NULL) {
	 dist = Distance (list->colorptr->pix, nodeptr->pix);
	 if (dist < mindist) {
	    mindist = dist;
	    nodeptr->closest = list->colorptr;
	 }
      }
      list = list->next;
   }
   return mindist != MaxDistance;
}


static int LoadImage8 (interp, tkwin, tkimage, file, maxval, format)
/*         ----------
 *
 *  Does a simple job of loading a 24 bit image into a 8 bit deep display
 */
     Tcl_Interp * interp;
     Tk_Window tkwin;
     tkImage * tkimage;
     FILE* file;
{
   pixel* imgbuf = NULL;
   pixel* pixrow = NULL;
   pixel* pixptr = NULL;
   pixel  pix;
   int status = TCL_OK;
   int wid = tkimage->width;
   int hgt = tkimage->height;
   colorNode * colorPtr, * closest;
   colorList colorHash [16*16*16];
   colorList * colorListPtr = colorHash;
   colorList allcolors = NULL;
   colorList clist, tmp;
   XColor pref, *pixcolor;
   int x, y, ncolors, nalloc, n, dist, mindist;
   long naccesses = 0;

   /* Init some vars */
   colorListPtr = colorHash;
   n = 16*16*16;
   while (n--) *colorListPtr++ = NULL;
   allcolors = NULL;

   /* Alloc imgbuf */
   imgbuf = (pixel*) malloc (sizeof(pixel)* wid * hgt);
   if (imgbuf == NULL) { status = TCL_ERROR; goto done; }

#ifdef DEBUG
printf ("\nLoading an image %d x %d\n", wid, hgt);
clockreset ();
#endif
         
   /* Load image into imgbuf - gather color frequencies */
   for (y = 0, pixrow = imgbuf; y < hgt; ++y, pixrow += wid) {
      if (!readppmrow (file, pixrow, wid, maxval, format)) {
	 status = TCL_ERROR;
	 goto done;
      }
      for (x = 0, pixptr = pixrow; x < wid; ++x, pixptr++) {
	 /* Search for an identical color in the hash table */
	 colorListPtr = & colorHash [256 * ((pixptr->r)>>4) +
				     16 * ((pixptr->g)>>4) + ((pixptr->b)>>4)];
	 clist = *colorListPtr;
      
	 if (clist != NULL) {
	    n = 0;	    
	    mindist = MaxDistance;
	    do {
	       register colorNode * this = clist->colorptr;
	       n++;
	       if (this->pix.r == pixptr->r && this->pix.g == pixptr->g && 
		   this->pix.b == pixptr->b) {
		  this->freq++;
		  goto found;
	       }
	       if ((dist = Distance (this->pix, *pixptr)) < mindist) {
		  mindist = dist;
		  closest = this;
	       }
	       clist = clist->next;
	    } while (clist != NULL);
	    if (n == MaxBucket) { 
	       closest->pix = *pixptr;
	       closest->freq++;
	       goto found;
	    }
	 }
	 colorPtr = (colorNode*) malloc (sizeof (colorNode));
	 colorPtr->xcolor = NULL;
	 colorPtr->closest = NULL;
	 colorPtr->freq = 1;
	 colorPtr->pix = *pixptr;
	 clist = (colorList) malloc (sizeof (struct colorListNode));
	 clist->colorptr = colorPtr;
	 clist->next = *colorListPtr;
	 *colorListPtr = clist;
      found:;
      }	
   }

#ifdef DEBUG
   clockmark ();
   printf ("Finished reading \n"); 
   printf ("Avg hash table accesses: %ld \n", naccesses/wid/hgt);
   fflush (stdout);
#endif
   
   /*
    * Gather all colors and sort them by freq
    */
   n = 16*16*16;
   allcolors = NULL;
   ncolors = 0;
   colorListPtr = &colorHash [0];
   while (n--) {
      clist = *colorListPtr++;
      while (clist != NULL) {
	 ncolors++;
	 tmp = (colorList) malloc (sizeof (struct colorListNode));
	 tmp->next = allcolors;
	 tmp->colorptr = clist->colorptr;
	 allcolors = tmp;
	 clist = clist->next;
      }
   } 

#ifdef DEBUG
   clockmark ();
   printf ("Finished gathering: %d colors\n", ncolors); fflush (stdout);
#endif

   ColorSortLong (&allcolors, ncolors);

#ifdef DEBUG
   clockmark ();
   printf ("Finished sorting colors\n"); fflush (stdout);
#endif

   /*
    * Try to allocate as many colors as possible 
    */
   nalloc = 0;
   while (allcolors != NULL) {
      pref.red = (unsigned long)256 * allcolors->colorptr->pix.r;
      pref.green = (unsigned long)256 * allcolors->colorptr->pix.g;
      pref.blue = (unsigned long)256 * allcolors->colorptr->pix.b;
      if (XAllocColor (Tk_Display (tkwin), Tk_Colormap (tkwin), &pref) == 0) {
	 break;
      }
      XFreeColors (Tk_Display (tkwin), Tk_Colormap (tkwin), 
		   &(pref.pixel), 1, 0L);
      allcolors->colorptr->xcolor = Tk_GetColorByValue (interp, tkwin, 
						Tk_Colormap (tkwin), &pref);
      clist = allcolors;
      allcolors = allcolors->next;
      clist->next = tkimage->neededColors;
      tkimage->neededColors = clist;
      nalloc++;
   }

#ifdef DEBUG
   clockmark ();
   printf ("Finished allocating: %d colors\n", nalloc); fflush (stdout);
#endif

   if (nalloc < 2 && ncolors >= 2) {
      status = TCL_ERROR;
      goto done;
   }

   /*
    * find 'close' colors to those that could not be allocated
    */
   while (allcolors != NULL) {
      colorPtr = allcolors->colorptr;
      n = (256*((colorPtr->pix.r)>>4) + 16*((colorPtr->pix.g)>>4) + 
	   ((colorPtr->pix.b)>>4));
      CloseColor (colorPtr, colorHash [n]);
      if (colorPtr->closest == NULL) {
	 CloseColor (colorPtr, tkimage->neededColors);
      }
      clist = allcolors;
      allcolors = allcolors->next;
      free (clist);
   }

#ifdef DEBUG
   clockmark ();
   printf ("Finished finding %d close colors\n", ncolors-nalloc); 
   fflush (stdout);
#endif

   /*
    * map pixels into xcolor pixels in image
    */
   for (y = 0, pixrow = imgbuf; y < hgt; y++, pixrow += wid) {
      for (x = 0, pixptr = pixrow; x < wid; ++x, pixptr++) {
	 clist = colorHash [256 * ((pixptr->r)>>4) +
			    16 * ((pixptr->g)>>4) + ((pixptr->b)>>4)];
	 closest = clist->colorptr;
	 mindist = Distance (*pixptr, closest->pix);
	 while (mindist > 0 && clist->next != NULL) {
	    clist = clist->next;	    
	    dist = Distance (*pixptr, clist->colorptr->pix);
	    if (dist < mindist) {
	       mindist = dist;
	       closest = clist->colorptr;
	    }
	 }

	 if (closest->xcolor != NULL) {
	    XPutPixel (tkimage->ximage, x, y, closest->xcolor->pixel);
	 }
	 else {
	    XPutPixel (tkimage->ximage, x, y, closest->closest->xcolor->pixel);
	 }
      }
   }

#ifdef DEBUG
   clockmark ();
   printf ("Finished loading\n"); fflush (stdout);
#endif

done:
   colorListPtr = &colorHash [0];
   n = 16*16*16;
   while (n--) {
      while (*colorListPtr != NULL) {
	 clist = *colorListPtr;
	 *colorListPtr = clist->next;
	 free (clist);
      }
      colorListPtr++;
   }
   while (allcolors != NULL) {
      clist = allcolors;
      allcolors = allcolors->next;
      free (clist->colorptr);
      free (clist);
   }
   free (imgbuf);
   return status;
}

static int SaveImage8 (interp, tkwin, tkimage, file, maxval, format)
/*         ----------
 *
 *  Does a simple job of writing 8 bit deep image into a ppm file
 */
     Tcl_Interp * interp;
     Tk_Window tkwin;
     tkImage * tkimage;
     FILE* file;
{
   int status = TCL_OK;
   int wid = tkimage->width;
   int hgt = tkimage->height;
   XColor colors [256];
   pixel colorpixel [256];
   int i, x, y;
   pixel * pixbuf, * pixptr;

   pixbuf = (pixel*) malloc (wid * sizeof (pixel));
   for (i = 0; i < 256; ++i) {
      colors [i].pixel = i;
      colors [i].flags = DoRed | DoGreen | DoBlue;
   }
   XQueryColors (Tk_Display (tkwin), Tk_Colormap (tkwin), colors, 256);
   for (i = 0; i < 256; ++i) {
      colorpixel [i].r = colors [i].red >> 8;
      colorpixel [i].g = colors [i].green >> 8;
      colorpixel [i].b = colors [i].blue >> 8;
   }
   for (y = 0; y < hgt; ++y) {
      pixptr = pixbuf;
      for (x = 0; x < wid; ++x) {
	 *pixptr++ = colorpixel [XGetPixel (tkimage->ximage, x, y)];
      }
      if (!writeppmrow (file, pixbuf, wid, maxval, format)) {
	 status = TCL_ERROR;
	 break;
      }
   }
   free (pixbuf);
   return status;
}
   
      
static int RasterImageLoad (interp, raster, imagetable, argc, argv)
/*         --------------- 
 *
 *  Command:
 *
 *    <path> image load <img_name> <ppm_filename> 
 */
     Tcl_Interp * interp;
     Tk_Raster* raster;
     Tcl_HashTable* imagetable;
     int argc;
     char * argv [];  
{  
   int wid, hgt, bytesperline, maxval;
   char * imagedata;
   FILE* ppmfile;
   char ppmtclfile [30];
   tkImage* tkimage;
   int format;
   int result;
   Tk_Window tkwin = GetRasterTkWin (raster);

   /*
    *  Get tkimage
    */
   if (argc < 2) {
      interp->result = "usage: pathname image load <name> <ppmfile>";
      return TCL_ERROR;
   }

   if (RasterGetNewImage (interp, imagetable, &tkimage, argv [0]) != TCL_OK) {
      return TCL_ERROR;
   }

   argc--;
   argv++;

   /* 
    *  Set up ppm file for reading
    */

   if (Tcl_VarEval (interp, "open \"", argv [0], "\" r", (char*)NULL) 
       != TCL_OK) {
      Tcl_AppendResult (interp, "could not open ppm file ", argv [0],
			"\n", (char*)NULL);
      return TCL_ERROR;
   }
   strcpy (ppmtclfile, interp->result);

   Tcl_GetOpenFile (interp, ppmtclfile, 0, 0, &ppmfile);

   if ((!readppminit (ppmfile, &wid, &hgt, &maxval, &format)) ||
       format != RPPM_FORMAT && format != PPM_FORMAT &&
       format != RPGM_FORMAT && format != PGM_FORMAT && 
       format != RPBM_FORMAT && format != PBM_FORMAT) {
      Tcl_AppendResult (interp, " file is not portable pixmap ",
			argv [0], "\n", (char*)NULL);
      Tcl_VarEval (interp, "close ", ppmtclfile, (char*)NULL);
      return TCL_ERROR;
   }

   /*
    *  Try to create ximage structure for image
    */
   bytesperline = (Tk_Depth (tkwin)+1)/8*wid;
   imagedata = malloc (bytesperline*hgt);
   if (imagedata == NULL) {
      Tcl_AppendResult (interp, "no memory for an image this big",(char*)NULL);
      DestroyTkImage (imagetable, tkimage);
      Tcl_VarEval (interp, "close ", ppmtclfile, (char*)NULL);
      return TCL_ERROR;
   }		      

   tkimage->width = wid;
   tkimage->height = hgt;
   tkimage->ximage = XCreateImage (Tk_Display (tkwin),
				   Tk_Visual (tkwin),
				   Tk_Depth (tkwin),
				   ZPixmap,
				   0,
				   imagedata,
				   (unsigned)wid,
				   (unsigned)hgt,
				   8,
				   bytesperline);

   /*
    *  Load Image code 
    */
   if (Tk_Depth (tkwin) <= 8) {
      result = LoadImage8 (interp, tkwin, tkimage, ppmfile, maxval, format);
   }
   else {
      result = TCL_ERROR;
   }

done:

   /* Cleanup */
   Tcl_VarEval (interp, "close ", ppmtclfile, (char*)NULL);
   if (result != TCL_OK) {
      /*
       *  Something wicked happened ...
       */
      DestroyTkImage (imagetable, tkimage);
   }
   
   SetRasterModifiedArea (raster, 0, 0, 0, 0);

   return result;
}


static int RasterImageSave (interp, raster, imagetable, argc, argv)
/*         --------------- 
 *
 *  Command:
 *
 *    <path> image save <img_name> <ppm_filename> 
 */
     Tcl_Interp * interp;
     Tk_Raster* raster;
     Tcl_HashTable* imagetable;
     int argc;
     char * argv [];  
{  
   FILE* ppmfile;
   char ppmtclfile [30];
   tkImage* tkimage;
   int result;
   Tk_Window tkwin = GetRasterTkWin (raster);

   /*
    *  Get tkimage
    */

   if (argc < 2) {
      interp->result = "usage: pathname image save <name> <ppmfile>";
      return TCL_ERROR;
   }

   if (RasterGetNewImage (interp, imagetable, &tkimage, argv [0]) != TCL_OK) {
      return TCL_ERROR;
   }

   argc--;
   argv++;

   /* 
    *  Set up ppm file for writing
    */

   if (Tcl_VarEval (interp, "open \"", argv [0], "\" w", (char*)NULL) 
       != TCL_OK) {
      Tcl_AppendResult (interp, "could not open ppm file ", argv [0],
			"\n", (char*)NULL);
      return TCL_ERROR;
   }
   strcpy (ppmtclfile, interp->result);

   if (Tcl_GetOpenFile (interp, ppmtclfile, 1, 1, &ppmfile) != TCL_OK) {
      return TCL_ERROR;
   }

   if (!writeppminit (ppmfile, tkimage->width, tkimage->height, 
		      255, RPPM_FORMAT)) {      
      Tcl_AppendResult (interp, "error writing ", argv [0], "\n", 
			(char*)NULL);
      return TCL_ERROR;
   }

   /*
    *  Save Image code 
    */
   if (Tk_Depth (tkwin) <= 8) {
      result = SaveImage8 (interp, tkwin, tkimage, ppmfile, 255, RPPM_FORMAT);
   }
   else {
      result = TCL_ERROR;
   }

   /* Cleanup */
   Tcl_VarEval (interp, "close ", ppmtclfile, (char*)NULL);

   SetRasterModifiedArea (raster, 0, 0, 0, 0);

   return result;
}

static int Init (interp, raster, dataptr)
/*         ----
 * 
 *  Initializes the image hash table for raster
 */
     Tcl_Interp * interp;
     Tk_Raster* raster;
     ClientData * dataptr;
{
   Tcl_HashTable * tablePtr;
   tablePtr = (Tcl_HashTable*) malloc (sizeof (Tcl_HashTable));
   Tcl_InitHashTable (tablePtr,TCL_STRING_KEYS);
   *((Tcl_HashTable **)dataptr) = tablePtr;
   return TCL_OK;
}

static void Free (raster, data)
/*          ----
 *
 *  Frees up the image hash table and the images there contained
 */
     Tk_Raster* raster;
     ClientData data;
{
   Tcl_HashTable* imagetable = (Tcl_HashTable*) data;
   Tcl_HashSearch search;
   Tcl_HashEntry* entryPtr;
   tkImage* image;

   for (entryPtr = Tcl_FirstHashEntry (imagetable, &search);
	entryPtr != NULL;
	entryPtr = Tcl_NextHashEntry (&search)) {
      image = (tkImage*) Tcl_GetHashValue (entryPtr);
      DestroyTkImage (imagetable, image);
   }    
   Tcl_DeleteHashTable (imagetable);
   free (imagetable);
}
   

static int Draw (interp, raster, data, argc, argv)
/*         ---- 
 *
 *  Entry point for all commands of the form:
 *
 *    <path> image option arg arg ... arg
 */
     Tcl_Interp * interp;
     Tk_Raster* raster;
     ClientData data;
     int argc;
     char * argv [];  
{  
   Tcl_HashTable* imagetable = (Tcl_HashTable*) data;
   if (argc < 3) {
      Tcl_AppendResult (interp, "usage: <path> image <option> <arg> ... <arg>",
			(char*)NULL);
      return TCL_ERROR;
   }
   if (strcmp (argv [2], "create") == 0) {
      return RasterImageCreate (interp, raster, imagetable, argc-3, argv+3);
   }
   if (strcmp (argv [2], "delete") == 0) {
      return RasterImageDelete (interp, raster, imagetable, argc-3, argv+3);
   }
   if (strcmp (argv [2], "info") == 0) {
      return RasterImageInfo (interp, raster, imagetable, argc-3, argv+3);
   }
   if (strcmp (argv [2], "get") == 0) {
      return RasterImageGet (interp, raster, imagetable, argc-3, argv+3);
   }
   if (strcmp (argv [2], "put") == 0) {
      return RasterImagePut (interp, raster, imagetable, argc-3, argv+3);
   }
   if (strcmp (argv [2], "save") == 0) {
      return RasterImageSave (interp, raster, imagetable, argc-3, argv+3);
   }
   if (strcmp (argv [2], "load") == 0) {
      return RasterImageLoad (interp, raster, imagetable, argc-3, argv+3);
   }
   Tcl_AppendResult (interp, argv [2], ": option must be create, delete ",
		     "info, get, put, load or save", (char*)NULL);
}


int RasterImageInit (interp)
/*  ---------------
 *
 *  Initializes the 'Image' commands for rasters
 */
     Tcl_Interp * interp;
{   
   return RasterAddPrimitive (interp, "image", Draw, Init, Free);
}


