/*
 * TkTree.cc - member routines for class TkTree,
 *             implementation of the TCL tree command
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1993 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  Allan
 * Brighton make no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 * -----------------------------------------------------------------------------
 */

#include <TkTree.h>
#include <TkTreeNode.h>
#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <strstream.h>
#include <string.h>

// tree config options
Tk_ConfigSpec TkTree::configSpecs_[4] = {
    {TK_CONFIG_PIXELS, "-parentdistance", "parentDistance", "ParentDistance",
	"30", Tk_Offset(TkTree::TkTreeOptions, parentDistance), 0},
    {TK_CONFIG_STRING, "-layout", "layout", "Layout",
	"horizontal", Tk_Offset(TkTree::TkTreeOptions, layout), 0},
    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	"10", Tk_Offset(TkTree::TkTreeOptions, borderWidth), 0},
    {TK_CONFIG_END, 0, 0, 0, 0, 0, 0}
};


/*
 * tree command
 * 
 * Syntax:
 * 
 *         tree pathName ?options?
 * 
 * Options:
 *         -parentDistance <distance> - the distance from node to its parent
 *         -layout <horizontal or vertical> - orientation of tree in camvas
 */
extern "C" 
int treeCmd(ClientData w, Tcl_Interp *interp, int argc, char* argv[])
{
    TkTree *t = new TkTree((Tk_Window)w, interp, argc, argv);
    return t->status();
}


/*
 * Constructor: initialize a new tree with the command line args
 * 
 * This constructor is called for each tree declared in tk. The destructor
 * is called when the tree widget is destroyed.
 */
TkTree::TkTree(Tk_Window w, Tcl_Interp *interp, int argc, char* argv[])
: interp_(interp),
  root_(new TkTreeNode(this, "", "")),
  status_(TCL_OK),
  options_(30, "horizontal", 10),
  tcl(interp)
{
    if (argc < 2) {
	Tcl_AppendResult(interp, 
			 "wrong # args:  should be \"", 
			 argv[0], " pathName ?options?\"", 0);
	status_ = TCL_ERROR;
	return;
    }

    // get the name of the parent canvas window from the tree's path
    String  path = argv[1];
    int i = path.rindex('.');

    if (i > 0)
        canvas_ = path.substr(0, i);

     if (i <= 0 
	 || Tcl_VarEval(interp, "winfo class ", canvas_.string(), 0) != TCL_OK
	 || strcmp(interp->result, "Canvas") != 0) {
        status_ = TCL_ERROR;
	Tcl_ResetResult(interp);
        Tcl_AppendResult(interp, 
			 "bad path name for tree: \"",
			 argv[1], "\", parent of tree should be a canvas widget", 
			 0);
 	return;
    }
    
    // create the widget window
    tkwin_ = Tk_CreateWindowFromPath(interp, w, argv[1], 0);
    if (!tkwin_) {
	status_ = TCL_ERROR;
	return;
    }

    Tk_SetClass(tkwin_, "Tree");
    if ((status_ = configureTree(argc-2, argv+2)) == TCL_OK) {
 	Tcl_CreateCommand(interp, argv[1], TkTree::treeWidgetCmd, 
			  (ClientData)this, TkTree::treeDeleteCmd);
    }
    else {
	Tk_DestroyWindow(tkwin_);
	tkwin_ = 0;
    }
    
    // return the name of the widget as a tcl result
    Tcl_SetResult(interp, argv[1], TCL_VOLATILE);
}


/*
 * destructor - clean up tree nodes when widget cmd is destroyed
 */
TkTree::~TkTree()
{
    delete root_;
    root_ = 0;
}


/*
 * fix the root tree node at left center (for horizontal trees)
 * or top center for vertical trees
 */
int TkTree::fixRootNode()
{
    if (Tcl_VarEval(interp_, 
		    "lindex [", 
		    canvas_.string(), 
		    " configure -scrollregion] 4",
		    0) 
	!= TCL_OK) 
        return TCL_ERROR;
        
    int x1, y1, x2, y2;
    if (sscanf(interp_->result, "%d %d %d %d", &x1, &y1, &x2, &y2) != 4)
        return TCL_ERROR;
     
    // center the root node at left center for horizontal trees
    // or top center for vertical trees
    if (horizontal())    
        root_->pos(Point(x1, (y2-y1)/2));
    else
        root_->pos(Point((x2-x1)/2, y1));
    
    return TCL_OK;
}


/*
 * ConfigureTree --
 * 
 * 	This procedure is called to process an argv/argc list, plus
 * 	the Tk option database, in order to configure (or
 * 	reconfigure) a tree widget.
 * 
 * Results:
 * 	The return value is a standard Tcl result.  If TCL_ERROR is
 * 	returned, then interp->result contains an error message.
 * 
 * Side effects:
 *         The root node may have to be repositioned if the layout is changed
 */
int TkTree::configureTree(int argc, char* argv[], int flags)
{
    if (Tk_ConfigureWidget(interp_, tkwin_, configSpecs_,
	    argc, argv, (char*)&options_, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    
    // set the pos of the root node incase the layout changed
    return fixRootNode();
}


/*
 * this is called for the configure widget command
 */
int TkTree::configure(int argc, char* argv[])
{
    if (argc == 2) {
	return Tk_ConfigureInfo(interp_, tkwin_, configSpecs_, (char*)&options_, 0, 0);
    } else if (argc == 3) {
	return Tk_ConfigureInfo(interp_, tkwin_, configSpecs_, (char*)&options_, argv[2], 0);
    } else {
	return configureTree(argc-2, argv+2, TK_CONFIG_ARGV_ONLY);
    }
}


/*
 * Implementation of the tree widget command
 * 
 * See the man page $TEDIR/man/mann/tree.n for a description of the
 * subcommands and options.
 * 
 */
int TkTree::treeWidgetCmd(ClientData clientData, Tcl_Interp* interp, int argc, char* argv[])
{   
    if (argc < 2) {
	Tcl_AppendResult(interp, 
			 "wrong # args: should be \"", 
			 argv[0], " command\"", 0);
	return TCL_ERROR;
    }
    
    char* cmd = argv[1];
    int length = strlen(cmd);
    TkTree* t = (TkTree*)clientData;
    Tcl_ResetResult(interp);
    
    switch (*cmd) {
    case 'a':
	if (strncmp(cmd, "addlink", length) == 0 && length > 1)
	    return t->addLink(argc, argv);

	if (strncmp(cmd, "ancestors", length) == 0 && length > 1)
	    return t->ancestors(argc, argv);
	break;

    case 'c':
	if (strncmp(cmd, "canvas", length) == 0 && length > 1) {	/* allan: 23.7.93 */
	    Tcl_AppendResult(interp, t->canvas().string(), 0);
	    return TCL_OK;
	}

	if (strncmp(cmd, "configure", length) == 0 && length > 1)
	    return t->configure(argc, argv);

	if (strncmp(cmd, "child", length) == 0 && length > 1)
	    return t->child(argc, argv);
	break;

    case 'd':
	if (strncmp(cmd, "draw", length) == 0)
	    return t->draw(argc, argv);
	break;

    case 'i':
	if (strncmp(cmd, "isleaf", length) == 0 && length > 2)
	    return t->isLeaf(argc, argv);

	if (strncmp(cmd, "isroot", length) == 0 && length > 2)
	    return t->isRoot(argc, argv);
	break;

    case 'm':
	if (strncmp(cmd, "movelink", length) == 0)
	    return t->moveLink(argc, argv);
	break;

    case 'n':
	if (strncmp(cmd, "nodeconfigure", length) == 0)
	    return t->nodeConfigure(argc, argv);
	break;

    case 'p':
	if (strncmp(cmd, "prune", length) == 0 && length > 1)
	    return t->prune(argc, argv);

	if (strncmp(cmd, "parent", length) == 0 && length > 1)
	    return t->parent(argc, argv);
	break;

    case 'r':
	if (strncmp(cmd, "root", length) == 0 && length > 1)
	    return t->root(argc, argv);

	if (strncmp(cmd, "rmlink", length) == 0 && length > 1)
	    return t->rmLink(argc, argv);
	break;
	
    case 's':
	if (strncmp(cmd, "sibling", length) == 0 && length > 1)
	    return t->sibling(argc, argv);

	if (strncmp(cmd, "subnodes", length) == 0 && length > 1)
	    return t->subnodes(argc, argv);
	break;
    }
    
    Tcl_AppendResult(interp, "unknown tree subcommand: \"", cmd, "\"", 0);
    return TCL_ERROR;
}


/*
 * This command is called when the tree widget command is destroyed
 * or the same name if created.
 */
void TkTree::treeDeleteCmd(ClientData clientData)
{
    TkTree* t = (TkTree*)clientData;
    delete t;
}


/*
 * find the named node and return a pointer to it or 0 if not found
 */
TkTreeNode* TkTree::findNode(const String& tag)
{
    TkTreeNode* node = (TkTreeNode*)root_->find(tag);
    if (!node) 
	Tcl_AppendResult(interp_, 
			 "can't find tree node: \"", 
			 tag.string(), "\"", 0);
    return node;
}


/*
 * add a new child node to the parent node in the tree
 */
int TkTree::addLink(int argc, char* argv[])
{
    if (argc < 5) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"", 
			 argv[0], " ", argv[1], 
			 " parent_tag child_tag line_tag ?options?\"", 
			 0);
	return TCL_ERROR;
    }
	
    String parent_tag(argv[2]), child_tag(argv[3]), line_tag(argv[4]);
    TkTreeNode* parent = findNode(parent_tag);
    if (!parent) 
	return TCL_ERROR;
    
    TkTreeNode* child = new TkTreeNode(this, child_tag, line_tag, 0, 0, 0, 0, 0);
    if (setupNode(child, argc-5, argv+5) != TCL_OK)
        return TCL_ERROR;
        
    parent->addLink(child);
    
    return TCL_OK;
}


/*
 * Set up the node's size, position and options
 */
int TkTree::setupNode(TkTreeNode* node, int argc, char* argv[])
{
    if (Tcl_VarEval(interp_, 
		    canvas_.string(), 
		    " bbox {", 
		    node->tag().string(), 
		    "}", 
		    0) != TCL_OK) 
        return TCL_ERROR;
    
    // get the size of the node
    int x1, y1, x2, y2;
    sscanf(interp_->result, "%d %d %d %d", &x1, &y1, &x2, &y2);	
    node->box(x1, y1, x2, y2);
    
    // get the other options
    int d;
    for (int i = 0; i < argc; i++) {
	if (strcmp(argv[i], "-border") == 0 && ++i < argc) {
	    if (sscanf(argv[i], "%d", &d) == 1)
	        node->border(d);
	}
	else if (strcmp(argv[i], "-remove") == 0 && ++i < argc)
	    node->removeCmd(argv[i]);
    }
    
    return TCL_OK;
}

  

/*
 * unhook the child subtree and make it a subtree of the parent tree
 */
int TkTree::moveLink(int argc, char* argv[])
{
    if (argc != 4) {
 	Tcl_AppendResult(interp_,"wrong # args: should be \"", argv[0], 
  	    " ", argv[1], " parent_tag child_tag ?options?\"", 0);
	return TCL_ERROR;
    }
    
    String child_tag(argv[2]), parent_tag(argv[3]);	
    TkTreeNode* parent = findNode(parent_tag);
    if (!parent) 
	return TCL_ERROR;
	
    TkTreeNode* child = findNode(child_tag);
    if (!child) 
	return TCL_ERROR;
	
    child->unLink ();
    parent->addLink(child);
    
    return TCL_OK;
}
    
  

/*
 * remove the named node and its subnodes from the tree
 */
int TkTree::rmLink(int argc, char* argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"", 
			 argv[0], " ", argv[1], " tag\"", 0);
	return TCL_ERROR;
    }
    String tag(argv[2]);  
    
    TkTreeNode* node = findNode(tag);
    if (!node) 
	return TCL_ERROR;
    
    node->rmLink();
    return TCL_OK;
}
    
    
/*
 * Remove and delete the nodes children
 */
int TkTree::prune(int argc, char* argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"",
			 argv[0], " ", argv[1], " tag\"", 0);
	return TCL_ERROR;
    }
    String tag(argv[2]);  
    
    TkTreeNode* node = findNode(tag);
    if (!node) 
	return TCL_ERROR;
    
    node->prune();
    return TCL_OK;
}

    
/*
 * This command recalculates the node's size and also allows the same options
 * as the "addlink" sub-command.
 */
int TkTree::nodeConfigure(int argc, char* argv[])
{
    if (argc < 3) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"", 
			 argv[0],  " ", argv[1], " tag ?options?\"", 0);
	return TCL_ERROR;
    }
    String tag(argv[2]);  
    
    TkTreeNode* node = findNode(tag);
    if (!node) 
	return TCL_ERROR;
    
    if (setupNode((TkTreeNode*)node, argc-3, argv+3) != TCL_OK)
        return TCL_ERROR;
        
    return TCL_OK;
}

    
/*
 * Make the named node the new root of the tree.
 */
int TkTree::root(int argc, char* argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"", 
			 argv[0], " ", argv[1], " tag\"", 0);
	return TCL_ERROR;
    }
    String tag(argv[2]);  
    
    TkTreeNode* node = findNode(tag);
    if (!node) 
	return TCL_ERROR;
    
    if (node->parent()) {
        node->unLink ();
        root_->prune();
        root_->addLink(node);
    }
    
    return TCL_OK;
}


/*
 * this command returns true if the node is a leaf node
 */
int TkTree::isLeaf(int argc, char* argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"", 
			 argv[0], " ", argv[1], " tag\"", 0);
	return TCL_ERROR;
    }
    String tag(argv[2]);  
    
    TkTreeNode* node = findNode(tag);
    if (!node) 
	return TCL_ERROR;

    Tcl_AppendResult(interp_, (node->child() ? "0" : "1"), 0);
    return TCL_OK;
}


/*
 * this command returns true if the node is a root node
 * 
 * (root_ is invisible, so its children are the root nodes seen)
 */
int TkTree::isRoot(int argc, char* argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"", 
			 argv[0], " ", argv[1], " tag\"", 0);
	return TCL_ERROR;
    }
    String tag(argv[2]);  
    
    TkTreeNode* node = findNode(tag);
    if (!node) 
	return TCL_ERROR;

    Tcl_AppendResult(interp_, ((node->parent() == root_) ? "1" : "0"), 0);
    return TCL_OK;
}

    
/*
 * draw the tree in the canvas
 */
int TkTree::draw(int argc, char* argv[])
{
    if (argc != 2) {
	Tcl_AppendResult(interp_, "wrong # args: should be \"", 
		argv[0], " ", argv[1], "\"", 0);
	return TCL_ERROR;
    }
    root_->drawTree();
    return TCL_OK;
}
    

/*
 * this command returns the name of the node's child node
 * or the empty string if there is none
 */
int TkTree::child(int argc, char* argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"", 
			 argv[0], " ", argv[1], " tag\"", 0);
	return TCL_ERROR;
    }
    String tag(argv[2]);  
    
    TkTreeNode* node = findNode(tag);
    if (!node) 
	return TCL_ERROR;

    Tcl_AppendResult(interp_, 
		     ((node->child()) ? node->child()->tag().string() : ""), 
		     0);
    return TCL_OK;
}


/*
 * This command returns a Tcl list of this node's subnodes
 */
int TkTree::subnodes(int argc, char* argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"", 
			 argv[0], " ", argv[1], " tag\"", 0);
	return TCL_ERROR;
    }
    String tag(argv[2]);  
    
    TkTreeNode* node = findNode(tag);
    if (!node) 
	return TCL_ERROR;

    for (DrawTreeNode* p = node->child(); p; p = p->sibling()) 
        Tcl_AppendResult(interp_, "{", p->tag().string(), "} ", 0);
    return TCL_OK;
}


/*
 * this command returns the name of the node's sibling node
 * or the empty string if there is none
 */
int TkTree::sibling(int argc, char* argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"", 
			 argv[0], " ", argv[1], " tag\"", 0);
	return TCL_ERROR;
    }
    String tag(argv[2]);  
    
    TkTreeNode* node = findNode(tag);
    if (!node) 
	return TCL_ERROR;

    Tcl_AppendResult(interp_, 
		     ((node->sibling()) ? node->sibling()->tag().string() : ""), 
		     0);
    return TCL_OK;
}


/*
 * this command returns the name of the node's parent node
 * or the empty string if there is none
 */
int TkTree::parent(int argc, char* argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"", 
			 argv[0], " ", argv[1], " tag\"", 0);
	return TCL_ERROR;
    }
    String tag(argv[2]);  
    
    TkTreeNode* node = findNode(tag);
    if (!node) 
	return TCL_ERROR;

    Tcl_AppendResult(interp_, 
		     ((node->parent()) ? node->parent()->tag().string() : ""), 
		     0);
    return TCL_OK;
}


/*
 * This command returns a Tcl list of this nodes ancestors
 */
int TkTree::ancestors(int argc, char* argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp_, 
			 "wrong # args: should be \"", 
			 argv[0], " ", argv[1], " tag\"", 0);
	return TCL_ERROR;
    }
    String tag(argv[2]);  
    
    TkTreeNode* node = findNode(tag);
    if (!node) 
	return TCL_ERROR;

    // count the ancestors
    int n = 0;
    const DrawTreeNode* p;
    for (p = node->parent(); p && p->tag().length(); p = p->parent()) {
	n++;
    }
    if (n == 0) 
	return TCL_OK;

    // allocate the array arg for the tcl list routines
    /* int */ argc = n;
    /* char**  */ argv = new char*[argc];
    for (p = node->parent(); p && p->tag().length(); p = p->parent()) {
	argv[--n] = (char*) p->tag().string();
    }   

    // create the tcl list
    char* list = Tcl_Merge(argc, argv);
    if (list) 
	Tcl_AppendResult(interp_, list, 0);
    else
	return TCL_ERROR;

    free(list);
    delete argv;
    return TCL_OK;
}

