############################################################################
#
# general.tcl:
#	This file contains general procedures used by a lot of others
#	or influencing a lot of windows...
#
# ClearMem {ProcList VarList}
#	Clears part of the memory by removing the Procedures with the
#	name from ProcList, and unsetting the global! variables from VarList
#
# Cache:Read {cachefile}
# Cache:Write {cachefile List}
#	read or write a cache file line by line and translate each line to
#	a list element or vv. Cache:Read returns the List.
#
# Refresh {}
#	Kill all the folder, show and edit windows.
#
# AdaptAliMenus {}
#	Adapt the "Mail To" menu in the Control window to the new alias list
#
# FindLastFile {Fol}
#       Find the last file in a folder, and return it.
#
# FlashSel {w {num 1} {delay 50}}
#	Flashes the current selection in $w $num times. delay gives the
#	delay between flashes.
#
# GetListboxSel {Listbox maxnum}
#       Get the current selection list from the named listbox. Return a list
#       with a maximum number of maxnum members if there is a selection, 
#       return {} if none. defaults to one entry return
#       TODO: maybe more efficient coding....
#
# MakeMenu {m List repeat Path options}
#	This function makes a menu $m with entries from list. All menu entries
#	will have the same options. 
#	each item will have the name fronm the list. If list contains a sublist,
#	a cascade menu will be created, and so on. repeat tells if the menu 
#	entry should be repeated in the cascade that follows:
#	List: {1 {2 2a {2b {2b1 2b2} 2c}}
#       repeat = 0                      repeat = 1
#       1                               1
#       2 - 2a                          2 - 2
#           2b - 2b1                        2a 
#                2b2                        2b - 2b
#           2c                                   2b1
#                                                2b2
#                                           2c
#
# 	If you want to use the name of the entry, use %Entry, for example: 
#	  MakeMenu .m $List 0 -command {puts stdout %Entry}
#	will generate "puts stdout 2b2" for the command
#	If you want the entry with the path put in %EntryPath
#	  MakeMenu .m $List 0 -command {puts stdout %EntryPath}
#	will generate "puts stdout 2/2b/2b2" for the command.
#	This will be prepended by Path, usually that entry is empty. it's 
#	only there for recursive calling.
#
# KillMenu {m}
#	Kill the entire menu cascade below this menu
#
# Pipe {w command {newline nonewline}}
#	Pipe the current X selection through command. If the last option is
#	new[line], a newline will be appended to the output. Insert the result 
#	at the cursor in the text window $w. If the selection comes out of this
#	window, replace it with the result.
#	
# SaveText {w Filename}
#       This saves the contents of a text widget (w) to a file
#
#############################################################################

proc Cache:Read cachefile {
    if {![file exists $cachefile]} {
        MsgWindow "No such cache file: $cachefile.\n"; return {}
    }
    set List {}
    set Fhandle [open $cachefile r]
    while {[gets $Fhandle Element] != -1} {
        lappend List $Element
    }
    close $Fhandle
    return $List
}

proc Cache:Write {cachefile List} {
    set Fhandle [open $cachefile w]
    foreach Element $List {puts $Fhandle $Element}
    close $Fhandle
}

proc Cache:Append {cachefile List} {
    set Fhandle [open $cachefile a+]
    foreach Element $List {puts $Fhandle $Element}
    close $Fhandle
}

proc Refresh {} {
    foreach Win [winfo children .] {
        switch [winfo class $Win] {
            Folder -
            EditMsg -
            ShowMsg {destroy $Win} 
        }
    }
}

proc ClearMem {ProcList VarList} {
    foreach PROC $ProcList {rename $PROC {}}
    foreach VAR $VarList {
      # is it part of an array?]
      if {[set i [string first {(} $VAR]] == -1} {global $VAR} \
      else {global [string range $VAR 0 [expr $i - 1]]}
      unset $VAR
    }
}

proc AdaptAliMenus {} {
    global Aliases PREF AliasTree

    # Control Window
    catch {destroy .tkmh.b.mail}
    KillMenu .tkmh.b.mail.m
    if $PREF(AliMenu) {
        menubutton .tkmh.b.mail -relief raised -text {Mail to} \
          -menu .tkmh.b.mail.m
        menu .tkmh.b.mail.m
        .tkmh.b.mail.m add command -label *NEW* -command "ComposeMsg"
        MakeMenu .tkmh.b.mail.m $Aliases 0 {} -command \
          "if \$PREF(ExpandAlias) {ComposeMsg \"\[SearchAlias %Entry 1\]\"} \
           else {ComposeMsg %Entry}"
    } else {
        button .tkmh.b.mail -text Mail -command ComposeMsg
    }
    pack before .tkmh.b.f .tkmh.b.mail {top expand fill}
    # Other windows: Folder and show; 'Forward To:'
    foreach w [winfo children .] {
      switch [winfo class $w] {
        Folder {
          catch "destroy $w.b.forw"
          KillMenu $w.b.forw.m
          if $PREF(AliMenu) {
            menubutton $w.b.forw -text "Forward To" -menu $w.b.forw.m
            menu $w.b.forw.m
            $w.b.forw.m add command -label *NEW* -command \
             "set Msg \[GetMsgNum [WF $w]\]
              if {\$Msg != {}} {ForwardMsg [WF $w] \$Msg}"
            MakeMenu $w.b.forw.m $Aliases 0 {} -command  \
             "set Msg \[GetMsgNum [WF $w]\]
              if {\$Msg != {}} {
                if \$PREF(ExpandAlias) {ForwardMsg [WF $w] \$Msg \"\[SearchAlias %Entry 1\]\"} \
                else {ForwardMsg [WF $w] \$Msg %Entry}}"
          } else {
            button $w.b.forw -text "Forward" -command \
             "set Msg \[GetMsgNum [WF $w]\]
              if {\$Msg != {}} {ForwardMsg [WF $w] \$Msg}"
          }
          pack before $w.b.copy $w.b.forw {top expand fill}
        }
        ShowMsg {
          catch "destroy $w.b.forw"
          KillMenu $w.b.forw.m
          if $PREF(AliMenu) {
            menubutton $w.b.forw -text "Forward To" -menu $w.b.forw.m
            menu $w.b.forw.m
            $w.b.forw.m add command -label *NEW* -command \
              "set Fol [lindex [WS $w] 0]; set Msg [lindex [WS $w] 1]
               ForwardMsg \$Fol \$Msg"
            MakeMenu $w.b.forw.m $Aliases 0 {} -command \
              "set Fol [lindex [WS $w] 0]; set Msg [lindex [WS $w] 1]
               if \$PREF(ExpandAlias) {ForwardMsg \$Fol \$Msg \"\[SearchAlias %Entry 1\]\"} \
               else {ForwardMsg \$Fol \$Msg %Entry}"
          } else {
            button $w.b.forw -text "Forward" -command \
             "set Fol [lindex [WS $w] 0]; set Msg [lindex [WS $w] 1]
              ForwardMsg \$Fol \$Msg"
          }
          pack before $w.b.copy $w.b.forw {top expand fill}
        }
      }
    }
}

proc FindLastFile {Fol} {
    global MH

    set cwd [pwd]
    cd [RF $MH(Path)/$Fol]
    set List [lsort -integer \
        [glob -nocomplain {[0-9]} {[0-9][0-9]} {[0-9][0-9][0-9]} \
          {[0-9][0-9][0-9][0-9]} {[0-9][0-9][0-9][0-9][0-9]}]]
    set LastMsg [lindex $List [expr [llength $List] - 1]]
    if {$LastMsg == ""} {
        set LastMsg {0}
    }
    cd $cwd
    return $LastMsg
}

proc FlashSel {w {num 1} {delay 50}} {
    set bgopt [$w configure -selectbackground]
    set fgopt [$w configure -selectforeground]
    set oldbg [lindex $bgopt [expr [llength $bgopt] - 1]]
    set oldfg [lindex $fgopt [expr [llength $fgopt] - 1]]
    for {set i 0} {$i < $num} {incr i} {
      $w configure -selectbackground $oldfg
      $w configure -selectforeground $oldbg; update
      after $delay
      $w configure -selectbackground $oldbg
      $w configure -selectforeground $oldfg; update
    }
}

proc GetListboxSel {Listbox {maxnum 1}} {
    global Result

    if {[set Sels [$Listbox curselection]] == ""} {
        return {}
    }
    set Result {}
    foreach Sel $Sels {
        lappend Result [$Listbox get $Sel]
    }
    if {[llength $Result] <= $maxnum} {
        return $Result
    }
    return [lrange $Result 0 [expr $maxnum - 1]]
}

proc MakeMenu {m List repeat Path args} {
    if {![winfo exists $m]} {menu $m}
    foreach Entry $List {
        if {$Path != {}} {set EntryPath $Path/[lindex $Entry 0]} \
        else {set EntryPath [lindex $Entry 0]}
        if {[llength $Entry] == 1} {
            regsub -all "%EntryPath" $args "$EntryPath" locargs
            regsub -all "%Entry" $locargs "$Entry" locargs
            eval "$m add command -label $Entry $locargs"
        } else {
            set Sub [lindex $Entry 0]
            $m add cascade -label $Sub -menu $m.m$Sub
            if $repeat {
                if {![winfo exists $m.m$Sub]} {menu $m.m$Sub}
                regsub -all "%EntryPath" $args "$EntryPath" locargs
                regsub -all "%Entry" $locargs "$Sub" locargs
                eval "$m.m$Sub add command -label $Sub $locargs"
            }
            set Rest [lindex $Entry 1]
            eval MakeMenu $m.m$Sub {$Rest} $repeat $EntryPath $args
        }
    }
}

proc KillMenu {m} {
    if {![winfo exists $m]} {return}
    set Subs [winfo children $m]
    foreach Sub $Subs {KillMenu $Sub}
    $m delete 0 last
    update
}

proc Pipe {w command {newline nonewline}} {
    if [catch {selection get}] return
    set Result [eval exec $command << {[selection get]}]
    if {![string first "new" $newline]} {append Result "\n"}
    if [catch {$w index sel.first} Selfirst] {
        # Not selected from w
        $w insert insert $Result
    } else {
        # selected from w
        $w delete sel.first sel.last
        $w insert $Selfirst $Result
        set Sellast [$w index "$Selfirst + [string length $Result] chars"]
        $w tag add sel $Selfirst $Sellast
    }
}

proc SaveText {w Filename} {
    # Maybe use error checking on the file... (Is it possible? 
    # does it exist? append?)
    set FHandle [open $Filename w]
    puts -nonewline $FHandle [$w get 1.0 end]
    close $FHandle
}

