#!./wish -f

# Program: tdcad (a vector drawing program)
# Author:  Tuan T. Doan
# Date:    5/10/93
# =========================================================================
# Copyright 1993 Tuan T. Doan
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies.  Tuan
# Doan make no representations about the suitability of this software
# for any purpose.  It is provided "as is" without express or implied
# warranty.  If you do use any part of the software, I would like to
# know about it.  Please send me mail at tdoan@bnr.ca
#
# DXF format is copyrighted by Autodesk, Inc.
# Hershey font was originally created by Dr. A. V. Hershey; format of the
#    font was distributed by James Hurt; converted to tk canvas item by
#    Tuan T. Doan
# =========================================================================

set gvar(HOMEDIR)   .
set gvar(ICONDIR)   $gvar(HOMEDIR)/icon
set gvar(BITMAPDIR) $gvar(HOMEDIR)/bitmaps
set gvar(VFONTDIR)  $gvar(HOMEDIR)/vfont
source $gvar(HOMEDIR)/print.tcl
source $gvar(HOMEDIR)/util.tcl
#source $gvar(HOMEDIR)/sincostan.tcl

set gvar(obj)           cursor
#   coords: used in drawing; pcoords: used in picking; ecoords: used in editing
set gvar(coords)        {}
set gvar(pcoords)       {}
set gvar(ecoords)       {}
#   cx,cy=current pts; lx,ly=last pts (relative current view)
set gvar(cx)            0
set gvar(cy)            0
set gvar(lx)            0
set gvar(ly)            0
set gvar(pastex)        0
set gvar(pastey)        0
set gvar(pat)           0
set gvar(linepat)       \"\"
set gvar(fill)          \"\"
set gvar(width)         1
set gvar(dcolor)        black
set gvar(icolor)        black
set gvar(arrowshape)    {8 10 3}
set gvar(capstyle)      butt
set gvar(joinstyle)     round
set gvar(tfontanchor)   nw
set gvar(tfontjust)     left
set gvar(tfontname)     courier
set gvar(tfontweight)   medium
set gvar(tfontslant)    r
set gvar(tfontpoint)    14
set gvar(tfont)         "-*-$gvar(tfontname)-$gvar(tfontweight)-$gvar(tfontslant)-normal--*-[expr 10*$gvar(tfontpoint)]-*"
set gvar(vfontplace)    horiz
set gvar(tracemode)     line
set gvar(bitmapanchor)  c
set gvar(gridx)         20
set gvar(gridy)         10
set gvar(griddx)        0
set gvar(griddy)        0
set gvar(pside)         8
set gvar(oscale)        1
set gvar(ozoom)         100
set gvar(ozoomtype)     100
set gvar(oscaletype)    abs
set gvar(orotate)       45
set gvar(osnap)         normal
set gvar(ogrid)         off
set gvar(oxhair)        on
set gvar(osavefile)     "test1.tkobj"
set gvar(oloadfile)     $gvar(HOMEDIR)/init.tkobj
set gvar(ocutbuffer)    {}
set gvar(onopick)       0
set gvar(onocut)        0
set gvar(opattern)      normal
set gvar(oarcstyle)     arc
set gvar(oarctype)      3pts
set gvar(oovaltype)     3pts
set gvar(olinetype)     2pts
set gvar(orecttype)     2pts
set gvar(otexttype)     bitmap
set gvar(osheartype)    x
set gvar(oaligntype)    left
set gvar(oarrowtype)    none
set gvar(olinestyle)    arrow
set gvar(olinestype)    extend
set gvar(oplacetype)    front
set gvar(ofliptype)     x
set gvar(osmoothtype)   yes
set gvar(ogrouptype)    yes
set gvar(oedittype)     cut
# do we want more than a single action from an object button?
set gvar(opersist)      0
# (m)illimeter, (c)entimeter, (i)nches, (p)oints
set gvar(u)             ""
set gvar(canvasx1)      -1000
set gvar(canvasy1)      -1000
set gvar(canvasx2)      1000
set gvar(canvasy2)      1000
set gvar(canvasw)       100
set gvar(canvash)       100
set gvar(closeenough)   5
# single bezier data
set gvar(bzctrl)        ""
set gvar(bzpts}         ""
set gvar(bzangle)       ""
set gvar(grpno)         1
set gvar(lastcmd)       {}

proc _translate {c dx dy} {
   $c move pick  $dx $dy
   $c move trace $dx $dy
}

proc _rotAngle {a b cx cy x y} {
#  cpu time: 10099745 9977346 10117669 10097582
   set tx [expr $x-$cx]
   set ty [expr $y-$cy]
   set x1 [expr $tx+$a*$ty]
   set y1 [expr $ty+$b*$x1]
   set x1 [expr $x1+$a*$y1]
   return "[expr $cx+$x1] [expr $cy+$y1]"
}

proc _rotAngleX {a b cx cy x y} {
#  cpu time: 7578209 7199538 7181162 7419897   ~25% faster
   set tx [expr round($x-$cx)]
   set ty [expr round($y-$cy)]
   set x1 [expr $tx*$a-$ty*$b]
   set y1 [expr $tx*$b+$ty*$a]
   return "[expr $cx+($x1>>14)] [expr $cy+($y1>>14)]"
}

proc _xShear {a cx cy x y} {
   # a=tan(theta/2)
   set tx [expr $x-$cx]
   set ty [expr $y-$cy]
   return "[expr $tx+$a*$ty+$cx] [expr $ty+$cy]"
}

proc _yShear {a cx cy x y} {
   # a=sin(theta)
   set tx [expr $x-$cx]
   set ty [expr $y-$cy]
   return "[expr $tx+$cx] [expr $ty+$a*$tx+$cy]"
}

proc _crossProduct {x1 y1 x2 y2} {
   return [expr $x1*$y2-$x2*$y1]
}
proc _dotProduct {x1 y1 x2 y2} {
   set d [expr sqrt(($x1*$x1+$y1*$y1)*($x2*$x2+$y2*$y2))]
   if {$d!=0} {
      return [expr acos(($x1*$x2+$y1*$y2)/$d)]
   } else {
      return 0
   }
}
proc _lineCoeff {x1 y1 x2 y2} {
   set c [expr $x2*$y1-$x1*$y2]
   set a [expr $y2-$y1]
   set b [expr $x1-$x2]
   return "$a $b $c"
}
proc _line2PtsDist {a b c x1 y1} {
   set a2 [expr $a*$a]
   set b2 [expr $b*$b]
   set d [expr sqrt($a2+$b2)]
   if {$d==0} {
      return 0
   } else {
      return [expr ($a*$x1+$b*$y1+$c)/$d]
   }
}
proc _ptsPerp2Line {a b c x1 y1} {
   set x  0
   set y  0
   set d  [expr $a*$a+$b*$b]
   set cp [expr $a*$y1-$b*$x1]
   if {$d!=0} {
      set x [expr (-$a*$c-$b*$cp)/$d]
      set y [expr ($a*$cp-$b*$c)/$d]
   }
   return "$x $y"
}

proc _2closestPts {x1 y1 x2 y2 x3 y3 x4 y4} {
   set dx  [expr $x1-$x3]
   set dy  [expr $y1-$y3]
   set d13 [expr $dx*$dx+$dy*$dy]
   set dx  [expr $x1-$x4]
   set dy  [expr $y1-$y4]
   set d14 [expr $dx*$dx+$dy*$dy]
   set dx  [expr $x2-$x3]
   set dy  [expr $y2-$y3]
   set d23 [expr $dx*$dx+$dy*$dy]
   set dx  [expr $x2-$x4]
   set dy  [expr $y2-$y4]
   set d24 [expr $dx*$dx+$dy*$dy]
   return "$d13 $d14 $d23 $d24"
}

proc _fillet2Lines {ab cd r} {
   set x1 [lindex $ab 0]
   set y1 [lindex $ab 1]
   set x2 [lindex $ab 2]
   set y2 [lindex $ab 3]
   set x3 [lindex $cd 0]
   set y3 [lindex $cd 1]
   set x4 [lindex $cd 2]
   set y4 [lindex $cd 3]

   set t1  [_2closestPts $x1 $y1 $x2 $y2 $x3 $y3 $x4 $y4]
   set d13 [lindex $t1 0]
   set d14 [lindex $t1 1]
   set d23 [lindex $t1 2]
   set d24 [lindex $t1 3]
   if {$d23>$d24} {
      set d2 $d24
      set swap2 1
   } else { 
      set d2 $d23
      set swap2 0
   }
   if {$d13>$d14} {
      set d1 $d14
   } else {
      set d1 $d13
   }
#  puts stderr "D13=$d13 D14=$d14 D23=$d23 D24=$d24"
   # can't do: a=a^b; b=a^b; a=a^b unless a & b are int
   if {$d1<$d2} {
#     puts stderr "SWAP 1st"
      set t5 $x1; set x1 $x2; set x2 $t5
      set t5 $y1; set y1 $y2; set y2 $t5
   }
   if {$swap2} {
#     puts stderr "SWAP 2nd"
      set t5 $x3; set x3 $x4; set x4 $t5
      set t5 $y3; set y3 $y4; set y4 $t5
   }

   set t1 [_lineCoeff $x1 $y1 $x2 $y2]
   set a1 [lindex $t1 0]
   set b1 [lindex $t1 1]
   set c1 [lindex $t1 2]
   set t1 [_lineCoeff $x3 $y3 $x4 $y4]
   set a2 [lindex $t1 0]
   set b2 [lindex $t1 1]
   set c2 [lindex $t1 2] 
   if {[expr $a1*$b2]==[expr $a2*$b1]} { return "" }
   set mx [expr ($x3+$x4)*0.5]
   set my [expr ($y3+$y4)*0.5]
   set d1 [_line2PtsDist $a1 $b1 $c1 $mx $my]
   if {$d1==0} { return "" }
   set mx [expr ($x1+$x2)*0.5]
   set my [expr ($y1+$y2)*0.5]
   set d2 [_line2PtsDist $a2 $b2 $c2 $mx $my]
   if {$d2==0} { return "" }
   if {$d1<=0} { set rr [expr -$r] } { set rr $r }
   set c1p [expr $c1-$rr*sqrt($a1*$a1+$b1*$b1)]
   if {$d2<=0} { set rr [expr -$r] } { set rr $r }
   set c2p [expr $c2-$rr*sqrt($a2*$a2+$b2*$b2)]
   set d   [expr $a1*$b2-$a2*$b1]
   set xc  [expr ($c2p*$b1-$c1p*$b2)/$d]
   set yc  [expr ($c1p*$a2-$c2p*$a1)/$d]
   set t1  [_ptsPerp2Line $a1 $b1 $c1 $xc $yc]
   set x2  [lindex $t1 0]
   set y2  [lindex $t1 1]
   set t1  [_ptsPerp2Line $a2 $b2 $c2 $xc $yc]
   set x3  [lindex $t1 0]
   set y3  [lindex $t1 1]
   set v1x [expr $x2-$xc]
   set v2x [expr $x3-$xc]
   # reverse math because of y's coords is flipped
   set v1y [expr $yc-$y2]
   set v2y [expr $yc-$y3]
   set sa  [expr atan2($v1y,$v1x)]
   set da  [_dotProduct $v1x $v1y $v2x $v2y]
   # reverse check because of y's coords is flipped
   if {[_crossProduct $v1x $v1y $v2x $v2y]<0} {set da [expr -$da]}
   return "$xc $yc $sa $da [list [list $x1 $y1 $x2 $y2]] [list [list $x3 $y3 $x4 $y4]]"
}

proc _closestPoint {c pts x y} {
   global gvar
   set ll [llength $pts]
   set rc -1
   set dist 100000
   for {set i 0; set j 1} {$i < $ll} {incr i 2; incr j 2} {
      set ix [lindex $pts $i]
      set iy [lindex $pts $j]
      set dx [expr {$x-$ix}]
      set dy [expr {$y-$iy}]
      set idist [expr $dx*$dx+$dy*$dy]
      if {$idist < $dist} {
         set rc $i
         set dist $idist
      }
   }
   return "$rc $dist"
}

proc _getAngle {dx dy} {
   set ta 90
   if {$dx!=0} {
      set ta [expr atan(1.0*$dy/$dx)]
      set ta [expr $ta*57.29578]
   }
   if {$dy>0} {
      if {$dx<0} {
         set ta [expr 180+$ta]
      }
   } else {
      if {$dx<=0} {
         set ta [expr 180+$ta]
      } else { 
         set ta [expr 360+$ta]
      }
   }
   return $ta
}

proc _getRadian {dx dy} {
   set ta 1.5707963
   if {$dx!=0} {
      set ta [expr atan(1.0*$dy/$dx)]
   }
   if {$dy>0} {
      if {$dx<0} {
         set ta [expr 3.1415927+$ta]
      }
   } else {
      if {$dx<=0} {
         set ta [expr 3.1415927+$ta]
      } else { 
         set ta [expr 6.2831853+$ta]
      }
   }
   return $ta
}

proc _doCmd {c} {
   global gvar
   set cmd [.cl.entry get]
   set t1 [lindex $cmd 0]
   set t2 [lrange $cmd 1 end]
   case $t1 in {
   {tcl} {
         catch {eval $t2} t5
         puts stderr "COMMAND: $t2 ($t5)"
      }
   {v*} {
         _drawVfont $c $t2
      }
   {bm bitmap} {
         set gvar(obitmap) $t2
         _drawBitmap $c $t2
      }
   {gridx} {
         set gvar(gridx) $t2
      }
   {gridy} {
         set gvar(gridy) $t2
      }
   {grid} {
         set gvar(gridx) $t2
         set gvar(gridy) $t2
      }
   {tx} {
         _translate $c $t2 0
      }
   {ty} {
         _translate $c 0 $t2
      }
   {s scale} {
         _setScale $c 1 $t2
      }
   {sx scalex} {
         _setScale $c 2 $t2
      }
   {sy scaley} {
         _setScale $c 3 $t2
      }
   {r rotate} {
         set gvar(orotate) $t2
         _setRotate $c pick
      }
   {pside*} {
         set gvar(pside) $t2
      }
   default {
         tk_dialog .dm "dialog" "Unknown command: $t1" warning 0 "Ok"
      }
   }
}

proc _createXhair {c} {
   global gvar
   $c delete xhair yhair
   $c create line [$c canvasx $gvar(canvasx1)$gvar(u)] 2$gvar(u) [$c canvasx $gvar(canvasx2)$gvar(u)] 2$gvar(u) -tags "xhair" -fill $gvar(icolor)
   $c create line 2$gvar(u) [$c canvasy $gvar(canvasy1)$gvar(u)] 2$gvar(u) [$c canvasy $gvar(canvasy2)$gvar(u)] -tags "yhair" -fill $gvar(icolor)
   set t1 [$c find below xhair]
   while {$t1!=""} {
      $c lower xhair $t1
      set t1 [$c find below xhair]
   }
   $c lower yhair xhair
}

proc _drawGrid {c xc yc dx dy ang} {
   global gvar
   # butt-slow; also point (xc,yc) is generated 4 times!
   # 0.017453293 / 2 = 0.0087266463       |
   # angle starts   ----------------> 0.0 v
   if {($ang%90)==0} {
      set t1 {$i$gvar(u) $j$gvar(u)}
   } else {
      set a [expr {-1*tan($ang*0.0087266463)}]
      set b [expr {-2*$a/(1+$a*$a)}]
      set t1 {[_rotAngle $a $b $xc $yc $i $j]}
   }
   for {set j $yc} {$j<1000} {incr j $dy} {
      for {set i $xc} {$i<1000} {incr i $dx} {
         eval $c create bitmap $t1 -tags "grid" \
            -fg $gvar(dcolor) -anchor center \
            -bitmap @$gvar(ICONDIR)/gridmark.cad
      }
      for {set i $xc} {$i>-1000} {incr i -$dx} {
         eval $c create bitmap $t1 -tags "grid" \
            -fg $gvar(dcolor) -anchor center \
            -bitmap @$gvar(ICONDIR)/gridmark.cad
      }
   }
   for {set j $yc} {$j>-1000} {incr j -$dy} {
      for {set i $xc} {$i<1000} {incr i $dx} {
         eval $c create bitmap $t1 -tags "grid" \
            -fg $gvar(dcolor) -anchor center \
            -bitmap @$gvar(ICONDIR)/gridmark.cad
      }
      for {set i $xc} {$i>-1000} {incr i -$dx} {
         eval $c create bitmap $t1 -tags "grid" \
            -fg $gvar(dcolor) -anchor center \
            -bitmap @$gvar(ICONDIR)/gridmark.cad
      }
   }
}

proc _cycleSnap {snap} {
#  puts stderr "CYCLE: $snap"
   case $snap in {
   {orth}    {return angle}
   {angle}   {return endpts}
   {endpts}  {return midpts}
   {midpts}  {return grid}
   {grid}    {return normal}
   {normal}  {return orth}
   }
}

proc _snapPoint {c x y x1 y1 x2 y2} {
   global gvar 
   case $gvar(osnap) in {
      {orth} {
         set dx [expr abs($x1-$x2)]
         set dy [expr abs($y1-$y2)]
         if {$dx>$dy} {
            set gvar(cx) $x1
            set gvar(cy) $y2
         } else {
            set gvar(cx) $x2
            set gvar(cy) $y1
         }
      }
      {angle} {
         set dx [expr $x1-$x2]
         set dy [expr $y1-$y2]
         set a  [expr round([_getAngle $dx $dy])]
         set a  [expr ($a-($a%$gvar(orotate)))*0.017453293]
#        set a  [expr round($a)/$gvar(orotate)*$gvar(orotate)*0.017453293]
         set dx [expr abs($dx)]
         set dy [expr abs($dy)]
         set r  [expr $dx+$dy-((($dx>$dy)?$dy:$dx)*0.5)]
         set gvar(cx) [expr $r*cos($a)+$x2]
         set gvar(cy) [expr $r*sin($a)+$y2]
#        puts stderr "ANGLE: $a - $gvar(cx) $gvar(cy)"
      }
      {grid} {
         set gvar(cx) [expr [$c canvasx $x $gvar(gridx)]+$gvar(griddx)]
         set gvar(cy) [expr [$c canvasy $y $gvar(gridy)]+$gvar(griddy)]
      }
      {endpts midpts perp} {
      }
      default {
         set gvar(cx) $x1
         set gvar(cy) $y1
      }
   }
}

proc _drawCursor {c x y snap} {
   global gvar
   set x1 [$c canvasx $x]
   set y1 [$c canvasy $y]
   set ll [llength $gvar(coords)]
   incr ll -1
   set y2 [lindex $gvar(coords) $ll]
   incr ll -1
   set x2 [lindex $gvar(coords) $ll]
   if $snap {
         case $gvar(osnap) in {
         {endpts midpts grid} {
               _snapPoint $c $x $y $x1 $y1 $x2 $y2
            }
         default {
               if {$x2!="" && $y2!=""} {
                  _snapPoint $c $x $y $x1 $y1 $x2 $y2
               } else {
                  set gvar(cx) $x1
                  set gvar(cy) $y1
               }
            }
         }
   } else {
      set gvar(cx) $x1
      set gvar(cy) $y1
   }
   set gvar(xylabel) "@[format "%.2f,%.2f" $gvar(cx) $gvar(cy)]<$gvar(orotate)"
   if {$gvar(oxhair)=="on"} {
      $c coords xhair $gvar(canvasx1) $gvar(cy) $gvar(canvasx2) $gvar(cy)
      $c coords yhair $gvar(cx) $gvar(canvasy1) $gvar(cx) $gvar(canvasy2)
   }
   if {[llength $gvar(coords)] > 0} {
      case $gvar(tracemode) in {
         {rorth} {
               set x1 [lindex $gvar(coords) 0]
               set y1 [lindex $gvar(coords) 1]
               if {$x1<$gvar(cx)} {
                  set xm $gvar(cx)
                  set ym $y1
               } else {
                  set xm $x1
                  set ym $gvar(cy)
               }
               set gvar(coords) [lreplace $gvar(coords) 2 3 $xm $ym]
               eval $c coords trace $gvar(coords) $gvar(cx) $gvar(cy)
            }
         {lorth} {
               set x1 [lindex $gvar(coords) 0]
               set y1 [lindex $gvar(coords) 1]
               if {$x1<$gvar(cx)} {
                  set xm $x1
                  set ym $gvar(cy)
               } else {
                  set xm $gvar(cx)
                  set ym $y1
               }
               set gvar(coords) [lreplace $gvar(coords) 2 3 $xm $ym]
               eval $c coords trace $gvar(coords) $gvar(cx) $gvar(cy)
            }
         {pline} { 
            foreach i [$c find withtag trace] {
               scan [$c coords $i] "%f %f %f %f" x1 y1 x2 y2
#              $c coords trace $x1 $y1 $gvar(cx) $gvar(cy)
               set t2 [$c coords $i]
               set ll [llength $t2]
               incr ll -1
               set t2 [lreplace $t2 $ll $ll $gvar(cy)]
               incr ll -1
               set t2 [lreplace $t2 $ll $ll $gvar(cx)]
               eval $c coords $i $t2
            }
         }
         {line} {
            $c coords trace $x2 $y2 $gvar(cx) $gvar(cy)
         }
         {rect} {
            if {$x1>$gvar(cx)} {
               set t5 $x1
               set x1 $gvar(cx)   
               set x2 $t5
               set t5 $y1
               set y1 $gvar(cy)
               set y2 $t5
            }
            $c coords trace $x1 $y1 $x2 $y2
         }
      }
   }
}

proc _restore {c} {
   global gvar
   $c dtag pick pick
   $c delete trace trace2
   set gvar(coords) ""
}

proc _dumpObj {c tag fd} {
   global gvar
   foreach j [$c find withtag $tag] {
      set opt ""
      foreach i [$c itemconfig $j] {
         set defopt  [lindex $i 3]
         set actopt  [lindex $i 4]
         if {[llength $i]==5 && $defopt!=$actopt} {
            set optname [lindex $i 0]
            set optval  $actopt
            case $optname in {
            {-tags} {
                  set t1 {}
                  foreach k $optval {
                     case $k in {
                     {pick current} {}
                     default        {lappend t1 $k}
                     }
                  }
                  set optval [list $t1]
               }
            {-text} {
                  regsub -all \n $optval \033 optval
                  set optval [list $optval]
               }
            {-stipple} {
#                 puts stderr "DO: $actopt - $gvar(BITMAPDIR)/$gvar(opattern)"
                  regsub -all $gvar(BITMAPDIR) $actopt {$gvar(BITMAPDIR)} optval
#                 set t1 [file dirname $actopt]
#                 if {[file dirname $t1]=="@$gvar(BITMAPDIR)"} {
#                    set optval "\@\$gvar(BITMAPDIR)/$gvar(opattern)/[file tail $actopt]"
#                 }
               }
            {-bitmap} {
#                 puts stderr "DO: $actopt - $gvar(ICONDIR)"
#                 set optval "\@\$gvar(ICONDIR)/[file tail $actopt]"
                  set optval $actopt
               }
            default {
               set optval [list $actopt]
            }
            }
            append opt "$optname $optval "
         }
      }
      set t1 [concat "$c create [$c type $j]" [$c coords $j] $opt]
      puts $fd "$t1"
      lappend result $t1
   }
   return $result
}


proc _quit {c} {
   if {[tk_dialog .dm "dialog" "Are you sure?" question 0 "No" "Yes"]==1} {
      destroy .
      exit 0
   }
}

proc _saveit {c} {
   global gvar
   if {[file exists $gvar(osavefile)]} {
      if {[tk_dialog .dm "dialog" "File exists! Overwrite?" question 0 "No" "Yes"]==0} {
         return
      }
   }
   set fd [open $gvar(osavefile) w+] 
   _busy "_dumpObj $c obj $fd"
   close $fd
}

proc _accessFile {w fname} {
   global gvar
   if {$fname==""} {return 0}
   if {[file isdirectory $fname]} {
      if {[file executable $fname]} {
         $w.e1 delete 0 end
         $w.e1 insert 0 $fname
#        puts stderr "ACCESS: [cd $fname]"
         $w.lb delete 0 end
         $w.lb insert end "../"
         foreach i [lsort [glob -nocomplain *]] {
            if {[file isdirectory $i]} {
               $w.lb insert end "$i/"
            } else {
               $w.lb insert end $i
            }
         }
         update
      } else {
         tk_dialog .dm "dialog" "Could not access file" info 0 "Ok"
      }
   } else {
      if {[file readable $fname]} {
         $w.e1 delete 0 end
         $w.e1 insert 0 $fname
         set gvar(grc) $fname
         return 1
      } else {
         tk_dialog .dm "dialog" "Could not access file" info 0 "Ok"
      }
   }
   return 0
}
proc _getFileFromListBox {w} {
   set t1 [$w.lb curselection]
   if {$t1!=""} {
      return [$w.lb get $t1]
   }
   return ""
}
proc _getFile {title} {
   global gvar
   set f .file
   catch {destroy $f}
   toplevel $f 
   label $f.l1 -text "${title}:"
   entry $f.e1 -relief sunken
   bind $f.e1 <Return> "if \[_accessFile $f \[$f.e1 get\]\] {destroy $f}"
   frame $f.l3
      scrollbar $f.l3.hs -orient horiz -relief sunken -command "$f.lb xview"
      frame     $f.l3.cn -height 20 -width 20 -relief sunken 
   pack $f.l3.cn -side right
   pack $f.l3.hs -side left -fill x -expand on
   listbox   $f.lb -xscroll "$f.l3.hs set" -yscroll "$f.vs set" -relief sunken
   bind $f.lb <Double-Button-1> "if \[_accessFile $f \[_getFileFromListBox $f\]\] {destroy $f}"
   scrollbar $f.vs -relief sunken -command "$f.lb yview"
   frame $f.cmd 
      button $f.cmd.b1 -text "CANCEL" -command "destroy $f"
      button $f.cmd.b2 -text "OK"     -command "if \[_accessFile $f \[$f.e1 get\]\] {destroy $f}"
   pack $f.cmd.b1 -side left -fill x -expand yes
   pack $f.cmd.b2 -side left -fill x -expand yes

   pack $f.l1   -side top -anchor w
   pack $f.e1   -side top -fill x -expand yes
   pack $f.cmd  -side bottom -fill x
   pack $f.l3   -side bottom -fill x
   pack $f.vs   -side right -fill y
   pack $f.lb   -side left -fill both -expand yes
   _accessFile $f "."
   tkwait window $f
}

proc _save {c} {
   global gvar
   set gvar(grc) ""
   _getFile "Save file"
   if {$gvar(grc)!=""} {
      set gvar(oloadfile) $gvar(grc)
      _busy {_saveit $c}
   }
}

proc _loadit {c} {
   global gvar
   $c delete obj
   if {[file exists $gvar(oloadfile)]} {
      set fd [open $gvar(oloadfile) r]
      while {[gets $fd line] > -1} {
         regsub -all \033 $line \n line
         set i [eval $line]
         puts stderr "Loading: $i"
         $c bind $i <Any-Enter> "_enterObj $c $i"
         $c bind $i <Any-Leave> "_leaveObj $c $i"
      }
      close $fd
   } else {
      tk_dialog .dm "dialog" "Could not access file" info 0 "Ok"
      _load $c
   }
}

proc _load {c} {
   global gvar
   set gvar(grc) ""
   _getFile "Load file"
   if {$gvar(grc)!=""} {
      set gvar(oloadfile) $gvar(grc)
      _busy {_loadit $c}
   }
   return
}

proc _drawBitmap {c fname} {
   global gvar font
#  set fname [.i1.bitmap.2 get]
   foreach i [glob -nocomplain $fname] {
      if {[file exists $i] && [regexp "c program text" [exec file $i]]} {
         set x [lindex $gvar(coords) 0]
         set y [lindex $gvar(coords) 1]
         eval "$c create bitmap $x $y -anchor $gvar(bitmapanchor) -bitmap @$i -tags \"obj\" -foreground $gvar(dcolor)"
         set x [expr $x+10]
         set y [expr $y+10]
         set gvar(coords) "$x $y"
      }
   }
   $c delete trace
   _moveMode $c
}

proc _drawVfont {c txt} {
   global gvar font
   if {$gvar(coords)!=""} {
      set cx [lindex $gvar(coords) 0]
      set cy [lindex $gvar(coords) 1]
      set fname $gvar(vfontname)
      set ltxt [string length $txt]
      if {$gvar(vfontplace)=="guide"} {
         set guideok 0
         foreach i [$c find closest $cx $cy 10 trace] {
#           puts stderr "GUIDE: $i"
            if {[$c type $i]=="line"} {
               set pts [$c coords $i]
               if {[llength $pts]>=[expr $ltxt*2]} {
                  set guideok 1
#                 puts stderr "GUIDE... $pts"
                  break
               }
            }
         }
         if {!$guideok} {
            tk_dialog .dm "dialog" "Could not find guide" error 0 "Ok"
            return
         }
         set t1 [lindex [_closestPoint $c $pts $cx $cy] 0]
         set pts "[lrange $pts $t1 end] [lrange $pts 0 [expr $t1-1]]"
      }
      for {set i 0} {$i < $ltxt} {incr i} {
         scan [string index $txt $i] "%c" t3
         if {[info exists font($fname,$t3)]} {
            foreach j $font($fname,$t3) {
               eval "$c create $j -tags \"trace2 obj vtext\" -fill $gvar(dcolor)"
            }
            scan [$c bbox trace2] "%d %d %d %d" x1 y1 x2 y2
            set t4 [$c create polygon $x1 $y1 $x2 $y1 $x2 $y2 $x1 $y2 \
               -tags "trace2 vtext$t3 obj" -fill ""]
            $c bind $t4 <Any-Enter> "_enterObj $c $t4"
            $c bind $t4 <Any-Leave> "_leaveObj $c $t4"
            $c addtag vt$t4 withtag vtext
            $c dtag vtext
#           puts stderr "$t4 - [$c find withtag vt$t4]"
            case $gvar(vfontplace) in {
            {horiz} {
                  set dx [expr $x2-$x1]
                  $c move trace2 [expr $cx+($dx*0.5)] $cy
                  set cx [expr $cx+$dx]
               }
            {vert} {
                  set dy [expr $y2-$y1]
                  $c move trace2 $cx [expr $cy+($dy*0.5)]
                  set cy [expr $cy+$dy]
               }
            {guide} {
                  set cx [lindex $pts [expr $i*2]]
                  set cy [lindex $pts [expr $i*2+1]]
                  $c move trace2 $cx $cy
               }
            }
            $c dtag trace2
         } else {
            set cx [expr $cx+20]
         }
      }
      set gvar(coords) ""
      $c delete trace
      _moveMode $c
   }
}

proc _enterObj {c obj} {
   global gvar

   set Itag  [$c gettags $obj]
   set Itype [$c type $obj]
#  puts stderr "TAG=$Itag TYPE=$Itype"
   # rectangle: bounding box - trace
   # polygon:   vfont
   if {[lsearch $Itag vt*]>-1} {
      set Itype vtext
   }
   case $Itype in {
   {bitmap} {}
   {arc line oval polygon rectangle} {
         set t2 [lindex [_closestPoint $c [$c coords $obj] $gvar(cx) $gvar(cy)] 1]
         if {$t2<100} {
            $c config -cursor circle
         } else {
            $c config -cursor cross
         }
      }
   {text} {
         $c config -cursor pencil
      }
   {vtext} {
         $c config -cursor cross
      }
   }
   case $gvar(osnap) in {
      {tangent} {
         set pts [$c coords $obj]
      }
      {perp} {
         set pts [$c coords $obj]
      }
      {endpts} {
         set pts [$c coords $obj]
         set t1 [_closestPoint $c $pts $gvar(cx) $gvar(cy)]
         set t2  [lindex $t1 1]
         set t1  [lindex $t1 0]
         set gvar(cx) [lindex $pts $t1]
         set gvar(cy) [lindex $pts [incr t1]]
      }
      {midpts} {
         set pts [$c coords $obj]
         set x1 [lindex $pts 0]
         set y1 [lindex $pts 1]
         set x2 [lindex $pts 2]
         set y2 [lindex $pts 3]
         set gvar(cx) [expr ($x1+$x2)*0.5]
         set gvar(cy) [expr ($y1+$y2)*0.5]
      }
   }
#  _labelObj $c $obj
}

proc _leaveObj {c i} {
   global gvar
   $c config -cursor left_ptr
}

proc _mkObj {obj} {
   global gvar
   set i [eval $obj] 
   set c [lindex $obj 0]
   $c bind $i <Any-Enter> "_enterObj $c $i"
   $c bind $i <Any-Leave> "_leaveObj $c $i"
   set gvar(coords) {}
   $c delete trace trace2
   _moveMode $c
   return $i
}
proc _editObj {c obj xc yc xs ys} {
   global gvar
   if {$xs<0.0000001} {set xs 0.0000001}
   if {$ys<0.0000001} {set ys 0.0000001}
   set gvar(lastcmd) {}
   foreach i [$c find withtag $obj] {
      set t1 "$c coords $i [$c coords $i]"
      lappend gvar(lastcmd) $t1
      $c scale $i $xc $yc $xs $ys
   }
#  puts stderr "EDIT: $gvar(lastcmd)"
   set gvar(obj) cursor
   $c config -cursor left_ptr
   _boundingBox $c $obj
}

proc _compute3ptsCircle {} {
   global gvar
   set x1 [lindex $gvar(coords) 0]
   set y1 [lindex $gvar(coords) 1]
   set x2 [lindex $gvar(coords) 2]
   set y2 [lindex $gvar(coords) 3]
   set x3 [lindex $gvar(coords) 4]
   set y3 [lindex $gvar(coords) 5]
   set tx2 [expr $x2-$x1]
   set ty2 [expr $y2-$y1]
   set tx3 [expr $x3-$x1]
   set ty3 [expr $y3-$y1]
   set st2 [expr $tx2*$tx2+$ty2*$ty2]
   set st3 [expr $tx3*$tx3+$ty3*$ty3]
   set xc  [expr ($ty3*$st2-$ty2*$st3)/($tx2*$ty3-$tx3*$ty2)*0.5]
   set yc  [expr ($tx3*$st2-$tx2*$st3)/($ty2*$tx3-$ty3*$tx2)*0.5]
   set r   [expr sqrt($xc*$xc+$yc*$yc)]
   return "[expr $xc+$x1] [expr $yc+$y1] $r"
}

proc _computeBezier {n} {
   global gvar

   set p0x [lindex $gvar(coords) 0]
   set p0y [lindex $gvar(coords) 1]
   set p1x [lindex $gvar(coords) 2]
   set p1y [lindex $gvar(coords) 3]
   set p2x [lindex $gvar(coords) 4]
   set p2y [lindex $gvar(coords) 5]
   set p3x [lindex $gvar(coords) 6]
   set p3y [lindex $gvar(coords) 7]

   set du [expr 1.0/($n-1)]
   set u 0.0
   set gvar(bzpts)   {}
   set gvar(bzangle) {}
   for {set i 0} {$i<$n} {incr i} {
      set up  [expr 1.0-$u]
      set b2  [expr $u*$u]
      set u2  [expr 3.0*$b2]
      set c2  3.0*(2.0*$u-$u2)
      set c1  3.0*(1.0-4.0*$u+$u2)
      set c0  3.0*$up*$up
      set dx  [expr $p3x*$u2+$p2x*$c2+$p1x*$c1-$p0x*$c0]
      set dy  [expr $p3y*$u2+$p2y*$c2+$p1y*$c1-$p0y*$c0]
      set b3  [expr $b2*$u]
      set b2  [expr 3.0*$b2*$up]
      set up2 [expr $up*$up]
      set b1  [expr 3.0*$u*$up2]
      set b0  [expr $up2*$up]
      set x   [expr $p3x*$b3+$p2x*$b2+$p1x*$b1+$p0x*$b0]
      set y   [expr $p3y*$b3+$p2y*$b2+$p1y*$b1+$p0y*$b0]
      lappend gvar(bzpts) "$x $y"
      set u   [expr $u+$du]
      set a   [expr atan(-1.0*$dx/$dy)]
      lappend gvar(bzangle) $a
   }
}

proc _drawNorm {c} {
   global gvar
   set i 0
   foreach pts $gvar(bzpts) {
      set x [lindex $pts 0]
      set y [lindex $pts 1]
      set a [lindex $gvar(bzangle) $i]
      eval $c create line $x $y [expr $x+20*cos($a)] [expr $y+20*sin($a)]
      incr i
   }
}

proc _draw3ptsArc {c} {
   global gvar
   set pts [_compute3ptsCircle]
   set xc  [lindex $pts 0]
   set yc  [lindex $pts 1]
   set r   [lindex $pts 2]
   set x1  [lindex $gvar(coords) 0]
   set y1  [lindex $gvar(coords) 1]
   set x3  [lindex $gvar(coords) 4]
   set y3  [lindex $gvar(coords) 5]
   set dx  [expr $x1-$xc]
   set dy  [expr $yc-$y1]
   set a1  [_getAngle $dx $dy]
   set dx  [expr $x3-$xc]
   set dy  [expr $yc-$y3]
   set a2  [_getAngle $dx $dy]
   set a2 [expr $a2-$a1]
   set x1  [expr $xc-$r]
   set y1  [expr $yc-$r]
   set x2  [expr $xc+$r]
   set y2  [expr $yc+$r]
   _mkObj "$c create arc $x1 $y1 $x2 $y2 -start $a1 -extent $a2 \
      -outline $gvar(dcolor) \
      -tags \"obj\" -style $gvar(oarcstyle) \
      -stipple @$gvar(BITMAPDIR/$gvar(opattern)/pat$gvar(pat).cad -fill $gvar(fill)"
   _moveMode $c
}

proc _draw3ptsCircle {c} {
   global gvar
   set pts [_compute3ptsCircle]
   set xc  [lindex $pts 0]
   set yc  [lindex $pts 1]
   set r   [lindex $pts 2]
   set x1  [expr $xc-$r]
   set y1  [expr $yc-$r]
   set x2  [expr $xc+$r]
   set y2  [expr $yc+$r]
   _mkObj "$c create oval $x1 $y1 $x2 $y2 -tags \"obj\" \
      -width $gvar(width) -fill $gvar(fill) \
      -outline $gvar(dcolor) \
      -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad"
   _moveMode $c
}

proc _draw3ptsOval {c} {
#  from Graphics Gem III
   global gvar
   set m  3
   set xp [expr [lindex $gvar(coords) 0]<<16]
   set yp [expr [lindex $gvar(coords) 1]<<16]
   set xq [expr [lindex $gvar(coords) 2]<<16]
   set yq [expr [lindex $gvar(coords) 3]<<16]
   set xk [expr [lindex $gvar(coords) 4]<<16]
   set yk [expr [lindex $gvar(coords) 5]<<16]
   set vx [expr $xk-$xq]
   set ux [expr $xk-$xp]
   set vy [expr $yk-$yq]
   set uy [expr $yk-$yp]
   set xj [expr $xp-$vx+32768]
   set yj [expr $yp-$vy+32768]
#  puts stderr "ARC3: $xj $yj"
   set w  [expr $ux >> (2*$m + 3)]
   incr ux -$w
   set w  [expr $w  >> (2*$m + 4)]
   incr ux -$w
   set w  [expr $w  >> (2*$m + 3)]
   incr ux -$w
   set w  [expr $vx >> ($m + 1)]
   incr ux $w
   set w  [expr $uy >> (2*$m + 3)]
   incr uy -$w
   set w  [expr $w  >> (2*$m + 4)]
   incr uy -$w
   set w  [expr $w  >> (2*$m + 3)]
   incr uy -$w
   set w  [expr $vy >> ($m + 1)]
   incr uy $w
   set pts ""
   for {set i [expr (102944<<$m)>>16]} {$i>=0} {incr i -1} {
      set x [expr ($xj+$vx) >> 16]
      set y [expr ($yj+$vy) >> 16]
      append pts "$x $y "
      incr ux [expr -($vx >> $m)]
      incr vx [expr  ($ux >> $m)]
      incr uy [expr -($vy >> $m)]
      incr vy [expr  ($uy >> $m)]
   }
   _mkObj "$c create line $pts -tags \"obj\" \
      -arrow $gvar(oarrowtype) -capstyle $gvar(capstyle) \
      -joinstyle $gvar(joinstyle) -width $gvar(width) \
      -fill $gvar(dcolor) \
      -stipple $gvar(linepat) -arrowshape [list $gvar(arrowshape)]"
   _moveMode $c
}

proc _drawStart {c x y} {
   global gvar
   case $gvar(obj) in {
   {cursor}  { 
              set Itag [$c gettags pick]
#             puts stderr "drawStart: $Itag"
              if {$Itag==""} {
                 _pickObj $c [$c find withtag current]
              }
              set gvar(startDragX) [$c canvasx $x]
              set gvar(startDragY) [$c canvasy $y]
             }
   {edit_n edit_nw edit_ne edit_e edit_se edit_s edit_sw edit_w}  {
              set gvar(ecoords) [$c bbox trace]
              $c delete trace
              eval $c create rectangle $gvar(ecoords) -tags "trace" -outline $gvar(dcolor)
             }
   }
}
proc _dragObj {c x y snap} {
   global gvar
   if $snap {
      set x [expr [$c canvasx $x $gvar(gridx)]+$gvar(griddx)]
      set y [expr [$c canvasy $y $gvar(gridy)]+$gvar(griddy)]
   } else {
      set x [$c canvasx $x]
      set y [$c canvasy $y]
   }
   set Itag [$c gettags pick]
   case $gvar(obj) in { 
   {edit_nw} {
              set pts [$c coords trace]
              set xc [lindex $gvar(ecoords) 2]
              set yc [lindex $gvar(ecoords) 3]
              set xs [expr 1.0*($x-$xc)/([lindex $pts 0]-$xc)]
              set ys [expr 1.0*($y-$yc)/([lindex $pts 1]-$yc)]
              if {$ys>0 && $xs>0} {
                 $c scale trace $xc $yc $xs $ys
              }
             }
   {edit_n}  {
              set pts [$c coords trace]
              set xc [lindex $gvar(ecoords) 2]
              set yc [lindex $gvar(ecoords) 3]
              set ys [expr 1.0*($y-$yc)/([lindex $pts 1]-$yc)]
              if {$ys>0} {
                 $c scale trace $xc $yc 1.0 $ys
              }
             }
   {edit_ne} {
              set pts [$c coords trace]
              set xc [lindex $gvar(ecoords) 0]
              set yc [lindex $gvar(ecoords) 3]
              set xs [expr 1.0*($x-$xc)/([lindex $pts 2]-$xc)]
              set ys [expr 1.0*($y-$yc)/([lindex $pts 1]-$yc)]
              if {$ys>0 && $xs>0} {
                 $c scale trace $xc $yc $xs $ys
              }
             }
   {edit_e}  {
              set pts [$c coords trace]
              set xc [lindex $gvar(ecoords) 0]
              set yc [lindex $gvar(ecoords) 1]
              set xs [expr 1.0*($x-$xc)/([lindex $pts 2]-$xc)]
              if {$xs>0} {
                 $c scale trace $xc $yc $xs 1.0
              }
             }
   {edit_se} {
              set pts [$c coords trace]
              set xc [lindex $gvar(ecoords) 0]
              set yc [lindex $gvar(ecoords) 1]
              set xs [expr 1.0*($x-$xc)/([lindex $pts 2]-$xc)]
              set ys [expr 1.0*($y-$yc)/([lindex $pts 3]-$yc)]
              if {$ys>0 && $xs>0} {
                 $c scale trace $xc $yc $xs $ys
              }
             }
   {edit_s}  {
              set pts [$c coords trace]
              set xc [lindex $gvar(ecoords) 0]
              set yc [lindex $gvar(ecoords) 1]
              set ys [expr 1.0*($y-$yc)/([lindex $pts 3]-$yc)]
              if {$ys>0} {
                 $c scale trace $xc $yc 1.0 $ys
              }
             }
   {edit_sw} {
              set pts [$c coords trace]
              set xc [lindex $gvar(ecoords) 2]
              set yc [lindex $gvar(ecoords) 1]
              set xs [expr 1.0*($x-$xc)/([lindex $pts 0]-$xc)]
              set ys [expr 1.0*($y-$yc)/([lindex $pts 3]-$yc)]
              if {$ys>0 && $xs>0} {
                 $c scale trace $xc $yc $xs $ys
              }
             }
   {edit_w}  {
              set pts [$c coords trace]
              set xc [lindex $gvar(ecoords) 2]
              set yc [lindex $gvar(ecoords) 3]
              set xs [expr 1.0*($x-$xc)/([lindex $pts 0]-$xc)]
              if {$xs>0} {
                 $c scale trace $xc $yc $xs 1.0
              }
             }
   {cursor} {set cursor [lindex [$c config -cursor] 4]
             if {$cursor=="circle"} {
                foreach i [$c find withtag pick] {
                   set t1 [lindex [_closestPoint $c [$c coords $i] $x $y] 0]
                   set pts [$c coords $i]
                   set pts [lreplace $pts $t1 [incr t1] $x $y]
#                  puts stderr "EDIT $i: $t1 - $pts"
                   eval $c coords $i $pts
                }
             } else {

             if {$Itag==""} {
                if {$gvar(pcoords)==""} {
                   set gvar(pcoords) "$x $y"
                   $c delete trace
                   $c create rectangle $x $y $x $y -tags "trace" -outline $gvar(dcolor)
                } else {
                   set x2 [lindex $gvar(pcoords) 0]
                   set y2 [lindex $gvar(pcoords) 1]
                   $c coords trace $x $y $x2 $y2
                }
             } else {
                set dx [expr $x-$gvar(startDragX)]
                set dy [expr $y-$gvar(startDragY)]
                $c move pick  $dx $dy
                $c move trace $dx $dy
                set gvar(startDragX) $x
                set gvar(startDragY) $y
             }
            
             }
            }
   }
}
proc _drawEnd {c x y} {
   global gvar
   case $gvar(obj) in {
   {cursor}  {
              append gvar(pcoords) " $x $y"
#             puts stderr "DRAWEND: $gvar(pcoords)"
              _pickArea $c
              set gvar(pcoords) ""
             }
   {edit_nw} {
              set xc [lindex $gvar(ecoords) 2]
              set yc [lindex $gvar(ecoords) 3]
              set xs [expr 1.0*([$c canvasx $x]-$xc)/([lindex $gvar(ecoords) 0]-$xc)]
              set ys [expr 1.0*([$c canvasy $y]-$yc)/([lindex $gvar(ecoords) 1]-$yc)]
              _editObj $c pick $xc $yc $xs $ys
             }
   {edit_n}  {
              set xc [lindex $gvar(ecoords) 2]
              set yc [lindex $gvar(ecoords) 3]
              set ys [expr 1.0*([$c canvasy $y]-$yc)/([lindex $gvar(ecoords) 1]-$yc)]
              _editObj $c pick $xc $yc 1 $ys
             }
   {edit_ne} {
              set xc [lindex $gvar(ecoords) 0]
              set yc [lindex $gvar(ecoords) 3]
              set xs [expr 1.0*([$c canvasx $x]-$xc)/([lindex $gvar(ecoords) 2]-$xc)]
              set ys [expr 1.0*([$c canvasy $y]-$yc)/([lindex $gvar(ecoords) 1]-$yc)]
              _editObj $c pick $xc $yc $xs $ys
             }
   {edit_e}  {
              set xc [lindex $gvar(ecoords) 0]
              set yc [lindex $gvar(ecoords) 1]
              set xs [expr 1.0*([$c canvasx $x]-$xc)/([lindex $gvar(ecoords) 2]-$xc)]
              _editObj $c pick $xc $yc $xs 1
             }
   {edit_se} {
              set xc [lindex $gvar(ecoords) 0]
              set yc [lindex $gvar(ecoords) 1]
              set xs [expr 1.0*([$c canvasx $x]-$xc)/([lindex $gvar(ecoords) 2]-$xc)]
              set ys [expr 1.0*([$c canvasy $y]-$yc)/([lindex $gvar(ecoords) 3]-$yc)]
              _editObj $c pick $xc $yc $xs $ys
             }
   {edit_s}  {
              set xc [lindex $gvar(ecoords) 0]
              set yc [lindex $gvar(ecoords) 1]
              set ys [expr 1.0*([$c canvasy $y]-$yc)/([lindex $gvar(ecoords) 3]-$yc)]
              _editObj $c pick $xc $yc 1 $ys  
             }
   {edit_sw} {
              set xc [lindex $gvar(ecoords) 2]
              set yc [lindex $gvar(ecoords) 1]
              set xs [expr 1.0*([$c canvasx $x]-$xc)/([lindex $gvar(ecoords) 0]-$xc)]
              set ys [expr 1.0*([$c canvasy $y]-$yc)/([lindex $gvar(ecoords) 3]-$yc)]
              _editObj $c pick $xc $yc $xs $ys
             }
   {edit_w}  {
              set xc [lindex $gvar(ecoords) 2]
              set yc [lindex $gvar(ecoords) 3]
              set xs [expr 1.0*([$c canvasx $x]-$xc)/([lindex $gvar(ecoords) 0]-$xc)]
              _editObj $c pick $xc $yc $xs 1
             }
   default   {_drawObj $c $x $y
             }
   }
}

proc _shearObj {c obj} {
   global gvar

   set t1 [$c find withtag $obj]
   if {$t1==""} {
      tk_dialog .dm "dialog" "No object picked" error 0 "Ok"
      return
   }
   case $gvar(osheartype) in {
      {x}  {
         set a [expr tan($gvar(orotate)*0.0087266463)]
         set pts [$c bbox $obj]
         set xc  [lindex $pts 0]
         set yc  [lindex $pts 1]
         foreach k $t1 {
            set pts [$c coords $k]
            set ll  [llength $pts]
            set t3 ""
            for {set i 0} {$i<$ll} {incr i} {
               set x1 [lindex $pts $i]
               set y1 [lindex $pts [incr i]]
               append t3 [_xShear $a $xc $yc $x1 $y1] " "
            }
            eval $c coords $k $t3
         }
         $c delete trace
         _boundingBox $c $obj
      }
      {y}  {
         set a [expr sin($gvar(orotate)*0.017453293)]
         set pts [$c bbox $obj]
         set xc  [lindex $pts 0]
         set yc  [lindex $pts 1]
         foreach k $t1 {
            set pts [$c coords $k]
            set ll  [llength $pts]
            set t3 ""
            for {set i 0} {$i<$ll} {incr i} {
               set x1 [lindex $pts $i]
               set y1 [lindex $pts [incr i]]
               append t3 [_yShear $a $xc $yc $x1 $y1] " "
            }
            eval $c coords $k $t3
         }
         $c delete trace
         _boundingBox $c $obj
      }
      {xy} {
         set t5 [time {_setRotate $c $obj} 1]
#        puts stderr "TIME: $t5"
      }
   }
}

proc _drawObj {c x y} {
   global gvar
   set gvar(pcoords) ""
   set x1 $gvar(cx)
   set y1 $gvar(cy)
   set t1 [llength $gvar(coords)]
   case $gvar(obj) in {
      {line} {
         case $gvar(olinetype) in {
         {lorth rorth} {
                 set gvar(tracemode) $gvar(olinetype)
                 case $t1 in {
                 {0} {
                      set gvar(coords) "$x1 $y1 $x1 $y1"
                      eval $c create line $gvar(coords) -tags "trace" -width $gvar(width) -fill $gvar(dcolor)
                     }
                 {4} {
                      append gvar(coords) " $x1 $y1"
                      _mkObj "$c create line $gvar(coords) -tags \"obj\" \
                         -arrow $gvar(oarrowtype) -capstyle $gvar(capstyle) \
                         -joinstyle $gvar(joinstyle) -width $gvar(width) \
                         -fill $gvar(dcolor) \
                         -stipple $gvar(linepat) -arrowshape [list $gvar(arrowshape)]"
                     }
                 }
                }
         {2pts} {
                 case $t1 in {
                 {0} {
#                     puts stderr "LINE: $x1 $gvar(cx) $y1 $gvar(cy)"
                      set gvar(coords) "$x1 $y1"
                      $c create line $x1 $y1 $x1 $y1 -tags "trace" -width $gvar(width) -fill $gvar(dcolor)
                     }
                 {2} {
                      append gvar(coords) " $x1 $y1"
                      _mkObj "$c create line $gvar(coords) -tags \"obj\" \
                         -arrow $gvar(oarrowtype) -capstyle $gvar(capstyle) \
                         -joinstyle $gvar(joinstyle) -width $gvar(width) \
                         -fill $gvar(dcolor) \
                         -stipple $gvar(linepat) -arrowshape [list $gvar(arrowshape)]"
                     }
                 }
                }
         {4pts} {
               case $t1 in {
                  {0} {
                     append gvar(coords) "$x1 $y1 "
                     $c create line $x1 $y1 $x1 $y1 -tags "trace" -fill $gvar(dcolor)
                  }
                  {2 4} {
                     $c addtag trace2 withtag trace
                     $c dtag trace
                     append gvar(coords) "$x1 $y1 "
                     $c create line $x1 $y1 $x1 $y1 -tags "trace" -fill $gvar(dcolor)
                  }   
                  {6} {
                     append gvar(coords) "$x1 $y1 "
                     set gvar(bzctrl) $gvar(coords)
                     _computeBezier 16
                     set bpts ""
                     foreach i $gvar(bzpts) {
                        append bpts "$i "
                     }
                     _mkObj "$c create line $bpts -tags \"obj bezier\" \
                         -arrow $gvar(oarrowtype) -capstyle $gvar(capstyle) \
                         -joinstyle $gvar(joinstyle) -width $gvar(width) \
                         -fill $gvar(dcolor) \
                         -stipple $gvar(linepat) -arrowshape [list $gvar(arrowshape)]"
#                    _drawNorm $c
                  }
               }
            }
         {npts} {
#                area = (1/2)*ABS((x0+x1)*(y1-y0) + ... + (xn+x0)*(y0-yn))
                 set ll [llength $gvar(coords)]
                 incr ll -1
                 set oy [lindex $gvar(coords) $ll]
                 incr ll -1
                 set ox [lindex $gvar(coords) $ll]
#                puts stderr "$ox $x1   $oy $y1"
                 if {$ox==$x1 && $oy==$y1} {
                    if {$gvar(pat)==0} {
                       _mkObj "$c create line $gvar(coords) -tags \"obj\" \
                          -arrowshape [list $gvar(arrowshape)] -arrow $gvar(oarrowtype) \
                          -joinstyle $gvar(joinstyle) -capstyle $gvar(capstyle) \
                          -fill $gvar(dcolor) \
                          -stipple $gvar(linepat) -width $gvar(width)"
                    } else {
                       _mkObj "$c create polygon $gvar(coords) -tags \"obj\" \
                          -fill $gvar(fill) \
                          -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad"
                    }
#                   $c delete trace2
                 } else {
                    $c addtag trace2 withtag trace
                    $c dtag trace
                    append gvar(coords) "$x1 $y1 "
                    $c create line $x1 $y1 $x1 $y1 -tags "trace" -fill $gvar(dcolor)
                 }
                }
         }
      }
      {oval} {
         case $gvar(oovaltype) in {
         {diameter} {
                 case $t1 in {
                 {0} {
                      append gvar(coords) "$x1 $y1 "
                      $c create line $x1 $y1 $x1 $y1 -tags "trace" -width $gvar(width) -fill $gvar(dcolor)
                     }
                 {2} {
                      set x2 [lindex $gvar(coords) 0]
                      set y2 [lindex $gvar(coords) 1]
                      set xt [expr $x1-$x2]
                      set yt [expr $y1-$y2]
                      set r  [expr sqrt($xt*$xt+$yt*$yt)*0.5]
                      set xt [expr ($x1+$x2)*0.5]
                      set yt [expr ($y1+$y2)*0.5]
                      set x1 [expr $xt-$r]
                      set y1 [expr $yt-$r]
                      set x2 [expr $xt+$r]
                      set y2 [expr $yt+$r]
                      _mkObj "$c create oval $x1 $y1 $x2 $y2 -tags \"obj\" \
                         -width $gvar(width) -fill $gvar(fill) \
                         -outline $gvar(dcolor) \
                         -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad"
                     }
                 }
                }
         {radius} {
                 case $t1 in {
                 {0} {
                      $c delete trace
                      append gvar(coords) "$x1 $y1 "
                      $c create line $x1 $y1 $x1 $y1 -tags "trace" -width $gvar(width) -fill $gvar(dcolor)
                     }
                 {2} {
                      set x2 [lindex $gvar(coords) 0]
                      set y2 [lindex $gvar(coords) 1]
                      set dx [expr $x1-$x2]
                      set dy [expr $y1-$y2]
                      set r [expr sqrt($dx*$dx+$dy*$dy)]
                      set x1 [expr $x2-$r]
                      set y1 [expr $y2-$r]
                      set x2 [expr $x2+$r]
                      set y2 [expr $y2+$r]
                      _mkObj "$c create oval $x1 $y1 $x2 $y2 -tags \"obj\" \
                         -width $gvar(width) -fill $gvar(fill) \
                         -outline $gvar(dcolor) \
                         -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad"
                     }
                 }
                }
         {3pts} {
                 set gvar(tracemode) point
                 case $t1 in {
                 {0} {
                      $c delete trace
                      append gvar(coords) "$x1 $y1 "
                      $c create bitmap $x1 $y1 -tags "trace" -foreground $gvar(icolor) \
                         -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
                     }
                 {2} {
                      append gvar(coords) "$x1 $y1 "
                      $c create bitmap $x1 $y1 -tags "trace" -foreground $gvar(icolor) \
                         -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
                     }
                 {4} { 
                      append gvar(coords) "$x1 $y1 "
                      _draw3ptsCircle $c
                     }
                 }
                }
         {2pts} {
                 case $t1 in {
                 {0} {
                      append gvar(coords) "$x1 $y1 "
                      $c create rectangle $x1 $y1 $x1 $y1 -tags "trace" -width $gvar(width) -outline $gvar(dcolor)
                     }
                 {2} {
                      set x2 [lindex $gvar(coords) 0]
                      set y2 [lindex $gvar(coords) 1]
                      _mkObj "$c create oval $x2 $y2 $x1 $y1 -tags \"obj\" \
                         -width $gvar(width) -fill $gvar(fill) \
                         -outline $gvar(dcolor) \
                         -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad"
                     }
                 }
                }
         }
      }
      {polygon} {
         case $t1 in {
            {0} {
               append gvar(coords) "$x1 $y1 "
               $c create line $x1 $y1 $x1 $y1 -tags "trace" -width $gvar(width) -fill $gvar(dcolor)
            }
            {2} {
               set cx [lindex $gvar(coords) 0]
               set cy [lindex $gvar(coords) 1]
               set t3 "$x1 $y1 "
               set rpd 0.017453293
               set a [expr 360.0/$gvar(pside)]
               set a [expr {-1*tan($a*$rpd*0.5)}]
               set b [expr {-2*$a/(1+$a*$a)}]
               for {set i 0} {$i < $gvar(pside)} {incr i} {
                  set t2 [_rotAngle $a $b $cx $cy $x1 $y1]
                  set x1 [lindex $t2 0]
                  set y1 [lindex $t2 1]
                  append t3 "$x1 $y1 "
               }
               _mkObj "$c create line $t3 -tags \"obj\" \
                 -arrow $gvar(oarrowtype) -capstyle $gvar(capstyle) \
                 -joinstyle $gvar(joinstyle) -width $gvar(width) \
                 -fill $gvar(dcolor) \
                 -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad \
                 -arrowshape [list $gvar(arrowshape)]"
            }
         }
      }
      {rect} {
         case $gvar(orecttype) in {
         {2pts} {
                 case $t1 in {
                 {0} {
                      append gvar(coords) "$x1 $y1 "
                      $c create rectangle $x1 $y1 $x1 $y1 -tags "trace" -width $gvar(width) -outline $gvar(dcolor)
                     }
                 {2} {
                      set x2 [lindex $gvar(coords) 0]
                      set y2 [lindex $gvar(coords) 1]
                      set t2 "$x2 $y2 $x1 $y2 $x1 $y1 $x2 $y1 $x2 $y2"
                      _mkObj "$c create line $t2 -tags \"obj\" \
                         -fill $gvar(dcolor) \
                         -width $gvar(width) -stipple $gvar(linepat) \
                         -joinstyle $gvar(joinstyle) -capstyle $gvar(capstyle)"
                     }
                 }
                }
         {2ptsfill} {
                 case $t1 in {
                 {0} {
                      append gvar(coords) "$x1 $y1 "
                      $c create rectangle $x1 $y1 $x1 $y1 -tags "trace" -width $gvar(width) -outline $gvar(dcolor)
                     }
                 {2} {
                      set x2 [lindex $gvar(coords) 0]
                      set y2 [lindex $gvar(coords) 1]
                      set t2 "$x2 $y2 $x1 $y2 $x1 $y1 $x2 $y1 $x2 $y2"
                      _mkObj "$c create polygon $t2 -tags \"obj\" \
                         -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad -fill $gvar(fill)"
                     }
                 }
                }
         {2ptsr} {
                 case $t1 in {
                 {0} {
                      append gvar(coords) "$x1 $y1 "
                      $c create rectangle $x1 $y1 $x1 $y1 -tags "trace" -width $gvar(width) -outline $gvar(dcolor)
                     }
                 {2} {
                      set x2 [lindex $gvar(coords) 0]
                      set y2 [lindex $gvar(coords) 1]
                      set t2 "$x2 $y2 "
                      append t2 "[expr {($x1+$x2)*0.5}] $y2 "
                      append t2 "$x1 $y2 "
                      append t2 "$x1 [expr {($y2+$y1)*0.5}] "
                      append t2 "$x1 $y1 "
                      append t2 "[expr {($x1+$x2)*0.5}] $y1 "
                      append t2 "$x2 $y1 "
                      append t2 "$x2 [expr {($y2+$y1)*0.5}] "
                      append t2 "$x2 $y2 "
                      _mkObj "$c create line $t2 -tags \"obj\" -smooth 1 \
                         -fill $gvar(dcolor) \
                         -width $gvar(width) -stipple $gvar(linepat) \
                         -joinstyle $gvar(joinstyle) -capstyle $gvar(capstyle)"
                     }
               }
              }
         {2ptsrfill} {
                 case $t1 in {
                 {0} {
                      append gvar(coords) "$x1 $y1 "
                      $c create rectangle $x1 $y1 $x1 $y1 -tags "trace" -width $gvar(width) -outline $gvar(dcolor)
                     }
                 {2} {
                      set x2 [lindex $gvar(coords) 0]
                      set y2 [lindex $gvar(coords) 1]
                      set t2 "$x2 $y2 "
                      append t2 "[expr {($x1+$x2)*0.5}] $y2 "
                      append t2 "$x1 $y2 "
                      append t2 "$x1 [expr {($y2+$y1)*0.5}] "
                      append t2 "$x1 $y1 "
                      append t2 "[expr {($x1+$x2)*0.5}] $y1 "
                      append t2 "$x2 $y1 "
                      append t2 "$x2 [expr {($y2+$y1)*0.5}] "
                      append t2 "$x2 $y2 "
                      _mkObj "$c create polygon $t2 -tags \"obj\" -smooth 1 \
                         -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad -fill $gvar(fill)"
                     }
               }
              }
         }
      }
      {text} {
         case $gvar(otexttype) in {
         {bitmap} {
                   $c dtag text
                   if {[$c type current]=="text"} {
                      $c icursor current @$x1,$y1
                      $c focus current
                      $c select from current @$x1,$y1
                      $c addtag text withtag current
                   } else {
                      $c delete trace
                      $c create bitmap $x1 $y1 -bitmap @$gvar(ICONDIR)/xhair.icon -anchor center \
                         -tags "trace" -foreground $gvar(icolor)
                      _mkObj "$c create text $x1 $y1 -tags \"obj text\" \
                         -fill $gvar(dcolor) \
                         -anchor $gvar(tfontanchor) -font $gvar(tfont) \
                         -justify $gvar(tfontjust)"
                      $c focus text
                   }
                  }
         {vector} {
                   set gvar(coords) "$x1 $y1"
                   $c delete trace
                   $c create bitmap $x1 $y1 -tags "trace" -foreground $gvar(icolor) \
                      -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
                  }
         }
      }
      {bitmap} {
         set gvar(coords) "$x1 $y1"
         $c delete trace
         $c create bitmap $x1 $y1 -tags "trace" -foreground $gvar(dcolor) \
            -bitmap @$gvar(ICONDIR)/xhair.icon -anchor center
      }
      {arc} {
         case $gvar(oarctype) in {
         {3pts} {
            set gvar(tracemode) point
            case $t1 in {
            {0} {
                 $c delete trace
                 append gvar(coords) "$x1 $y1 "
                 $c create bitmap $x1 $y1 -tags "trace" -foreground $gvar(icolor) \
                     -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
                }
            {2} {
                 append gvar(coords) "$x1 $y1 "
                 $c create bitmap $x1 $y1 -tags "trace" -foreground $gvar(icolor) \
                     -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
                }
            {4} { 
                 append gvar(coords) "$x1 $y1 "
                 _draw3ptsArc $c
                }
            }
         }
         {2pts} {
            set gvar(tracemode) line
            case $t1 in {
            {0} {
                 append gvar(coords) "$x1 $y1 "
                 $c create line $x1 $y1 $x1 $y1 -tags "trace" -fill $gvar(dcolor)
                }
            {2} {
                 append gvar(coords) "$x1 $y1 "
                 $c addtag trace2 withtag trace
                 $c dtag trace
                 set x2 [lindex $gvar(coords) 0]
                 set y2 [lindex $gvar(coords) 1]
                 $c create line $x2 $y2 $x2 $y2 -tags "trace" -fill $gvar(dcolor)
                 set x2 [lindex $gvar(coords) 2]
                 set y2 [lindex $gvar(coords) 3]
                 $c create line $x2 $y2 $x2 $y2 -tags "trace" -fill $gvar(dcolor)
                }
            {4} {
                 append gvar(coords) "$x1 $y1 "
                 _draw3ptsOval $c
                }
            }
         }
         {4pts} {
            case $t1 in {
            {0} {
                 append gvar(coords) "$x1 $y1 "
                 $c create rectangle $x1 $y1 $x1 $y1 -tags "trace" -outline $gvar(dcolor)
                }
            {2} {
                 set x2 [lindex $gvar(coords) 0]
                 set y2 [lindex $gvar(coords) 1]
                 append gvar(coords) "$x1 $y1 "
                 set cx [expr {($x1+$x2)*0.5}]
                 set cy [expr {($y1+$y2)*0.5}]
                 append gvar(coords) "$cx $cy "
                 eval "$c create oval [$c coords trace] -tags \"trace2\"" -outline $gvar(dcolor)
                 $c delete trace
                 $c create line $cx $cy $cx $cy -tags "trace" -width $gvar(width) -fill $gvar(dcolor)
                 set gvar(tracemode) line
                }
            {6} {
                 set x1 [lindex [$c coords trace] 2]
                 set y1 [lindex [$c coords trace] 3]
                 append gvar(coords) "$x1 $y1 "
                 eval "$c create line [$c coords trace] -tags \"trace2\" -fill $gvar(dcolor)"
                 set gvar(tracemode) line
                }
            {8} {
                 set x1 [lindex [$c coords trace] 2]
                 set y1 [lindex [$c coords trace] 3]
                 append gvar(coords) "$x1 $y1 "
                 # (x1,y1) (x2,y2) (cx,cy) (ax1,ay1) (ax2,ay2)
                 set cx [lindex $gvar(coords) 4]
                 set cy [lindex $gvar(coords) 5]
                 set x1 [lindex $gvar(coords) 6]
                 set y1 [lindex $gvar(coords) 7]
                 set x2 [lindex $gvar(coords) 8]
                 set y2 [lindex $gvar(coords) 9]
#                puts stderr "$cx $cy $x1 $y1 $x2 $y2"
                 set bx1 [lrange $gvar(coords) 0 1]
                 set bx2 [lrange $gvar(coords) 2 3]
                 set dx [expr [lindex $bx2 0]-[lindex $bx1 0]]
                 set dy [expr [lindex $bx2 1]-[lindex $bx1 1]]
                 set ecc [expr 1.0*$dx/$dy]
                 set dx [expr $x1-$cx]
                 set dy [expr $cy-$y1]
                 set a1 [_getAngle $dx [expr $dy*$ecc]]
                 set dx [expr $x2-$cx]
                 set dy [expr $cy-$y2]
                 set a2 [_getAngle $dx [expr $dy*$ecc]]
                 set a2 [expr $a2-$a1]
#                puts stderr "$a1 $a2"
                 _mkObj "$c create arc $bx1 $bx2 -start $a1 -extent $a2 \
                    -outline $gvar(dcolor) \
                    -tags \"obj\" -style $gvar(oarcstyle) \
                    -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad -fill $gvar(fill)"
                }
            }    
         }
         }
      }

   }
   set gvar(lx) $x
   set gvar(ly) $y
}

proc _enterPick {c obj} {
   global gvar
   # top_left_corner       top_side       top_right_corner
   # left_side             cross          right_side
   # bottom_left_corner    bottom_side    bottom_right_corner
   set pts [$c coords $obj]
#  puts stderr "PICK: $pts"
   set x1  [lindex $pts 0]
   set y1  [lindex $pts 1]
   set x2  [lindex $pts 2]
   set y2  [lindex $pts 3]
   set xm  [expr ($x1+$x2)*0.5]
   set ym  [expr ($y1+$y2)*0.5]
   set pts "$x1 $y1 $xm $y1 $x2 $y1 $x2 $ym $x2 $y2 $xm $y2 $x1 $y2 $x1 $ym"
   set t1 [_closestPoint $c $pts $gvar(cx) $gvar(cy)]
   set t2 [lindex $t1 0]
   set t1 [lindex $t1 1]
   if {$t1>500} {
      $c config -cursor cross
      set gvar(obj) cursor
      return
   }
   case $t2 in {
      {0}   { $c config -cursor top_left_corner
              set gvar(obj) edit_nw }
      {2}   { $c config -cursor top_side
              set gvar(obj) edit_n }
      {4}   { $c config -cursor top_right_corner
              set gvar(obj) edit_ne }
      {6}   { $c config -cursor right_side
              set gvar(obj) edit_e }
      {8}   { $c config -cursor bottom_right_corner
              set gvar(obj) edit_se }
      {10}  { $c config -cursor bottom_side
              set gvar(obj) edit_s }
      {12}  { $c config -cursor bottom_left_corner
              set gvar(obj) edit_sw }
      {14}  { $c config -cursor left_side
              set gvar(obj) edit_w }
   }
}
proc _leavePick {c obj} {
   global gvar
   set gvar(obj) cursor
   $c config -cursor left_ptr
}

proc _ptsFromLine {xa ya xb yb} {
#  if (xb-xa)*(yb-ya)-(xp-xa)*(yb-ya)=positive means left
#                                     negative means right
}

proc _breakLines {c obj} {
   global gvar
   foreach k [$c find withtag $obj] {
      set pts [$c coords $k]
      for {set i 0} {$i<[expr [llength $pts]-3]} {incr i 2} {
         set j  $i
         set t1 [lrange $pts $j [incr j 3]]
         _mkObj "$c create line $t1 -tags \"obj\" \
            -arrow $gvar(oarrowtype) -capstyle $gvar(capstyle) \
            -joinstyle $gvar(joinstyle) -width $gvar(width) \
            -fill $gvar(dcolor) \
            -stipple $gvar(linepat) -arrowshape [list $gvar(arrowshape)]"
      }
      $c delete $k
   }
}

proc _getLineCircle {c obj} {
   set rc {}
   set kc ""
   set kl ""
   foreach i $obj {
      case [$c type $i] in {
      {oval} {
            set pts [$c coords $i]
            set x1 [lindex $pts 0]
            set y1 [lindex $pts 1]
            set x2 [lindex $pts 2]
            set y2 [lindex $pts 3]
            set dx [expr $x1-$x2]
            set dy [expr $y1-$y2]
            if {[expr abs($dx-$dy)]>5} {continue}
            set xc [expr ($x1+$x2)*0.5]
            set yc [expr ($y1+$y2)*0.5]
            set r  [expr abs($dx*0.5)]
            set kc "$xc $yc $r"
         }
      {line} {
            set kl [$c coords $i]
         }
      }
   }
   if {$kl==""} {
      tk_dialog .dm "dialog" "Need a line" warning 0 "Ok"
      return
   }
   if {$kc==""} {
      tk_dialog .dm "dialog" "Need a circle" warning 0 "Ok"
      return
   }
#  puts stderr "$kl $kc"
   return "$kl $kc"
}
proc _get2Circles {c obj} {
   set rc {}
   set k  0
   foreach i $obj {
      if {[$c type $i]=="oval"} {
         set pts [$c coords $i]
         set x1 [lindex $pts 0]
         set y1 [lindex $pts 1]
         set x2 [lindex $pts 2]
         set y2 [lindex $pts 3]
         set dx [expr $x1-$x2]
         set dy [expr $y1-$y2]
         if {[expr abs($dx-$dy)]>5} {continue}
         set xc [expr ($x1+$x2)*0.5]
         set yc [expr ($y1+$y2)*0.5]
         set r  [expr abs($dx*0.5)]
         set l($k) $i
         lappend rc [list $xc $yc $r]
         incr k
         if {$k>1} {
            return "$l(0) $l(1) $rc"
         }
      }
   }
   tk_dialog .dm "dialog" "Need 2 circles; found $k" warning 0 "Ok"
   return ""
}
proc _get2Lines {c obj} {
   set k 0
   foreach i $obj {
      if {[$c type $i]=="line"} {
         set l($k) $i
         incr k
      }
      if {$k>1} {break}
   }
   if {$k<2} {
      tk_dialog .dm "dialog" "Need 2 lines; found $k" warning 0 "Ok"
      return ""
   }
   set ab    [$c coords $l(0)]
   set cd    [$c coords $l(1)]
   if {[llength $ab]!=4 || [llength $cd]!=4} {
      tk_dialog .dm "dialog" "Must be 2pts lines" warning 0 "Ok"
      return ""
   }
   return "$l(0) $l(1) [list $ab] [list $cd]"
}

proc _perpLine {ab cd} {
   set x1 [lindex $ab 0]
   set y1 [lindex $ab 1]
   set x2 [lindex $ab 2]
   set y2 [lindex $ab 3]
   set x3 [lindex $cd 0]
   set y3 [lindex $cd 1]
   set x4 [lindex $cd 2]
   set y4 [lindex $cd 3]
   set m1 [expr 1.0*($y1-$y2)/($x2-$x1)]
   set m2 [expr -1.0/$m1]
   set b1 [expr 1.0*$y1/($m1*$x1)]
   set b2 [expr 1.0*$y3/($m2*$x3)]
   set xi [expr 1.0*($b2-$b1)/($m1-$m2)]
   set yi [expr $m1*$xi+$b1]
#  puts stderr "PERP: ($x1,$y1-$x2,$y2) ($x3,$y3-$x4,$y4) $xi $yi"
   return "$xi $yi"
}

proc _intersectCirclesX {x1 y1 r1 x2 y2 r2 xp yp} {
   global gvar
   set eps   0.00001
   set limit 100
   set t 0
   set xt $xp
   set yt $yp
   set r1s [expr $r1*$r1]
   set r2s [expr $r2*$r2]
   while {$t<$limit} {
      set xt1  [expr $xt+$x1]
      set yt1  [expr $yt+$y1]
      set xt2  [expr $xt+$x2]
      set yt2  [expr $yt+$y2]
      set fnf  [expr $r1s-$xt1*$xt1-$yt1*$yt1]
      set fng  [expr $r2s-$xt2*$xt2-$yt2*$yt2]
      set fnfx [expr -2*$xt1]
      set fnfy [expr -2*$yt1]
      set fngx [expr -2*$xt2]
      set fngy [expr -2*$yt2]
      set d [expr $fnfx*$fngy - $fnfy*$fngx]
      set x [expr $xt+($fnf*$fngy+$fnfy*$fng)/$d]
      set y [expr $yt+($fnfx*$fng+$fnf*$fngx)/$d]
      if {[expr abs($x-$xt)]<$eps && [expr abs($y-$yt)]<$eps} { 
#        puts stderr "Isect CIRCLE: XI=$x YI=$y"
         .c.c create bitmap $x $y -tags "trace" -foreground $gvar(icolor) \
            -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
         return
      }
      incr t
      set xt $x
      set yt $y
   }
#  puts stderr "Isect CIRCLE: no solution"
}

proc _intersectCircles {c1x c1y c1r c2x c2y c2r} {
   set dx [expr $c1x-$c2x]
   set dy [expr $c2y-$c1y]
   set r  [_getAngle $dx $dy]
   set ra [expr $r-90]
   set a  [expr -tan($ra*0.0087266463)]
   set b  [expr -2*$a/(1+$a*$a)]
   set t3 [_rotAngle $a $b $c2x $c2y $c1x $c1y]
   set xc [lindex $t3 0]
   set yc [lindex $t3 1]
   set x1 [expr $xc-$c1r]
   set y1 [expr $yc-$c1r]
   set x2 [expr $xc+$c1r]
   set y2 [expr $yc+$c1r]
#  _mkObj "$c create oval $x1 $y1 $x2 $y2 -tags \"obj\" \
#     -width $gvar(width) -fill $gvar(fill) \
#     -outline $gvar(dcolor) \
#     -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad"
   set dx [expr $xc-$c2x]
   set dy [expr $yc-$c2y]
#  puts stderr "dx=$dx $c2x - dy=$dy $c2y"
   set s1 [expr ($c2r*$c2r-$dy*$dy-$c1r*$c1r)/(-2*$c1r*$dy)]
#  puts stderr "SIN=$s1"
   # |s1| > circle does not intersect
   # s1>1 indicates circle inside; s1<-1 indicates circle outside
   if {[expr abs($s1)]<=1} {
      set s1 [expr asin($s1)]
#     puts stderr "2C: $c2x $c2y - $xc $yc - A=[expr $s1*57.29578]"
      set xa [expr $c2x+$c1r*cos($s1)]
      set xb [expr $c2x-$c1r*cos($s1)]
      set y1 [expr $c2y+$dy-$c1r*sin($s1)]
      set a  [expr -tan(-$ra*0.0087266463)]
      set b  [expr -2*$a/(1+$a*$a)]
      set t3 [_rotAngle $a $b $c2x $c2y $xa $y1]
      set xa [lindex $t3 0]
      set ya [lindex $t3 1]
      set t3 [_rotAngle $a $b $c2x $c2y $xb $y1]
      set xb [lindex $t3 0]
      set yb [lindex $t3 1]
      return "$xa $ya $xb $yb"
   }
   return ""
}

proc _intersectLineCircle {x1 y1 x2 y2 xc yc r} {
   global gvar
   set i   [expr $x2-$x1]        
   set j   [expr $y2-$y1]
   set a   [expr $i*$i+$j*$j]
   set b   [expr 2*$i*($x1-$xc)+2*$j*($y1-$yc)]
   set c   [expr $xc*$xc+$yc*$yc+$x1*$x1+$y1*$y1+2*(-$xc*$x1-$yc*$y1)-$r*$r]
   set d   [expr $b*$b-4*$a*$c]
   if {$d<0} {
      return "" 
   }
   if {$d==0} {
      set t1  [expr -$b*0.5/$a]
      set x1i [expr $x1+($x2-$x1)*$t1]
      set y1i [expr $y1+($y2-$y1)*$t1]
      return "$t1 $x1i $y1i"
   }
   set d   [expr sqrt($d)]
   set t1  [expr (-$b+$d)*0.5/$a]
   set t2  [expr (-$b-$d)*0.5/$a]
   set x1i [expr $x1+($x2-$x1)*$t1]
   set y1i [expr $y1+($y2-$y1)*$t1]
   set x2i [expr $x1+($x2-$x1)*$t2]
   set y2i [expr $y1+($y2-$y1)*$t2]
   return "$t1 $t2 $x1i $y1i $x2i $y2i"
}

proc _intersectLines {ab cd} {
   global gvar

#  AB=A+r(B-A)   0<=r<=1
#  CD=C+s(D-C)   0<=s<=1
#  0<=r<=1 && 0<=s<=1         intersection exists
#  r<0 || r>1 || s<0 || s>1   no intersection
#  denom=0 in r               lines are parallel
#  numer=0 in r               lines are coincident
#  r>1                        extend AB
#  r<0                        extend BA
#  s>1                        extend CD
#  s<0                        extend DC

   set xa [lindex $ab 0]
   set ya [lindex $ab 1]
   set xb [lindex $ab 2]
   set yb [lindex $ab 3]
   set xc [lindex $cd 0]
   set yc [lindex $cd 1]
   set xd [lindex $cd 2]
   set yd [lindex $cd 3]

   set denom [expr 1.0*($xb-$xa)*($yd-$yc)-($yb-$ya)*($xd-$xc)]
   if {$denom==0} {
      tk_dialog .dm "dialog" "Intersect: lines are parallel" warning 0 "Ok"
      return
   }
   set r     [expr (($ya-$yc)*($xd-$xc)-($xa-$xc)*($yd-$yc))/$denom]
   set s     [expr (($ya-$yc)*($xb-$xa)-($xa-$xc)*($yb-$ya))/$denom]
   set xi    [expr $xa+$r*($xb-$xa)]
   set yi    [expr $ya+$r*($yb-$ya)]
   if {$r>0.5} {
      set ab [lreplace $ab 2 3 $xi $yi]
   } else {
      set ab [lreplace $ab 0 1 $xi $yi]
   }
   if {$s>0.5} {
      set cd [lreplace $cd 2 3 $xi $yi]
   } else {
      set cd [lreplace $cd 0 1 $xi $yi]
   }

#  puts stderr "XI=$xi YI=$yi r=$r s=$s ab=$ab cd=$cd"
#  eval $c create line $ab -tags "traceit" -width 5
#  eval $c create line $cd -tags "traceit" -width 5
#  $c create bitmap $xi $yi -tags "traceit" -foreground $gvar(icolor) \
#     -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
#  update
#  after 500
#  $c delete traceit
   return "[list $ab] [list $cd]"
}

proc _interSect {c obj} {
   global gvar
   case $gvar(olinestype) in {
   {extend} {
         set t1 [_get2Lines $c [$c find withtag $obj]]
         if {$t1==""} { return }
         set t2 [_intersectLines [lindex $t1 2] [lindex $t1 3]]
         if {$t2==""} { return }
#        puts stderr "InterSECT: $t1 - $t2"
         eval $c coords [lindex $t1 0] [lindex $t2 0]
         eval $c coords [lindex $t1 1] [lindex $t2 1]
      } 
   {perp} {
         set t1 [_get2Lines $c [$c find withtag $obj]]
         if {$t1==""} { return }
         set t2 [_perpLine [lindex $t1 2] [lindex $t1 3]]
         $c create bitmap [lindex $t2 0] [lindex $t2 1] -tags "trace" -fg $gvar(icolor) -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
      }
   {chamfer} {
         set t1 [_getLineCircle $c [$c find withtag $obj]]
         if {$t1==""} { return }
         set x1 [lindex $t1 0]
         set y1 [lindex $t1 1]
         set x2 [lindex $t1 2]
         set y2 [lindex $t1 3]
         set xc [lindex $t1 4]
         set yc [lindex $t1 5]
         set r  [lindex $t1 6]
         set t2 [eval _intersectLineCircle $t1]
         if {$t2==""} { return }
         if {[llength $t2]>3} {
            eval $c create bitmap [lrange $t2 2 3] -tags "trace" -fg $gvar(icolor) -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
            eval $c create bitmap [lrange $t2 4 5] -tags "trace" -fg $gvar(icolor) -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
         } else {
            eval $c create bitmap [lrange $t2 1 2] -tags "trace" -fg $gvar(icolor) -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
         }
#        puts stderr "LINECIRC: $t2"
      }
   {2circ} {
         set t1 [_get2Circles $c [$c find withtag $obj]]
         if {$t1==""} { return }
         set c1 [lindex $t1 2]
         set c2 [lindex $t1 3]
         set c1x [lindex $c1 0]
         set c1y [lindex $c1 1]
         set c1r [lindex $c1 2]
         set c2x [lindex $c2 0]
         set c2y [lindex $c2 1]
         set c2r [lindex $c2 2]
         set t2 [_intersectCircles $c1x $c1y $c1r $c2x $c2y $c2r]
         if {$t2==""} { return }
         $c create bitmap [lindex $t2 0] [lindex $t2 1] -tags "trace" -fg $gvar(icolor) -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
         $c create bitmap [lindex $t2 2] [lindex $t2 3] -tags "trace" -fg $gvar(icolor) -anchor center -bitmap @$gvar(ICONDIR)/xhair.icon
      }
   {fillet} {
         set t1 [_get2Lines $c [$c find withtag $obj]]
         if {$t1==""} { return }
         set t2 [_fillet2Lines [lindex $t1 2] [lindex $t1 3] 20]
         if {$t2==""} { return }
         set xc [lindex $t2 0]
         set yc [lindex $t2 1]
         set sa [expr [lindex $t2 2]*57.29578]
         set da [expr [lindex $t2 3]*57.29578]
         set x1 [expr $xc-20]
         set y1 [expr $yc-20]
         set x2 [expr $xc+20]
         set y2 [expr $yc+20]
#        puts stderr "FILLET: $t2 - $xc $yc $sa $da"
         _mkObj "$c create arc $x1 $y1 $x2 $y2 \
            -start $sa -extent $da -fill $gvar(dcolor) \
            -style arc -tags \"obj\""
         eval $c coords [lindex $t1 0] [lindex $t2 4]
         eval $c coords [lindex $t1 1] [lindex $t2 5]
      }
   }
}

proc _labelObj {c obj} {
   global gvar
   foreach k [$c find withtag $obj] {
      set pts [$c coords $k]
      case [$c type $k] in {
      {oval} {
         set x [expr ([lindex $pts 0]+[lindex $pts 2])*0.5]
         set y [expr ([lindex $pts 1]+[lindex $pts 3])*0.5]
         set k [$c create bitmap $x $y -tags "trace" \
            -foreground $gvar(dcolor) -anchor center \
            -bitmap @$gvar(ICONDIR)/pickbox.cad]
         $c lower $k $obj
         }
      {arc} {
         set t1 {}
         set x1 [lindex $pts 0]
         set y1 [lindex $pts 1]
         set x2 [lindex $pts 2]
         set y2 [lindex $pts 3]
         set xc [expr ($x1+$x2)*0.5]
         set yc [expr ($y1+$y2)*0.5]
         lappend t1 [list $xc $yc]
         set rx  [expr ($x2-$x1)*0.5]
         set ry  [expr ($y2-$y1)*0.5]
         set sa [expr [lindex [$c itemconf $k -start] 4]*0.017453293]
         set ea [expr $sa+[lindex [$c itemconf $k -extent] 4]*0.017453293]
#        puts stderr "ARC: $rx $ry $sa $ea"
         lappend t1 [list [expr $xc+$rx*cos($sa)] [expr $yc-$ry*sin($sa)]]
         lappend t1 [list [expr $xc+$rx*cos($ea)] [expr $yc-$ry*sin($ea)]]
         foreach pts $t1 {
            set k [eval $c create bitmap $pts \
               -tags "trace" -foreground $gvar(dcolor) -anchor center \
               -bitmap @$gvar(ICONDIR)/pickbox.cad]
            $c lower $k $obj
         }
         }
      default {
         for {set i 0; set j 1} {$i<[llength $pts]} {incr i 2; incr j 2} {
            set x [lindex $pts $i]
            set y [lindex $pts $j]
            set k [$c create bitmap $x $y -tags "trace" \
               -foreground $gvar(dcolor) -anchor center \
               -bitmap @$gvar(ICONDIR)/pickbox.cad]
            $c lower $k $obj
         }
         }
      }
   }
}

proc _boundingBox {c obj} {
   global gvar
   set gvar(ecoords) [$c bbox $obj]
   $c delete trace
   if {$gvar(ecoords)!=""} {
#     scan $t1 "%f %f %f %f" x1 y1 x2 y2
#     set xm [expr $x1+($x2-$x1)*0.5]
#     set ym [expr $y1+($y2-$y1)*0.5]
#     set pts(all) "$x1 $y1 $xm $y1 $x2 $y1 $x2 $ym $x2 $y2 $xm $y2 $x1 $y2 $x1 $ym $x1 $y1"
#     set i [eval $c create line $pts(all) -tags "trace" -fill $gvar(dcolor)]
      set i [eval $c create rectangle $gvar(ecoords) -tags "trace" -outline $gvar(dcolor)]
      $c bind $i <Any-Enter> "_enterPick $c $i"
      $c bind $i <Any-Leave> "_leavePick $c $i"
   }
}

proc _pickObj {c t2} {
   global gvar
   
   set result 0
   
#  puts stderr "$t2 - withtag=[$c find withtag current]"
   set objs ""
   foreach i $t2 {
      set po [$c gettags $i]
      if {$po=="" ||
          [lsearch $po trace]>-1 ||
          [lsearch $po xhair]>-1 || 
          [lsearch $po yhair]>-1} {
         _restore $c
         return -1
      }
      set t1 [lsearch $po cgrp*]
      if {$t1>-1} {
         append objs [lindex $po $t1] " "
         continue
      }
      set t1 [lsearch $po grp*]
      if {$t1>-1} {
         continue
      }
      set t1 [lsearch $po vtext*]
      if {$t1>-1} {
         append objs "VTEXT$i "
         continue
      }
      set t1 [lsearch $po vt*]
      if {$t1>-1} {
         continue
      }
      append objs "$i "
   }
#  puts stderr "PICKED: $objs"
   foreach i $objs {
      case $i in {
      {VTEXT*} {
            scan $i "VTEXT%d" k
            set k [$c find withtag $k]
            $c addtag pick withtag $k
            $c addtag pick withtag vt$k
         }
      {cgrp*} {
            scan $i "cgrp%d" k
            $c addtag pick withtag $i
            $c addtag pick withtag grp$k
         }
      default {
            $c addtag pick withtag $i
         }
      }
   }
   _boundingBox $c pick
   return [llength $objs]
}

proc _pickArea {c} {
   global gvar
   if {[llength $gvar(pcoords)]<3} {
      set gvar(onopick) [_pickObj $c [$c find withtag current]]
      if {$gvar(onopick)<1} { 
         $c dtag pick pick
         $c delete trace trace2
#        puts stderr "UNPICK"
      }
   } else {
      set t2 [eval $c find enclosed [$c coords trace]]
#     puts stderr "PICKAREA: $t2"
      $c delete trace trace2
      if {$t2!=""} {
         set gvar(onopick) [_pickObj $c "$t2"]
      }
      set gvar(pcoords) ""
   }
}

proc _canvasClear {c} {
   $c delete obj
   _restore $c
}

proc _canvasInfo {c} {
   global gvar
   set f .infoframe
   catch {destroy $f}
   toplevel $f
   wm title $f "Canvas Info"
   wm minsize $f 200 200
   foreach i {arc bitmap line oval polygon rectangle text window vfont group} {
      set val($i) 0
   }
   foreach i [$c find withtag obj] {
      set iTag [$c gettags $i]
      if {[lsearch $iTag cgrp*]>-1} {
         incr val(group)
      } elseif {[lsearch $iTag vtext*]>-1} {
         incr val(vfont)
      } else {
         incr val([$c type $i])
      }
   }
   foreach i {arc bitmap line oval polygon rectangle text window vfont group} {
      frame $f.$i
         label $f.$i.1 -text [format "%10ss" $i] -font courr12
         label $f.$i.2 -text [format "%-10s" $val($i)] -relief sunken -bd 2
      pack $f.$i.1 $f.$i.2 -side left -fill x -expand yes -pady 3
      pack $f.$i -side top -fill x
   }
   frame $f.f2
      button $f.f2.1 -bd 5 -text "DONE" -command "destroy $f"
   pack $f.f2.1 -side left -fill x -expand yes
   pack $f.f2 -side top -fill x
}

proc _alignObj {c} {
   global gvar
   set Itag [$c find withtag pick]
   if {$Itag==""} {return}
   set k [lindex $Itag 0]
   set t1 [$c bbox $k]
   case $gvar(oaligntype) in {
   {left} {
         foreach i $Itag {
            set t2 [$c bbox $i]
            set dx [expr [lindex $t1 0]-[lindex $t2 0]]
            $c move $i $dx 0
         }
      }
   {right} {
         foreach i $Itag {
            set t2 [$c bbox $i]
            set dx [expr [lindex $t1 2]-[lindex $t2 2]]
            $c move $i $dx 0
         }
      }
   {up} {
         foreach i $Itag {
            set t2 [$c bbox $i]
            set dy [expr [lindex $t1 1]-[lindex $t2 1]]
            $c move $i 0 $dy
         }
      }
   {down} {
         foreach i $Itag {
            set t2 [$c bbox $i]
            set dy [expr [lindex $t1 3]-[lindex $t2 3]]
            $c move $i 0 $dy
         }
      }
   {xcenter} {
         set x1 [expr ([lindex $t1 0]+[lindex $t1 2])*0.5]
         foreach i $Itag {
            set t2 [$c bbox $i]
            set x2 [expr ([lindex $t2 0]+[lindex $t2 2])*0.5]
            set dx [expr $x1-$x2]
            $c move $i $dx 0
         }
      }
   {ycenter} {
         set y1 [expr ([lindex $t1 1]+[lindex $t1 3])*0.5]
         foreach i $Itag {
            set t2 [$c bbox $i]
            set y2 [expr ([lindex $t2 1]+[lindex $t2 3])*0.5]
            set dy [expr $y1-$y2]
            $c move $i 0 $dy
         }
      }
   }
   _boundingBox $c pick
}

proc _moveObj {c dx dy} {
   global gvar
   $c move pick $dx $dy
   $c move trace $dx $dy
}

proc _deleteObj {c} {
   global gvar
   set gvar(lastcmd) {}
   foreach i [$c find withtag pick] {
      lappend gvar(lastcmd) "_mkObj [_dumpObj $c $i stdout]"
   }
   $c delete pick
   _restore $c
   _leavePick $c pick
}
proc _unDo {c} {
   global gvar
   foreach i $gvar(lastcmd) {
      eval $i
   }
   $c delete trace
}

proc _cutObj {c} {
   global gvar
   if {$gvar(ocutbuffer)!=""} {}
   set gvar(ocutbuffer) {}
   foreach i [$c find withtag pick] {
      lappend gvar(ocutbuffer) $i
   }
   set gvar(onocut) [llength $gvar(ocutbuffer)]
   set gvar(pastex) 0
   set gvar(pastey) 0
}
proc _pasteObj {c dx dy} {
   global gvar
   if {$gvar(ocutbuffer)==""} { return }
   $c dtag pick
   incr gvar(pastex) $dx
   incr gvar(pastey) $dy
   foreach j $gvar(ocutbuffer) {
      set opt {}
      foreach i [$c itemconfig $j] {
         if {[llength $i]==5 && [lindex $i 3]!=[lindex $i 4]} {
            lappend opt [lindex $i 0]
            lappend opt [lindex $i 4]
         }
      }
      set t1 [_mkObj "$c create [$c type $j] [$c coords $j] $opt"]
      $c move $t1 $gvar(pastex) $gvar(pastey)
      $c addtag pick withtag $t1
#     puts stderr "PASTE: $t1=[$c gettag $t1]"
   }
#  puts stderr "PASTE...[$c find withtag pick]"
   _boundingBox $c pick
}

proc _editPick {c} {
   global gvar
   case $gvar(oedittype) {
      {cut}   { _deleteObj $c }
      {copy}  { _cutObj $c }
      {paste} { _pasteObj $c 10 10 }
   }
}

proc _groupObj {c} {
   global gvar
   case $gvar(ogrouptype) in {
   {yes} {
         set t1 [$c bbox pick]
         if {$t1!=""} {
            set t3 "obj"
            foreach i [$c find withtag pick] {
               $c addtag grp$gvar(grpno) withtag $i
               append t3 " grp$i"
            }
            scan $t1 "%f %f %f %f" x1 y1 x2 y2
            set t2 [$c create rectangle $x1 $y1 $x2 $y2 -tags "pick" -outline ""]
            $c bind $t2 <Any-Enter> "_enterObj $c $t2"
            $c bind $t2 <Any-Leave> "_leaveObj $c $t2"
            append t3 " cgrp$gvar(grpno)"
#           puts stderr "GROUP: $t3"
            foreach i $t3 {
               $c addtag $i withtag $t2
            }
            $c delete trace
            _boundingBox $c $t2
            incr gvar(grpno)
         }
      }
   {no} {
         $c delete trace trace2
         set cg ""
         foreach i [$c find withtag pick] {
            set t2 [$c gettag $i]
            set t1 [lsearch $t2 cgrp*]
#           puts stderr "UG: $i - $t2 - $t1"
            if {$t1>-1} {
               scan [lindex $t2 $t1] "cgrp%d" k
               append cg " $k"
            }
         }
         set cg [lsort -integer -decreasing $cg]
         set k  [lindex $cg 0]
#        puts stderr "UNGROUP: $k"
         $c dtag grp$k
         $c delete cgrp$k
         $c dtag pick
#        foreach i [lrange $cg 1 end] {
#           $c addtag pick withtag cgrp$i
#           $c addtag pick withtag grp$i
#        }
      }
   }
}

proc _placeObj {c} {
   global gvar
   case $gvar(oplacetype) in {
   {front} {
         set t1 [$c find above pick]
         set t2 [$c gettags $t1]
         if {$t2=="obj"} {
            $c raise pick $t1
         }
      }
   {back} {
         set t1 [$c find below pick]
         set t2 [$c gettags $t1]
         if {$t2=="obj"} {
            $c lower pick $t1  
         }
      }
   }
}

proc _flipObj {c} {
   global gvar
   case $gvar(ofliptype) in {
   {x} {
         set sx -1.0
         set sy  1.0
      }
   {y} {
         set sx  1.0
         set sy -1.0
      }
   }
   set t2 [$c find withtag pick]
   if {$t2==""} {return}
   scan [$c bbox pick] "%f %f %f %f" x1 y1 x2 y2
   set cx [expr {($x1+$x2)*0.5}]
   set cy [expr {($y1+$y2)*0.5}]
   foreach i $t2 {
      $c move $i [expr {-1*$cx}] [expr {-1*$cy}]
      set t1 [$c coords $i]
      set k 0
      set result {}
      foreach j $t1 {
         if {[expr {$k%2}]==0} {
            append result "[expr {$sx*$j}] "
         } else {
            append result "[expr {$sy*$j}] "
         }
         incr k
      }
      eval "$c coords $i $result"
      $c move $i $cx $cy
   }
}

proc _smoothObj {c} {
   global gvar
   set t1 [$c find withtag pick]
   foreach i $t1 {
      set t2 [$c type $i]
      case $t2 in {
      {line polygon} {$c itemconfig $i -smooth $gvar(osmoothtype)}
      }
   }
}

proc _setBitmap {c tag} {
   global gvar
   set t1 [$c find withtag $tag]
   foreach i $t1 {
      set t2 [$c type $i]
      case $t2 in {
         {bitmap} {
            eval $c itemconfig $i -anchor $gvar(bitmapanchor)
         }
      }
   }
}

proc _setPattern {c} {
   global gvar
   foreach i {0 1 2 3 4 5 6 7 8} {
      .p.$i configure -bitmap @$gvar(BITMAPDIR)/$gvar(opattern)/pat$i.cad -fg $gvar(dcolor)
   }
#  .p.8 configure -bitmap @$gvar(BITMAPDIR)/$gvar(opattern)/pat7.cad -fg white
}

proc _setArcstyle {c} {
   global gvar
   set t1 [$c find withtag pick]
   foreach i $t1 {
      set t2 [$c type $i]
      case $t2 in {
         {arc} {
            eval $c itemconfig $i -style $gvar(oarcstyle)
         }
      }
   }
}
   
proc _setFill {c p tag} {
   global gvar
   .p.10 configure -bitmap @$gvar(BITMAPDIR)/$gvar(opattern)/pat$p.cad
   case $p in {
      {0} {
         set gvar(pat) $p
         set gvar(fill) \"\"
      }
      {8} {
         set gvar(pat) 7
         set gvar(fill) white
      }
      default {
         set gvar(pat) $p
         set gvar(fill) $gvar(dcolor)
      }
   }
   set t1 [$c find withtag $tag]
   foreach i $t1 {
      set t2 [$c type $i]
      case $t2 in {
         {line arc oval} {
            eval $c itemconfig $i -fill $gvar(fill) -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad
         }
         {polygon} {
            if {[lsearch [$c gettags $i] vtext*] < 0} {
               eval $c itemconfig $i -fill $gvar(fill) -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad
            }
         }
         {rectangle} {
            if {[lsearch [$c gettags $i] cgrp*] < 0} {
               eval $c itemconfig $i -fill $gvar(fill) -stipple @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad
            }
         }
      }
   }
}

proc _setRotate {c tag} {
   global gvar cos sin
   set ra [string trim $gvar(orotate)]
   set t2 [$c bbox $tag]
   if {![regexp {^(-?[0-9]+)$} $ra] || $t2==""} {
      return
   }
   scan $t2 "%f %f %f %f" x1 y1 x2 y2
   set cx [expr {($x2+$x1)*0.5}]
   set cy [expr {($y2+$y1)*0.5}]
   set t1 [$c find withtag $tag]
   set a [expr {-tan($ra*0.0087266463)}]
   set b [expr {-2*$a/(1+$a*$a)}]
#  set a [set cos($gvar(orotate))]
#  set b [set sin($gvar(orotate))]
   foreach k $t1 {
      set t2 [$c type $k]
      case $t2 in {
         {oval line rectangle polygon} {
            set t3 ""
            set pts [$c coords $k]
            set ll  [llength $pts]
            for {set i 0} {$i<$ll} {incr i} {
               set x1 [lindex $pts $i]
               set y1 [lindex $pts [incr i]]
               append t3 [_rotAngle $a $b $cx $cy $x1 $y1] " "
            }
            eval $c coords $k $t3
         }
      }
   }
   $c delete trace
   _boundingBox $c $tag
}

proc _setWidth {c tag value} {
   global gvar
   set t1 [$c find withtag $tag]
   set gvar(width) $value
   foreach i $t1 {
      set t2 [$c type $i]
      case $t2 in {
         {arc oval rectangle line} {$c itemconfig $i -width $value}
      }
   }
}

proc _setFont {c tag} {
   global gvar
   set t1 [$c find withtag $tag]
   set gvar(tfont) "-*-$gvar(tfontname)-$gvar(tfontweight)-$gvar(tfontslant)-normal--*-[expr {10*$gvar(tfontpoint)}]-*"
   foreach i $t1 {
      set t2 [$c type $i]
      case $t2 in {
         {text} {$c itemconfig $i -font $gvar(tfont) -just $gvar(tfontjust) \
                    -anchor $gvar(tfontanchor)
                 $c delete trace
                 _boundingBox $c $tag
         }
      }
   }
}

proc _loadVfont {} {
   global gvar font
   set fname $gvar(vfontname)
   if {[file exist $gvar(VFONTDIR)/$fname.tkfont]} {
      set fd [open $gvar(VFONTDIR)/$fname.tkfont r]
      while {[gets $fd line] > -1} {
         set t1 [lindex $line 0]
         set font($fname,$t1) [lrange $line 2 end]
      }
      close $fd
   }
}

proc _setVfont {c tag} {
   global gvar
   _loadVfont
}

proc _lineStyle {c} {
   global gvar
   case $gvar(olinestyle) in {
   {arrow}    {_arrowShape .arrowstyle}
   {cap}      {_capStyle .capstype}
   {join}     {_joinStyle .joinstyle}
   }
}

proc _arrowLoc {c tag} {
   global gvar
   set t1 [$c find withtag $tag]
   if {$t1!=""} {
      foreach i $t1 {
         set t2 [$c type $i]
         case $t2 in {
            {line} {$c itemconfig $i -arrow $gvar(oarrowtype)}
         }
      }
   }
}

proc _setArrow {c type value} {
   global gvar
   case $type in {
      {ah} {
         set gvar(temp) "[lindex $gvar(temp) 0] [lindex $gvar(temp) 1] $value"
         $c itemconfig test -arrowshape $gvar(temp)
      }
      {bw} {
         set gvar(temp) "$value [lindex $gvar(temp) 1] [lindex $gvar(temp) 2]"
         $c itemconfig test -arrowshape $gvar(temp)
      }
      {aw} {
         set gvar(temp) "[lindex $gvar(temp) 0] $value [lindex $gvar(temp) 2]"
         $c itemconfig test -arrowshape $gvar(temp)
      }
   }
}

proc _keepvalue {w x} {
   global gvar
   set $x $gvar(temp)
   .w.1 set $gvar(width)
   destroy $w
}

proc _arrowShape {w} {
   global gvar
   catch {destroy $w}
   toplevel $w
   set gvar(temp) $gvar(arrowshape)
   frame $w.c
      canvas $w.c.2 -h 150 -relief raised -bd 3
      foreach i {{0 50 100 50} {50 0 50 100} {150 0 250 100} {150 100 250 0}} {
         eval "$w.c.2 create line $i -arrow both -tags \"test\" \
            -arrowshape [list $gvar(arrowshape)] -width $gvar(width) \
            -fill $gvar(dcolor)"
      }
      $w.c.2 move test 25 25
   pack append $w.c $w.c.2 {top}
   
   frame $w.s
      set j 1
      foreach i {lineWidth arrowHeight arrowWidth baseWidth} {
         scale $w.s.$j -label $i -length 100 -width 10 -from 1 -to 30 \
            -orient horiz
         pack append $w.s $w.s.$j {left padx 5}
         incr j
      }
      $w.s.1 set $gvar(width)
      $w.s.2 set [lindex $gvar(arrowshape) 2]
      $w.s.3 set [lindex $gvar(arrowshape) 1]
      $w.s.4 set [lindex $gvar(arrowshape) 0]
      $w.s.1 configure -command "_setWidth $w.c.2 test"
      $w.s.2 configure -command "_setArrow $w.c.2 ah"
      $w.s.3 configure -command "_setArrow $w.c.2 aw"
      $w.s.4 configure -command "_setArrow $w.c.2 bw"

   frame $w.b
      button $w.b.1 -text "DONE" -bd 5 -command "_keepvalue $w gvar(arrowshape)"
      button $w.b.2 -text "CANCEL" -bd 5 -command "destroy $w"
   pack append $w.b $w.b.1 {left expand fillx} \
                    $w.b.2 {left expand fillx}

   pack append $w $w.c {top expand fill} \
                  $w.s {top fill} \
                  $w.b {top expand fillx}
}

proc _capStyle {w} {
   global gvar
   set gvar(temp) $gvar(capstyle)
   catch {destroy $w}
   toplevel $w
   frame $w.c
      canvas $w.c.2 -h 150 -relief raised -bd 3
      foreach i {{0 50 100 50} {50 0 50 100} {150 0 250 100} {150 100 250 0}} {
         eval "$w.c.2 create line $i -tags test -capstyle $gvar(temp) -width $gvar(width) -fill $gvar(dcolor)"
      }
      $w.c.2 move test 25 25
   pack append $w.c $w.c.2 {top}
   
   frame $w.s
      scale $w.s.1 -label "lineWidth" -length 100 -width 10 -from 1 -to 30 \
         -orient horiz
      pack append $w.s $w.s.1 {left padx 5}
      $w.s.1 set $gvar(width)
      $w.s.1 configure -command "_setWidth $w.c.2 test"
      set j 2
      foreach i {butt projecting round} {
         radiobutton $w.s.$j -text $i -variable gvar(temp) -value $i \
            -command "$w.c.2 itemconfig test -capstyle $i"
         pack append $w.s $w.s.$j {left padx 5}
         incr j
      }

   frame $w.b
      button $w.b.1 -text "DONE" -bd 5 -command "_keepvalue $w gvar(capstyle)"
      button $w.b.2 -text "CANCEL" -bd 5 -command "destroy $w"
   pack append $w.b $w.b.1 {left expand fillx} \
                    $w.b.2 {left expand fillx}

   pack append $w $w.c {top expand fill} \
                  $w.s {top fill} \
                  $w.b {top expand fillx}
}

proc _joinStyle {w} {
   global gvar
   catch {destroy $w}
   set gvar(temp) $gvar(joinstyle)
   toplevel $w
   frame $w.c
      canvas $w.c.2 -h 150 -relief raised -bd 3
      foreach i {{0 0 0 100 100 100} {125 0 175 100 225 0 275 100}} {
         eval "$w.c.2 create line $i -tags test -joinstyle $gvar(temp) -width $gvar(width) -fill $gvar(dcolor)"
      }
      $w.c.2 move test 25 25
   pack append $w.c $w.c.2 {top}
   
   frame $w.s
      scale $w.s.1 -label "lineWidth" -length 100 -width 10 -from 1 -to 30 \
         -orient horiz
      pack append $w.s $w.s.1 {left padx 5}
      $w.s.1 set $gvar(width)
      $w.s.1 configure -command "_setWidth $w.c.2 test"
      set j 2
      foreach i {miter bevel round} {
         radiobutton $w.s.$j -text $i -variable gvar(temp) -value $i \
            -command "$w.c.2 itemconfig test -joinstyle $i"
         pack append $w.s $w.s.$j {left padx 5}
         incr j
      }

   frame $w.b
      button $w.b.1 -text "DONE" -bd 5 -command "_keepvalue $w gvar(joinstyle)"
      button $w.b.2 -text "CANCEL" -bd 5 -command "destroy $w"
   pack append $w.b $w.b.1 {left expand fillx} \
                    $w.b.2 {left expand fillx}

   pack append $w $w.c {top expand fill} \
                  $w.s {top fill} \
                  $w.b {top expand fillx}
}

proc _zoomCanvas {c {obj all}} {
   global gvar
   set sc [expr 1.0*$gvar(ozoomtype)/$gvar(ozoom)]
   set xc [$c canvasx $gvar(canvasw)]
   set yc [$c canvasy $gvar(canvash)]
   $c scale $obj $xc $yc $sc $sc
   set gvar(ozoom) $gvar(ozoomtype)
}

proc _setScale {c dir sc} {
   global gvar
   set t1 [$c bbox pick]
   if {$t1!=""} {
      scan $t1 "%f %f %f %f" x1 y1 x2 y2
      set cx [expr {($x2+$x1)*0.5}]
      set cy [expr {($y2+$y1)*0.5}]
      if {![regexp {^((0?\.?[1-9][0-9]*)|([0-9]+\.[0-9]+))$} $sc]} { return }
      set t3 [expr {1.0/$gvar(oscale)}]
      case $dir in {
      {1} {set t4 "$t3 $t3"
           set t5 "$sc $sc"
          }
      {2} {set t4 "$t3 1.0"
           set t5 "$sc 1.0"
          }
      {3} {set t4 "1.0 $t3"
           set t5 "1.0 $sc"
          }
      }
      if {$gvar(oscaletype)=="abs"} {
         eval "$c scale pick $cx $cy $t4"
      }
      eval "$c scale pick $cx $cy $t5"
      set gvar(oscale) $sc
      $c delete trace
      _boundingBox $c pick
   }
}

proc _printit {c} {
   global gvar
   $c coords xhair 0 0 0 0
   $c coords yhair 0 0 0 0
   $c delete trace trace2
   $c dtag pick
   _canvasprint $c
}

proc _changeButton {bt bn} {
   global gvar
   set k ""
#  puts stderr "changeButton: $bt $bn $k"
   case $bt in {
      {.cmd} {
         case $bn in {
            {edit}     { set k oedittype;    set m {cut copy paste} }
            {lines}    { set k olinestype;   set m {extend chamfer fillet} }
            {smooth}   { set k osmoothtype;  set m {yes no} }
            {group}    { set k ogrouptype;   set m {yes no} }
            {flip}     { set k ofliptype;    set m {x y} }
            {place}    { set k oplacetype;   set m {front back} }
            {zoom}     { set k ozoomtype;    set m {25 50 100 150 200 400} }
            {style}    { set k olinestyle;   set m {arrow cap join} }
            {arrow}    { set k oarrowtype;   set m {none first last both} }
            {align}    { set k oaligntype;   set m {left right up down xcenter ycenter} }
            {shear}    { set k osheartype;   set m {x y xy} }
         }
      }
      {.o} {
         case $bn in {
            {text}     { set k otexttype;    set m {bitmap vector} }
            {rect}     { set k orecttype;    set m {2pts 2ptsfill 2ptsr 2ptsrfill} }
            {line}     { set k olinetype;    set m {2pts npts 4pts lorth rorth} }
            {oval}     { set k oovaltype;    set m {2pts 3pts radius diameter} }
            {arc}      { set k oarctype;     set m {2pts 3pts 4pts} }
         }
      }
   }
   if {$k!=""} {
      set i [lsearch $m $gvar($k)]
      incr i
      if {$i>=[llength $m]} {set i 0}
      set m [lindex $m $i]
      $bt.$bn configure -bitmap @$gvar(ICONDIR)/$bn$m.cad
      set gvar($k) $m
      if {$bt==".o"} {
         _pressButton $bt $bn
      }
   }
}

proc _pressButton {bt bn} {
   global gvar
   foreach i {move line oval rect polygon text arc bitmap guide morph} {
      .o.$i configure -relief raised
   }
      case $bn in {
         {move} { 
            _restore .c.c
            set gvar(obj) cursor 
         }
         default {
            _restore .c.c
            set gvar(obj) $bn
         }
      }
      case $bn in {
         {line polygon} {
               set gvar(tracemode) line
            }
         {rect oval arc} {
               set gvar(tracemode) rect
            }
         default {
               set gvar(tracemode) point
            }
      }
      $bt.$bn configure -relief sunken
}

proc _moveMode {c} {
   global gvar
#  $c delete trace trace2
   set gvar(coords)  {}
   set gvar(pcoords) {}
   set gvar(ecoords) {}
   if {!$gvar(opersist)} {
      _pressButton .o move
   }
}

#=start=======================================================================

foreach i $argv {
   case $i in {
   {-c} {option add *background [format {#%2x%2x%2x} 238 169 184]
        }
   {-r} {option add *activeBackground white
         option add *activeForeground black
         option add *selectBackground white
         option add *selectForeground black
         option add *insertBackground white
         option add *insertForeground black
         option add *background black
         option add *foreground white
         set gvar(dcolor) white
        }
   default {
      set gvar(oloadfile) $i
      set gvar(osavefile) $i
        }
   }
}

proc _hscroll {a b n} {
   $a xview $n
   $b xview $n
}
proc _vscroll {a b n} {
   $a yview $n
   $b yview $n
}

wm minsize . 500 500

frame .c -relief raised -bd 3
   frame .c.l
      frame  .c.l.tlc -height 30 -width 30 -relief sunken -bd 0
      set t1 "0 $gvar(canvasy1)$gvar(u) 30 $gvar(canvasy2)$gvar(u)"
      canvas .c.l.lr -scrollregion $t1 -w 30 -h $gvar(canvash)$gvar(u) -relief sunken
      frame  .c.l.blc -height 20 -width 30 -relief sunken -bd 0
   pack .c.l.tlc -side top
   pack .c.l.blc -side bottom
   pack .c.l.lr  -side top -fill y -expand yes
   frame .c.t
      set t1 "$gvar(canvasx1)$gvar(u) 0 $gvar(canvasx2)$gvar(u) 30"
      canvas .c.t.tr -scrollregion $t1 -w $gvar(canvasw)$gvar(u) -h 30 -relief sunken
      frame  .c.t.trc -height 30 -width 20 -relief sunken -bd 0
   pack .c.t.trc -side right
   pack .c.t.tr  -side left -fill x -expand yes
   frame .c.b
      scrollbar .c.b.hs -orient horiz -relief sunken -command "_hscroll .c.c .c.t.tr"
      frame  .c.b.brc -height 20 -width 20 -relief sunken -bd 0
   pack .c.b.brc -side right
   pack .c.b.hs  -side left -fill x -expand yes
   set t1 "$gvar(canvasx1)$gvar(u) $gvar(canvasy1)$gvar(u) $gvar(canvasx2)$gvar(u) $gvar(canvasy2)$gvar(u)"
   canvas .c.c -scrollregion $t1 \
      -w $gvar(canvasw) -h $gvar(canvash) \
      -xscroll ".c.b.hs set" -yscroll ".c.vs set" -close $gvar(closeenough)
   scrollbar .c.vs -relief sunken -command "_vscroll .c.c .c.l.lr"
pack .c.l   -side left   -fill y
pack .c.t   -side top    -fill x
pack .c.b   -side bottom -fill x
pack .c.vs  -side right  -fill y
pack .c.c   -side left   -fill both -expand yes

# ruler codes
.c.t.tr create line $gvar(canvasx1)$gvar(u) 29 $gvar(canvasx2)$gvar(u) 29
set dx [expr ($gvar(canvasx2)-$gvar(canvasx1))/20.0]
for {set x $gvar(canvasx1)} {$x<=$gvar(canvasx2)} {set x [expr $x+$dx]} {
   set t1 $x
   .c.t.tr create line $t1$gvar(u) 19 $t1$gvar(u) 29
   set t1 [expr $t1+$dx*0.5]
   .c.t.tr create line $t1$gvar(u) 25 $t1$gvar(u) 29
   .c.t.tr create text $x$gvar(u) 19 -font helvr08 -text $x$gvar(u) -anchor s
}
.c.l.lr create line 29 $gvar(canvasy1)$gvar(u) 29 $gvar(canvasy2)$gvar(u)
set dy [expr ($gvar(canvasy2)-$gvar(canvasy1))/20.0]
for {set y $gvar(canvasy1)} {$y<=$gvar(canvasy2)} {set y [expr $y+$dy]} {
   set t1 $y
   .c.l.lr create line 19 $t1$gvar(u) 29 $t1$gvar(u)
   set t1 [expr $t1+$dy*0.5]
   .c.l.lr create line 25 $t1$gvar(u) 29 $t1$gvar(u)
   .c.l.lr create text 1 $y$gvar(u) -font helvr08 -text $y$gvar(u) -anchor w
}

frame .m -bd 2 -relief sunken
   menubutton .m.file -text "File" -menu .m.file.m
   menu .m.file.m
      .m.file.m add command -label "Clear" -command "_canvasClear .c.c"
      .m.file.m add command -label "UnDo" -command  "_unDo .c.c"
      .m.file.m add command -label "Info"  -command "_canvasInfo .c.c"
      .m.file.m add command -label "Print" -command "_printit .c.c"
      .m.file.m add command -label "Dump"  -command "_busy {_dumpObj .c.c obj stdout}"
      .m.file.m add command -label "Save"  -command "_save .c.c"
      .m.file.m add command -label "Load"  -command "_load .c.c"
      .m.file.m add command -label "Quit"  -command "_quit .c.c"
   menubutton .m.bfont -text "BFont" -menu .m.bfont.m
   menu .m.bfont.m
      set font(name)   {courier helvetica times symbol {new century schoolbook}}
      set font(weight) {medium bold}
      set font(slant)  {r o i}
      set font(point)  {8 10 12 14 18 24}
      set font(just)   {left center right}
      set font(anchor) {nw n ne w c e sw s se}
      foreach i {name weight slant point just anchor} {
         .m.bfont.m add cascade -label $i -menu .m.bfont.m.$i
         menu .m.bfont.m.$i
         foreach j $font($i) {
            .m.bfont.m.$i add radio -label $j -variable gvar(tfont$i) \
               -value $j -command "_setFont .c.c pick"
         }
      }
   menubutton .m.vfont -text "VFont" -menu .m.vfont.m
   menu .m.vfont.m
      set vfont(name) ""
      foreach i [glob $gvar(VFONTDIR)/*.tkfont] {
         append vfont(name) "[file tail [file root $i]] "
      }
      foreach i {name} {
         .m.vfont.m add cascade -label $i -menu .m.vfont.m.$i
         menu .m.vfont.m.$i
         foreach j $vfont($i) {
            .m.vfont.m.$i add radio -label $j -variable gvar(vfont$i) \
               -value $j -command "_setVfont .c.c pick"
         }
      }
   menubutton .m.misc -text "Misc" -menu .m.misc.m
   menu .m.misc.m
      .m.misc.m add cascade -label "Crosshair" -menu .m.misc.m.xhair
      menu .m.misc.m.xhair
         .m.misc.m.xhair add radio -label "On" \
            -variable gvar(oxhair) -value on  -command "_createXhair .c.c"
         .m.misc.m.xhair add radio -label "Off" \
            -variable gvar(oxhair) -value off -command ".c.c delete xhair yhair"
      .m.misc.m add cascade -label "Scale" -menu .m.misc.m.scale
      menu .m.misc.m.scale
         .m.misc.m.scale add radio -label "Absolute" \
            -variable gvar(oscaletype) -value abs
         .m.misc.m.scale add radio -label "Relative" \
            -variable gvar(oscaletype) -value rel
      .m.misc.m add cascade -label "Grid" -menu .m.misc.m.grid
      menu .m.misc.m.grid
         .m.misc.m.grid add radio -label "On" \
            -variable gvar(ogrid) -value on  \
            -command "_busy \"_drawGrid .c.c 0 0 50 50 0\""
         .m.misc.m.grid add radio -label "Off" \
            -variable gvar(ogrid) -value off -command ".c.c delete grid"
      .m.misc.m add cascade -label "Snap" -menu .m.misc.m.snap
      menu .m.misc.m.snap
         foreach i {normal grid endpts midpts} {
            .m.misc.m.snap add radio -label $i -variable gvar(osnap) -value $i \
               -command ""
         }
      .m.misc.m add cascade -label "Pattern" -menu .m.misc.m.pat
      menu .m.misc.m.pat
      foreach i [glob $gvar(BITMAPDIR)/*] {
         set j [file tail [file root $i]]
         .m.misc.m.pat add radio -label $j -variable gvar(opattern) \
               -value $j -command "_setPattern .c.c"
      }
      .m.misc.m add cascade -label "ArcStyle" -menu .m.misc.m.arc
      menu .m.misc.m.arc
      foreach i {arc pieslice chord} {
         .m.misc.m.arc add radio -label $i -variable gvar(oarcstyle) \
               -value $i -command "_setArcstyle .c.c"
      }
      .m.misc.m add cascade -label "Bitmap" -menu .m.misc.m.bitmap
      menu .m.misc.m.bitmap
      foreach i {nw n ne w c e sw s se} {
         .m.misc.m.bitmap add radio -label $i -variable gvar(bitmapanchor) \
               -value $i -command "_setBitmap .c.c pick"
      }
      
pack append .m .m.file      {left fillx} \
               .m.bfont     {left fillx} \
               .m.vfont     {left fillx} \
               .m.misc      {left fillx}

frame .s
   label .s.nopickl   -text "Obj="
   label .s.nopick    -textvariable gvar(onopick)
   label .s.nocutl    -text "Buf="
   label .s.nocut     -textvariable gvar(onocut)
   label .s.obj       -textvariable gvar(obj)
   label .s.coords    -textvariable gvar(xylabel)
   label .s.snap      -textvariable gvar(osnap)
pack append .s .s.nopickl {left} \
               .s.nopick  {left} \
               .s.nocutl  {left} \
               .s.nocut   {left} \
               .s.obj     {left} \
               .s.snap    {left} \
               .s.coords  {right}

frame .o
   set j 1
   foreach i {move line oval rect polygon text arc bitmap guide morph} {
      set bm $i.cad
      case $i in {
      {text}                  { set bm $i$gvar(otexttype).cad }
      {rect}                  { set bm $i$gvar(orecttype).cad }
      {line}                  { set bm $i$gvar(olinetype).cad }
      {oval}                  { set bm $i$gvar(oovaltype).cad }
      {arc}                   { set bm $i$gvar(oarctype).cad }
      }
      button .o.$i -bitmap @$gvar(ICONDIR)/$bm -bd 2
      bind .o.$i <Button-1>        "set gvar(opersist) 0; _pressButton .o $i"
      bind .o.$i <Shift-Button-1>  "set gvar(opersist) 1; _pressButton .o $i"
      bind .o.$i <Button-2>        "_changeButton .o $i"
      pack append .o .o.$i {top}
      incr j
   }
   .o.morph configure -state disabled
   .o.guide configure -state disabled
   .o.move            configure -command { _moveMode .c.c }
   .o.move invoke

frame .cmd
   set j 1
   foreach i {align arrow style place group smooth flip edit zoom shear lines break} {
      set bm $i.cad
      case $i in {
      {smooth}                { set bm $i$gvar(osmoothtype).cad }
      {group}                 { set bm $i$gvar(ogrouptype).cad }
      {flip}                  { set bm $i$gvar(ofliptype).cad }
      {place}                 { set bm $i$gvar(oplacetype).cad }
      {zoom}                  { set bm $i$gvar(ozoomtype).cad }
      {style}                 { set bm $i$gvar(olinestyle).cad }
      {align}                 { set bm $i$gvar(oaligntype).cad }
      {arrow}                 { set bm $i$gvar(oarrowtype).cad }
      {shear}                 { set bm $i$gvar(osheartype).cad }
      {lines}                 { set bm $i$gvar(olinestype).cad }
      {edit}                  { set bm $i$gvar(oedittype).cad }
      }
      button .cmd.$i -bitmap @$gvar(ICONDIR)/$bm -bd 2
      bind .cmd.$i <Button-2>        "_changeButton .cmd $i"
      pack append .cmd .cmd.$i {top}
      incr j
   }
   .cmd.place           configure -command { _placeObj .c.c }
   .cmd.smooth          configure -command { _smoothObj .c.c }
   .cmd.group           configure -command { _groupObj .c.c }
   .cmd.flip            configure -command { _flipObj .c.c }
   .cmd.edit            configure -command { _editPick .c.c }
   .cmd.align           configure -command { _alignObj .c.c }
   .cmd.arrow           configure -command { _arrowLoc .c.c pick }
   .cmd.style           configure -command { _lineStyle .c.c }
   .cmd.zoom            configure -command { _zoomCanvas .c.c }
   .cmd.lines           configure -command { _interSect .c.c pick }
   .cmd.shear           configure -command { _shearObj .c.c pick }
   .cmd.break           configure -command { _breakLines .c.c pick }

frame .p -bd 2 -relief sunken
   label  .p.10 -bitmap @$gvar(BITMAPDIR)/$gvar(opattern)/pat$gvar(pat).cad -relief sunken -bd 2 -width 16 -height 16
   pack append .p .p.10 {left padx 5 pady 5}
   foreach i {0 1 2 3 4 5 6 7 8} {
      button .p.$i -relief raised -bd 2 -width 16 -height 16 \
         -command "_setFill .c.c $i pick"
      pack append .p .p.$i {left pady 5}
   }
   _setPattern .c.c

frame .w 
   scale  .w.1 -label "Width" -length 100 -width 10 -from 1 -to 30 \
      -orient horiz -command "_setWidth .c.c pick"
   pack append .w .w.1 {left}

frame .i1
   frame .i1.bfont
      label  .i1.bfont.1 -text "BFont" -relief flat
      entry  .i1.bfont.2 -relief sunken -textvariable gvar(tfont)
      bind   .i1.bfont.2 <Return> "_setFont .c.c pick"
   pack append .i1.bfont .i1.bfont.1 {left} \
                         .i1.bfont.2 {left fillx expand}
pack append .i1 .i1.bfont  {left fillx expand}

frame .cl
   label .cl.label -text "Command:"
   entry .cl.entry -relief sunken -bd 3 -textvariable gvar(ocommand)
   bind  .cl.entry  <Return> "_busy {_doCmd .c.c}"
pack .cl.label  -side left
pack .cl.entry  -side left -fill x -expand yes

pack append . .m      {top fillx} \
              .s      {top fillx} \
              .cl     {bottom fillx} \
              .p      {bottom fillx} \
              .w      {bottom fillx} \
              .i1     {bottom fillx} \
              .o      {left filly} \
              .cmd    {left filly} \
              .c      {right expand fill}

bind .c.c <Button-2>               { .c.c scan mark -%x -%y }
bind .c.c <B2-Motion>              { .c.c scan dragto -%x -%y }
bind .c.c <Enter>                  { focus %W }
#bind .c.c <KeyPress-Shift_L>      { set gvar(osnap) orth }
#bind .c.c <KeyRelease-Shift_L>    { set gvar(osnap) normal }
bind .c.c <Control-s>              { set gvar(osnap) [_cycleSnap $gvar(osnap)] }
bind .c.c <Shift-Motion>           { _drawCursor .c.c %x %y 1 }
bind .c.c <Motion>                 { _drawCursor .c.c %x %y 0 }
bind .c.c <Button-1>               { _drawStart .c.c %x %y }
bind .c.c <Shift-B1-Motion>        { _dragObj .c.c %x %y 1 }
bind .c.c <B1-Motion>              { _dragObj .c.c %x %y 0 }
bind .c.c <ButtonRelease-1>        { _drawEnd .c.c %x %y }
bind .c.c <Shift-Button-1>         { _drawStart .c.c %x %y }
bind .c.c <Shift-ButtonRelease-1>  { _drawEnd .c.c %x %y }

bind .c.c <Shift-Up>               { _moveObj .c.c 0 -1 }
bind .c.c <Up>                     { _moveObj .c.c 0 -10 }
bind .c.c <Shift-Down>             { _moveObj .c.c 0 1 }
bind .c.c <Down>                   { _moveObj .c.c 0 10 }
bind .c.c <Shift-Right>            { _moveObj .c.c 1 0 }
bind .c.c <Right>                  { _moveObj .c.c 10 0 }
bind .c.c <Shift-Left>             { _moveObj .c.c -1 0 }
bind .c.c <Left>                   { _moveObj .c.c -10 0 }

bind .c.c <BackSpace>              { _deleteObj .c.c }
bind .c.c <Delete>                 { _deleteObj .c.c }

proc _btextDelete {w} {
   set t1 ""
   set t2 [catch {$w index text sel.first} t1]
   if {[string range $t1 0 8] != "selection"} {
      $w dchars text sel.first sel.last
   } else {
      set x [expr {[$w index text insert] - 1}]
       if {$x >= 0} {$w dchars text $x}
   }
}
proc _btextMove {w i} {
   set x [expr {[$w index text insert] + $i}]
   set xm [$w index text end]
   if {$x >= 0 && $x <= $xm} {$w icursor text $x}
}

.c.c bind text <KeyPress>       { .c.c insert text insert %A }
.c.c bind text <Shift-KeyPress> { .c.c insert text insert %A }
.c.c bind text <Shift-1>        { .c.c select adjust current @%x,%y }
.c.c bind text <Return>         { .c.c insert text insert \n }
.c.c bind text <Control-h>      { _btextDelete .c.c }
.c.c bind text <Delete>         { _btextDelete .c.c }
.c.c bind text <BackSpace>      { _btextDelete .c.c }
.c.c bind text <Control-u>      { .c.c dchars text 0 end }
.c.c bind text <Left>           { _btextMove .c.c -1 }
.c.c bind text <Right>          { _btextMove .c.c 1 }
.c.c bind text <Button1-Motion> { .c.c select to text @%x,%y }
.c.c bind text <ButtonPress-2>  { catch {%W insert text insert [selection get] } }

_createXhair .c.c
set gvar(vfontname) gotheng
_setVfont .c.c pick
_busy {_loadit .c.c}
