################################################################
####		       Latex Mode                           ####
################################################################

set LatexText {}

##
##  Building a latex file from the template
##

proc latexBuildFile {text bboxFlag args} {
  global LatexDocStyle
  
  ## build latex file content ##
  ##  args are additional style files  ##

  #  get the style files from template  #
  #  and add the styles in args  #

  set LatexDocStyle [string trim $LatexDocStyle]
  if {[regexp {^\{[^\}]*\}$} $LatexDocStyle]} {
    set optStyles ""
  } else {
    if {![regexp {^\[([^]]*)\]\{([^\}]*)\}$} $LatexDocStyle foo optStyles mainStyle]} {
      error "Invalid latex document style. See Latex options"
    }
  }

  set optStyles "[split $optStyles ","] $args"
  set docst "\[[join $optStyles ","]\]\{$mainStyle\}"

  # accent commands are redefined in latex tabular environment #
  regsub -all {\\'} $text {\a'} text
  regsub -all {\\`} $text {\a`} text
  # remove blank lines that cause boxdims to break #
  regsub -all "^\[ \]*\n" $text {} text

  set result "\
\\documentstyle$docst
\\pagestyle{empty}
\\setlength{\\abovedisplayskip}{0pt}
\\begin{document}
"
  if {$bboxFlag} {
    append result "\\boxdims\{"
  }

  append result "
\\begin{minipage}{\\hsize}
\\begin{tabbing}
$text
\\end{tabbing}
\\end{minipage}
"
  if {$bboxFlag} {
    append result "\}"
  }

  append result "
\\end{document}\n"
  
  return $result
}
  
##
##  Latex compilation
##

proc latex2dvi {inpath} {
  global TmpDir

  set cwd [pwd]
  cd $TmpDir
  catch {exec echo q | latex $inpath} output
  
  set status [regexp -nocase {output[ ]*written[ ]*on[ ]*([^.]*.dvi)}\
	      $output foo dvipath]

  cd $cwd
  
  if {! $status} {
    error $output
  } else {
    return $output
  }
}  

proc latex2ps {inpath outpath} {
  global TmpDir

  set oldpwd [pwd]
  cd $TmpDir  

  catch {exec echo q | latex $inpath} output
  
  set status [regexp -nocase {output[ ]*written[ ]*on[ ]*([^.]*.dvi)}\
	      $output foo dvipath]

  if {! $status} {
    cd $oldpwd
    error "Couldn't find latex dvi output file\n $output"
  }
  
  catch {exec dvips $dvipath -n 1 -o $outpath} dvioutput

  catch {exec rm $dvipath}

  cd $oldpwd
  return
}

#
#  LatexCompile:
#      compiles selected latex items and displays them as images
#

proc LatexCompile {} {
  global LatexText LatexResolution
  
  set ids [[cv] find withtag s latex]
  if {[lempty $ids]} {
    warn "No latex item selected"
    return
  }

  foreach id $ids {
    set both [[cv] find withtag \
	      [getTopGroupTags $id]]
    if {[lindex $both 0] == $id} {
      set frameId [lindex $both 1]
    } else {
      set frameId [lindex $both 0]
    }

    # get latex text and position #
    set text [keylget LatexText $id]
    lassign [[cv] coords $frameId] x1 y1 x2 y2
    set xc [expr ($x1+$x2)/2]
    set yc [expr ($y1+$y2)/2]
    set width [expr $x2 - $x1]

    ##  Create image item  ##
    saveMsg
    msg "Converting Latex to Postscript ..."

    #  latex file  #
    set latexfile [mkTmpFile latex]
    set st [open $latexfile "w"]
    puts $st [latexBuildFile $text 0]
    close $st

    msg {Converting Latex to Postscript ...}
    #  latex -> ps  #
    set psfile [mkTmpFile ps]
    if {[catch {latex2ps $latexfile $psfile} err]} {
      catch {exec rm $latexfile $psfile}
      restoreMsg
      restoreCursor
      textBox "Latex error:" $err "dismiss"
      return
    }

    catch {exec rm $latexfile}

    #  ps -> ppm  #
    msg "Converting Postscript to PPM ..."
  
    set ppmfile [mkTmpFile ppm]
    set cmd "echo \"showpage\" | gs -r$LatexResolution -sDEVICE=ppmraw -sOutputFile=$ppmfile $psfile -dNODISPLAY 2>&1 > /dev/null"

    if {[catch {system "$cmd"} err]} {
      warn "Postscript to PPM conversion failed."
      catch {exec rm $psfile $ppmfile}
      restoreMsg
      restoreCursor
      return
    }

    catch {exec rm $psfile}

    msg "Stripping PPM image ..."
  
    #  ppm -> picasso  #
    set picfile [mkTmpFile image]
    if {[catch {ppm2pic -strip $ppmfile $picfile 0} err]} {
      warn $err
      catch {exec rm $ppmfile $picfile}
      restoreMsg
      restoreCursor
      
      return
    }

    catch {exec rm $ppmfile}

     #  destroy old item  #
    [cv] delete $id
    [cv] delete $frameId
    [cv] delete sFr
    #  create image item  #

    set imId [eval "[cv] create image $xc $yc \
                    -path $picfile \
                    -width $width \
                    -tags inCreation"]

    keylset LatexText $imId \
      [keylget LatexText $id]
    keyldel LatexText $id
    keylset ImageFile $imId $picfile
  }
  
  restoreMsg
  restoreCursor
  createEnd_hook
  redrawFrames
}

#
#  LatexUncompile :
#    Replace an image latex object by a new style latex object
#

proc LatexUncompile {} {
  global LatexText LatexAnchor anchor2just
  global LatexDims LatexDrawingRatio
  
  set sIds [[cv] find withtag s]
  set ids {}
  foreach id $sIds {
    if {([[cv] type $id] == "image") &&
      [keylget LatexText $id {}]} {
	lappend ids $id
      }
  }
  
  if {[lempty $ids]} {
    warn "No latex item selected"
    return
  }

  [cv] delete sFr
  waitCursor
  
  foreach id $ids {
    lassign [[cv] coords $id] xc yc
    set w2 [expr [lindex [[cv] itemconfigure $id -width] 4]/2]
    set h2 [expr [lindex [[cv] itemconfigure $id -height] 4]/2]
    set text [keylget LatexText $id]

    #  New latex items impose some modifs on the text  #
    regsub {\\\\\\\\} $text {\\\\} text
    if {[regsub -all {\\begin\{center\}} $text {} text]} {
      regsub -all {\\end\{center\}} $text {} text
    }
    regsub -all "^\[ \]*\n" $text {} text
    if {[regsub -all {\\\[} $text {$ \\displaystyle } text]} {
      regsub -all {\\\]} $text { $} text
    }

    switch -exact -- $LatexAnchor {
      nw { set xt [expr $xc - $w2]; set yt [expr $yc - $h2] }
      w  { set xt [expr $xc - $w2]; set yt $yc              }
      sw { set xt [expr $xc - $w2]; set yt [expr $yc + $h2] }
      n  { set xt $xc;              set yt [expr $yc - $h2] }
      center { set xt $xc; set yt $yc }
      s  { set xt $xc;              set yt [expr $yc + $h2] }
      ne { set xt [expr $xc + $w2]; set yt [expr $yc - $h2] }
      e  { set xt [expr $xc + $w2]; set yt $yc              }
      se { set xt [expr $xc + $w2]; set yt [expr $yc + $h2] }
    }

    saveMsg
    msg "Computing latex bounding box..."
    if {[catch {computeLatexDims $text} dims]} {
      restoreCursor
      textBox "Latex error:" $dims "dismiss"
      continue
    } 

    [cv] delete $id
    
    set textId [[cv] create text $xc $yc \
		-anchor $LatexAnchor \
		  -justify [keylget anchor2just $LatexAnchor] \
		  -text $text]
    [cv] itemconfigure $textId \
      -tags "s sOfr tGr.$textId Gr.$textId latex"
    
    keylset LatexDims $textId $dims

    set bbw2 [expr [lindex $dims 0] / (2*$LatexDrawingRatio)]
    set bbh2 [expr [lindex $dims 1] / (2*$LatexDrawingRatio)]

    set x1 [expr $xc - $bbw2]
    set x2 [expr $xc + $bbw2]
    set y1 [expr $yc - $bbh2]
    set y2 [expr $yc + $bbh2]

    fitTextInRect $textId $x1 $y1 $x2 $y2
    set frameId [[cv] create rectangle $x1 $y1 $x2 $y2 \
		 -outline green -fill green \
		   -tags "s sOfr tGr.$textId Gr.$textId ltxFr"]

    keylset LatexText $textId $text
    keyldel LatexText $id
  }
  
  restoreMsg
  restoreCursor
  redrawFrames
  [cv] raise ltxFr
  [cv] raise latex
}

##
##  Computes the bbox in pixels of the postscript generated
##    by compiling a latex object. Doesn't take care
##    of LatexDrawingRatio.
##

set LatexDims {}

proc computeLatexDims {text} {
  #  Generate the style file  #
  #  (TeX macro to compile the dimensions of the latex-generated box  #

  set styfile [mkTmpFile sty]
  set st [open "$styfile.sty" "w"]
  puts $st "\\def\\boxdims#1\{"
  puts $st "\\setbox\\@tempboxa\\hbox\{#1\}"
  puts $st "\\typeout\{BOXWIDTH \\the\\wd\\@tempboxa\}"
  puts $st "\\setbox\\@tempboxa\\vbox\{#1\}"
  puts $st "\\newdimen\\tmpboxheight"
  puts $st "\\tmpboxheight=\\ht\\@tempboxa"
  puts $st "\\advance\\tmpboxheight\\dp\\@tempboxa"
  puts $st "\\typeout\{BOXHEIGHT \\the\\tmpboxheight\}"
  puts $st "\}"
  close $st

  #  latex file  #
  set latexfile [mkTmpFile latex]
  set st [open $latexfile "w"]

  if {[catch {latexBuildFile $text 1 $styfile} text]} {
    close $st
    catch {rm $latexfile}
    error $text
  }

  puts $st $text
  close $st

  catch {latex2dvi $latexfile} output
  catch {exec rm $latexfile $styfile.sty}

  if {![regexp {BOXWIDTH[ ]+([^p]*)pt} $output foo w] ||
    ![regexp {BOXHEIGHT[ ]+([^p]*)pt} $output foo h]} {
      error $output
    }

  return "[[cv] convert ${w}p] [[cv] convert ${h}p]"
}

#
#  Recomputes all latex item bboxes
#

proc updateLatexBboxes {args} {
  global LatexDims LatexDrawingRatio

  [cv] delete sFr
  
  foreach id $args {
    set both [[cv] find withtag \
	      [getTopGroupTags $id]]
    if {[lindex $both 0] == $id} {
      set frameId [lindex $both 1]
    } else {
      set frameId [lindex $both 0]
    }

    lassign [keylget LatexDims $id] w h
    set w [expr $w / $LatexDrawingRatio]
    set h [expr $h / $LatexDrawingRatio]
    
    lassign [[cv] coords $id] xc yc
    set cf [[cv] itemconfigure $id -anchor]
    set anchor [lindex [[cv] itemconfigure $id -anchor] 4]
    if {$anchor == {}} { set anchor [lindex $cf 3] }
    switch $anchor {
      n -
      s -
      center { set x1 [expr $xc - $w/2] }
      nw -
      w -
      sw { set x1 $xc }
      ne -
      e -
      se { set x1 [expr $xc - $w] }
    }
    set x2 [expr $x1 + $w]
    
    switch $anchor {
      nw -
      n -
      ne { set y1 $yc }
      w -
      center -
      e { set y1 [expr $yc - $h/2] }
      sw -
      s -
      se { set y1 [expr $yc - $h] }
    }
    set y2 [expr $y1 + $h]

    [cv] coords $frameId $x1 $y1 $x2 $y2
  }
  redrawFrames
}

#
#  SetLatexAnchor:
#    set selected latex items anchor according to LatexAnchor
#

set anchor2just {
  {nw left} {w left} {sw left}
  {n center} {center center} {s center}
  {ne right} {e right} {se right}
}

proc SetLatexAnchor {} {
  global LatexAnchor anchor2just
  
  foreach id [[cv] find withtag s latex] {
    #  find the associated frameId  #
    set both [[cv] find withtag \
	      [getTopGroupTags $id]]
    if {[lindex $both 0] == $id} {
      set frameId [lindex $both 1]
    } else {
      set frameId [lindex $both 0]
    }

    [cv] itemconfigure $id \
      -anchor $LatexAnchor \
      -justify [keylget anchor2just $LatexAnchor]
    
    lassign [[cv] bbox $id] tx1 ty1 tx2 ty2
    lassign [[cv] coords $frameId] x1 y1 x2 y2
    
  #  Don't use coords cause what we wan't is to ensure  #
  #  that the text bbox is inside the rect.  #

    switch $LatexAnchor {
      nw {
	[cv] move $id [expr $x1 - $tx1] [expr $y1 - $ty1]
      }
      w {
	[cv] move $id \
	  [expr $x1 - $tx1] [expr ($y1 + $y2 - $ty1 - $ty2)/2]
      }
      sw {
	[cv] move $id \
	  [expr $x1 - $tx1] [expr $y2 - $ty2]
      }
      n {
	[cv] move $id \
	  [expr ($x1 + $x2 - $tx1 - $tx2)/2] [expr $y1 - $ty1]
      }
      center {
	[cv] move $id \
	  [expr ($x1 + $x2 - $tx1 - $tx2)/2] [expr ($y1 + $y2 - $ty1 - $ty2)/2]
      }
      s {
	[cv] move $id \
	  [expr ($x1 + $x2 - $tx1 - $tx2)/2] [expr $y2 - $ty2]
      }
      ne {
	[cv] move $id \
	  [expr $x2 - $tx2] [expr $y1 - $ty1]
      }
      e {
	[cv] move $id \
	  [expr $x2 - $tx2] [expr ($y1 + $y2 - $ty1 - $ty2)/2]
      }
      se {
	[cv] move $id \
	  [expr $x2 - $tx2] [expr $y2 - $ty2]
      }
    }
  }
}    
    
##
##  Latex object creation  
##

#  initial cursor positions  #
set LtxC0 {}
#  initial item texts  #
set LtxT0 {}

foreach spec $LatexEnvs {
  set ci [string first ">|<" $spec]
  lappend LtxC0 [max $ci 0]
  regsub {>[|]<} $spec {} spec
  lappend LtxT0 $spec
}

proc clatexStart {x y} {
  global currentStyle LatexText ModeState
  global LtxT0 LtxC0

  saveMsg

  set inc [[cv] find withtag inCreation]
  set cur [lindex [[cv] find withtag current] 0]

  if {($inc != "") && ($inc != $cur)} {
    # an item is in creation and user clicked elsewhere #
    # => end creation #
    if {[clatexEnd] != "done"} {
      return
    }
  }
  
  if {($inc != "") && ($inc == $cur)} {
    # user clicked into the latex item he's creating #
    set i [lsearch -exact $LtxT0 \
	   [lindex [[cv] itemconfigure $inc -text] 4]]
    if {$i >= 0} {
      # user clicked twice to get a different starting environment #
      incr i
      set i [expr $i % [llength $LtxT0]]

      [cv] itemconfigure $inc \
	-text [lindex $LtxT0 $i]
      [cv] icursor $cur [lindex $LtxC0 $i]
    } else {
      [cv] icursor $cur @$x,$y
    }
    return
  }

  ##  No item is being created  ##
  set tags [[cv] gettags $cur]
  if {[lmember $tags "latex"] ||
    [lmember $tags "ltxFr"]} {
      #  user wants to modify an old item  #
      # => destroy the frame item #

      deselect_all
      set both [[cv] find withtag \
		[getTopGroupTags $cur]]
      if {[lmember [[cv] gettags [lindex $both 0]] "latex"]} {
	set textId [lindex $both 0]
	set frameId [lindex $both 1]
      } else {
	set textId [lindex $both 1]
	set frameId [lindex $both 0]
      }

      #  place the text at the exact latex position  #
      lassign [[cv] coords $frameId] x1 y1 x2 y2

      switch [lindex [[cv] itemconfigure $textId -anchor] 4] {
	nw {
	  [cv] coords $textId $x1 $y1
	}
	w {
	  [cv] coords $textId $x1 [expr ($y1+$y2)/2]
	}
	sw {
	  [cv] coords $textId $x1 $y2
	}
	n {
	  [cv] coords $textId [expr ($x1+$x2)/2] $y1
	}
	center {
	  [cv] coords $textId [expr ($x1+$x2)/2] [expr ($y1+$y2)/2]
	}
	s {
	  [cv] coords $textId [expr ($x1+$x2)/2] $y2
	}
	ne {
	  [cv] coords $x2 $y1
	}
	e {
	  [cv] coords $x2 [expr ($y1+$y2)/2]
	}
	se {
	  [cv] coords $x2 $y2
	}
      }
      
      [cv] delete $frameId
      createStart_hook
      [cv] itemconfigure $textId \
	-width 0 \
	-text [keylget LatexText $textId] \
	-font [lindex [[cv] itemconfigure $textId -font] 3]
      
      [cv] addtag inCreation withtag $textId
      [cv] focus $textId
      bind_item_emacs_like $textId
      [cv] icursor $textId @$x,$y
      set ModeState 1
    } else {
      ## New item creation ##
      deselect_all
      createStart_hook

      global LatexAnchor anchor2just
      
      set newitem [[cv] create text $x $y \
		   -anchor $LatexAnchor \
		     -justify [keylget anchor2just $LatexAnchor] \
		     -text [lindex $LtxT0 0] \
		     -tags {inCreation latex}]

      [cv] icursor $newitem [lindex $LtxC0 0]
      [cv] focus $newitem
      bind_item_emacs_like $newitem
      set ModeState 1
    }
}

proc clatexEnd {} {
  global LatexText LatexResolution  ImageFile
  global LtxT0 ModeState

  [cv] focus {}
  set item [[cv] find withtag inCreation]
  if {$item == ""} { return done }
  
  if {[[cv] type $item] != "text"} {
    set ModeState 0
    return
  }

  waitCursor

  # get latex text and position #
  set text [string trim [lindex [[cv] itemconfigure $item -text] 4]]

  # if empty item, done #
  set i [lsearch -exact $LtxT0 $text]
  if {$i >= 0} {
    #  destroy text item  #
    focus .
    [cv] focus {}
    [cv] delete $item
    restoreCursor
    createEnd_hook
    set ModeState 0
    return done
  }

  ##  Compute latex dims  ##
  global LatexDims LatexDrawingRatio

  saveMsg
  msg "Computing latex bounding box..."
  if {[catch {computeLatexDims $text} dims]} {
    restoreCursor
    textBox "Latex error:" $dims "dismiss"
    focus [cv]
    [cv] focus inCreation
    return notdone
  } 

  keylset LatexDims $item $dims

  set bbw [expr [lindex $dims 0] / $LatexDrawingRatio]
  set bbh [expr [lindex $dims 1] / $LatexDrawingRatio]

  lassign [[cv] coords $item] xc yc
  set cf [[cv] itemconfigure $item -anchor]
  set anchor [lindex $cf 4]
  if {$anchor == {}} { set anchor [lindex $cf 3] }
  switch $anchor {
    n -
    s -
    center { set x1 [expr $xc - $bbw/2] }
    nw -
    w -
    sw { set x1 $xc }
    ne -
    e -
    se { set x1 [expr $xc - $bbw] }
  }
  set x2 [expr $x1 + $bbw]

  switch $anchor {
    nw -
    n -
    ne { set y1 $yc }
    w -
    center -
    e { set y1 [expr $yc - $bbh/2] }
    sw -
    s -
    se { set y1 [expr $yc - $bbh] }
  }
  set y2 [expr $y1 + $bbh]
  
  #  item creation  #
  fitTextInRect $item $x1 $y1 $x2 $y2
  set tags [[cv] gettags $item]
  set tgtag [lindex $tags [lsearch -glob $tags tGr*]]
  if {$tgtag == ""} {
    set tgtag tGr.$item
    set gtag Gr.$item
    lappend tags $tgtag $gtag
  } else {
    set gtag [lindex $tags [lsearch -glob $tags Gr*]]
  }

  lappend tags s sOfr
  [cv] itemconfigure $item -tags $tags

  set frameId [[cv] create rectangle $x1 $y1 $x2 $y2 \
	       -outline green -fill green \
		 -tags "$tgtag $gtag s sOfr ltxFr"]

  keylset LatexText $item $text

  restoreMsg
  restoreCursor
  [cv] focus {}
  focus .
  grab release [cv]

  [cv] dtag $item inCreation
  putFrame $tgtag

  [cv] raise ltxFr
  [cv] raise latex
  set ModeState 0
  return done
}


