# mxedit.bindings

# Default bindings for the mxedit widget.  This imports the names
# of various menus in order to establish keyboard bindings and menu
# accelerators in a consistent manner via the mxMenuBind procedure.
#
# This also includes bindings for the search/replace and command entries
#
# Copyright (c) 1992-3 Xerox Corporation.
# Use and copying of this software and preparation of derivative works based
# upon this software are permitted. Any distribution of this software or
# derivative works must comply with all applicable United States export
# control laws. This software is made available AS IS, and Xerox Corporation
# makes no warranty about the software, its performance or its conformity to
# any specification.
#

# mxShowBindings --
#	Pop up a new window to display key bindings.
#	Note that we seem to get them back in reverse order
#	to which they were defined, so mxBindings should
#	define boring ones first.

proc mxShowBindings { args } {
    global mxedit
    set newWindow [mxOpen {}]
    send $newWindow {mxInsert Keystroke\ Bindings:\n}
    send $newWindow {mxInsert ---------\ --------\n}
    if {[llength $args] == 0} {set args [bind $mxedit]}
    foreach binding $args {
	send $newWindow [list mxInsert [format {\n%-20s => "%s"} $binding [bind $mxedit $binding]]]
    }
    send $newWindow mxClean
    send $newWindow {mxSee 0.0}

}

# mxBasicBindings --
#	Set up ASCII input and a couple other bindings for an mxedit widget
#	that probably don't need to be changed on a per-user basis
#	The use of the "Any" modifier is to support folks that
#	like to type with the silly caps lock key down.

proc mxBasicBindings { widget } {

    # Set up ascii input
    mxBindLatin1 $widget
    bind $widget <Return> "mxNewline ; mxSee caret"
    bind $widget <Tab> "$widget ! insert %A"
    bind $widget <BackSpace> "mxDeleteBackChar"

    bind $widget <Lock-Key-Return> "mxNewline ; mxSee caret"
    bind $widget <Lock-Key-Tab> "$widget ! insert %A"
    bind $widget <Lock-Key-BackSpace> "mxDeleteBackChar"
}

# mxClearMouseBindings --
#	Clear all bindings for buttons

proc mxClearMouseBindings { widget } {
    foreach sequence [bind $widget] {
	case $sequence {
	    {*Button* *B1* *B2* *B3*} {
		bind $widget $sequence {}
	    }
	}
    }
}

# mxRestoreMouseBindings --
#	Restore mouse bindings based on the proc variables that
#	identify which procedures to execute

proc mxRestoreMouseBindings { widget } {
    global mxScrollBindingProc mxSelectBindingProc
    foreach proc {mxScrollBindingProc mxSelectBindingProc} {
	set cmd [list [set $proc] $widget]
	# cmd is like {mxScrollBindingProc .mx.edit}
	if [catch $cmd msg] {
	    tkerror "$cmd failed: $msg"
	}
    }
}

# mxClearKeyBindings --
#	Clear all bindings for keystrokes

proc mxClearKeyBindings { widget } {
    # Clear any old menu bindings
    mxMenuUnBindAll

    # Clear any left-over bindings
    foreach sequence [bind $widget] {
	case $sequence {
	    {<Any-Key>} {
		# leave this one alone
	    }
	    {*Key*} {
		bind $widget $sequence {}
	    }
	}
    }
}

# mxRestoreKeyBindings --
#	Restore mouse bindings based on the proc variables that
#	identify which procedures to execute

proc mxRestoreKeyBindings { widget } {
    global mxKeyBindingProc
    set cmd [list [set mxKeyBindingProc] $widget]
    # cmd is like {mxKeyBindings .mx.edit}
    if [catch $cmd msg] {
	tkerror "$cmd failed: $msg"
    }
}

# mxScrollButton --
#	Set up a mouse button as the drag-scroller
#	B is 1, 2, or 3
#	modifier is left out, or Shift, Control, Mod1, etc.
#	speed is the relative motion accelerator

proc mxScrollButton { widget B {modifier NONE} {speed 1} } {
    if {$modifier != "NONE"} {
	set modifier ${modifier}-
    } else {
	set modifier {}
    }
    bind $widget <${modifier}Button-${B}>	{mxScanMark %W %y}
    bind $widget <${modifier}B${B}-Motion>	"mxScanDragto %W %y $speed"
    bind $widget <Any-ButtonRelease-${B}>	{mxScanDone %W}    
}

# mxScrollBindings --
#	Default scroll bindings (middle button)
#
proc mxScrollBindings { widget } {

    mxScrollButton $widget 2
    mxScrollButton $widget 2 Shift 4	;# 4x speedup
    mxScrollButton $widget 3 Shift 10	;# 10x speedup

    # Mark ourself as the scroll binding procedure
    mxSetScrollBindingProc mxScrollBindings
}

# mxOldScrollBindings --
#	Old scroll bindings (shift-left button)
#
proc mxOldScrollBindings { widget } {

    mxScrollButton $widget 1 Shift 4
    mxScrollButton $widget 3 Shift 10

    mxSetScrollBindingProc mxOldScrollBindings
}

# mxOpenwinScrollBindings --
#	Right button scrolling
#
proc mxOpenwinScrollBindings { widget } {

    mxScrollButton $widget 3
    mxScrollButton $widget 3 Shift 4

    mxSetScrollBindingProc mxOpenwinScrollBindings
}

# mxSelectButton --
#	Set up a mouse button as the Selection
#	B is 1, 2, or 3
#	modifier is left out, or Shift, Control, Mod1, etc.
#	how is:
#		Anchor - selection anchored to caret
#		Adjust - adjust anchored selection
#		Float  - selection detached
#		Fadjust - adjust detached selection

proc mxSelectButton { widget B {how Anchor} {modifier NONE} } {
    if {$modifier != "NONE"} {
	set modifier ${modifier}-
    } else {
	set modifier {}
    }
    case $how {
	Anchor {
	    # This button starts an anchored selection (mxCaretSelFoo)
	    bind $widget <${modifier}Button-$B> {
		mxHistory ignore {mxCaretNoSelChar %W @%x,%y}
	    }
	    bind $widget <${modifier}Double-Button-$B> {
		mxHistory ignore {mxCaretSelWord %W @%x,%y}
	    }
	    bind $widget <${modifier}Triple-Button-$B> {
		mxHistory ignore {mxCaretSelLine %W @%x,%y}
	    }
	    bind $widget <${modifier}B${B}-Motion> {
		mxHistory ignore {mxCaretSelMove %W @%x,%y}
	    }
	    bind $widget <${modifier}ButtonRelease-$B> {
		mxHistory ignore {mxMouseUp %W ; mxRedoBarrier %W ; %W markparen @%x,%y}
	    }
	}
	Adjust {
	    # This button extends an existing selection (mxCaretSelAdjust)
	    set cmd [bind $widget <${modifier}Button-$B>]
	    if {! [string match *mxCaretNoSelChar* $cmd]} {
		# Careful not to stomp this part of Anchor binding
		bind $widget <${modifier}Button-$B> {
		    mxHistory ignore {mxCaretSelAdjust %W @%x,%y}
		}
	    }
	    bind $widget <${modifier}B${B}-Motion> {
		mxHistory ignore {mxCaretSelAdjust %W @%x,%y}
	    }
	    bind $widget <${modifier}ButtonRelease-$B> {
		mxHistory ignore {mxMouseUp %W ; mxRedoBarrier %W}
	    }
	}
	Float {
	    # This button starts a detached selection (mxSelFoo)
	    bind $widget <${modifier}Button-$B> {
		mxHistory ignore {mxSelChar %W @%x,%y}
	    }
	    bind $widget <${modifier}Double-Button-$B> {
		mxHistory ignore {mxSelWord %W @%x,%y}
	    }
	    bind $widget <${modifier}Triple-Button-$B> {
		mxHistory ignore {mxSelLine %W @%x,%y}
	    }
	    bind $widget <${modifier}B${B}-Motion> {
		mxHistory ignore {mxSelMove %W @%x,%y}
	    }
	    bind $widget <${modifier}ButtonRelease-$B> {
		mxHistory ignore {mxMouseUp %W ; mxRedoBarrier %W}
	    }
	}
	Fadjust {
	    # This button extends a detached selection (mxSelAdjust)
	    set cmd [bind $widget <${modifier}Button-$B>]
	    if {! [string match *mxSelChar* $cmd]} {
		bind $widget <${modifier}Button-$B> {
		    mxHistory ignore {mxSelAdjust %W @%x,%y}
		}
	    }
	    bind $widget <${modifier}B${B}-Motion> {
		mxHistory ignore {mxSelAdjust %W @%x,%y}
	    }
	    bind $widget <${modifier}ButtonRelease-$B> {
		mxHistory ignore {mxMouseUp %W ; mxRedoBarrier %W}
	    }
	}
    }
}

# mxSelectionBindings --
#	Default mouse behavior (left and right buttons)
#
proc mxSelectionBindings { widget } {
    # Start anchored selection with button 1
    mxSelectButton $widget 1 Anchor
    # Extend anchored selection with B3 or Shift-Button1
    mxSelectButton $widget 1 Adjust Shift
    mxSelectButton $widget 3 Adjust

    # Start a floating selection with Control-Button1, extend with Control-B3
    mxSelectButton $widget 1 Float Control
    mxSelectButton $widget 3 Fadjust Control

    mxSetSelectBindingProc mxSelectionBindings
}

proc mxSelectionBindingsOriginal {widget} {
    bind $widget <ButtonRelease-1> {
	mxHistory ignore "mxMouseUp %W ; mxRedoBarrier %W ; %W markparen @%x,%y"
    }
    bind $widget <ButtonRelease-3> {
	mxHistory ignore "mxMouseUp %W ; mxRedoBarrier %W"
    }
    #
    # Unmodified Button-1 starts a selection anchored to the caret.
    #
    bind $widget <Button-1> {
	mxHistory ignore "mxCaretNoSelChar %W @%x,%y"
    }
    bind $widget <Double-Button-1> {
	mxHistory ignore "mxCaretSelWord %W @%x,%y"
    }
    bind $widget <Triple-Button-1> {
	mxHistory ignore "mxCaretSelLine %W @%x,%y"
    }
    bind $widget <B1-Motion> {
	mxHistory ignore "mxCaretSelMove %W @%x,%y"
    }
    bind $widget <Shift-Button-1> {
	mxHistory ignore "mxCaretSelAdjust %W @%x,%y"
    }
    bind $widget <Button-3> {
	mxHistory ignore "mxCaretSelAdjust %W @%x,%y"
    }
    bind $widget <B3-Motion> {
	mxHistory ignore "mxCaretSelAdjust %W @%x,%y"
    }
    #
    # Control-Button-1 starts a selection detached from the caret.
    #
    bind $widget <Control-Button-1> {
	mxHistory ignore "mxSelChar %W @%x,%y"
    }
    bind $widget <Control-B1-Motion> {
	mxHistory ignore "mxSelMove %W @%x,%y"
    }
    bind $widget <Control-ButtonRelease-1> {
	mxHistory ignore "mxMouseUp %W"
    }
    bind $widget <Control-Double-Button-1> {
	mxHistory ignore "mxSelWord %W @%x,%y"
    }
    bind $widget <Control-Triple-Button-1> {
	mxHistory ignore "mxSelLine %W @%x,%y"
    }
    bind $widget <Control-Shift-Button-1> {
	mxHistory ignore "mxSelAdjust %W @%x,%y"
    }
    bind $widget <Control-Button-3> {
	mxHistory ignore "mxSelAdjust %W @%x,%y"
    }
    bind $widget <Control-B3-Motion> {
	mxHistory ignore "mxSelAdjust %W @%x,%y"
    }

    mxSetSelectBindingProc mxSelectionBindings
}

# mxOpenwinSelectionBindings --
#	OpenWin mouse behavior (Paste key accellerator)
#
proc mxOpenwinSelectionBindings { widget } {
    # Get left-click selection for starters
    mxSelectButton $widget 1 Anchor
    mxSelectButton $widget 1 Adjust	;# Set Adjust *after* Anchor
    mxSelectButton $widget 2 Adjust

    mxSelectButton $widget 1 Float Control
    mxSelectButton $widget 1 Fadjust Control
    mxSelectButton $widget 2 Fadjust Control

    # The Paste key (on my Sun4 keyboard) is F18, plus
    # it gets mapped to Mod5 by the Openwin server

    # Do paste on button-up to allow pseudo-shift-select
    bind $widget <Key-F18> {}				;# Nuke this binding
    bind $widget <KeyRelease-F18> { mxPaste }		;# Move to key-up

    # Mod5-Button-1 starts a selection detached from the caret.
    mxSelectButton $widget 1 Float Mod5
    mxSelectButton $widget 2 Fadjust Mod5
    # and then releasing the paste key causes the insert.
    bind $widget <Mod5-KeyRelease-F18> { mxPaste }

    mxSetSelectBindingProc mxOpenwinSelectionBindings
}

# mxMotifSelectionBindings --
#	Motif-style mouse behavior (left starts and drags selections)
#
proc mxMotifSelectionBindings { widget } {
    # Get left-click selection for starters
    mxSelectButton $widget 1 Anchor
    mxSelectButton $widget 1 Adjust	;# Set Adjust *after* Anchor
    mxSelectButton $widget 1 Adjust Shift

    mxSelectButton $widget 1 Float Control
    mxSelectButton $widget 1 Fadjust Control

    mxMiddlePaste

    mxSetSelectBindingProc mxMotifSelectionBindings
}

# mxRightDrag -
#	Move dragging over to button 3
#
proc mxRightDrag { { widget NONE} } {
    global mxedit
    if {$widget == "NONE"} {
	set widget $mxedit
    }
    bind $widget <Button-3> {%W scan mark %y}
    bind $widget <B3-Motion> {%W scan dragto %y}
    bind $widget <Shift-Button-3> {%W scan mark %y}
    bind $widget <Shift-B3-Motion> {%W scan dragto %y 4}
}

# mxMiddlePaste -
#	Use middle button to paste or move the selection
#
proc mxMiddlePaste { { widget NONE} } {
    global mxedit
    if {$widget == "NONE"} {
	set widget $mxedit
    }
    bind $widget <Button-2> {mxPaste}
    bind $widget <Shift-Button-2> {mxMoveSel}
}

# mxBindings --
# This sets up the keystroke bindings for an mxedit widget.
# This procedure uses mxMenuBind to set up keyboard accelerators.
# mxMenuBind keeps the menu display up-to-date with the keyboard bindings.
#

proc mxBindings { widget } {
    mxKeyBindings $widget
    mxScrollBindings $widget
    mxSelectionBindings $widget
}

proc mxKeyBindings { widget } {

    # Set up ASCII input
    mxBasicBindings $widget

    # Mark-related bindings
    mxBindMarks $widget

    # Keyboard selection
    mxMarkBindings $widget

    # Clear any old menu bindings
    foreach menu {File Edit Search} {
	mxMenuUnBindAll $menu
    }

    ### FILE OPERATIONS
    mxMenuBind <Control-s> File "Save"
    mxMenuBind <Control-q> File "Quit"
    mxMenuBind <Control-o> File "Open new window"
    mxMenuBind <Mod1-o>	   File "Open file SEL"
    mxMenuBind <Mod1-s>	   File "Switch to file SEL"
    mxMenuBind <Escape><s> File "Switch to previous file"
    mxMenuBind <Control-t> File "Switch to tag SEL"
    mxMenuBind <Mod1-t>    File "Open on tag SEL"
    mxMenuBind <Control-g> File "Go to line SEL"

    # Open a new window, using the current word as the tag
    bind $widget <Escape><t> {
	set _t [mxMark [mxMark caret forw 1 word] back 1 char]
	mxSelection set [mxMark $_t back 1 word] $_t
	mxApplyToSelection mxTagOpen
    }

    ### EDITING OPERATIONS

    mxMenuBind <Control-u> Edit "Undo"
    mxMenuBind <Control-a> Edit "Do Again"
    mxMenuBind <Mod1-c>    Edit "Copy SEL"
    mxMenuBind <Control-d> Edit "Cut SEL"
    mxMenuBind <Escape>d   Edit "Delete SEL"
    mxMenuBind <Control-v> Edit "Paste"
    mxMenuBind <Mod1-v>    Edit "Move SEL"

    ### INDENTATION

    # Indent the line at the insertion point
    global mxIndent
    if {![info exists mxIndent]} {
	set mxIndent 4
    }
    mxMenuBind <Control-p>  Edit "Indent line"
    mxMenuBind <Control-y>  Edit "Outdent line"
    mxMenuBind <Mod1-p>     Edit "Indent SEL"
    mxMenuBind <Mod1-y>     Edit "Outdent SEL"

    ### SEARCH/REPLACE OPERATIONS

    mxMenuBind <Control-f> Search "Forward"
    mxMenuBind <Mod1-f>    Search "Forward for SEL"
    mxMenuBind <Control-b> Search "Backward"
    mxMenuBind <Mod1-b>    Search "Backward for SEL"
    mxMenuBind <Control-r> Search "Replace"
    mxMenuBind <Mod1-r>    Search "Replace in SEL"

    # Clear and focus on the replace window
    bind $widget <Escape><r>	{ mxReplaceClear }
    # Make sure the find window is visible
    bind $widget <Escape><f>	{ mxFindClear }

    ### KEYBOARD EDITTING

    # Delete forward character
    bind $widget <Control-l> { mxDeleteForwChar }

    # Delete backward character
    bind $widget <Control-h> { mxDeleteBackChar }

    bind $widget <Delete> { mxDeleteBackChar }

    # Delete backward word
    bind $widget <Control-w> { mxDeleteBackWord }

    # Delete forward word
    bind $widget <Control-n> { mxDeleteForwWord }

    # Delete to end of line
    bind $widget <Control-e> { mxDeleteEndOfLine }

    # Delete the current line
    bind $widget <Control-x> { mxDeleteLine }

    # Open a line below this one
    bind $widget <Control-j> { mxOpenLineBelow }

    # Open a line above this one
    bind $widget <Control-k> { mxOpenLineAbove }

    ### KEYBOARD MOTION

    # Backward one character
    bind $widget <Mod1-h> { mxBack1char }

    # Forward one character
    bind $widget <Mod1-l> { mxForw1char }

    # Down one line
    bind $widget <Mod1-j> { mxDown1Line }
    bind $widget <Mod1-Return> { mxNextLine }

    # Up one line
    bind $widget <Mod1-k> { mxUp1Line }

    # Backward one word
    bind $widget <Mod1-w> { mxBack1Word }

    # Forward one word
    bind $widget <Mod1-n> { mxForw1Word }

    # To end-of-line
    bind $widget <Mod1-e> { mxEndOfLine }

    # Miscellany

    # Change focus to the command entry widget
    bind $widget <Control-c> { mxCommandFocus }

    bind $widget <Shift-Control-question> { mxCaretInfo }

}

proc mxMarkBindings { {widget NONE} } {
    global mxedit

    if {$widget == "NONE"} {
	set widget $mxedit
    }
    bind $widget <Control-space> { mxCaretSelChar %W caret ; set mxMarkActive 1 }
    bind $widget <Mod1-space> { mxExchDotMark }
}

proc mxKillBindings { {widget NONE} } {
    global mxedit
    if {$widget == "NONE"} {
	set widget $mxedit
    }
    bind $widget <Escape>d	{ emacsKill sel }
    bind $widget <Control-e>	{ emacsKill line }
    bind $widget <Control-l>	{ emacsKill char }
    bind $widget <Control-h>	{ emacsKill backchar }

    bind $widget <Mod1-w>	{ emacsKill sel copy }
    bind $widget <Escape>w	{ emacsKill sel copy }

    bind $widget <Escape>v	{ emacsYank }
    bind $widget <Escape>y	{ emacsYankPop }
}

#####################################################################
# These are additions to the entry widget bindings that rightfully
# belong in tk.tcl, but I don't want folks to have to modify that.
# These add mxedit-like bindings to entry widgets.

proc mxEntryBindings {} {
    bind Entry <Control-l> {tk_entryDelRight %W}
    bind Entry <Mod1-h> { tk_entryBack1char %W }
    bind Entry <Mod1-l> { tk_entryForw1char %W }
    bind Entry <Mod1-e> { tk_entryEndOfLine %W }
    
    # Make the left button start a selection,
    # but dragging won't adjust it.  This is to help
    # preserve selections in the main editing window.
    bind Entry <1> {
	%W icursor @%x
	%W select from @%x
	if {[lindex [%W config -state] 4] == "normal"} {focus %W}
    }
    bind Entry <B1-Motion> {}

    # Make the third button extend the selection
    bind Entry <3> { %W select adjust @%x }
    bind Entry <B3-Motion> { %W select to @%x }
    bind Entry <Shift-3> { %W select adjust @%x }
}

#####################################################################
# mxCommandBindings --

proc mxCommandBindings { entry } {
    bind $entry <Return>		{ mxDoCmd }
    bind $entry <Control-Return>	{ mxEditFocus }
    bind $entry <Control-q>		{ mxCommandHide ; mxEditFocus }
    bind $entry <Control-f>		{ mxSearchFocus }
    bind $entry <Control-r>		{ mxReplaceFocus }
}

# mxSearchBindings --
#	Extra bindings for the search entry widget

proc mxSearchBindings { entry } {
    bind $entry <Return>	{ mxFindInner forward }
    bind $entry <Control-f>	{ mxFindInner forward }
    bind $entry <Control-f>	{ mxFindInner forward }
    bind $entry <Control-Return> { mxFindInner backward }
    bind $entry <Control-r>	{ mxReplaceFocus }
    bind $entry <Control-q>	{ mxHideFind ; mxEditFocus }
    bind $entry <Control-c>	{ mxEditFocus }
}

# mxReplaceBindings --
#	Extra bindings for the replace entry widget

proc mxReplaceBindings { entry } {
    bind $entry <Return>	{ mxFindInner replace }
    bind $entry <Control-r>	{ mxFindInner replace }
    bind $entry <Mod1-r>	{ mxFindInner replaceSel }
    bind $entry <Escape>	{ info library ;# Null op }
    bind $entry <Escape><r>	{ mxFindInner replaceEverywhere }
    bind $entry <Control-f>	{ mxSearchFocus }
    bind $entry <Control-q>	{ mxHideFind ; mxEditFocus }
    bind $entry <Control-c>	{ mxEditFocus }
}


# mxBindLatin1 --
# Bind the Latin 1 characters to insert themselves.

proc mxBindLatin1 { widget } {
    bind $widget <Any-Key> {%W ! insert %A}
    bind $widget <Escape> { set _foo ""}
}

# mxBindMarks --
#	Set up bindings to make and move to marks.

proc mxBindMarks { widget } {
    bind $widget <Escape><m> { set _foo "" }
    bind $widget <Mod1-m> { set _foo "" }
    bind $widget <Escape><m><a> {set mxMarka [mxMark caret]}
    bind $widget <Mod1-m><a> {mxSee $mxMarka; mxCaret $mxMarka}
    bind $widget <Escape><m><b> {set mxMarkb [mxMark caret]}
    bind $widget <Mod1-m><b> {mxSee $mxMarkb; mxCaret $mxMarkb}
    bind $widget <Escape><m><c> {set mxMarkc [mxMark caret]}
    bind $widget <Mod1-m><c> {mxSee $mxMarkc; mxCaret $mxMarkc}
    bind $widget <Escape><m><d> {set mxMarkd [mxMark caret]}
    bind $widget <Mod1-m><d> {mxSee $mxMarkd; mxCaret $mxMarkd}
    bind $widget <Escape><m><e> {set mxMarke [mxMark caret]}
    bind $widget <Mod1-m><e> {mxSee $mxMarke; mxCaret $mxMarke}
    bind $widget <Escape><m><f> {set mxMarkf [mxMark caret]}
    bind $widget <Mod1-m><f> {mxSee $mxMarkf; mxCaret $mxMarkf}
    bind $widget <Escape><m><g> {set mxMarkg [mxMark caret]}
    bind $widget <Mod1-m><g> {mxSee $mxMarkg; mxCaret $mxMarkg}
    bind $widget <Escape><m><h> {set mxMarkh [mxMark caret]}
    bind $widget <Mod1-m><h> {mxSee $mxMarkh; mxCaret $mxMarkh}
    bind $widget <Escape><m><i> {set mxMarki [mxMark caret]}
    bind $widget <Mod1-m><i> {mxSee $mxMarki; mxCaret $mxMarki}
    bind $widget <Escape><m><j> {set mxMarkj [mxMark caret]}
    bind $widget <Mod1-m><j> {mxSee $mxMarkj; mxCaret $mxMarkj}
    bind $widget <Escape><m><k> {set mxMarkk [mxMark caret]}
    bind $widget <Mod1-m><k> {mxSee $mxMarkk; mxCaret $mxMarkk}
    bind $widget <Escape><m><l> {set mxMarkl [mxMark caret]}
    bind $widget <Mod1-m><l> {mxSee $mxMarkl; mxCaret $mxMarkl}
    bind $widget <Escape><m><m> {set mxMarkm [mxMark caret]}
    bind $widget <Mod1-m><m> {mxSee $mxMarkm; mxCaret $mxMarkm}
    bind $widget <Escape><m><n> {set mxMarkn [mxMark caret]}
    bind $widget <Mod1-m><n> {mxSee $mxMarkn; mxCaret $mxMarkn}
    bind $widget <Escape><m><o> {set mxMarko [mxMark caret]}
    bind $widget <Mod1-m><o> {mxSee $mxMarko; mxCaret $mxMarko}
    bind $widget <Escape><m><p> {set mxMarkp [mxMark caret]}
    bind $widget <Mod1-m><p> {mxSee $mxMarkp; mxCaret $mxMarkp}
    bind $widget <Escape><m><q> {set mxMarkq [mxMark caret]}
    bind $widget <Mod1-m><q> {mxSee $mxMarkq; mxCaret $mxMarkq}
    bind $widget <Escape><m><r> {set mxMarkr [mxMark caret]}
    bind $widget <Mod1-m><r> {mxSee $mxMarkr; mxCaret $mxMarkr}
    bind $widget <Escape><m><s> {set mxMarks [mxMark caret]}
    bind $widget <Mod1-m><s> {mxSee $mxMarks; mxCaret $mxMarks}
    bind $widget <Escape><m><t> {set mxMarkt [mxMark caret]}
    bind $widget <Mod1-m><t> {mxSee $mxMarkt; mxCaret $mxMarkt}
    bind $widget <Escape><m><u> {set mxMarku [mxMark caret]}
    bind $widget <Mod1-m><u> {mxSee $mxMarku; mxCaret $mxMarku}
    bind $widget <Escape><m><v> {set mxMarkv [mxMark caret]}
    bind $widget <Mod1-m><v> {mxSee $mxMarkv; mxCaret $mxMarkv}
    bind $widget <Escape><m><w> {set mxMarkw [mxMark caret]}
    bind $widget <Mod1-m><w> {mxSee $mxMarkw; mxCaret $mxMarkw}
    bind $widget <Escape><m><x> {set mxMarkx [mxMark caret]}
    bind $widget <Mod1-m><x> {mxSee $mxMarkx; mxCaret $mxMarkx}
    bind $widget <Escape><m><y> {set mxMarky [mxMark caret]}
    bind $widget <Mod1-m><y> {mxSee $mxMarky; mxCaret $mxMarky}
    bind $widget <Escape><m><z> {set mxMarkz [mxMark caret]}
    bind $widget <Mod1-m><z> {mxSee $mxMarkz; mxCaret $mxMarkz}
}


