# Bindings for browsing (no editing). Includes scrolling, traversal, selection


# Selection code

proc select_next_line {t} {
	if {[catch "$t index sel.last"]} { set start insert
	} elseif {([$t compare insert < sel.first]) ||
		  ([$t compare insert > sel.last])} {
		$t tag remove sel sel.first sel.last
		set start insert
	} else {set start sel.first}
	if {($start == "insert")} {	set near $start
	} else { set near sel.last}
	if {([$t get $near] == "\n")} {
		$t tag add sel $start "$near +1 chars"
		} else {$t tag add sel $start "$near lineend"}
	move_insert $t sel.last
}

proc select_region {t} {
	if {![catch {set m [$t index mark]}]} {
		if {[$t compare $m <= insert]} {
			set start $m
			set end insert
		} else {set start insert
			set end $m}
		catch {$t tag remove sel sel.first sel.last}
		$t tag add sel $start $end
	} else {beep
}}

proc select_all {t} {
	catch "$t tag remove sel sel.first sel.last"
	$t tag add sel 1.0 end
}


# Cursor movement code

# Moves insert specified by index.
proc move_insert {t index} {
	if {[$t compare insert == $index]} {beep}
	$t mark set insert $index
	$t yview -pickplace insert
}

# Used to naintain column while traversing across lines that are too short.
set column 0

proc adjacent_line {t d} {
	global column
	scan [$t index insert] "%d.%d" r c
	if {[$t compare insert == "insert $d lines"]} {
		if {([string first "+" $d] != -1)} {
			$t mark set insert end ; beep
		} else {$t mark set insert 1.0 ; beep}
	} elseif {($c < $column) && [$t compare insert == {insert lineend}]} {
		$t mark set insert "insert $d lines linestart"
		if {[$t compare "insert +$column chars" <= "insert lineend"]} {
			$t mark set insert "insert +$column chars"
		} else {$t mark set insert "insert lineend"}
	} elseif {[$t compare "insert $d lines" == "insert $d lines lineend"]} {
		$t mark set insert "insert $d lines"
		set column $c
	} else {$t mark set insert "insert $d lines"}

	if {[$t compare insert != {insert lineend}]} {	set column 0}
	$t yview -pickplace insert
}

proc center_cursor {t {v ""}} {
	if {($v == "")} {set v [lindex [$t configure -height] 4]}
	$t yview "insert -[expr "$v/2"] lines"
}

proc is_cursor_on_screen {t {v ""}} {
	if {($v == "")} {set v [lindex [$t configure -height] 4]}
	if {([$t compare insert < @0,0]) ||
	    ([$t compare insert >= "@0,0 +$v lines"])} {
		return 0} else {return 1
}}

proc ensure_cursor_on_screen {t {v ""}} {
	if {($v == "")} {set v [lindex [$t configure -height] 4]}
	if {![is_cursor_on_screen $t $v]} {
		$t mark set insert "@0,0 +[expr "$v/2"] lines"
}}

proc end_of_page {t} {
	set v [lindex [$t configure -height] 4]
	move_insert $t "@0,0 +$v lines -1 chars"
}

proc page_down {t} {
	set v [lindex [$t configure -height] 4]
	$t yview "@0,0 +[incr v -1] lines"
	ensure_cursor_on_screen $t $v
}

proc page_up {t} {
	set v [lindex [$t configure -height] 4]
	$t yview "@0,0 -[incr v -1] lines"
	ensure_cursor_on_screen $t $v
}


# Misc. browsing functions

proc exchange_dot_and_mark {t} {
	if {[catch {set m [$t index mark]}]} {beep ; return}
	$t mark set mark insert
	$t mark set insert $m
	$t yview -pickplace insert
}

proc goto_where {t l e} {
	set where [$e get]
	if {[regexp {[0-9]*} $where howmuch] && ($howmuch == $where)} {
		set where "$where.0 linestart"
	}
	destroy_f_entry $t $l $e
	move_insert $t $where
}

proc goto {t f} {
	create_f_entry $t $f.gotol $f.gotoe
	$f.gotol configure -text "Goto:"
	bind $f.gotoe <Return> "goto_where $t $f.gotol $f.gotoe"
	bind $f.gotoe <KP_Enter> "[bind $f.gotoe <Return>]"
}

proc quit_beth {} {
	global quit_hook
	if {[info exists quit_hook]} {eval $quit_hook}
	exit
}


# Browse bindings. f is a frame widget to put messages in.
# q is a button that exits.
proc browsebind {f} {
	bind Text <Key> {if {[regexp . %A]} {beep}}
	bind Text <Return> {beep}
	bind Text <KP_Enter> {beep}
	bind Text <Delete> {beep}
	bind Text <Control-Key> {if {[regexp . %A]} {beep}}
	bind Text <Meta-Key> {if {[regexp . %A]} {beep}}
	bind Text <Control-Meta-Key> {if {[regexp . %A]} {beep}}

	bind Text <Control-space> "%W mark set mark insert"
	bind Text <Control-a> "move_insert %W {insert linestart}"
	bind Text <Control-b> "move_insert %W {insert -1 chars}"
	bind Text <Meta-b> "move_insert %W {insert -1 chars wordstart}"
	bind Text <Control-c> "%W tag add sel 1.0 ; %W tag remove sel 1.0 end"
	bind Text <Control-e> "move_insert %W {insert lineend}"
	bind Text <Control-f> "move_insert %W {insert +1 chars}"
	bind Text <Meta-f> "move_insert %W {insert wordend}"
	bind Text <Meta-g> "goto %W $f"
	bind Text <Control-g> "beep ;
				catch \{destroy_f_entry %W $f.gotol $f.gotoe\}"
	bind Text <Control-K> "select_next_line %W"
	bind Text <Control-l> "center_cursor %W"
	bind Text <Control-n> "adjacent_line %W +1"
	bind Text <Control-p> "adjacent_line %W -1"
	bind Text <Meta-q> "quit_beth"
	bind Text <Control-v> "page_down %W"
	bind Text <Meta-v> "page_up %W"
	bind Text <Control-W> "select_region %W"
	bind Text <Meta-w> "select_all %W"
	bind Text <Control-x> "exchange_dot_and_mark %W"
	bind Text <Control-z> "%W yview {@0,0 +1 lines}
			ensure_cursor_on_screen %W"
	bind Text <Meta-z> "%W yview {@0,0 -1 lines}
			ensure_cursor_on_screen %W"
	bind Text <Meta-comma> "move_insert %W @0,0"
	bind Text <Meta-period> "end_of_page %W"
	bind Text <Meta-less> "move_insert %W 1.0"
	bind Text <Meta-greater> "move_insert %W end"

# Duplicate bindings
	bind Text <Control-k> [bind Text <Control-K>]
	bind Text <Control-w> [bind Text <Control-W>]
	bind Text <space> [bind Text <Control-v>]
	bind Text <Prior> [bind Text <Meta-v>]
	bind Text <Shift-Prior> [bind Text <Meta-less>]
	bind Text <Next> [bind Text <Control-v>]
	bind Text <Shift-Next> [bind Text <Meta-greater>]
	bind Text <Up> [bind Text <Control-p>]
	bind Text <Shift-Up> [bind Text <Meta-comma>]
	bind Text <Down> [bind Text <Control-n>]
	bind Text <Shift-Down> [bind Text <Meta-period>]
	bind Text <Left> [bind Text <Control-b>]
	bind Text <Shift-Left> [bind Text <Control-a>]
	bind Text <Right> [bind Text <Control-f>]
	bind Text <Shift-Right> [bind Text <Control-e>]
	bind Text <Select> [bind Text <Control-w>]
	bind Text <Shift-Select> [bind Text <Meta-w>]
}

browsebind $frame
