# Binding procedures to protect text with certain tags.

# Some tag routines

# Searches ranges for index. More efficient than list search, since ranges
# is ordered.
proc text_list_bisearch {t index ranges} {
	set l [llength $ranges]
	if {[$t compare $index == [lindex $ranges 0]]} {return 0}
	if {[$t compare $index == [lindex $ranges [expr $l - 1]]]} {return [expr $l - 1]}

	set l [expr $l / 2]
	set i $l
	set r [lindex $ranges $i]
	while {[$t compare $index != $r]} {
		if {$l != 1} {set l [expr $l / 2]}
		if {[$t compare $index < $r]} {	incr i -$l
		} else {			incr i $l}
		set r [lindex $ranges $i]
	}
	return $i
}

# Line $t tag nextrange, but returns the range whose start is previous to index
proc tag_prevrange {t tag index} {
	set ranges [$t tag ranges $tag]
	set nextrange [$t tag nextrange $tag "$index +1c"]
	if {$nextrange == ""} {
		set i [llength $ranges]
	} else {set i [text_list_bisearch $t [lindex $nextrange 0] $ranges]}
	return [lrange $ranges [expr $i - 2] [expr $i - 1]]
}

# Returns the range covering $index, or "" if tag does not exist on index.
proc tag_thisrange {t tag index} {
	if {([lsearch [$t tag names $index] $tag] < 0) &&
		([lsearch [$t tag names "$index -1c"] $tag] < 0)} {return ""}
	return [tag_prevrange $t $tag $index]
}

# Returns the range covering $index, or "" if tag does not exist on index.
proc old_tag_thisrange {t tag index} {
	set start $index
	if {([lsearch [$t tag names $index] $tag] < 0) ||
		([lsearch [$t tag names "$index -1c"] $tag] < 0)} {return ""}
	set index [$t index "$index -1 lines"]
	set range [$t tag nextrange $tag $index $start]
	while {($range == "")} {
		set new_index [$t index "$index -1 lines"]
		if {[$t compare $index == $new_index]} {
			set range [$t tag nextrange 1.0]
			break
		}
		set index $new_index
		set range [$t tag nextrange $tag $index $start]
	}
	while {[$t compare [lindex $range 1] > $start]} {
		set index [lindex $range 1]
		set range [$t tag nextrange $tag $index $start]
}}


# Like the tag ranges command, but clips the ranges by start and end.
proc tag_ranges {t tag start end} {
	set result ""
	set trace $start
	if {[lsearch [$t tag names $start] $tag] >= 0} {
		lappend result $start
		set range [tag_thisrange $t $tag $start]
		if {[$t compare [lindex $range 1] >= $end]} {
			lappend result $end
			return $result
		} else {lappend $result [lindex $range 1]
			set trace [lindex $range 1]
	}}

	while {[set range [$t tag nextrange $tag $trace $end]] != ""} {
		lappend result [lindex $range 0]
		if {[$t compare [lindex $range 1] >= $end]} {
			lappend result $end
			return $result
		} else {lappend result [lindex $range 1]
			set trace [lindex $range 1]
	}}
	return $result
}

# Copies text from t1: start end to t2 starting at new_start
# Also copies all tags.
proc copy_region {t1 t2 start end new_start} {
	set start [$t1 index $start]
	set end [$t1 index $end]
	set new_start [$t2 index $new_start]
	$t2 insert $new_start [$t1 get $start $end]
	foreach tag [$t1 tag names] {
		set ranges [tag_ranges $t1 $tag $start $end]
		set l [llength $ranges]
		set trace $start	 ; set new_trace $new_start
		for {set i 0} {$i < $l} {incr i 2} {
			set r1 [lindex $ranges $i]
			set offset [string length [$t1 get $trace $r1]]
			set trace [$t1 index "$trace + $offset c"]
			set new_trace [$t2 index "$new_trace + $offset c"]
			set new_first $new_trace

			set r2 [lindex $ranges [expr $i + 1]]
			set offset [string length [$t1 get $trace $r2]]
			set trace [$t1 index "$trace + $offset c"]
			set new_trace [$t2 index "$new_trace + $offset c"]
			set new_last $new_trace

			$t2 tag add $tag $new_first $new_last
}}}


# Preventing selection.
# Tags that shouldn't be selected over should be appended to
# prevent_select_tags
set prevent_select_tags ""

# Adjusts selection to not include a tagged text at either end.
proc prevent_select {t} {
	global prevent_select_tags
	foreach marker {sel.first sel.first-1c sel.last sel.last-1c} {
	foreach tag $prevent_select_tags {
	if {[lsearch [$t tag names $marker] $tag] >= 0} {
		set list [tag_thisrange $t $tag "$marker linestart"]
		$t tag remove sel [lindex $list 0] "[lindex $list 1] +1c"
}}}}

# Bind tag to not be partially selected. (It can still come inside a range
# of selected text, but not at either end)
proc prevent_select_bind {} {
	global Keys
	parse_bindings Text \
"S-B1-Motion S-Button-1 B1-Motion Triple-Button-1 Double-Button-1 Button-1 \
 C-K $Keys(C_W) $Keys(M_w)"	{+catch {prevent_select %W}}

	if {[info globals regions_defined] != ""} {parse_bindings Text \
C-j				{+catch {prevent_select %W}}}

	if {[info globals balancebind_loaded] != ""} {parse_bindings Text \
M-j				{+catch {prevent_select %W}}}
}


prevent_select_bind


# Moving prevention
# Tags that shouldn't be moved over should be appended
# to prevent_move_tags
set prevent_move_tags ""

# If insert is over a char with tag on it, move it to the end of the tag.
# If back is not empty, move it to the beginning instead.
# Assumes tag begins at linestart and gives an extra space of buffering.
proc prevent_move {t {back ""}} {
	global prevent_move_tags
	if {[winfo class $t] != "Text"} {return}
	foreach tag $prevent_move_tags {
		if {([lsearch [$t tag names insert] $tag] < 0) &&
			([lsearch [$t tag names insert-1c] $tag] < 0)} {return}
		if {$back != ""} {
			$t mark set insert "[lindex [tag_thisrange $t $tag \
						"insert linestart"] 0]-1c"
		} else {$t mark set insert "[lindex [tag_thisrange $t $tag \
						"insert linestart"] 1]+1c"}
}}

# Prevent text with tag to be traversed over by keyword traversal or mouse.
proc prevent_move_bind {} {
	global Keys
	parse_bindings Text \
"B1-Motion Triple-Button-1 Double-Button-1 Button-1 $Keys(C_a) $Keys(C_e) \
 $Keys(C_f) M-f C-l C-x $Keys(M_comma) $Keys(M_period)" \
			 	{+prevent_move %W} \
"$Keys(C_b) M-b"		{+prevent_move %W back}
	parse_bindings all \
"$Keys(C_n) $Keys(C_p) $Keys(C_v) $Keys(M_v) C-z M-z $Keys(M_less) \
 $Keys(M_greater)"		{+prevent_move %W}

	if {[info globals balancebind_loaded] != ""} {parse_bindings Text \
"M-A M-E"			{+prevent_move %W}}
	if {[info globals regions_defined] != ""} {parse_bindings Text \
"C-A C-E C-N"			{+prevent_move %W} \
C-P				{+prevent_move %W back}
		if {[info globals collapsebind_defined] != ""} {parse_bindings Text \
C-O				{+prevent_move %W}
}}}


prevent_move_bind


if $edit_flag {
# Preventing edit
# Tags that shouldn't be editable should be appended to
# prevent_edit_tags
set prevent_edit_tags ""

# Protects all chars under tag from being edited. Cmd is the editing that
# gets done, and it works on the range of chars between start and end.
# End defaults to start if unspecified.
# End and start may not be adjacent to any chars with tag.
proc prevent_edit {t cmd {start insert} {end ""}} {
	if {[catch {$t index $start}]} {beep ; return 0}
	if {$end == ""} {set end $start}
# Make sure start isn't tagged.
	global prevent_edit_tags
	foreach tag $prevent_edit_tags {
		if {[lsearch [$t tag names "$start-1c"] $tag] >= 0} {
			beep ; return 0}
# Make sure nothing between start and end is tagged.
		set indices [$t tag nextrange $tag $start]
		if {[llength $indices] == 2} {
			if {[$t compare [lindex $indices 0] < "$end +1c"]} {
				beep ; return 0}}
	}
	eval $cmd
	return 1
}

# Keeps a tagged text from being killed.
proc prevent_kill_region {t} {
	if {![catch {set m [$t index mark]}]} {
		if {[$t compare $m <= insert]} {
			set start $m
			set end insert
		} else {set start insert
			set end $m}
		prevent_edit $t "kill_region $t" $start $end
	} else {beep ; return 0}
}

# Keeps a character from being inserted within a tag.
proc prevent_self_insert {t c} {
	if {(![regexp . $c])} {return}
	global prevent_edit_tags overwrite_mode
	foreach tag $prevent_edit_tags {
		if {[lsearch [$t tag names insert] $tag] >= 0} {beep ; return 0}
		if {[lsearch [$t tag names insert-1c] $tag] >= 0} {beep ; return 0}
		if {$overwrite_mode && ([$t get insert+1c] != "\n") &&
			([lsearch [$t tag names insert+2c] $tag] >= 0)} {
			beep ; return 0}
	}
	$t insert insert $c
	foreach tag $prevent_edit_tags {$t tag remove $tag insert-1c}
	if {$overwrite_mode && ([$t get insert+1c] != "\n")} {$t delete insert}
	global modified	;	set modified 1
	$t yview -pickplace insert
}

# Since many functions use delete_range, we redefine it.
rename delete_range normal_delete_range
proc delete_range {t start end {dont_undo ""}} {
	prevent_edit $t [list normal_delete_range $t $start $end $dont_undo] $start $end
}

# Since several functions use this, we redefine it
rename filter_word normal_filter_word
proc filter_word {t filter} {
	prevent_edit $t	[list normal_filter_word $t $filter] insert {insert wordend}
}


# Prevent any chars with tag from being edited.
proc prevent_edit_bind {} {
	global Keys
# Protect all text before prompt from editing commands.
	parse_bindings Text \
Key		"prevent_self_insert %W %A" \
Tab		"prevent_self_insert %W %A" \
C-k		"prevent_edit %W [list [bind Text <Control-k>]] \
		 insert {insert lineend +1c}" \
$Keys(C_m)	"prevent_edit %W [list [bind Text <Control-m>]]" \
C-o		"prevent_edit %W [list [bind Text <Control-o>]]" \
C-q		"prevent_edit %W [list [bind Text <Control-q>]]" \
C-t		"prevent_edit %W [list [bind Text <Control-t>]] \
		 insert-1c insert+1c" \
M-t		"prevent_edit %W [list [bind Text <Meta-t>]] \
		 {insert -2c wordstart -2c wordstart} {insert wordend}" \
C-w		"prevent_kill_region %W" \
$Keys(C_y)	"prevent_edit %W [list [bind Text <Control-y>]]"

	if {[info globals regions_defined] != ""} {parse_bindings Text \
C-D	"prevent_edit %W [list [bind Text <Control-D>]] \
		insert \[region_end %W insert\]" \
C-H	"prevent_edit %W [list [bind Text <Control-H>]] \
		\[region_prev %W insert\] insert" \
C-U	"prevent_edit %W [list [bind Text <Control-U>]] \
		\[region_prev %W insert\] \[region_end %W insert\]"}

	if {[info globals balancebind_loaded] != ""} {parse_bindings Text \
M-D	"prevent_edit %W [list [bind Text <Meta-D>]]" \
M-H	"prevent_edit %W [list [bind Text <Meta-H>]]" \
M-U	"prevent_edit %W [list [bind Text <Meta-U>]]"
# A very inadequate solution, but we don't know the boundaries until another
# keypress.
}}

prevent_edit_bind
}
