/*
 *=============================================================================
 *                                  tSippUtil.c
 *-----------------------------------------------------------------------------
 * Utility procedures.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippUtil.c,v 5.8 1996/10/26 19:00:11 markd Exp $
 *============================================================================
 */

#include "tSippInt.h"
#include <varargs.h>

extern double pow ();

/*
 * Implied column 3 of the Transf_mat.
 */
static double matrixCol3 [] = {0.0, 0.0, 0.0, 1.0};

/*=============================================================================
 * TSippConvertFraction --
 *   Convert a string that should be in the range 0 to 1 to a double and do
 *   error checking.
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o strBuf (I) - String containing the number.
 *   o valuePtr (O) - The converted number is returned here.
 * Returns:
 *   TRUE if the number is valid, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
bool
TSippConvertFraction (tSippGlobPtr, strBuf, valuePtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *strBuf;
    double       *valuePtr;
{
    if (Tcl_GetDouble (tSippGlobPtr->interp, strBuf, valuePtr) != TCL_OK)
        return FALSE;
    if ((*valuePtr < 0.0) || (*valuePtr > 1.0)) {
        Tcl_AppendResult (tSippGlobPtr->interp,
                          "Expected a number in the range 0..1, got: ",
                          strBuf, (char *) NULL);
        return FALSE;
    }
    return TRUE;
}

/*=============================================================================
 * TSippConvertUnsignedDbl --
 *   Convert a string that should be in a double >= 0.0 with error checking.
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o strBuf (I) - String containing the number.
 *   o valuePtr (O) - The converted number is returned here.
 * Returns:
 *   TRUE if the number is valid, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
bool
TSippConvertUnsignedDbl (tSippGlobPtr, strBuf, valuePtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *strBuf;
    double       *valuePtr;
{
    if (Tcl_GetDouble (tSippGlobPtr->interp, strBuf, valuePtr) != TCL_OK)
        return FALSE;
    if (*valuePtr < 0.0) {
        Tcl_AppendResult (tSippGlobPtr->interp,
                          "Expected a number >= 0.0, got: ",
                          strBuf, (char *) NULL);
        return FALSE;
    }
    return TRUE;
}

/*=============================================================================
 * TSippConvertPosUnsigned --
 *   Convert a string that should be in a integer > 0 with error checking.
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o strBuf (I) - String containing the number.
 *   o valuePtr (O) - The converted number is returned here.
 * Returns:
 *   TRUE if the number is valid, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
bool
TSippConvertPosUnsigned (tSippGlobPtr, strBuf, valuePtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *strBuf;
    unsigned     *valuePtr;
{
    if (Tcl_GetUnsigned (tSippGlobPtr->interp, strBuf, valuePtr) != TCL_OK)
        return FALSE;
    if (*valuePtr == 0) {
        Tcl_AppendResult (tSippGlobPtr->interp, "Expected a number > 0, got: ",
                          strBuf, (char *) NULL);
        return FALSE;
    }
    return TRUE;
}

/*=============================================================================
 * TSippConvertColor --
 *   Convert a three element list of color values to float point numbers and
 * validate that they are in range.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o listStr (I) - Tcl list containing the color values to convert.
 *   o colorPtr (O) - The converted color values are returned here.
 * Returns:
 *   TRUE if the list and numbers are valid, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
bool
TSippConvertColor (tSippGlobPtr, listStr, colorPtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *listStr;
    Color        *colorPtr;
{
    int    listArgc;
    char **listArgv;

    if (Tcl_SplitList (tSippGlobPtr->interp, listStr, &listArgc,
                       &listArgv) != TCL_OK)
        return FALSE;

    if (listArgc != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp,
                          "color must be a list of three elements, got \"",
                          listStr, "\"", (char *) NULL);
        goto errorCleanup;
    }
    if (!TSippConvertFraction (tSippGlobPtr, listArgv [0], &colorPtr->red))
        goto errorCleanup;
    if (!TSippConvertFraction (tSippGlobPtr, listArgv [1], &colorPtr->grn))
        goto errorCleanup;
    if (!TSippConvertFraction (tSippGlobPtr, listArgv [2], &colorPtr->blu))
        goto errorCleanup;

    sfree (listArgv);
    return TRUE;
    
  errorCleanup:
    sfree (listArgv);
    return FALSE;
}

/*=============================================================================
 * TSippFormatColor --
 *   Format a color as a list of floating point color values.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o color (I) - The color to convert.
 *   o listStr (I) - Tcl list of color values is returned here, it must have
 *     3 * TCL_DOUBLE_SPACE characters of space.
 *-----------------------------------------------------------------------------
 */
void
TSippFormatColor (tSippGlobPtr, color, listStr)
    tSippGlob_t  *tSippGlobPtr;
    Color         color;
    char         *listStr;
{
    char  *bufPtr;

    bufPtr = listStr;
    Tcl_PrintDouble (tSippGlobPtr->interp,
                     color.red,
                     bufPtr);
    bufPtr = bufPtr + strlen (bufPtr);
    *bufPtr++ = ' ';

    Tcl_PrintDouble (tSippGlobPtr->interp,
                     color.grn,
                     bufPtr);
    bufPtr = bufPtr + strlen (bufPtr);
    *bufPtr++ = ' ';

    Tcl_PrintDouble (tSippGlobPtr->interp,
                     color.blu,
                     bufPtr);
}

/*=============================================================================
 * TSippFormatIntColor --
 *   Format a integer color as a list of floating point color values.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o color (I) - The color to convert.
 *   o listStr (I) - Tcl list of color values is returned here, it must have
 *     3 * TCL_DOUBLE_SPACE characters of space.
 *-----------------------------------------------------------------------------
 */
void
TSippFormatIntColor (tSippGlobPtr, color, listStr)
    tSippGlob_t  *tSippGlobPtr;
    u_char       *color;
    char         *listStr;
{
    char  *bufPtr;

    bufPtr = listStr;
    Tcl_PrintDouble (tSippGlobPtr->interp,
                     ((double) color [TSIPP_RED]) / 255.0,
                     bufPtr);
    bufPtr = bufPtr + strlen (bufPtr);
    *bufPtr++ = ' ';

    Tcl_PrintDouble (tSippGlobPtr->interp,
                     ((double) color [TSIPP_GREEN]) / 255.0,
                     bufPtr);
    bufPtr = bufPtr + strlen (bufPtr);
    *bufPtr++ = ' ';

    Tcl_PrintDouble (tSippGlobPtr->interp,
                     ((double) color [TSIPP_BLUE]) / 255.0,
                     bufPtr);
}

/*=============================================================================
 * TSippConvertOpacity --
 *   Convert a list specifying opacity values to float point numbers and
 *  validate that they are in range.  The list may be either a single value, or
 *  a set of red, blue and green values.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o listStr (I) - Tcl list containing the opacity values to convert.
 *   o opacityPtr (O) - The converted opacity values are returned here.
 * Returns:
 *   TRUE if the list and numbers are valid, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
bool
TSippConvertOpacity (tSippGlobPtr, listStr, opacityPtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *listStr;
    Color        *opacityPtr;
{
    int    listArgc;
    char **listArgv;

    if (Tcl_SplitList (tSippGlobPtr->interp, listStr, &listArgc,
                       &listArgv) != TCL_OK)
        return FALSE;

    if (!((listArgc == 1) || (listArgc == 3))) {
        Tcl_AppendResult (tSippGlobPtr->interp, "opacity must be a single ",
                          "number or a list of three elements, got \"",
                          listStr, "\"", (char *) NULL);
        goto errorCleanup;
    }
    if (listArgc == 1) {
        if (!TSippConvertFraction (tSippGlobPtr, listArgv [0],
                                   &opacityPtr->red))
            goto errorCleanup;
        opacityPtr->blu = opacityPtr->grn = opacityPtr->red;
    } else {
        if (!TSippConvertFraction (tSippGlobPtr, listArgv [0],
                                   &opacityPtr->red))
            goto errorCleanup;
        if (!TSippConvertFraction (tSippGlobPtr, listArgv [1],
                                   &opacityPtr->grn))
            goto errorCleanup;
        if (!TSippConvertFraction (tSippGlobPtr, listArgv [2],
                                   &opacityPtr->blu))
            goto errorCleanup;
    }
    sfree (listArgv);
    return TRUE;
    
errorCleanup:
    sfree (listArgv);
    return FALSE;
}

/*=============================================================================
 * TSippConvertAngleRad --
 *   Convert a string that should represents a angle to radians.   The value
 *   may be in degrees or radians.  If it is prefixed with 'D' it is taken
 *   to be degrees, if it is prefixed with 'R', it is taken to be radians.
 *   If neither is specified, radians is assumed.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o strBuf (I) - String containing the angle
 *   o valuePtr (O) - The converted angle is returned here (in radians).
 * Returns:
 *   TRUE if the number is valid, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
bool
TSippConvertAngleRad (tSippGlobPtr, strBuf, valuePtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *strBuf;
    double       *valuePtr;
{
    char *strStartPtr = strBuf;

    if ((strStartPtr [0] == 'D') || (strStartPtr [0] == 'R'))
        strStartPtr++;

    if (Tcl_GetDouble (tSippGlobPtr->interp, strStartPtr, valuePtr) != TCL_OK)
        return FALSE;

    if (strBuf [0] == 'D')
       *valuePtr = (*valuePtr * M_PI) / 180.0;
    return TRUE;
}

/*=============================================================================
 * TSippConvertAngleDeg --
 *   Convert a string that should represents a angle to degrees.   The value
 *   may be in degrees or radians.  If it is prefixed with 'D' it is taken
 *   to be degrees, if it is prefixed with 'R', it is taken to be radians.
 *   If neither is specified, degrees is assumed.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o strBuf (I) - String containing the angle
 *   o valuePtr (O) - The converted angle is returned here (in degress).
 * Returns:
 *   TRUE if the number is valid, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
bool
TSippConvertAngleDeg (tSippGlobPtr, strBuf, valuePtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *strBuf;
    double       *valuePtr;
{
    char *strStartPtr = strBuf;

    if ((strStartPtr [0] == 'D') || (strStartPtr [0] == 'R'))
        strStartPtr++;

    if (Tcl_GetDouble (tSippGlobPtr->interp, strStartPtr, valuePtr) != TCL_OK)
        return FALSE;

    if (strBuf [0] == 'R')
       *valuePtr = (*valuePtr * 180.0) / M_PI;
    return TRUE;
}

/*=============================================================================
 * TSippConvert2DPoint --
 *   Convert a list of two numbers repersenting a point in 2D space to floaing
 * point numbers.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o listStr (I) - A Tcl list containing the vector values to convert.
 *   o xCoordPtr (O) - The X coordinate is returned here.
 *   o yCoordPtr (O) - The Y coordinate is returned here.
 * Returns:
 *   TRUE if the list and numbers are valid, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
bool
TSippConvert2DPoint (tSippGlobPtr, listStr, xCoordPtr, yCoordPtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *listStr;
    double       *xCoordPtr;
    double       *yCoordPtr;
{
    int    listArgc;
    char **listArgv;

    if (Tcl_SplitList (tSippGlobPtr->interp, listStr, &listArgc,
                       &listArgv) != TCL_OK)
        return FALSE;

    if (listArgc != 2) {
        Tcl_AppendResult (tSippGlobPtr->interp, 
                          "a 2D point must be a list of two elements, got \"",
                          listStr, "\"", (char *) NULL);
        goto errorCleanup;
    }
   
    if (Tcl_GetDouble (tSippGlobPtr->interp, listArgv [0],
                       xCoordPtr) != TCL_OK)
       goto errorCleanup;
    if (Tcl_GetDouble (tSippGlobPtr->interp, listArgv [1],
                       yCoordPtr) != TCL_OK)
       goto errorCleanup;

    sfree (listArgv);
    return TRUE;

errorCleanup:
    sfree (listArgv);
    return FALSE;
}

/*=============================================================================
 * TSippConvertVertex --
 *   Convert a list contain a list of a triple of numbers repersenting a
 * vertex X, Y and Z into vertex.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o listStr (I) - A Tcl list containing the vertex values to convert.
 *   o vertexPtr (O) - The converted vertex is returned here.
 * Returns:
 *   TRUE if the list and numbers are valid, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
bool
TSippConvertVertex (tSippGlobPtr, listStr, vertexPtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *listStr;
    Vector       *vertexPtr;
{
    int      vertexArgc;
    char   **vertexArgv;

    if (Tcl_SplitList (tSippGlobPtr->interp, listStr, &vertexArgc,
                       &vertexArgv) != TCL_OK)
        return FALSE;

    if (vertexArgc != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp, "vertex or vector must be",
                         " a list of three elements, got \"", listStr,
                          "\"", (char *) NULL);
        goto errorCleanup;
    }

    if (Tcl_GetDouble (tSippGlobPtr->interp, vertexArgv [0],
                       &vertexPtr->x) != TCL_OK)
       goto errorCleanup;
    if (Tcl_GetDouble (tSippGlobPtr->interp, vertexArgv [1],
                       &vertexPtr->y) != TCL_OK)
       goto errorCleanup;
    if (Tcl_GetDouble (tSippGlobPtr->interp, vertexArgv [2],
                       &vertexPtr->z) != TCL_OK)
       goto errorCleanup;
    sfree (vertexArgv);
    return TRUE;

errorCleanup:
    sfree (vertexArgv);
    return FALSE;
}

/*=============================================================================
 * TSippConvertMatrix --
 *   Convert a list of list of floating point numbers representing a 4x4 matrix
 * into a matrix.  The first list element represents the first row, etc.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o listStr (I) - A Tcl list containing the matrix
 *   o matrixPtr (O) - The converted mattix is returned here as a 4x3 matrix.
 * Returns:
 *   TRUE if the matrix is OK,  FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
bool
TSippConvertMatrix (tSippGlobPtr, listStr, matrixPtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *listStr;
    Transf_mat   *matrixPtr;
{
    int      numRows, numColumns, row, column; 
    char   **rowArgv, **columnArgv;
    double   value, expectValue;

    /*
     * Split each row, then convert the columns in the row.
     */

    if (Tcl_SplitList (tSippGlobPtr->interp, listStr, &numRows,
                       &rowArgv) != TCL_OK)
        return FALSE;
    if (numRows != 4) {
        Tcl_AppendResult (tSippGlobPtr->interp, "matrix must have 4 rows, ",
                          "got \"", listStr, "\"", (char *) NULL);
        goto rowErrorExit;
    }
    for (row = 0; row < 4; row++) {
        if (Tcl_SplitList (tSippGlobPtr->interp, rowArgv [row], &numColumns,
                           &columnArgv) != TCL_OK)
            goto rowErrorExit;
        if (numColumns != 4) {
            Tcl_AppendResult (tSippGlobPtr->interp, 
                              "matrix must have 4 columns, got \"",
                              rowArgv [row], "\"", (char *) NULL);
            goto colErrorExit;
        }
        for (column = 0; column < 3; column++) {
            if (Tcl_GetDouble (tSippGlobPtr->interp, columnArgv [column],
                               &matrixPtr->mat [row][column]) != TCL_OK)
                goto colErrorExit;
        }
        /*
         * Validate that the value of column 3 but don't save it.
         */
        if (numColumns == 4) {
            if (Tcl_GetDouble (tSippGlobPtr->interp, columnArgv [column],
                               &value) != TCL_OK)
                goto colErrorExit;

            expectValue = (row == 3) ? 1.0 : 0.0;
            if (value != expectValue) {
                sprintf (tSippGlobPtr->interp->result,
                        "matrix row %d, column 3 must have a value of %3.1f",
                        row, expectValue);
                goto colErrorExit;
            }
        }
        sfree (columnArgv);
    }
    sfree (rowArgv);
    return TRUE;

colErrorExit:
    sfree (columnArgv);
rowErrorExit:
    sfree (rowArgv);
    return FALSE;
}

/*=============================================================================
 * TSippFormatMatrix --
 *   Format a 4x4 matrix as a list of lists.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o matrixPtr (I) - The matrix to format.
 * Returns:
 *   A dynamically alloced string contain the lists.
 *-----------------------------------------------------------------------------
 */
char *
TSippFormatMatrix (tSippGlobPtr, matrixPtr)
   tSippGlob_t  *tSippGlobPtr;
   Transf_mat   *matrixPtr;
{
    int    row; 
    char  *rowArgv [4], *bufPtr;
    char   rowBuffers [4][4 * TCL_DOUBLE_SPACE];

    rowArgv [0] = rowBuffers [0];
    rowArgv [1] = rowBuffers [1];
    rowArgv [2] = rowBuffers [2];
    rowArgv [3] = rowBuffers [3];

    /*
     * Build each row in the form "%g %g %g %g".
     */
    for (row = 0; row < 4; row++) {
        bufPtr = rowArgv [row];

        Tcl_PrintDouble (tSippGlobPtr->interp,
                         matrixPtr->mat [row][0],
                         bufPtr);
        bufPtr = bufPtr + strlen (bufPtr);
        *bufPtr++ = ' ';

        Tcl_PrintDouble (tSippGlobPtr->interp,
                         matrixPtr->mat [row][1],
                         bufPtr);
        bufPtr = bufPtr + strlen (bufPtr);
        *bufPtr++ = ' ';

        Tcl_PrintDouble (tSippGlobPtr->interp,
                         matrixPtr->mat [row][2],
                         bufPtr);
        bufPtr = bufPtr + strlen (bufPtr);
        *bufPtr++ = ' ';

        Tcl_PrintDouble (tSippGlobPtr->interp,
                         matrixCol3 [row],
                         bufPtr);
    }
    return Tcl_Merge (4, rowArgv);
}

/*=============================================================================
 * TSippHandleListConvert --
 *   Convert a list of handles into an array of generic pointers.  This assumes
 * that the handle entry consist of a single pointer.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o tableHdrPtr (I) - Pointer to the table header for the handles to
 *     convert.
 *   o listPtr (I) - The list to convert.
 *   o handleListPtr (O) - A handle list structure, which be contain the
 *     array of generic pointers.    If NULL is specified, the list is not
 *     returned. HandleListFree must be called to clean up this structure.
 *   o handleEntryListPtr (O) - A handle list structure, which be contain the
 *     array of generic pointers to the actual handle entry.  This is required
 *     if the handle entries are to be freed.  If NULL is specified, the list
 *     is not returned. HandleListFree must be called to clean up this
 *     structure.
 *   o handleArgvPtr (O) - A pointer to the argv returned from spliting the
 *     handle list is returned here.  Must be freed after use.  If NULL, then
 *     the list argv is not returned.
 * Returns:
 *   TRUE if the conversion succeeds, FALSE and an error in
 * tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
bool
TSippHandleListConvert (tSippGlobPtr, handleTblPtr, listPtr, handleListPtr,
                        handleEntryListPtr, handleListArgvPtr)
    tSippGlob_t     *tSippGlobPtr;
    void            *handleTblPtr;
    char            *listPtr;
    handleList_t    *handleListPtr;
    handleList_t    *handleEntryListPtr;
    char          ***handleListArgvPtr;
{
    int     handleArgc, idx, idx2;
    char  **handleArgv;
    void  **handleEntryPtr, **checkListPtr;

    if (Tcl_SplitList (tSippGlobPtr->interp, listPtr, &handleArgc,
                       &handleArgv) != TCL_OK)
        return FALSE;

    if (handleListPtr != NULL) {
        if (handleArgc > HANDLE_LIST_STATIC_SIZE) {
            handleListPtr->ptr =
                (void **) smalloc (handleArgc * sizeof (void *));
        } else {
            handleListPtr->ptr = handleListPtr->staticArray;
        }
        handleListPtr->len = handleArgc;
    }

    if (handleEntryListPtr != NULL) {
        if (handleArgc > HANDLE_LIST_STATIC_SIZE) {
            handleEntryListPtr->ptr = (void **) 
                smalloc (handleArgc * sizeof (void *));
        } else {
            handleEntryListPtr->ptr = handleEntryListPtr->staticArray;
        }
        handleEntryListPtr->len = handleArgc;
    }

    for (idx = 0; idx < handleArgc; idx++) {
        handleEntryPtr = (void **)
            Tcl_HandleXlate (tSippGlobPtr->interp, handleTblPtr,
                             handleArgv [idx]);
        if (handleEntryPtr == NULL)
            goto errorExit;
        if (handleListPtr != NULL)
            handleListPtr->ptr [idx] = *handleEntryPtr;
        if (handleEntryListPtr != NULL)
            handleEntryListPtr->ptr [idx] = handleEntryPtr;
    }

    /*
     * Check for duplicate entries.
     */
    checkListPtr = (handleListPtr != NULL) ? handleListPtr->ptr :
                                             handleEntryListPtr->ptr;
    for (idx = 0; idx < handleArgc - 1; idx++) {
        for (idx2 = idx + 1; idx2 < handleArgc; idx2++)
            if (checkListPtr [idx2] == checkListPtr [idx]) {
                 Tcl_AppendResult (tSippGlobPtr->interp, "duplicate handle ",
                                  "in list: ", handleArgv [idx],
                                  (char *) NULL);
                 goto errorExit;
            }
    }

    if (handleListArgvPtr != NULL)
        *handleListArgvPtr = handleArgv;
    else
        sfree (handleArgv);
    return TRUE;

errorExit:
    if (handleListPtr != NULL)
        TSippHandleListFree (handleListPtr);
    if (handleEntryListPtr != NULL)
        TSippHandleListFree (handleEntryListPtr);
    sfree (handleArgv);
    return FALSE;
}

/*=============================================================================
 * TSippHandleListFree --
 *    Free the array in an handle handle list, if it was dynamically allocated.
 *
 * Parameters:
 *   o handleListPtr (I) - The handle list structure.
 *-----------------------------------------------------------------------------
 */
void
TSippHandleListFree (handleListPtr)
    handleList_t  *handleListPtr;
{
    if (handleListPtr->ptr != handleListPtr->staticArray)
        sfree (handleListPtr->ptr);
}

/*=============================================================================
 * TSippInitCmds --
 *   Given a command table, initialize the Tcl commands listed in it, with
 * the Tcl SIPP globals as client data.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o cmdTablePtr (I) - A table of commands and there executors.  Terminated
 *     by a entry of NULLs.
 *-----------------------------------------------------------------------------
 */
void
TSippInitCmds (tSippGlobPtr, cmdTablePtr)
    tSippGlob_t       *tSippGlobPtr;
    tSippTclCmdTbl_t  *cmdTablePtr;
{
    while (cmdTablePtr->name != NULL) {
        Tcl_CreateCommand (tSippGlobPtr->interp, 
                          cmdTablePtr->name, cmdTablePtr->proc,
                          (ClientData) tSippGlobPtr, (void (*)())NULL);
        cmdTablePtr++;
    }
}

/*=============================================================================
 * TSippParseTextureMapping --
 *   Utility procedure to parse thetexture mapping parameter supplied to
 * the object-creation primitive commands.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *     Errors are returned in interp->result.
 *   o textureStr (I) - The texture mapping string to parse.
 *   o texturePtr (O) - The parsed texture mapping is returned here.
 *   o invalidList (I) - A list  of texture mappings that are not valid for the
 *     command, Terminated by an entry containing -1.  NULL is all are valid.
 * Returns:
 *   TRUE if all is ok, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
bool
TSippParseTextureMapping (tSippGlobPtr, textureStr, texturePtr, invalidList)
    tSippGlob_t    *tSippGlobPtr;
    char           *textureStr;
    int            *texturePtr;
    int            *invalidList;
{
    int texture;

    if ((textureStr [0] == '\0') || (STREQU (textureStr, "NATURAL"))) {
        texture = NATURAL;
    } else
    if (STREQU (textureStr, "CYLINDRICAL")) {
        texture = CYLINDRICAL;
    } else
    if (STREQU (textureStr, "SPHERICAL")) {
        texture = SPHERICAL;
    } else
    if (STREQU (textureStr, "WORLD")) {
        texture = WORLD;
    } else {
        Tcl_AppendResult (tSippGlobPtr->interp, "expected one of ",
                          "\"NATURAL\", \"CYLINDRICAL\", \"SPHERICAL\", or ",
                          "\"WORLD\", got \"", textureStr, "\"",
                          (char *) NULL);
        return FALSE;
    }

    if (invalidList != NULL) {
        int idx;

        for (idx = 0; invalidList [idx] != -1; idx++) {
            if (texture == invalidList [idx]) {
                Tcl_AppendResult (tSippGlobPtr->interp, "texture mapping ",
                                  textureStr, " is not valid for this command",
                                  (char *) NULL);
                return FALSE;
            }
        }
    }

    *texturePtr = texture;
    return TRUE;
}

/*=============================================================================
 * TSippNotWhileRendering --
 *   Return an error message as a command result when a command to modify SIPP
 * is used during an event update while rendering is taking place.
 *
 * Parameters:
 *   o interp (I) - Error message is set in interp->result.
 * Returns:
 *   Always returns TCL_ERROR so this function maybe used in a return
 * statement.
 *-----------------------------------------------------------------------------
 */
int
TSippNotWhileRendering (interp)
    Tcl_Interp *interp;
{
    interp->result = "SIPP state may not be modified while rendering";
    return TCL_ERROR;
}

/*=============================================================================
 * TSippAtEOF --
 *   Return an error message in the interp that a TSIPP managed file is at
 * EOF.
 *
 * Parameters:
 *   o interp (I) - A pointer to the Tcl interpreter.
 *   o handle (I) - The handle to include in the message.
 * Returns:
 *   TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
int
TSippAtEOF (interp, handle)
    Tcl_Interp *interp;
    char       *handle;
{
    Tcl_AppendResult (interp, handle, " is at the end of file",
                      (char *) NULL);
    return TCL_ERROR;
}

/*=============================================================================
 * TSippNotReadable --
 *   Return an error message in the interp that a TSIPP managed file is not
 * open for read access.
 *
 * Parameters:
 *   o interp (I) - A pointer to the Tcl interpreter.
 *   o handle (I) - The handle to include in the message.
 * Returns:
 *   TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
int
TSippNotReadable (interp, handle)
    Tcl_Interp *interp;
    char       *handle;
{
    Tcl_AppendResult (interp, handle, " is not open for read access",
                      (char *) NULL);
    return TCL_ERROR;
}

/*=============================================================================
 * TSippNotWritable --
 *   Return an error message in the interp that a TSIPP managed file is not
 * open for write access.
 *
 * Parameters:
 *   o interp (I) - A pointer to the Tcl interpreter.
 *   o handle (I) - The handle to include in the message.
 * Returns:
 *   TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
int
TSippNotWritable (interp, handle)
    Tcl_Interp *interp;
    char       *handle;
{
    Tcl_AppendResult (interp, handle, " is not open for write access",
                      (char *) NULL);
    return TCL_ERROR;
}

/*=============================================================================
 * TSippAddHist --
 *    Append a history string to a HISTORY comment (compatible with rle_addhist
 * but does not lose memory.
 *
 * Parameters:
 *   o commentsPtr (I/O) - Pointer to comment array.
 *   o argv (I) - Command line history to add to comments.
 *-----------------------------------------------------------------------------
 */
void
TSippAddHist (commentsPtr, argv)
    char   ***commentsPtr;
    char    **argv;
{
    time_t	 dateTime;
    char        *dateTimeStr, *oldHistory, *newHistory;
    int          length, idx;

    time (&dateTime);
    dateTimeStr = ctime (&dateTime); /* Includes a new-line */

    oldHistory = TSippCommentGet (*commentsPtr,
                                  "HISTORY");

    /*
     * Determine size of the buffer for the new comment.  It is in the form:
     *   "old-history\n\tcommand on date-time\n\t"
     */

    length = 0;
    if (oldHistory != NULL)
        length += strlen (oldHistory);

    for (idx = 0; argv [idx] != NULL; idx++)
        length += strlen (argv [idx]) + 1;  /* arg plus space */

    length += 3 + strlen (dateTimeStr) + 3; /* on date \n\t\0 */

    /*
     * Build up the comment.
     */
    newHistory = (char *) alloca (length);
    newHistory [0] = '\0';

    if (oldHistory != NULL)
        strcat (newHistory, oldHistory);
    for (idx = 0; argv [idx] != NULL; idx++) {
        strcat (newHistory, argv [idx]);
        strcat (newHistory, " ");
    }
    strcat (newHistory, "on ");
    strcat (newHistory, dateTimeStr);
    strcat (newHistory, "\t");

    TSippCommentPut (commentsPtr,
                     "HISTORY",
                     newHistory);
}

/*=============================================================================
 * TSippCallCmd --
 *    Call a Tcl command from C.  This is a quicker than calling Tcl_Eval.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o args (I) - Command and a NULL terminate variable number of arguments to
 *     the command.
 * Returns:
 *   The result code from calling the command.
 *-----------------------------------------------------------------------------
 */
int
TSippCallCmd (tSippGlobPtr, va_alist)
    tSippGlob_t  *tSippGlobPtr;
    va_dcl
{
    va_list       args;
    int           idx, argc;
    char        **argv;
    Tcl_CmdInfo   cmdInfo;

    /*
     * Count the arguments.
     */
    va_start (args);
    for (argc = 0; (va_arg (args, char *) != NULL); argc++)
        continue;
    va_end (args);

    /*
     * Build up the argument vector.
     */
    argv = (char **) alloca ((argc + 1) * (sizeof (char *)));

    va_start (args);
    for (idx = 0; idx < argc; idx++) {
        argv [idx] = va_arg (args, char *);
    }
    va_end (args);
    
    /*
     * Find the Tcl command to execute.
     */
    if (!Tcl_GetCommandInfo (tSippGlobPtr->interp, argv [0], &cmdInfo)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "command \"", argv [0],
                          "\" not found, called internally by tsipp.",
                          (char *) NULL);
        return TCL_ERROR;
    }

    /*
     * Call the command via the saved pointer.
     */
    return cmdInfo.proc (cmdInfo.clientData,
                         tSippGlobPtr->interp,
                         argc,
                         argv);
}
